package org.herac.tuxguitar.io.exporter;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Iterator;
import java.util.List;

import org.eclipse.swt.graphics.Point;
import org.herac.tuxguitar.io.UnsupportedFormatException;
import org.herac.tuxguitar.io.gp.GPFormatException;
import org.herac.tuxguitar.song.managers.MeasureManager;
import org.herac.tuxguitar.song.managers.SongManager;
import org.herac.tuxguitar.song.models.Component;
import org.herac.tuxguitar.song.models.Duration;
import org.herac.tuxguitar.song.models.InstrumentString;
import org.herac.tuxguitar.song.models.Measure;
import org.herac.tuxguitar.song.models.Note;
import org.herac.tuxguitar.song.models.Song;
import org.herac.tuxguitar.song.models.SongTrack;

public class ASCIITabOutputStream {
	
	private static final String[] TONIC_NAMES = new String[]{"C","C#","D","D#","E","F","F#","G","G#","A","A#","B"}; 
	
	private static final int MAX_LINE_LENGTH = 80;
	
	private SongManager manager;
	private PrintStream stream;
	private ASCIIOutputStream out;
	
	
	public ASCIITabOutputStream(PrintStream stream){
		this.stream = stream;
	}

	public ASCIITabOutputStream(String fileName) throws FileNotFoundException{
		this(new PrintStream(fileName));
	}
	
	public void writeSong(Song song){
		this.manager = new SongManager();
		this.manager.setSong(song);
		out = new ASCIIOutputStream(this.stream);
		
		drawSong();
		
		out.flush();
	}
	
	private void drawSong(){
		Song song = manager.getSong();	
		
		//Propiedades de cancion
		out.drawStringLine("Title: " + song.getName());
		out.drawStringLine("ALbum: " + song.getAlbum());
		out.drawStringLine("Author: " + song.getAuthor());
		out.drawStringLine("Interpret: " + song.getInterpret());
		
		Iterator it = song.getTracks().iterator();
		while(it.hasNext()){
			SongTrack track = (SongTrack)it.next();
			
			out.nextLine();
			drawTrack(track);
			out.nextLine();		
			
		}				
	}
	
	private void drawTrack(SongTrack track){
		//Propiedades de pista
		out.nextLine();
		out.drawStringLine("Track " + track.getNumber() + ": " + track.getName());
		
		//Obtengo los nombres de la afinacion, y el ancho maximo que ocupa
		String[] tunning = new String[track.getStrings().size()];
		int maxTunningLength = 1;		
		for(int i = 0; i < track.getStrings().size();i++){
			InstrumentString string = (InstrumentString)track.getStrings().get(i);
			tunning[i] = TONIC_NAMES[(string.getValue() % TONIC_NAMES.length)];
			maxTunningLength = Math.max(maxTunningLength,tunning[i].length());
		}
		
		int nextMeasure = 0;		
		boolean eof = false;
		while(!eof){			
			out.nextLine();
			int index = nextMeasure;
			for(int i = 0; i < track.getStrings().size();i++){
				InstrumentString string = (InstrumentString)track.getStrings().get(i);
			
				//Dibujo la afinacion de la cuerda
				out.drawTuneSegment(tunning[i],maxTunningLength);
				
				for(int j = index; j < track.getMeasures().size(); j++){
					Measure measure = (Measure)track.getMeasures().get(j);				
					drawMeasure(measure,string);
					nextMeasure = (j + 1);
					
					//Calculo si era el ultimo compas
					eof = (manager.getTrackManager().isLastMeasure(track,measure));										
					
					//Si se supero el ancho maximo, bajo de linea	
					Point position = out.getPosition();
					if(position.x > MAX_LINE_LENGTH){						
						break;
					}
				}
				//Cierro los compases
				out.drawBarSegment();
				out.nextLine();
			}
			out.nextLine();
		}
		out.nextLine();
	}
	
	private void drawMeasure(Measure measure,InstrumentString string){
		MeasureManager measureManager = manager.getMeasureManager(); 
		List components = measureManager.getComponents(measure);
		measureManager.orderComponents(components);
		
		//Abro el compas
		out.drawBarSegment();
		out.drawStringSegments(1);
		Component component = next(components,measure.getStart(),string.getNumber());
		while(component != null){
			int outLength = 0;
			//Si es nota y en la misma cuerda, la dibujo
			if(component instanceof Note && ((Note)component).getString() == string.getNumber()){
				Note note = (Note)component;
				outLength = (Integer.toString(note.getValue()).length() - 1);
				out.drawNote(note.getValue());
			}
			//dejo el espacio
			else{
				out.drawStringSegments(1);	
			}
			
			//Agrego espacios correspondientes hasta el proximo pulso.
			out.drawStringSegments(getDurationScaping(component.getDuration()) - outLength);
			
			component = next(components,component.getStart() + component.getDuration().getTime(),string.getNumber());
		}
		
	}
	
	private Component next(List components,long start,int string){
		Component next = null;
		for(int i = 0;i < components.size();i++){
			Component component = (Component)components.get(i);
			
			if(component.getStart() >= start && (next == null || component.getStart() < next.getStart())){
				next = component;
			}else if(component.getStart() >= start && (next == null || component.getStart() == next.getStart())){
				if(component instanceof Note && ((Note)component).getString() == string){
					next = component;					
				}
			}			
		}
		return next;
	}
	
	private int getDurationScaping(Duration duration){
		int spacing = 1;

		if(duration.getValue() >= Duration.SIXTEENTH){
			spacing = 2;
		}
		else if(duration.getValue() >= Duration.EIGHTH){
			spacing = 3;
		}
		else if(duration.getValue() >= Duration.QUARTER){
			spacing = 4;
		}
		else if(duration.getValue() >= Duration.HALF){
			spacing = 5;
		}		
		else if(duration.getValue() >= Duration.WHOLE){
			spacing = 6;
		}
		
		return spacing;
	}
	
	public static void main(String[] s){
		SongManager manager = new SongManager();		
		try {
			manager.open("/home/julian/test.gp4");
			manager.autoCompleteSilences();
			
			ASCIITabOutputStream out = new ASCIITabOutputStream(System.out);
			
			out.writeSong(manager.getSong());
		} catch (GPFormatException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (UnsupportedFormatException e) {
			e.printStackTrace();
		}
		System.exit(0);
	}
}
