package org.herac.tuxguitar.gui.plugins;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import org.herac.tuxguitar.gui.TuxGuitar;
import org.herac.tuxguitar.gui.util.TuxGuitarFileUtils;

public class PluginManager {

	public static final String PROPERTY_PLUGIN_PATH = "plugin.path";
	
	public static final String PROPERTY_PLUGIN_CLASS = "plugin.class";
	
	private static final String LANGUAGE_FOLDER = "lang";

	private List plugins;

	private List pluginItems;
	
	public PluginManager(){
		this.plugins = new ArrayList();
		this.pluginItems = new ArrayList();
		this.loadPlugins();
	}
    
	public List getPlugins(){
		return this.plugins;
	}
	
	public List getPluginItems(){
		return this.pluginItems;
	}
	
    /**
     * Load language files from lang folder
     *
     */
    private void loadPlugins(){
    	// we need plugin_*.properties files only
        final FilenameFilter filter = new FilenameFilter() {
            public boolean accept(File dir, String name) {
                return (name.indexOf("plugin_") != -1 && name.indexOf(".properties") != -1);
            }
        };          
        // get the files
        String prefix = TuxGuitarFileUtils.PLUGINS_PREFIX;
        if(prefix != null){        	
        	File path = new File(prefix);
        	if(path != null){
        		String[] files = path.list(filter);
        
        		// now iterate over them
        		for(int i = 0;i < files.length;i++){        	
        			try {
						loadPLugin(prefix + File.separator + files[i]);
					} catch (FileNotFoundException e) {
						e.printStackTrace();
					} catch (IOException e) {
						e.printStackTrace();
					} catch (InstantiationException e) {
						e.printStackTrace();
					} catch (IllegalAccessException e) {
						e.printStackTrace();
					} catch (ClassNotFoundException e) {
						e.printStackTrace();
					}        			        	
        		}
        	}
        }
    }    
    
    private void loadPLugin(String pluginProperties) throws FileNotFoundException, IOException, InstantiationException, IllegalAccessException, ClassNotFoundException{    	
    	try{
    	Properties properties = new Properties();    		
		properties.load(new FileInputStream(pluginProperties));
			
		String pluginPath = properties.getProperty(PROPERTY_PLUGIN_PATH,new String());
		String pluginClass = properties.getProperty(PROPERTY_PLUGIN_CLASS,new String());
			
		File file = new File(pluginPath);
		
		if(file != null && file.exists()){		
			ClassLoader loader = null;
								
			if(file.isDirectory()){
				loader = new URLClassLoader(new URL[]{ new File(pluginPath).toURL() },getClass().getClassLoader());
			}else if(file.getName().endsWith(".jar")){			
				loader = new JarClassLoader(new JarFile(pluginPath));									
			}			
			
			if(loader != null){
				Class loadedClass = loader.loadClass(pluginClass);
				if(loadedClass != null){
					Object object = loadedClass.newInstance();
					if(object instanceof TGPlugin){		
						TGPlugin plugin = (TGPlugin)object;
						
						plugins.add(plugin);
						if(plugin.getItem().isEnabled()){
							pluginItems.add(plugin);
						}
					}
				}
			}
		}
    	}catch(Exception e){    		
    		TuxGuitar.instance().showErrorMessage(e);
    	}catch(Error e){    		
    		TuxGuitar.instance().showErrorMessage(e);
    	}
    }
    
    
    public void initPLugins(){
    	Iterator it = plugins.iterator();
    	while(it.hasNext()){
    		TGPlugin plugin = (TGPlugin)it.next();
    		plugin.init(TuxGuitar.instance());
    	}
    }
    
    public void closePLugins(){
    	Iterator it = plugins.iterator();
    	while(it.hasNext()){
    		TGPlugin plugin = (TGPlugin)it.next();
    		plugin.close();
    	}
    }
    
    private void showErrorMessage(){
    	
    }
}


class JarClassLoader extends ClassLoader{

	private Map loadedClasses;
	
	public JarClassLoader(JarFile jarFile){
		this.loadedClasses = new HashMap();
		this.loadJarFile(jarFile);		
	}
	
	public void loadJarFile(JarFile jarFile){
		Enumeration enumeration = jarFile.entries();
		while (enumeration.hasMoreElements()) {
			JarEntry jarEntry = (JarEntry) enumeration.nextElement();
			if(jarEntry.getName().endsWith(".class")){
				defineClass( jarFile, jarEntry) ;
			}
		}
	}
	
	public Class defineClass(JarFile jarFile,JarEntry jarEntry) {
		Class loadedClass = null;
		try {
			InputStream is = jarFile.getInputStream(jarEntry);
			int available = is.available();
			byte data[] = new byte[available];
			is.read(data);
				
			loadedClass = defineClass(null,data, 0, data.length);
				
			loadedClasses.put(loadedClass.getName(),loadedClass);													
		}catch(IOException e) {
			e.printStackTrace();
		}catch(Error e) {
			System.out.println(jarEntry.getName());
		}
		return loadedClass;
	}
	
	public Class loadClass(final String className) {
		try {
			Class loadedclass = new ClassLoader() {
				public Class findClass(String name) {
					return (Class)loadedClasses.get(name);
				}
			}.loadClass(className);
			return loadedclass;
		}catch(Exception e) {
			e.printStackTrace();		
		}
		return null;
	}
	

} 