package org.herac.tuxguitar.gui.marker;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;
import org.herac.tuxguitar.gui.TuxGuitar;
import org.herac.tuxguitar.gui.undo.undoables.JoinedUndoable;
import org.herac.tuxguitar.gui.undo.undoables.custom.UndoableChangeMarker;
import org.herac.tuxguitar.song.managers.SongManager;
import org.herac.tuxguitar.song.models.Marker;
import org.herac.tuxguitar.song.models.RGBColor;

public class MarkerEditor {
	public static final int STATUS_NEW = 1;
	public static final int STATUS_EDIT = 2;
	
	private int status;
	private Marker marker;
	private Shell dialog;
	private Spinner measureSpinner;
	private Text titleText;
	private Button colorButton;

	public MarkerEditor(Marker marker) {
		this(marker,STATUS_NEW);
	}	
	
	public MarkerEditor(Marker marker,int status) {
		this.marker = (Marker)marker.clone();
		this.status = status;
	}		

	public void show(Shell shell) {
		dialog = new Shell(shell, SWT.DIALOG_TRIM | SWT.APPLICATION_MODAL);
		dialog.setLayout(new GridLayout());
		dialog.setText(TuxGuitar.getProperty("marker.editor"));

		// ----------------------------------------------------------------------
		Composite marker = new Composite(dialog, SWT.NONE);
		marker.setLayout(new GridLayout(4, false));
		marker.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		// Measure Number
		final int measureCount = TuxGuitar.instance().getSongManager()
				.countMeasures();
		Label measureLabel = new Label(marker, SWT.NULL);
		measureLabel.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, true,true));
		measureLabel.setText(TuxGuitar.getProperty("measure"));
		GridData spinnerData = new GridData();
		spinnerData.widthHint = 26;
		spinnerData.verticalAlignment = SWT.CENTER;
		spinnerData.grabExcessVerticalSpace = true;
		this.measureSpinner = new Spinner(marker, SWT.BORDER);
		this.measureSpinner.setLayoutData(spinnerData);
		this.measureSpinner.setMinimum(1);
		this.measureSpinner.setMaximum(measureCount);
		this.measureSpinner.setSelection(this.marker.getMeasure());
		this.measureSpinner.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				int selection = measureSpinner.getSelection();
				if (selection < 1) {
					measureSpinner.setSelection(1);
				} else if (selection > measureCount) {
					measureSpinner.setSelection(measureCount);
				}
			}
		});

		// Color
		Label colorLabel = new Label(marker, SWT.NULL);
		colorLabel.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, true, true));
		colorLabel.setText(TuxGuitar.getProperty("color"));
		this.colorButton = new Button(marker, SWT.PUSH);
		this.setButtonColor(colorButton, new Color(dialog.getDisplay(),this.marker.getColor().getR(), this.marker.getColor().getG(),this.marker.getColor().getB()));

		colorButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				ColorDialog dlg = new ColorDialog(dialog);

				dlg.setRGB(dialog.getDisplay().getSystemColor(SWT.COLOR_BLACK).getRGB());
				dlg.setText(TuxGuitar.getProperty("choose-color"));
				RGB rgb = dlg.open();
				if (rgb != null) {
					setButtonColor(colorButton, new Color(dialog.getDisplay(),rgb));
				}
			}
		});

		// Title
		Label titleLabel = new Label(marker, SWT.NULL);
		titleLabel.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, true, true));
		titleLabel.setText(TuxGuitar.getProperty("title"));
		this.titleText = new Text(marker, SWT.BORDER);
		this.titleText.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true,true, 3, 1));
		this.titleText.setText(this.marker.getTitle());

		// ------------------BUTTONS--------------------------
		Composite buttons = new Composite(dialog, SWT.NONE);
		buttons.setLayout(new GridLayout(2, false));
		buttons.setLayoutData(new GridData(SWT.END, SWT.FILL, true, true));

		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true);
		data.minimumWidth = 80;
		data.minimumHeight = 25;

		final Button buttonOK = new Button(buttons, SWT.PUSH);
		buttonOK.setText(TuxGuitar.getProperty("ok"));
		buttonOK.setLayoutData(data);
		buttonOK.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent arg0) {
				updateMarker();
				dialog.dispose();
			}
		});

		Button buttonCancel = new Button(buttons, SWT.PUSH);
		buttonCancel.setText(TuxGuitar.getProperty("cancel"));
		buttonCancel.setLayoutData(data);
		buttonCancel.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent arg0) {
				dialog.dispose();
			}
		});

		dialog.pack();
		dialog.open();

		int x = shell.getBounds().x + (shell.getBounds().width - dialog.getSize().x) / 2;
		int y = shell.getBounds().y + (shell.getBounds().height - dialog.getSize().y) / 2;
		dialog.setLocation(x, y);

		while (!dialog.isDisposed()) {
			if (!dialog.getDisplay().readAndDispatch()) {
				dialog.getDisplay().sleep();
			}

		}

	}

	private void setButtonColor(Button button, Color color) {
		Image image = new Image(dialog.getDisplay(), 25, 25);
		GC gc = new GC(image);
		gc.setBackground(color);
		gc.fillRectangle(0, 0, 25, 25);
		gc.dispose();
		button.setImage(image);
		this.marker.setColor(new RGBColor(color.getRed(), color.getGreen(),color.getBlue()));
	}

	private void updateMarker() {	
		int oldMeasure = this.marker.getMeasure();
		this.marker.setMeasure(this.measureSpinner.getSelection());
		this.marker.setTitle(this.titleText.getText());
		this.marker = (Marker)this.marker.clone();
		
		SongManager manager = TuxGuitar.instance().getSongManager();
		
		//comienza el undoable
		JoinedUndoable joinedUndoable = new JoinedUndoable();
		
		if(this.status == STATUS_EDIT && oldMeasure != this.marker.getMeasure()){
			UndoableChangeMarker undoable = UndoableChangeMarker.startUndo(manager.getMarker(oldMeasure));			
			TuxGuitar.instance().getSongManager().removeMarker(oldMeasure);			
			joinedUndoable.addUndoableEdit(undoable.endUndo(null));
		}
		UndoableChangeMarker undoable = UndoableChangeMarker.startUndo(manager.getMarker(this.marker.getMeasure()));		
		TuxGuitar.instance().getSongManager().updateMarker(this.marker);
		joinedUndoable.addUndoableEdit(undoable.endUndo(this.marker));

		// termia el undoable
		TuxGuitar.instance().getTablatureEditor().getUndoManager().addEdit(joinedUndoable.endUndo());
	}
}
