package org.herac.tuxguitar.play.models.tg_player;

import java.io.OutputStream;
import java.util.List;

import javax.sound.midi.MidiSystem;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Receiver;
import javax.sound.midi.Soundbank;

import org.herac.tuxguitar.play.models.Player;
import org.herac.tuxguitar.song.managers.SongManager;
import org.herac.tuxguitar.song.models.Duration;
import org.herac.tuxguitar.song.models.Song;
import org.herac.tuxguitar.song.models.SongChannel;
import org.herac.tuxguitar.song.models.SongTrack;


public final class SongPlayer implements Player{
	private SongManager manager;
	private Receiver receiver;
	private ChannelControl channelControl;
	private NotePlayer notePlayer;
	private BeatIterator beatIterator;
	private boolean running;
    private boolean paused;
    
	public SongPlayer(SongManager manager) {
		this.manager = manager;
	}
	
	public void loadTestReceiver() {
		this.receiver = new ReceiverImpl();		
	}		
	
	public void loadDefaultReceiver() {
		try {			
			this.receiver = MidiSystem.getReceiver();			
		} catch (MidiUnavailableException e) {
			e.printStackTrace();
		}		
	}	
	
	public void loadImplReceiver() {
		this.receiver = new ReceiverImpl();		
	}		


	private synchronized void playNotes(NotePlayer notePlayer){			
		notePlayer.checkNotes();
		sleep(10);													
	}

	public static void sleep(long time){				
		/*
		try {			
			Thread.sleep(time);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		*/
		long init = System.currentTimeMillis();
		while(System.currentTimeMillis() < (init + time)){
			try {			
				Thread.sleep(time / 3);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
	}
	
	public static long getTime(int tempo,long time){		
		long millis = (1000 * time / Duration.QUARTER_TIME);
		return (long)((double)60.00 / (double)tempo * (double)millis);
	}	
	

	public void init() {
		loadDefaultReceiver();
		//this.receiver = new ReceiverImpl();
		this.channelControl = new ChannelControl(this.receiver);
		this.notePlayer = new NotePlayer(this.receiver);
		this.beatIterator = new BeatIterator(notePlayer);			
	}

	public void close() {
		if(this.receiver != null){
			this.receiver.close();
		}
	}

	public void play() {
		new Thread(new Runnable() {				
			public void run() {				
				play(manager.getSong());
			}				
		}).start();	
	}

	
	private void play(final Song song){	
		setRunning(true);
							
		channelControl.update(song);
		beatIterator.nextBeat(song);
		long nextTickPosition = beatIterator.getNextTickPosition();
		long beatDuration = beatIterator.getBeatDuration();
		long startedAt = System.currentTimeMillis();	
		while(nextTickPosition > 0 && isRunning()){		
			while(System.currentTimeMillis() < (startedAt + beatDuration) && isRunning()){			
				playNotes(notePlayer);
			}			
						
			beatIterator.nextBeat(song);
			nextTickPosition = beatIterator.getNextTickPosition();
			
			beatDuration = beatIterator.getBeatDuration();
			startedAt = System.currentTimeMillis();	
		}		
		if(isRunning()){
			while(System.currentTimeMillis() < (startedAt + beatDuration)){			
				playNotes(notePlayer);								
			}				
		}
		notePlayer.stopAll();
		setRunning(false);
	}	
	
	public void stop() {
		setRunning(false);
	}
	
    public void pause(){
    	this.stop();
    	this.setPaused(true);
    }	

	public void reset() {
		close();
		init();
	}

	public boolean isRunning() {
		return this.running;
	}

	public void setRunning(boolean running) {
		this.running = running;
	}

	public boolean isRealTimeSequencer() {
		return true;
	}

	public void playBeat(SongTrack track, List notes) {		
	}

	public void updateChannels() {		
	}

	public void updateChannel(SongChannel channel) {		
	}

	public void setTickPosition(long position) {
		beatIterator.setTickPosition(position);
	}

	public void setTickPosition(long position,long startMove) {
		setTickPosition(position);
	}	
	
	public long getTickPosition() {
		return this.beatIterator.getTickPosition();
	}

	
	
	public boolean isPaused() {
		return paused;
	}

	public void setPaused(boolean paused) {
		this.paused = paused;
	}

	public Soundbank getDefaultSoundbank() {
		return null;
	}

	public void write(OutputStream out) {
		
	}	
}
