package org.herac.tuxguitar.io.gp;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.herac.tuxguitar.song.models.BendEffect;
import org.herac.tuxguitar.song.models.Duration;
import org.herac.tuxguitar.song.models.InstrumentString;
import org.herac.tuxguitar.song.models.Measure;
import org.herac.tuxguitar.song.models.Note;
import org.herac.tuxguitar.song.models.NoteEffect;
import org.herac.tuxguitar.song.models.Song;
import org.herac.tuxguitar.song.models.SongChannel;
import org.herac.tuxguitar.song.models.SongTrack;
import org.herac.tuxguitar.song.models.Tempo;
import org.herac.tuxguitar.song.models.TimeSignature;
import org.herac.tuxguitar.song.models.TrackColor;

public class GP4InputStream extends InputStream {
    private static final String supportedVersions[] = { "FICHIER GUITAR PRO v4.00", "FICHIER GUITAR PRO v4.06", "FICHIER GUITAR PRO L4.06" };
    private String version;
    private InputStream inputStream;
    private int offset;

    public GP4InputStream(InputStream inputStream) {
        super();
        this.inputStream = inputStream;
        this.offset = 0;
    }

    public GP4InputStream(String fileName) throws FileNotFoundException {
        this(new FileInputStream(new File(fileName)));
    }

    public void close() throws IOException {
        this.inputStream.close();
    }

    public int read() throws IOException {
        this.offset++;
        return this.inputStream.read();
    }

    private void readVersion(){
		try {
			if(this.version == null){
				this.version = readStringByte(30);
			}
		} catch (IOException e) {
			this.version = "NOT_SUPPORTED";
		}    	
    }    
    
    public static boolean isSupportedVersion(String version) {
        for (int i = 0; i < supportedVersions.length; i++) {
            if (version.equals(supportedVersions[i])) {
                return true;
            }
        }
        return false;
    }

    public boolean isSupportedVersion(){    
    	try{
    		readVersion();
    		return isSupportedVersion(version);
    	}catch(Exception e){
    		return false;
    	}catch(Error e){
    		return false;
    	}       		
    }    
    
    public Song readSong() throws IOException, GPFormatException {
    	readVersion();
        if (!isSupportedVersion(version)) {
            throw new GPFormatException("Unsuported Version");
        }

        String title = readStringIntegerPlusOne();

        String subtitle = readStringIntegerPlusOne();

        String interpret = readStringIntegerPlusOne();

        String album = readStringIntegerPlusOne();

        String songAuthor = readStringIntegerPlusOne();

        String copyright = readStringIntegerPlusOne();

        String pieceAuthor = readStringIntegerPlusOne();

        String instructions = readStringIntegerPlusOne();

        int nbNotes = readInt();
        String note = "";
        for (int i = 0; i < nbNotes; i++) {
            note += readStringIntegerPlusOne();
            note += "\n";
        }

        boolean tripletFeel = readBoolean();

        int trackNumber = readInt();
        for (int i = 0; i < 5; i++) {
            int measureNumber = readInt();
            String lines = readStringInteger();
        }

        int tempoValue = readInt();

        int key = readByte();

        int octave = readInt();

        List channels = new ArrayList();
        int[] instruments = new int[64];
        for (int i = 0; i < 64; i++) {
        	channels.add(new SongChannel((short)i,
    					(short)i, 
    					(short)readInt(),
    					toChannelShort(readByte()),
    					toChannelShort(readByte()),
    					toChannelShort(readByte()),
    					toChannelShort(readByte()),
    					toChannelShort(readByte()),
    					toChannelShort(readByte()),
    					false,
    					false));       
            byte[] b = { 0, 0 };
            read(b);
        }

        TimeSignature timeSignature = new TimeSignature(4, new Duration(4));

        int numberOfMeasures = readInt();
        int numberOfTracks = readInt();

        List measures = new ArrayList();
        if (numberOfMeasures > 0) {
            for (int i = 0; i < numberOfMeasures; i++) {
                Measure measure = createMeasure((i + 1),timeSignature);
                measures.add(measure);
            }
        }

        List tracks = new ArrayList();
        for (int i = 0; i < numberOfTracks; i++) {
            tracks.add(createTrack((i + 1), channels));
        }

        long nextMeasureStart = 1000;
        for (int i = 0; i < numberOfMeasures; i++) {
            Measure currMeasure = (Measure) measures.get(i);
            Measure lastMeasure = null;
            currMeasure.setStart(nextMeasureStart);
            for (int j = 0; j < numberOfTracks; j++) {
                SongTrack track = (SongTrack) tracks.get(j);
                Measure measure = (Measure) currMeasure.clone();

                Tempo tempo = new Tempo(tempoValue);
                addMeasureComponents(track.getStrings(), measure, track.getMeasures(), tempo);
                
                //checkeo que el tempo coincida
                if(lastMeasure != null && lastMeasure.getTempo().getValue() != tempo.getValue()){
                	measure.setTempo((Tempo)lastMeasure.getTempo().clone());      
                }else{
                	measure.setTempo(tempo);       
                }
                lastMeasure = measure;                                       
                
                track.getMeasures().add(measure);
            }
            nextMeasureStart += currMeasure.getLength();
        }

        return new Song(title,interpret,album,songAuthor,tracks,Song.MAX_VOLUME);
    }

    private long addNotes(long start, List notes, List trackStrings, List currTrackMeasures, Tempo tempo) throws IOException,
            GPFormatException {
        NoteEffect effect = new NoteEffect();
        
        int header = readUnsignedByte();

        if ((header & 0x40) != 0) {
            int beatStatus = readUnsignedByte();
            boolean emptyBeat = (beatStatus == 0x00);
            boolean restBeat = (beatStatus == 0x02);
        }

        boolean dottedNotes = ((header & 0x01) != 0);

        Duration duration = parseDuration(readByte());
        duration.setDotted(dottedNotes);

        if ((header & 0x20) != 0) {
            int tuplet = readInt();
            //-------Verifico el tupleto--------------------
            switch (tuplet) {
            case 3:
                duration.getTupleto().setEnters(3);
                duration.getTupleto().setTimes(2);
                break;
            case 5:
                duration.getTupleto().setEnters(5);
                duration.getTupleto().setTimes(4);
                break;                
            case 6:
                duration.getTupleto().setEnters(6);
                duration.getTupleto().setTimes(4);
                break;
            case 7:
                duration.getTupleto().setEnters(7);
                duration.getTupleto().setTimes(4);
                break;
            case 9:
                duration.getTupleto().setEnters(9);
                duration.getTupleto().setTimes(8);
                break;
            case 10:
                duration.getTupleto().setEnters(10);
                duration.getTupleto().setTimes(8);
                break;
            case 11:
                duration.getTupleto().setEnters(11);
                duration.getTupleto().setTimes(8);
                break;
            case 12:
                duration.getTupleto().setEnters(12);
                duration.getTupleto().setTimes(8);
                break;                  
            }
        }

        if ((header & 0x02) != 0) {
            readChordDiagram();
        }

        if ((header & 0x04) != 0) {
            String text = readStringIntegerPlusOne();
        }

        if ((header & 0x08) != 0) {
            readBeatEffects((NoteEffect)effect.clone(),duration);
        }

        if ((header & 0x10) != 0) {
            readMixChange(tempo);
        }

        int stringsPlayed = readUnsignedByte();
        List strings = getPlayedStrings(stringsPlayed, trackStrings);

        for (int i = strings.size() - 1; i >= 0; i--) {
            InstrumentString string = (InstrumentString) strings.get(i);
            Note note = parseNote(start, string, duration, notes, currTrackMeasures,(NoteEffect)effect.clone());
            if (note != null) {
                notes.add(note);
            }
        }

        return duration.getTime();
    }

    private List getPlayedStrings(int stringsPlayed, List trackStrings) {
        List strings = new ArrayList();
        if ((stringsPlayed & (1 << 0)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(6)).clone());
        }
        if ((stringsPlayed & (1 << 1)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(5)).clone());
        }
        if ((stringsPlayed & (1 << 2)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(4)).clone());
        }
        if ((stringsPlayed & (1 << 3)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(3)).clone());
        }
        if ((stringsPlayed & (1 << 4)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(2)).clone());
        }
        if ((stringsPlayed & (1 << 5)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(1)).clone());
        }
        if ((stringsPlayed & (1 << 6)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(0)).clone());
        }
        return strings;
    }

    private Duration parseDuration(byte value) {
        Duration duration = null;
        switch (value) {
        case -2:
            duration = new Duration(Duration.WHOLE);
            break;
        case -1:
            duration = new Duration(Duration.HALF);
            break;
        case 0:
            duration = new Duration(Duration.QUARTER);
            break;
        case 1:
            duration = new Duration(Duration.EIGHTH);
            break;
        case 2:
            duration = new Duration(Duration.SIXTEENTH);
            break;
        case 3:
            duration = new Duration(Duration.THIRTY_SECOND);
            break;
        case 4:
            duration = new Duration(Duration.SIXTY_FOURTH);
            break;
        }
        return duration;
    }

    private int getTiedNoteValue(int string, List notes, List measures) {
        if (!notes.isEmpty()) {
            for (int nIdx = notes.size() - 1; nIdx >= 0; nIdx--) {
                Note note = (Note) notes.get(nIdx);
                if (note.getString() == string) {
                    return note.getValue();
                }
            }
        }
        if (!measures.isEmpty()) {
            for (int mIdx = measures.size() - 1; mIdx >= 0; mIdx--) {
                Measure measure = (Measure) measures.get(mIdx);
                for (int nIdx = measure.getNotes().size() - 1; nIdx >= 0; nIdx--) {
                    Note note = (Note) measure.getNotes().get(nIdx);
                    if (note.getString() == string) {
                        return note.getValue();
                    }
                }
            }
        }
        return -1;
    }

    private boolean readBoolean() throws IOException {
        return (read() == 1);
    }

    private byte readByte() throws IOException {
        return (byte) read();
    }

    private TrackColor readColor() throws IOException {
        int r = readUnsignedByte();
        int g = readUnsignedByte();
        int b = readUnsignedByte();
        read();
        
        return new TrackColor(r,g,b);
    }

    private int readInt() throws IOException {
        int integer = 0;
        byte[] b = { 0, 0, 0, 0 };

        read(b);
        integer = ((b[3] & 0xff) << 24) | ((b[2] & 0xff) << 16) | ((b[1] & 0xff) << 8) | (b[0] & 0xff);

        return integer;
    }

    private void readMarker() throws IOException {
        String name = readStringIntegerPlusOne();
        readColor();
    }

    private Measure createMeasure(int number,TimeSignature currTimeSignature) throws IOException {
        int header = readUnsignedByte();

        int numerator = 0;
        if ((header & 0x01) != 0) {
            numerator = readByte();
        }

        int denominator = 0;
        if ((header & 0x02) != 0) {
            denominator = readByte();
        }

        boolean repeatStart = ((header & 0x04) != 0);

        int numberOfRepetitions = 0;
        if ((header & 0x08) != 0) {
            numberOfRepetitions = readByte();
        }

        int NumberOfAlternateEnding = 0;
        if ((header & 0x10) != 0) {
            NumberOfAlternateEnding = readByte();
        }

        if ((header & 0x20) != 0) {
            readMarker();
        }

        if ((header & 0x40) != 0) {
            int type = readByte();
            readByte();
        }

        if ((header & 0x80) != 0) {
        }
        
        boolean doubleBar = ((header & 0x80) != 0);

        if (numerator > 0) {
            currTimeSignature.setNumerator(numerator);
        }
        if (denominator > 0) {
            currTimeSignature.setDenominator(new Duration(denominator));
        }

        return new Measure(number,0, new ArrayList(), new ArrayList(), (TimeSignature) currTimeSignature.clone(), new Tempo(120),1,0, repeatStart,
                numberOfRepetitions);
    }

    private void addMeasureComponents(List trackStrings, Measure measure, List currTrackMeasures, Tempo tempo) throws IOException,
            GPFormatException {
        long nextNoteStart = measure.getStart();
        int numberOfBeats = readInt();
        for (int i = 0; i < numberOfBeats; i++) {
            nextNoteStart += addNotes(nextNoteStart, measure.getNotes(), trackStrings, currTrackMeasures, tempo);
        }

    }

    private Note parseNote(long start, InstrumentString string, Duration currDuration, List currMeasureNotes, List currTrackMeasures,NoteEffect effect)
            throws IOException {
        int header = readUnsignedByte();

        boolean accentuated = ((header & 0x40) != 0);
        boolean dotted = ((header & 0x02) != 0);
        boolean ghostNote = ((header & 0x04) != 0);

        boolean tiedNote = false;
        boolean deadNote = false;
        if ((header & 0x20) != 0) {
            int noteType = readUnsignedByte();
            tiedNote = (noteType == 0x02);
            effect.setDeadNote((noteType == 0x03));
        }

        if ((header & 0x01) != 0) {
            byte duration = readByte();
            byte tuplet = readByte();
        }

        if ((header & 0x10) != 0) {
            byte dynamic = readByte();
        }

        byte numberOfFret = 0;
        if ((header & 0x20) != 0) {
            numberOfFret = readByte();
        }

        if ((header & 0x80) != 0) {
            byte fingeringLeftHand = readByte();
            byte fingeringRightHand = readByte();
        }

        if ((header & 0x08) != 0) {
            readNoteEffects(effect,currDuration);
        }

        int value = numberOfFret;
        if (numberOfFret >= 0 || tiedNote) {
            if (tiedNote) {
                value = getTiedNoteValue(string.getNumber(), currMeasureNotes, currTrackMeasures);
            }

            return new Note(value, start, (Duration) currDuration.clone(), 64, string.getNumber(),tiedNote,effect);
        }

        return null;
    }

    private String readStringByte(int expectedLength) throws IOException {
        byte[] b;
        String str;
        int realLength = readUnsignedByte();

        if (expectedLength != 0) {
            b = new byte[expectedLength];
        } else {
            b = new byte[realLength];
        }

        read(b);
        str = new String(b, 0, realLength);
        return str;
    }

    private String readStringInteger() throws IOException {
        byte[] b;
        String str;
        int length = readInt();

        b = new byte[length];
        read(b);

        str = new String(b);
        return str;
    }

    private String readStringIntegerPlusOne() throws IOException {
        byte[] b;
        String str;
        int lengthPlusOne = readInt();
        int length = lengthPlusOne - 1;

        if (length != read()) {
            throw new IOException();
        }

        b = new byte[length];
        read(b);

        str = new String(b);
        return str;
    }

    private SongTrack createTrack(long number, List channels) throws IOException {
        int header = readUnsignedByte();

        boolean isDrumsTrack = ((header & 0x01) != 0);
        boolean is12StringedGuitarTrack = ((header & 0x02) != 0);
        boolean isBanjoTrack = ((header & 0x04) != 0);

        String name = readStringByte(40);

        int numberOfStrings = readInt();

        List strings = new ArrayList(numberOfStrings);

        for (int i = 0; i < 7; i++) {
            int tunning = readInt();
            if (numberOfStrings > i) {
                strings.add(new InstrumentString(i + 1, tunning));
            }
        }

        int port = readInt();

        int channelIndex = readInt();

        int effects = readInt();

        int numberOfFrets = readInt();

        int capo = readInt();

        TrackColor color = readColor();
        
        return new SongTrack(number,name,parseChannel(channels,channelIndex,effects), new ArrayList(), strings,color);
    }
    
    private SongChannel parseChannel(List channels, int channelIndex,int effectChannel) {
    	SongChannel channel = (SongChannel) channels.get(channelIndex - 1);

        int instrument = channel.getInstrument();
        if (instrument == -1) {
            channel.setInstrument((short)0);
        }
        if(!channel.isPercusionChannel()){
        	channel.setEffectChannel((short)(effectChannel - 1));
        }
        
        return channel;
    }


    private int readUnsignedByte() throws IOException {
        return read();
    }

    private void readChordType() throws IOException {
        readUnsignedByte();
    }

    private void readRoot() throws IOException {
        readByte();
    }

    private void readTonalityType(int numBytes) throws IOException {
        if (numBytes == 1) {
            int type = readUnsignedByte();
        } else if (numBytes == 4) {
            int type = readInt();
        }
    }

    private String readChordName() throws IOException {
        byte[] nameB;
        char[] nameC;
        int i;
        int max;

        nameB = new byte[21];
        nameC = new char[20];
        read(nameB, 0, 21);
        max = 20;
        if (nameB[0] < max) {
            max = nameB[0];
        }

        for (i = 1; i <= max; i++) {
            nameC[i - 1] = (char) nameB[i];
        }
        return (String.valueOf(nameC, 0, max));
    }

    private void readChordDiagram() throws IOException, GPFormatException {
        int header;
        long aux;
        long skip;
        int i;

        header = readUnsignedByte();

        if ((header & 0x01) == 0) {
            throw new GPFormatException("Cannot Read Chord Diagram");
        }

        boolean sharp = readBoolean();

        this.skip(3);

        readRoot();

        readChordType();

        int nineElevenThirteen = readUnsignedByte();

        int bass = readInt();

        readTonalityType(4);

        int addedNote = readUnsignedByte();

        String name = readChordName();

        this.skip(2);

        readTonalityType(1);

        readTonalityType(1);

        readTonalityType(1);

        int baseFret = readInt();

        for (i = 1; i <= 7; i++) {
            int fret = readInt();
        }

        int numBarres = readUnsignedByte();

        for (i = 1; i <= 5; i++) {
            int fretOfBarre = readUnsignedByte();
        }
        for (i = 1; i <= 5; i++) {
            int barreStart = readUnsignedByte();
        }
        for (i = 1; i <= 5; i++) {
            int barreEnd = readUnsignedByte();
        }

        aux = this.skip(8);

        for (i = 1; i <= 7; i++) {
            int fingering = readByte();
        }
        boolean chordFingeringDisplayed = readBoolean();
    }

    private void readGraceNote() throws IOException {
        int i;
        byte b[] = new byte[4];

        for (i = 0; i < 4; i++) {
            b[i] = (byte) readUnsignedByte();
        }
    }

    private void readBend(NoteEffect effect,Duration duration) throws IOException {
        byte type = readByte();
        int value = readInt();
        
        BendEffect bend = new BendEffect();        
        int numPoints = readInt();
        for (int i = 0; i < numPoints; i++) {
            
            int bendPosition = readInt();
            int bendValue = readInt();
            byte bendVibrato = readByte();
            
            bend.addPoint((int)(bendPosition * BendEffect.MAX_POSITION_LENGTH / 60),(bendValue * 8 / 100));           
        }
        if(!bend.getPoints().isEmpty()){
            effect.setBend(bend);
        }        
    }

    private void readNoteEffects(NoteEffect noteEffect,Duration duration) throws IOException {
        int header1;
        int header2;
        int b;

        header1 = readUnsignedByte();
        header2 = readUnsignedByte();

        if ((header1 & 0x01) != 0) {
            readBend(noteEffect,duration);
        }

        if ((header1 & 0x10) != 0) {
            readGraceNote();
        }

        if ((header2 & 0x04) != 0) {
            readUnsignedByte();
        }

        if ((header2 & 0x08) != 0) {
            noteEffect.setSlide(true);
            readByte();            
        }

        if ((header2 & 0x10) != 0) {
            b = readByte();
        }

        if ((header2 & 0x20) != 0) {
            byte fret = readByte();
            byte period = readByte();
        }

        if ((header1 & 0x08) != 0) {
        }

        if ((header1 & 0x02) != 0) {
            noteEffect.setHammer(true);
        }

        if ((header2 & 0x40) != 0) {
            noteEffect.setVibrato(true);
        }

        if ((header2 & 0x02) != 0) {
        }

        if ((header2 & 0x01) != 0) {
        }
    }

    private void readBeatEffects(NoteEffect noteEffect,Duration currDuration) throws IOException {
        int header[] = { 0, 0 };

        header[0] = readUnsignedByte();
        header[1] = readUnsignedByte();

        if ((header[0] & 0x20) != 0) {
            int effect = readUnsignedByte();
            switch (effect) {
            case 0:
                break;
            case 1:
                break;
            case 2:
                break;
            case 3:
                break;
            default:
                throw new IOException();
            }
        }

        if ((header[1] & 0x04) != 0) {
            readBend(noteEffect,currDuration);
        }

        if ((header[0] & 0x40) != 0) {
            int durationValue = readByte();
            durationValue = readByte();
        }

        if ((header[1] & 0x01) != 0) {
            //Rasgueado
        }

        if ((header[1] & 0x02) != 0) {
            //Pickstroke
            readByte();
        }
    }

    private void readMixChange(Tempo tempo) throws IOException {
        int pos[] = new int[8];
        int i;
        int n;
        int aux;
        n = 0;
        for (i = 0; i < 7; i++) {
            aux = readByte();
            if ((i != 0) && (aux != -1)) {
                pos[n] = i;
                n++;
            }
        }
        aux = readInt();
        if (aux != -1) {
            tempo.setValue(aux);
            pos[n] = i;
            n++;
        }

        for (i = 0; i < n; i++) {
            aux = readByte();
        }
        int applyToAllTracks = readUnsignedByte();
    }
    
    private short toChannelShort(byte b){
    	short s = (short)b;
    	s = (short)((s * (short)127) / (short)16);
    	return (s <= 127)?s:127;
    }

}

