package org.herac.tuxguitar.io.gp;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.herac.tuxguitar.song.models.BendEffect;
import org.herac.tuxguitar.song.models.Duration;
import org.herac.tuxguitar.song.models.InstrumentString;
import org.herac.tuxguitar.song.models.Measure;
import org.herac.tuxguitar.song.models.Note;
import org.herac.tuxguitar.song.models.NoteEffect;
import org.herac.tuxguitar.song.models.Song;
import org.herac.tuxguitar.song.models.SongChannel;
import org.herac.tuxguitar.song.models.SongTrack;
import org.herac.tuxguitar.song.models.Tempo;
import org.herac.tuxguitar.song.models.TimeSignature;
import org.herac.tuxguitar.song.models.TrackColor;

/**
 * @author julian
 * 
 * TODO To change the template for this generated type comment go to Window - Preferences - Java - Code Style - Code Templates
 */
public class GP3InputStream extends InputStream {
    private static final String GP3_VERSION = "FICHIER GUITAR PRO v3.00";
    private String version;
    private InputStream inputStream;
    private int offset;

    public GP3InputStream(InputStream inputStream) {
        super();
        this.inputStream = inputStream;
        this.offset = 0;
    }

    public GP3InputStream(String fileName) throws FileNotFoundException {
        this(new FileInputStream(new File(fileName)));
    }

    /**
     * Closes the stream.
     */
    public void close() throws IOException {
        this.inputStream.close();
    }

    public int read() throws IOException {
        this.offset++;
        return this.inputStream.read();
    }

    public static boolean isSupportedVersion(String version) {
        return (version.equals(GP3_VERSION));
    }

    public boolean isSupportedVersion(){
    	try{
    		readVersion();
    		return isSupportedVersion(version);	  
    	}catch(Exception e){
    		return false;
    	}catch(Error e){
    		return false;
    	}   		
    }
    
    private void readVersion(){
		try {
			if(this.version == null){
				this.version = readStringByte(30);
			}
		} catch (IOException e) {
			this.version = "NOT_SUPPORTED";
		}    	
    }
    
    public Song readSong() throws IOException, GPFormatException {
        readVersion();
        if (!isSupportedVersion(version)) {
            throw new GPFormatException("Unsuported Version");
        }

        String title = readStringIntegerPlusOne();

        String subtitle = readStringIntegerPlusOne();

        String interpret = readStringIntegerPlusOne();

        String album = readStringIntegerPlusOne();

        String songAuthor = readStringIntegerPlusOne();

        String copyright = readStringIntegerPlusOne();

        String pieceAuthor = readStringIntegerPlusOne();

        String instructions = readStringIntegerPlusOne();

        int nbNotes = readInt();
        String note = "";
        for (int i = 0; i < nbNotes; i++) {
            note += readStringIntegerPlusOne();
            note += "\n";
        }

        boolean tripletFeel = readBoolean();

        int tempoValue = readInt();

        int key = readInt();

        List channels = new ArrayList();
        int[] instruments = new int[64];
        for (int i = 0; i < 64; i++) {
        	channels.add(new SongChannel((short)i,
        				(short)i, 
        				(short)readInt(),
        				toChannelShort(readByte()),
        				toChannelShort(readByte()),
        				toChannelShort(readByte()),
        				toChannelShort(readByte()),
        				toChannelShort(readByte()),
        				toChannelShort(readByte()),
        				false,
        				false));           
            byte[] b = { 0, 0 };
            read(b);
        }

        TimeSignature timeSignature = new TimeSignature(4, new Duration(4));
        int numberOfMeasures = readInt();
        int numberOfTracks = readInt();

        List measures = new ArrayList();
        if (numberOfMeasures > 0) {
            for (int i = 0; i < numberOfMeasures; i++) {
                Measure measure = createMeasure((i + 1),timeSignature);
                measures.add(measure);
            }
        }

        List tracks = new ArrayList();
        for (int i = 0; i < numberOfTracks; i++) {
            tracks.add(createTrack((i + 1), channels));
        }

        long nextMeasureStart = 1000;
        for (int i = 0; i < numberOfMeasures; i++) {
            Measure currMeasure = (Measure) measures.get(i);
            Measure lastMeasure = null;
            currMeasure.setStart(nextMeasureStart);
            for (int j = 0; j < numberOfTracks; j++) {
                SongTrack track = (SongTrack) tracks.get(j);
                Measure measure = (Measure) currMeasure.clone();

                Tempo tempo = new Tempo(tempoValue);
                addMeasureComponents(track.getStrings(), measure, track.getMeasures(), tempo);
                
                //checkeo que el tempo coincida
                if(lastMeasure != null && lastMeasure.getTempo().getValue() != tempo.getValue()){
                	measure.setTempo((Tempo)lastMeasure.getTempo().clone());      
                }else{
                	measure.setTempo(tempo);       
                }
                lastMeasure = measure;   

                track.getMeasures().add(measure);
            }
            nextMeasureStart += currMeasure.getLength();
        }

        int i = 0;
        int b;
        boolean different = false;
        b = read();
        while (b != -1) {
            i++;
            if (!different && i > 4) {
                different = true;
            }
            b = read();
        }

        return new Song(title,interpret,album,songAuthor,tracks,Song.MAX_VOLUME);

    }

    private List getPlayedStrings(int stringsPlayed, List trackStrings) {
        List strings = new ArrayList();
        if ((stringsPlayed & (1 << 0)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(6)).clone());
        }
        if ((stringsPlayed & (1 << 1)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(5)).clone());
        }
        if ((stringsPlayed & (1 << 2)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(4)).clone());
        }
        if ((stringsPlayed & (1 << 3)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(3)).clone());
        }
        if ((stringsPlayed & (1 << 4)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(2)).clone());
        }
        if ((stringsPlayed & (1 << 5)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(1)).clone());
        }
        if ((stringsPlayed & (1 << 6)) != 0) {
            strings.add(((InstrumentString) trackStrings.get(0)).clone());
        }
        return strings;
    }

    private Duration parseDuration(byte value) {
        Duration duration = null;
        switch (value) {
        case -2:
            duration = new Duration(Duration.WHOLE);
            break;
        case -1:
            duration = new Duration(Duration.HALF);
            break;
        case 0:
            duration = new Duration(Duration.QUARTER);
            break;
        case 1:
            duration = new Duration(Duration.EIGHTH);
            break;
        case 2:
            duration = new Duration(Duration.SIXTEENTH);
            break;
        case 3:
            duration = new Duration(Duration.THIRTY_SECOND);
            break;
        case 4:
            duration = new Duration(Duration.SIXTY_FOURTH);
            break;
        }
        return duration;
    }

    private int getTiedNoteValue(int string, List notes, List measures) {
        if (!notes.isEmpty()) {
            for (int nIdx = notes.size() - 1; nIdx >= 0; nIdx--) {
                Note note = (Note) notes.get(nIdx);
                if (note.getString() == string) {
                    return note.getValue();
                }
            }
        }
        if (!measures.isEmpty()) {
            for (int mIdx = measures.size() - 1; mIdx >= 0; mIdx--) {
                Measure measure = (Measure) measures.get(mIdx);
                for (int nIdx = measure.getNotes().size() - 1; nIdx >= 0; nIdx--) {
                    Note note = (Note) measure.getNotes().get(nIdx);
                    if (note.getString() == string) {
                        return note.getValue();
                    }
                }
            }
        }
        return -1;
    }

    private boolean readBoolean() throws IOException {
        return (read() == 1);
    }

    private byte readByte() throws IOException {
        return (byte) read();
    }

    private TrackColor readColor() throws IOException {
        int r = readUnsignedByte();
        int g = readUnsignedByte();
        int b = readUnsignedByte();
        read();
    
        return new TrackColor(r,g,b);
    }

    private int readInt() throws IOException {
        int integer = 0;
        byte[] b = { 0, 0, 0, 0 };

        read(b);
        integer = ((b[3] & 0xff) << 24) | ((b[2] & 0xff) << 16) | ((b[1] & 0xff) << 8) | (b[0] & 0xff);

        return integer;
    }

    private void readMarker() throws IOException {
        String name = readStringIntegerPlusOne();
        readColor();
    }

    private Measure createMeasure(int number,TimeSignature currTimeSignature) throws IOException {
        int header = readUnsignedByte();

        int numerator = 0;
        if ((header & 0x01) != 0) {
            numerator = readByte();
        }

        int denominator = 0;
        if ((header & 0x02) != 0) {
            denominator = readByte();
        }

        boolean repeatStart = ((header & 0x04) != 0);

        int numberOfRepetitions = 0;
        if ((header & 0x08) != 0) {
            numberOfRepetitions = readByte();
        }

        int NumberOfAlternateEnding = 0;
        if ((header & 0x10) != 0) {
            NumberOfAlternateEnding = readByte();
        }

        if ((header & 0x20) != 0) {
            readMarker();
        }

        if ((header & 0x40) != 0) {
            int type = readByte();
            readByte();
        }

        if ((header & 0x80) != 0) {
        }
        
        boolean doubleBar = ((header & 0x80) != 0);

        if (numerator > 0) {
            currTimeSignature.setNumerator(numerator);
        }
        if (denominator > 0) {
            currTimeSignature.setDenominator(new Duration(denominator));
        }

        return new Measure(number,0, new ArrayList(), new ArrayList(), (TimeSignature) currTimeSignature.clone(), new Tempo(120),1,0 ,repeatStart,numberOfRepetitions);
    }

    private void addMeasureComponents(List trackStrings, Measure measure, List currTrackMeasures, Tempo tempo) throws IOException,
            GPFormatException {
        long nextNoteStart = measure.getStart();
        int numberOfBeats = readInt();
        for (int i = 0; i < numberOfBeats; i++) {
            nextNoteStart += addNotes(nextNoteStart, measure.getNotes(), trackStrings, currTrackMeasures, tempo);
        }

    }

    private long addNotes(long start, List notes, List trackStrings, List currTrackMeasures, Tempo tempo) throws IOException,
            GPFormatException {
        NoteEffect effect = new NoteEffect();
        
        int header = readUnsignedByte();

        if ((header & 0x80) != 0) {
            //TODO ver que es 0x80
            header &= 0x3f;
        }

        if ((header & 0x40) != 0) {
            int beatStatus = readUnsignedByte();
            boolean emptyBeat = (beatStatus == 0x00);
            boolean restBeat = (beatStatus == 0x02);
        }

        boolean dottedNotes = ((header & 0x01) != 0);

        Duration duration = parseDuration(readByte());
        duration.setDotted(dottedNotes);

        if ((header & 0x20) != 0) {
            int tuplet = readInt();
            //-------Verifico el tupleto--------------------
            switch (tuplet) {
            case 3:
                duration.getTupleto().setEnters(3);
                duration.getTupleto().setTimes(2);
                break;
            case 5:
                duration.getTupleto().setEnters(5);
                duration.getTupleto().setTimes(4);
                break;                
            case 6:
                duration.getTupleto().setEnters(6);
                duration.getTupleto().setTimes(4);
                break;
            case 7:
                duration.getTupleto().setEnters(7);
                duration.getTupleto().setTimes(4);
                break;
            case 9:
                duration.getTupleto().setEnters(9);
                duration.getTupleto().setTimes(8);
                break;
            case 10:
                duration.getTupleto().setEnters(10);
                duration.getTupleto().setTimes(8);
                break;
            case 11:
                duration.getTupleto().setEnters(11);
                duration.getTupleto().setTimes(8);
                break;
            case 12:
                duration.getTupleto().setEnters(12);
                duration.getTupleto().setTimes(8);
                break;                
            }
        }

        if ((header & 0x02) != 0) {
            readChordDiagram();
        }

        if ((header & 0x04) != 0) {
            String text = readStringIntegerPlusOne();
        }

        if ((header & 0x08) != 0) {
            readBeatEffects(effect);
        }

        if ((header & 0x10) != 0) {
            readMixChange(tempo);
        }

        int stringsPlayed = readUnsignedByte();
        List strings = getPlayedStrings(stringsPlayed, trackStrings);
       
        for (int i = strings.size() - 1; i >= 0; i--) {
            InstrumentString string = (InstrumentString) strings.get(i);
            Note note = parseNote(start, string, duration, notes, currTrackMeasures,(NoteEffect)effect.clone());
            if (note != null) {
                notes.add(note);
            }
        }

        return duration.getTime();
    }

    private Note parseNote(long start, InstrumentString string, Duration currDuration, List currMeasureNotes, List currTrackMeasures,NoteEffect effect)
            throws IOException {
        int header = readUnsignedByte();

        boolean accentuated = ((header & 0x40) != 0);
        boolean dotted = ((header & 0x02) != 0);
        boolean ghostNote = ((header & 0x04) != 0);

        boolean tiedNote = false;
        boolean deadNote = false;
        if ((header & 0x20) != 0) {
            int noteType = readUnsignedByte();
            tiedNote = (noteType == 0x02);
            effect.setDeadNote((noteType == 0x03));
        }

        if ((header & 0x01) != 0) {
            byte duration = readByte();
            byte tuplet = readByte();
        }

        if ((header & 0x10) != 0) {
            byte dynamic = readByte();
        }

        byte numberOfFret = 0;
        if ((header & 0x20) != 0) {
            numberOfFret = readByte();
        }

        if ((header & 0x80) != 0) {
            byte fingeringLeftHand = readByte();
            byte fingeringRightHand = readByte();
        }

        if ((header & 0x08) != 0) {
            readNoteEffects(effect,currDuration);
        }

        int value = numberOfFret;
        if (numberOfFret >= 0 || tiedNote) {
            if (tiedNote) {
                value = getTiedNoteValue(string.getNumber(), currMeasureNotes, currTrackMeasures);
            }

            return new Note(value, start,(Duration)currDuration.clone(), 64,string.getNumber(),tiedNote,effect);
        }

        return null;
    }

    private String readStringByte(int expectedLength) throws IOException {
        byte[] b;
        String str;
        int realLength = readUnsignedByte();

        if (expectedLength != 0) {
            b = new byte[expectedLength];
        } else {
            b = new byte[realLength];
        }

        read(b);
        str = new String(b, 0, realLength);
        return str;
    }

    private String readStringInteger() throws IOException {
        byte[] b;
        String str;
        int length = readInt();

        b = new byte[length];
        read(b);

        str = new String(b);
        return str;
    }

    private String readStringIntegerPlusOne() throws IOException {
        byte[] b;
        String str;
        int lengthPlusOne = readInt();
        int length = lengthPlusOne - 1;

        if (lengthPlusOne > 0) {
            int r = read();
            if (length != r) {
                throw new IOException();
            }

            b = new byte[length];
            read(b);
            str = new String(b);
        } else {
            int r = read();
            str = "";
        }

        return str;
    }

    private SongTrack createTrack(long number, List channels) throws IOException {
        int header = readUnsignedByte();

        boolean isDrumsTrack = ((header & 0x01) != 0);
        boolean is12StringedGuitarTrack = ((header & 0x02) != 0);
        boolean isBanjoTrack = ((header & 0x04) != 0);

        String name = readStringByte(40);

        int numberOfStrings = readInt();

        List strings = new ArrayList(numberOfStrings);

        for (int i = 0; i < 7; i++) {
            int tunning = readInt();
            if (numberOfStrings > i) {
                strings.add(new InstrumentString(i + 1, tunning));
            }
        }

        int port = readInt();

        int channelIndex = readInt();

        int effects = readInt();

        int numberOfFrets = readInt();

        int capo = readInt();

        TrackColor color = readColor();

        return new SongTrack(number,name,parseChannel(channels,channelIndex,effects),new ArrayList(), strings,color);
    }
    
    
    private SongChannel parseChannel(List channels, int channelIndex,int effectChannel) {
    	SongChannel channel = (SongChannel) channels.get(channelIndex - 1);

        int instrument = channel.getInstrument();
        if (instrument == -1) {
            channel.setInstrument((short)0);
        }
        if(!channel.isPercusionChannel()){
        	channel.setEffectChannel((short)(effectChannel - 1));
        }
        
        return channel;
    }

    private int readUnsignedByte() throws IOException {
        return read();
    }

    private void readChordDiagram() throws IOException, GPFormatException {
        int header = readUnsignedByte();

        if ((header & 0x01) == 0) {
            String name = readStringIntegerPlusOne();
            int base = readInt();

            if (base != 0) {
                for (int i = 0; i < 6; i++) {
                    int fret = readInt();
                }
            }
        } else {
            boolean sharp = readBoolean();

            skip(3);

            int root = readInt();

            int chordType = readInt();

            int nineElevenThirteen = readInt();

            int bass = readInt();

            int tonalityType = readUnsignedByte();

            skip(3);

            int addedNote = readUnsignedByte();

            String name = readStringByte(34);

            int baseFret = readInt();

            for (int i = 0; i < 6; i++) {
                int fret = readInt();
            }

            for (int i = 0; i < 28; i++) {
                int n = readUnsignedByte();
            }

            for (int i = 0; i < 7; i++) {
                readUnsignedByte();
            }

            int n = readUnsignedByte();
        }

    }

    private void readGraceNote() throws IOException {
        byte b[] = new byte[4];
        read(b);
    }

    private void readBend(NoteEffect effect,Duration duration) throws IOException {
        byte type = readByte();
        int value = readInt();

        BendEffect bend = new BendEffect();
        int numPoints = readInt();
        for (int i = 0; i < numPoints; i++) {

            int bendPosition = readInt();
            int bendValue = readInt();
            byte bendVibrato = readByte();
            
            bend.addPoint((int)(bendPosition * BendEffect.MAX_POSITION_LENGTH / 60),(bendValue * 8 / 100));                       
        }
        if(!bend.getPoints().isEmpty()){
            effect.setBend(bend);
        }
    }

    private void readNoteEffects(NoteEffect effect,Duration duration) throws IOException {
        int header;

        header = readUnsignedByte();

        if ((header & 0x01) != 0) {
            readBend(effect,duration);
        }

        if ((header & 0x10) != 0) {
            readGraceNote();
        }

        if ((header & 0x04) != 0) {
            effect.setSlide(true);
        }

        if ((header & 0x08) != 0) {
        }

        if ((header & 0x02) != 0) {
            effect.setHammer(true);
        }
    }

    private void readBeatEffects(NoteEffect noteEffect) throws IOException {
        int header = readUnsignedByte();

        if ((header & 0x20) != 0) {
            int effect = readUnsignedByte();
            if (effect == 0) {
                int value = readInt();
            } else {
                switch (effect) {
                case 1:
                    break;
                case 2:
                    break;
                case 3:
                    break;

                default:
                    throw new IOException("Unexpected value in effect: " + effect);
                }

                int dummy = readInt();
            }
        }

        if ((header & 0x40) != 0) {
            int durationValue = readByte();
            durationValue = readByte();
        }
        
        if ((header & 0x01) != 0) {
            noteEffect.setVibrato(true);
            //Vibrato
        }        
        if ((header & 0x02) != 0) {
            noteEffect.setVibrato(true);
            //Wide Vibrato
        }
        if ((header & 0x04) != 0) {
            //Harmonic Natural
        }
        if ((header & 0x08) != 0) {
            //Harmonic Artificial
        }
        if ((header & 0x10) != 0) {
            // Fade in
        }
    }

    private void readMixChange(Tempo tempo) throws IOException {
        int pos[] = new int[8];
        int i;
        int n;
        int aux;

        n = 0;
        for (i = 0; i < 7; i++) {
            aux = readByte();
            if ((i != 0) && (aux != -1)) {
                pos[n] = i;
                n++;
            }
        }
        aux = readInt();
        if (aux != -1) {
            tempo.setValue(aux);

            pos[n] = i;
            n++;
        }
        for (i = 0; i < n; i++) {
            aux = readByte();
        }

    }

    
    
    private short toChannelShort(byte b){
    	short s = (short)b;
    	s = (short)((s * (short)127) / (short)16);
    	return (s <= 127)?s:127;
    }
}