package org.herac.tuxguitar.gui.table;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.ScrollBar;
import org.herac.tuxguitar.gui.TablatureEditor;
import org.herac.tuxguitar.gui.TuxGuitar;
import org.herac.tuxguitar.gui.actions.composition.ChangeInfoAction;
import org.herac.tuxguitar.gui.actions.track.ChangeTrackPropertiesAction;
import org.herac.tuxguitar.gui.actions.track.GoToTrackAction;
import org.herac.tuxguitar.gui.tab.MeasureCoords;
import org.herac.tuxguitar.gui.tab.SongTrackCoords;
import org.herac.tuxguitar.song.models.SongChannel;
import org.herac.tuxguitar.song.models.SongTrack;
import org.herac.tuxguitar.song.models.TrackColor;


public class SongTableInfo extends Composite{    
	private static final int TABLE_COLUMN_NUMBER = 0;
	private static final int TABLE_COLUMN_NAME = 1;
	private static final int TABLE_COLUMN_INSTRUMENT = 2;
	private static final int TABLE_COLUMN_CHANNEL = 3;
	private static final int TABLE_COLUMN_EFFECT_CHANNEL = 4;
	
	private static final Color[] BACKGROUNDS = new Color[]{
		new Color(TuxGuitar.instance().getDisplay(),255,255,255),
		new Color(TuxGuitar.instance().getDisplay(),238,238,238),
		new Color(TuxGuitar.instance().getDisplay(),192,192,192)
	};
	
	private TablatureEditor editor;
	private ScrollBar hSroll;
	private ScrolledComposite scrolledComposite;
	private SongTable table;
	private Composite trackMeasures;
	private Button trackMeasuresHeader;
	private Canvas painter;
	private int measureLength;
	
	private long lastTrackSelection;
	private MeasureCoords lasPlayedMeasure;
	
	public SongTableInfo(Composite parent,TablatureEditor editor) {
        super(parent,SWT.H_SCROLL | SWT.BORDER);
        this.setLayout(new FillLayout());
        this.editor = editor;
        this.init();
        
        this.hSroll = getHorizontalBar();
        this.hSroll.addListener(SWT.Selection, new Listener() {
            public void handleEvent(Event e) {
                painter.redraw();
            }
          });
         
        this.redraw();
    }
	
	private void init(){
	    this.scrolledComposite = new ScrolledComposite(this,SWT.V_SCROLL);
		GridLayout layout = new GridLayout();
		layout.marginTop = 0;
		layout.marginBottom = 0;
		layout.marginLeft = 0;
		layout.marginRight = 0;
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.horizontalSpacing = 0;
		layout.verticalSpacing = 0;
		scrolledComposite.setLayout(layout);

		SashForm sashForm = new SashForm(this.scrolledComposite, SWT.HORIZONTAL);	    
		sashForm.SASH_WIDTH = 2;
		
		initTable(sashForm);
		initTrackMeasures(sashForm);
		
		this.scrolledComposite.setContent(sashForm);		
		this.scrolledComposite.setExpandHorizontal(true);
		this.scrolledComposite.setExpandVertical(true);		
	}
	
	private void initTable(Composite parent){
	    this.table = new SongTable(parent,5,editor);
	    this.loadProperties();

	    this.table.setWeights(new int[] { 1,7,5,3,2});	    
	    
	    this.updateTable();
		

	}
	
	private void initTrackMeasures(Composite parent){
	    Composite composite = new Composite(parent,SWT.NONE);
	    composite.setLayout(new FillLayout(SWT.VERTICAL));	 
	 
		this.trackMeasures = new Composite(composite,SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginTop = 0;
		layout.marginBottom = 0;
		layout.marginLeft = 0;
		layout.marginRight = 0;
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.horizontalSpacing = 0;
		layout.verticalSpacing = 0;
		this.trackMeasures.setLayout(layout);		
		this.trackMeasuresHeader = new Button(this.trackMeasures,SWT.PUSH);		
		this.trackMeasuresHeader.setLayoutData(new GridData(SWT.FILL,SWT.TOP,true,false,0,0));
		
		this.trackMeasuresHeader.addMouseListener(new MouseAdapter() {
			public void mouseDoubleClick(MouseEvent e) {
				TuxGuitar.instance().getAction(ChangeInfoAction.NAME).process(e);
			}
		});
		
		
		this.painter = new Canvas(this.trackMeasures,SWT.DOUBLE_BUFFERED);
		this.painter.setLayoutData(new GridData(SWT.FILL,SWT.FILL,true,true,0,0));	
		this.painter.addPaintListener(new PaintListener() {
			public void paintControl(PaintEvent e) {
				updateTrackMeasuresHeader();
				paintMeasures(e.gc,-hSroll.getSelection(),0);	
				e.gc.dispose();
			}		
		});
		
		this.painter.addMouseListener(new MouseAdapter() {
			public void mouseDown(MouseEvent e) {
				int trackIndex = getSelectedTrackIndex(e.y);
				if(trackIndex >= 0){
				    SongTrackCoords track = (SongTrackCoords)editor.getTablature().getSongCoords().getTrackCoords().get(trackIndex);
				    int measureIndex = getSelectedMeasureIndex(track,e.x);
					    
					if(measureIndex >= 0){
						MeasureCoords measure = (MeasureCoords)track.getMeasuresCoords().get(measureIndex);
						editor.getTablature().getCaret().moveTo(track,measure,measure.getFirstComponent());
						TuxGuitar.instance().getTransport().gotoMeasure(measure.getMeasure());
						TuxGuitar.instance().redraw();
					}						
				}
			}
		});
		
	}



	private void updateTable(){	 		
		List tracks = this.editor.getTablature().getSongCoords().getTrackCoords();
		int length = tracks.size();
				
		this.table.clearColumnsAfter(length);
		this.makeTableItems(length);
		
		for(int i = 0;i < tracks.size();i++){
			final SongTrackCoords track = (SongTrackCoords)tracks.get(i);
			String number = Long.toString(track.getTrack().getNumber());
			String name = track.getTrack().getName();
			String instrument = editor.getSongManager().getInstrumentName(track.getTrack().getChannel().getInstrument());
	    	
			updateTableItems(i,new String[] {number,name,instrument},track);
	    }
		this.updateItemBackgrounds();
		
	    table.layout();		
	}
	
	private void makeTableItems(int count){
	    GridData gridData = new GridData(SWT.FILL,SWT.CENTER,true,true,0,0);
	    gridData.heightHint = 22;
	    
	    int currentItems = table.getColumn(TABLE_COLUMN_NUMBER).getItemCount();
	    for(int i = currentItems;i < count;i ++){	    	
	    	final int trackIndex = i;
	    	
            CLabel item = new CLabel(table.getColumn(TABLE_COLUMN_NUMBER),SWT.LEFT); 
            item.setLayoutData(gridData);
            item.addMouseListener(new ItemSelectionListener());
            
            item = new CLabel(table.getColumn(TABLE_COLUMN_NAME),SWT.LEFT); 
            item.setLayoutData(gridData);
            item.addMouseListener(new ItemSelectionListener());
            
            item = new CLabel(table.getColumn(TABLE_COLUMN_INSTRUMENT),SWT.LEFT);
            item.setLayoutData(gridData);
            item.addMouseListener(new ItemSelectionListener());
            
            
            final Combo channelCombo = new Combo(table.getColumn(TABLE_COLUMN_CHANNEL),SWT.LEFT);
            channelCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
            channelCombo.setData(new ChannelList());   
            updateChannelCombo(channelCombo,(ChannelList)channelCombo.getData(),trackIndex);                        
            
            final Combo effectChannelCombo = new Combo(table.getColumn(TABLE_COLUMN_EFFECT_CHANNEL),SWT.LEFT);
            effectChannelCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
            effectChannelCombo.setData(new ChannelList());            
            updateEffectChannelCombo(effectChannelCombo,(ChannelList)effectChannelCombo.getData(),trackIndex);
            
            
            channelCombo.addSelectionListener(new SelectionAdapter() {
    			public void widgetSelected(SelectionEvent e) {

    		        if(e.widget.getData() instanceof ChannelList){
        				SongChannel channel = ((SongTrack)TuxGuitar.instance().getSongManager().getSong().getTracks().get(trackIndex)).getChannel();
        		        boolean[] usedChannels = TuxGuitar.instance().getSongManager().getUsedChannels();
        		        boolean[] usedEffectChannels = TuxGuitar.instance().getSongManager().getUsedEffectChannels();    		        	
    		        	ChannelList channelItems = (ChannelList)e.widget.getData();
    		        	int idx = channelItems.getChannel(channelCombo.getSelectionIndex());
    		        	if(!usedChannels[idx]){
    		        		channel.setChannel((short)idx);					
    		        		channel.setEffectChannel(channel.getChannel());					
    					}else{
    						SongChannel tempChannel = TuxGuitar.instance().getSongManager().getUsedChannel(idx);					
    						channel.setChannel(tempChannel.getChannel());
    						channel.setEffectChannel(tempChannel.getEffectChannel());
    						channel.setVolume(tempChannel.getVolume());
    						channel.setBalance(tempChannel.getBalance());
    						channel.setChorus(tempChannel.getChorus());
    						channel.setInstrument(tempChannel.getInstrument());
    						channel.setPhaser(tempChannel.getPhaser());
    						channel.setReverb(tempChannel.getReverb());
    						channel.setTremolo(tempChannel.getTremolo());
    						channel.setSolo(tempChannel.isSolo());
    						channel.setMute(tempChannel.isMute());
    					}
    					fireChannelChange(channel);
    					
    					TuxGuitar.instance().getMixer().update();
    		        }
    			}		
    		});
            
            effectChannelCombo.addSelectionListener(new SelectionAdapter() {
    			public void widgetSelected(SelectionEvent e) {    				
    				if(e.widget.getData() instanceof ChannelList){
    					SongChannel channel = ((SongTrack)TuxGuitar.instance().getSongManager().getSong().getTracks().get(trackIndex)).getChannel();
    					ChannelList effectChannelItems = (ChannelList)e.widget.getData();
    					int idx = effectChannelItems.getChannel(effectChannelCombo.getSelectionIndex());
    					channel.setEffectChannel((short)idx);		
    					fireChannelChange(channel);
    					
    					TuxGuitar.instance().getMixer().update();
    				}
    			}
    		
    		});

	    }
		List tracks = TuxGuitar.instance().getSongManager().getSong().getTracks();
		for(int i = 0;i < tracks.size();i++){
			Combo channelCombo = (Combo)this.table.getColumn(TABLE_COLUMN_CHANNEL).getItem(i);
			Combo effectChannelCombo = (Combo)this.table.getColumn(TABLE_COLUMN_EFFECT_CHANNEL).getItem(i);
			
			updateChannelCombo(channelCombo,(ChannelList)channelCombo.getData(),i);
			updateEffectChannelCombo(effectChannelCombo,(ChannelList)effectChannelCombo.getData(),i);
		}		

	}
	
	private void updateItemBackgrounds(){		
	    this.lastTrackSelection = this.editor.getTablature().getCaret().getSongTrackCoords().getTrack().getNumber();
	    
	    int currentItems = table.getColumn(TABLE_COLUMN_NUMBER).getItemCount();
	    for(int i = 0;i < currentItems;i ++){	    		    	
	    	CLabel item = (CLabel)table.getColumn(TABLE_COLUMN_NUMBER).getItem(i);	    	
	    	Color background = BACKGROUNDS[(i % 2)];
	    	long trackNumber = Long.parseLong(item.getText());
	    	if(trackNumber == lastTrackSelection){
	    		background = BACKGROUNDS[2];
	    	}	    	
	    	item.setBackground(background);
	    	
	    	item = (CLabel)table.getColumn(TABLE_COLUMN_NAME).getItem(i); 
	    	item.setBackground(background);
	    	
	    	item = (CLabel)table.getColumn(TABLE_COLUMN_INSTRUMENT).getItem(i); 
	    	item.setBackground(background);	
	    }
	}
	
	private void updateTableItems(int index,String[] texts,Object data){
	    for(int i = 0;i < texts.length;i ++){	        
	    	CLabel item = (CLabel)table.getColumn(i).getItem(index); 
            item.setData(data);
            item.setText(texts[i]);            
	    }
	}

	private void updateTrackMeasuresHeader(){
		((GridData)this.trackMeasuresHeader.getLayoutData()).heightHint = table.getColumn(TABLE_COLUMN_NUMBER).getHeaderHeight();
		this.trackMeasuresHeader.setText(this.editor.getSongManager().getSong().getName());
	}
	
	private void paintMeasures(GC gc,int fromX,int fromY){		
		int width = this.trackMeasures.getBounds().width;
		int itemSpan = table.getColumn(TABLE_COLUMN_EFFECT_CHANNEL).getItemHeight();		
		
		int x = fromX;
		int y = 0;
		int firstTableItemY = 0;
		
		boolean playing = this.editor.getSongManager().getPlayer().isRunning();
		this.lasPlayedMeasure = null;
		List tracks = this.editor.getTablature().getSongCoords().getTrackCoords();
		int trackLength = tracks.size();
		for(int i = 0;i < trackLength;i++){
			SongTrackCoords track = (SongTrackCoords)tracks.get(i);
		
			//Calculo el Y acorde a la tabla
			Control control = table.getColumn(TABLE_COLUMN_NUMBER).getItem(i);
			Rectangle controlBounds = control.getBounds();
			if(i == 0){
				firstTableItemY = controlBounds.y;
			}
			y = fromY + (controlBounds.y - firstTableItemY);
			
			gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_GRAY));
			gc.fillRectangle(0,y,width,itemSpan - 1);

			TrackColor trackColor = track.getTrack().getColor();
			Color color = new Color(getDisplay(),trackColor.getR(),trackColor.getG(),trackColor.getB());
			gc.setBackground(color);

			List measures = track.getMeasuresCoords();
			this.measureLength = measures.size();
			for(int j = 0;j < measureLength;j++){
				MeasureCoords measure = (MeasureCoords)measures.get(j);
				gc.fillRectangle(x,y,itemSpan - 1,itemSpan - 1);
				
				if((playing && measure.isPlaying()) || (!playing && measure.hasCaret())){
					gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_BLACK));
					gc.fillRectangle(x + 4,y + 4,itemSpan - 9,itemSpan - 9);
					gc.setBackground(color);
					this.lasPlayedMeasure = measure;
				}
				
				x += itemSpan;
			}
			x = fromX;
			y += itemSpan;
			
			color.dispose();		
		}
		gc.setBackground(getDisplay().getSystemColor(SWT.COLOR_BLACK));
		
		this.trackMeasures.layout();
	}

	
	private int getSelectedTrackIndex(int y){
	    int itemSpan = table.getColumn(TABLE_COLUMN_EFFECT_CHANNEL).getItemHeight();	
		List tracks = editor.getTablature().getSongCoords().getTrackCoords();
		int trackLength = tracks.size();
		int trackY = 0;
		for(int i = 0;i < trackLength;i++){
			if(y >= trackY && y < (trackY + itemSpan)){
				return i;
			}
			trackY += itemSpan;
		}
		return 0;
	}
	
	private int getSelectedMeasureIndex(SongTrackCoords track,int x){
	    int itemSpan = table.getColumn(TABLE_COLUMN_EFFECT_CHANNEL).getItemHeight();		
		List measures = track.getMeasuresCoords();
		int measureLength = measures.size();
		int measureX = -hSroll.getSelection();;
		for(int i = 0;i < measureLength;i++){
			if(x >= measureX && x < (measureX + itemSpan)){
				return i;
			}
			measureX += itemSpan;
		}
		return 0;
	}
	
    public void updateScrolls(){
        int itemSpan = table.getColumn(TABLE_COLUMN_EFFECT_CHANNEL).getItemHeight();
        this.scrolledComposite.setMinSize(0, itemSpan + itemSpan * table.getColumn(TABLE_COLUMN_EFFECT_CHANNEL).getItemCount() + 10);
             
        int width = (editor.getTablature().getCaret().getSongTrackCoords().getMeasuresCoords().size() * itemSpan);
        this.hSroll.setMaximum(width);
        this.hSroll.setThumb(Math.min(width ,this.painter.getClientArea().width));
        
        this.hSroll.setVisible(width > this.painter.getClientArea().width);
    }
    
    
    public void loadProperties(){        
    	this.updateTable();    	
        this.table.getColumn(TABLE_COLUMN_NUMBER).setName(TuxGuitar.getProperty("track.number"));
	    this.table.getColumn(TABLE_COLUMN_NAME).setName(TuxGuitar.getProperty("track.name"));
	    this.table.getColumn(TABLE_COLUMN_INSTRUMENT).setName(TuxGuitar.getProperty("track.instrument"));	    
	    this.table.getColumn(TABLE_COLUMN_CHANNEL).setName(TuxGuitar.getProperty("track.channel.channel"));
	    this.table.getColumn(TABLE_COLUMN_EFFECT_CHANNEL).setName(TuxGuitar.getProperty("track.channel.effect-channel"));	    
    }
    
    public void fireUpdate(){
        updateTable();		
    }

    public int getHeight(){
    	return this.table.getBounds().height;
    }
    
    public void redraw(){     
        if(!super.isDisposed()){
        	if(editor.getTablature().getCaret().getSongTrackCoords().getTrack().getNumber() != lastTrackSelection){
        		updateItemBackgrounds();
        	}
        	painter.redraw();        	
            updateScrolls();
            super.redraw();
        }
   }
    
   public void redrawPlayingMode(){     
       if(!super.isDisposed()){	
       		if(editor.getTablature().getCaret().getSongTrackCoords().getTrack().getNumber() != lastTrackSelection){
       			updateItemBackgrounds();
       		}    	   
    	   if(this.lasPlayedMeasure == null || !this.lasPlayedMeasure.isPlaying()){
    		   painter.redraw();
    	   }
       }
   }
    
    
    private void updateChannelCombo(Combo channelCombo,ChannelList channelItems,int trackIndex){    	
    	channelCombo.removeAll();
    	channelItems.clear();
    	
    	SongChannel channel = ((SongTrack)TuxGuitar.instance().getSongManager().getSong().getTracks().get(trackIndex)).getChannel();
    	
        boolean[] usedChannels = TuxGuitar.instance().getSongManager().getUsedChannels();
        boolean[] usedEffectChannels = TuxGuitar.instance().getSongManager().getUsedEffectChannels();
        if(channel.isPercusionChannel()){
    		channelCombo.add(Integer.toString(SongChannel.DEFAULT_PERCUSION_CHANNEL));        		        		
    		channelCombo.select(0);        		
    		channelItems.addChannel(0,0);
    	}else{
    		int itemIndex = 0;        
    		for(int i = 0;i < usedChannels.length;i++){
    			if(i != SongChannel.DEFAULT_PERCUSION_CHANNEL){
    				if(!usedEffectChannels[i] || (usedEffectChannels[i] && usedChannels[i]) || (channel.getChannel() == i && channel.getEffectChannel() == channel.getChannel())){
    					String itemName = new String();
    					if(usedChannels[i] && (channel.getChannel() != i || TuxGuitar.instance().getSongManager().countTracksForChannel(i) > 1)){
    						itemName = i + " " + TuxGuitar.getProperty("track.channel.link");
    					}else{
    						itemName =  i + " " + TuxGuitar.getProperty("track.channel.free");
    					}
    					channelCombo.add(itemName);
        		
    					if(i == channel.getChannel()){
    						channelCombo.select(itemIndex);
    					}
        		
    					channelItems.addChannel(itemIndex,i);
    					itemIndex ++;
    				}
    			}
    		}
        }    	
        
    }
    
    private void updateEffectChannelCombo(Combo effectChannelCombo,ChannelList effectChannelItems,int trackIndex){    	
    	effectChannelCombo.removeAll();
    	effectChannelItems.clear();
    	
    	SongChannel channel = ((SongTrack)TuxGuitar.instance().getSongManager().getSong().getTracks().get(trackIndex)).getChannel();
    	
        boolean[] usedChannels = TuxGuitar.instance().getSongManager().getUsedChannels();
        boolean[] usedEffectChannels = TuxGuitar.instance().getSongManager().getUsedEffectChannels();
        if(channel.isPercusionChannel()){
        	effectChannelCombo.add(Integer.toString(SongChannel.DEFAULT_PERCUSION_CHANNEL));        		        		
        	effectChannelCombo.select(0);        		
        	effectChannelItems.addChannel(0,0);
    	}else{
        
    		int itemIndex = 0;
    		for(int i = 0;i < usedEffectChannels.length;i++){
    			if(i != SongChannel.DEFAULT_PERCUSION_CHANNEL){
    				if((!usedEffectChannels[i] || channel.getEffectChannel() == i) && (!usedChannels[i] || i == channel.getChannel())){        		
    					effectChannelCombo.add(Integer.toString(i));
        		
    					if(i == channel.getEffectChannel()){
    						effectChannelCombo.select(itemIndex);
    					}
    					effectChannelItems.addChannel(itemIndex,i);
    					itemIndex ++;
    				}
    			}
    		}    	
    	}
    }
    
    
	public void fireChannelChange(SongChannel channel){
		List tracks = TuxGuitar.instance().getSongManager().getSong().getTracks();
		for(int i = 0;i < tracks.size();i++){
			SongTrack track = (SongTrack)tracks.get(i);
			if(track.getChannel().getChannel() == channel.getChannel()){
				track.getChannel().setEffectChannel(channel.getEffectChannel());
				track.getChannel().setVolume(channel.getVolume());
				track.getChannel().setBalance(channel.getBalance());
				track.getChannel().setChorus(channel.getChorus());
				track.getChannel().setReverb(channel.getReverb());
				track.getChannel().setPhaser(channel.getPhaser());
				track.getChannel().setTremolo(channel.getTremolo());
				track.getChannel().setSolo(channel.isSolo());
				track.getChannel().setMute(channel.isMute());				
				
			}

		}
		updateTable();
		redraw();
	}
    
    
    private class ItemSelectionListener extends MouseAdapter{
        
        public void mouseDown(MouseEvent e) {
            TuxGuitar.instance().getAction(GoToTrackAction.NAME).process(e);            
        }
        
        public void mouseDoubleClick(MouseEvent e) {
            TuxGuitar.instance().getAction(ChangeTrackPropertiesAction.NAME).process(e);	            
        }
        
    }
    
    

    private class ChannelList{
    	private List channelIndexs;
    	
    	public ChannelList(){    
    		this.channelIndexs = new ArrayList();
    	}
    	
    	public void addChannel(int index,int channel){
    		this.channelIndexs.add(new ChannelIndex(index,channel));
    	}
    	
    	public int getChannel(int index){
    		Iterator it = this.channelIndexs.iterator();
    		while (it.hasNext()) {
				ChannelIndex channelIndex = (ChannelIndex) it.next();
				if(index == channelIndex.getIndex()){
					return channelIndex.getChannel();
				}
			}
    		return -1;
    	}
    	
    	public void clear(){
    		this.channelIndexs.clear();
    	}
    }
    
    private class ChannelIndex{
    	private int index;
    	private int channel;
    	
    	public ChannelIndex(int index,int channel){
    		this.index = index;
    		this.channel = channel;
    	}

		public int getChannel() {
			return channel;
		}

		public int getIndex() {
			return index;
		}
    	    	
    }
    
	public static void disposeColors(){
		for(int i = 0;i < BACKGROUNDS.length;i++){
			BACKGROUNDS[i].dispose();	
		}
		
	}
}
