package org.herac.tuxguitar.gui.printer;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.ScrollBar;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.herac.tuxguitar.gui.TuxGuitar;
import org.herac.tuxguitar.gui.keybindings.SpecialKeys;

public class PrintPreview{
	private Shell dialog;
	private PrintDocument document;
	private PrintPage currentPage;
	private Composite previewComposite;
	private Composite pageComposite;
	private Text currentText;
	private Button previous;
	private Button next;
	
	public PrintPreview(PrintDocument document){
		this.document = document;
	}
	
	public void showPreview(Shell parent){
		this.dialog = new Shell(parent,SWT.DIALOG_TRIM | SWT.APPLICATION_MODAL);
		this.dialog.setLayout(new GridLayout());
		this.dialog.setText(TuxGuitar.getProperty("print.preview"));
		
		this.initToolBar();
		this.initPreviewComposite();
		this.changePage(1);
		
		this.dialog.setMaximized(true);
		this.dialog.open();

		int x = parent.getBounds().x + (parent.getBounds().width - dialog.getSize().x) / 2;
		int y = parent.getBounds().y + (parent.getBounds().height - dialog.getSize().y) / 2;
		this.dialog.setLocation(x, y);		
		
		while (!dialog.isDisposed()) {
            if (!dialog.getDisplay().readAndDispatch()) {
            	dialog.getDisplay().sleep();
            }

        }				
	}
	
	private void initToolBar(){
		Composite composite = new Composite(this.dialog,SWT.NONE);
		composite.setLayout(new GridLayout(5,false));
		composite.setLayoutData(new GridData(SWT.FILL,SWT.FILL,true,false));		
		
		this.previous = new Button(composite,SWT.ARROW | SWT.LEFT);
		this.currentText = new Text(composite,SWT.BORDER);
		this.currentText.setLayoutData(new GridData(20,10));
		this.next = new Button(composite,SWT.ARROW | SWT.RIGHT);
		Label maxPages = new Label(composite,SWT.NONE);
		            
        GridData data = new GridData(SWT.RIGHT,SWT.FILL,true,true);    
        data.minimumWidth = 80;
        data.minimumHeight = 25;    
        
		Button exit = new Button(composite,SWT.PUSH);
		exit.setLayoutData(data);
		
		this.currentText.addKeyListener(new KeyAdapter() {
			public void keyReleased(KeyEvent e) {
				if(e.keyCode == SpecialKeys.ENTER){
					try{
						Integer number = new Integer(currentText.getText());
						changePage(number.intValue());
					}catch(NumberFormatException exception){
						changePage(currentPage.getPageNumber() + 1);
					}
				}
			}				
		});
		this.previous.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if(currentPage != null){
					changePage(currentPage.getPageNumber());
				}
			}		
		});
		this.next.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if(currentPage != null){
					changePage(currentPage.getPageNumber() + 2);
				}
			}		
		});
		exit.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				dialog.dispose();
			}		
		});
		maxPages.setText("To " + this.document.getPages().size());
		exit.setText("Close");
	}
	
	private void initPreviewComposite(){
		this.previewComposite = new Composite(this.dialog,SWT.BORDER | SWT.V_SCROLL);
		this.previewComposite.setLayout(new GridLayout());
		this.previewComposite.setLayoutData(new GridData(SWT.FILL,SWT.FILL,true,true));
		this.previewComposite.setBackground(this.previewComposite.getDisplay().getSystemColor(SWT.COLOR_GRAY));

		this.pageComposite = new Composite(this.previewComposite,SWT.BORDER | SWT.DOUBLE_BUFFERED);
		this.pageComposite.setLayout(new GridLayout());
		this.pageComposite.setBackground(this.previewComposite.getDisplay().getSystemColor(SWT.COLOR_WHITE));
		this.pageComposite.addPaintListener(new PaintListener() {		
			public void paintControl(PaintEvent e) {
				if(currentPage != null){
					int vScroll = previewComposite.getVerticalBar().getSelection();
					Image image = new Image(dialog.getDisplay(),currentPage.getData());					
					e.gc.drawImage(image,50,20 - vScroll);
					
					updateScroll();
					image.dispose();
				}
			}		
		});
		GridData pageData = new GridData();
		pageData.horizontalAlignment = SWT.CENTER;
		pageData.verticalAlignment = SWT.CENTER;
		pageData.grabExcessHorizontalSpace = true;
		pageData.grabExcessVerticalSpace = true;
		pageData.widthHint = this.document.getPageSize().width + 70;
		pageData.heightHint = this.document.getPageSize().height;
		this.pageComposite.setLayoutData(pageData);
				        
        this.previewComposite.getVerticalBar().addListener(SWT.Selection, new Listener() {
          public void handleEvent(Event e) {
            pageComposite.redraw();
          }
        });            
	}
	
	private void updateScroll(){
		ScrollBar vBar = this.previewComposite.getVerticalBar(); 
		Rectangle bounds = pageComposite.getBounds();
		Rectangle client = pageComposite.getClientArea();        
		vBar.setMaximum(document.getPageSize().height);
		vBar.setThumb(Math.min(document.getPageSize().height, client.height));      
	}

	private void changePage(int number){
		if(this.document != null){
			int pageCount = this.document.getPages().size();			
			if(number > 0 && number <= pageCount){			
				this.currentPage = (PrintPage)this.document.getPages().get((number - 1));
				this.currentText.setText(Integer.toString(number));
				this.pageComposite.redraw();		
			}else if(this.currentPage != null){
				this.currentText.setText(Integer.toString(this.currentPage.getPageNumber() + 1));
			}
			
			this.previous.setEnabled(this.currentPage.getPageNumber() > 0);
			this.next.setEnabled((this.currentPage.getPageNumber() + 1)< pageCount);
		}else{
			this.currentText.setEnabled(false);
			this.previous.setEnabled(false);
			this.next.setEnabled(false);
		}
		
	}
}
