package org.herac.tuxguitar.gui.keybindings;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.herac.tuxguitar.gui.TuxGuitar;
import org.herac.tuxguitar.gui.util.MessageDialog;
import org.herac.tuxguitar.gui.util.ConfirmDialog;

public class KeyBindingEditor {
	private Shell dialog;
	private List availableActions;
	private List actionKeyBindings;
	private Table table;
	private Button remove;
	private Button save;
	private boolean hasChanges;
	
	public KeyBindingEditor(List availableActions){
		this.availableActions = availableActions;
		this.hasChanges = false;
		this.makeActionKeyBindings();
	}
	
	public void show(Shell parent){
		this.dialog = new Shell(parent,SWT.DIALOG_TRIM |SWT.APPLICATION_MODAL);

		this.dialog.setText(TuxGuitar.getProperty("key-bindings-editor"));
		this.dialog.setLayout(new GridLayout());
    
		this.initCombo(this.dialog);
		this.initTable(this.dialog);
		this.notifyAction(false);
					
		this.dialog.pack();
		this.dialog.open();

		int x = parent.getBounds().x + (parent.getBounds().width - dialog.getSize().x) / 2;
		int y = parent.getBounds().y + (parent.getBounds().height - dialog.getSize().y) / 2;
		this.dialog.setLocation(x, y);
	}
	
	public void initCombo(Composite parent){
		Composite composite = new Composite(parent,SWT.NONE);
		composite.setLayout(new GridLayout(2,false));
		composite.setLayoutData(new GridData(SWT.FILL,SWT.FILL,true,true));   
		
		final Combo combo = new Combo(composite,SWT.NONE);
		combo.add(TuxGuitar.getProperty("key-bindings-editor-action-select"));
		combo.select(0);
		for(int i = 0;i < this.availableActions.size();i++){
			String actionName  = (String)this.availableActions.get(i);
			combo.add(getActionDescription(actionName));
		}

        GridData data = new GridData(SWT.FILL,SWT.FILL,true,true);    
        data.minimumWidth = 80;
        data.minimumHeight = 25;   		
		
		final Button button = new Button(composite,SWT.PUSH);
		button.setLayoutData(data);
		button.setText(TuxGuitar.getProperty("add"));		
		button.setEnabled(false);
		
		
		combo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				button.setEnabled(combo.getSelectionIndex() > 0);
			}		
		});
		
		button.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if(combo.getSelectionIndex() > 0){
					KeyBinding keyBinding = new KeySelector().select(dialog.getShell());
					if(keyBinding != null){
						String actionName = (String)availableActions.get(combo.getSelectionIndex() - 1);
						actionKeyBindings.add(new ActionKeyBinding(actionName,keyBinding));					
						notifyAction(true);
					}
				}
				combo.select(0);
				button.setEnabled(false);
			}		
		});
	}
	
	public void initTable(Composite parent){
		Composite tableComposite = new Composite(parent,SWT.NONE);
		tableComposite.setLayout(new GridLayout());
		tableComposite.setLayoutData(new GridData(SWT.FILL,SWT.FILL,true,true));   
        
	    this.table = new Table(tableComposite, SWT.BORDER);
	    this.table.setLayoutData(new GridData(350,200));
	    
	    TableColumn actionColumn = new TableColumn(table, SWT.LEFT);
	    TableColumn shortcutColumn = new TableColumn(table, SWT.LEFT);
	    actionColumn.setText(TuxGuitar.getProperty("key-bindings-editor-action-column"));
	    shortcutColumn.setText(TuxGuitar.getProperty("key-bindings-editor-shortcut-column"));
	    actionColumn.setWidth(250);
	    shortcutColumn.setWidth(100);
	    table.setHeaderVisible(true);

	    loadTableItems();
	
	    
        Composite buttons = new Composite(parent, SWT.NONE);
        buttons.setLayout(new GridLayout(3,false));
        buttons.setLayoutData(new GridData(SWT.END,SWT.FILL,true,true));    	
        
        GridData data = new GridData(SWT.FILL,SWT.FILL,true,true);    
        data.minimumWidth = 80;
        data.minimumHeight = 25;    
	    
	    this.remove = new Button(buttons,SWT.PUSH);
	    this.save = new Button(buttons,SWT.PUSH);
	    Button exit = new Button(buttons,SWT.PUSH);
	    
	    remove.setText(TuxGuitar.getProperty("remove"));	    
	    save.setText(TuxGuitar.getProperty("save"));	
	    exit.setText(TuxGuitar.getProperty("exit"));	    	  
	    
	    remove.setLayoutData(data);
	    save.setLayoutData(data);
	    exit.setLayoutData(data);
	    
	    table.addMouseListener(new MouseAdapter() {
			public void mouseUp(MouseEvent e) {
				notifyAction(false);			
			}		    		    
			public void mouseDoubleClick(MouseEvent e) {
				TableItem item = getSelectedItem();
				if(item != null){	
					ActionKeyBinding actionKeyBinding = (ActionKeyBinding)item.getData();		
					new KeySelector(actionKeyBinding.getKeyBinding()).select(dialog.getShell());
					notifyAction(true);
				}				
			}		
		});

	    
	    remove.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				TableItem item = getSelectedItem();
				if(item != null){
					ActionKeyBinding actionKeyBinding = (ActionKeyBinding)item.getData();	
					actionKeyBindings.remove(actionKeyBinding);
					notifyAction(true);
				}				
			}		
		});
	    save.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				TuxGuitar.instance().getkeyBindingManager().reset(actionKeyBindingsToMap());
				TuxGuitar.instance().getkeyBindingManager().saveKeyBindings();
				removeChanges();
				notifyAction(false);
			}		
		});	    
	    exit.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if(hasChanges){
					if(!new ConfirmDialog(table.getShell(),TuxGuitar.getProperty("key-bindings-editor-save-question")).confirm()){
						return;
					}
				}				
				dialog.dispose();
			}		
		});
	}

	private void clearTableItems(){
		TableItem[] items = this.table.getItems();
		for(int i = 0;i < items.length;i++){
			items[i].dispose();
		}
	}
	
	private void loadTableItems(){
		clearTableItems();
		Iterator it = actionKeyBindings.iterator();
		while (it.hasNext()) {
			ActionKeyBinding actionkeyBinding = (ActionKeyBinding) it.next();
			
			TableItem item = new TableItem(this.table, SWT.NONE);
			
			String actionName = actionkeyBinding.getActionName();
			String shortcut = actionkeyBinding.getKeyBinding().toString();
			
			item.setText(new String[] { getActionDescription(actionName),shortcut });
			item.setData(actionkeyBinding);	
			
			
		}				
	}
	
	private void removeChanges(){
		this.hasChanges = false;
	}
	
	private void notifyAction(boolean hasChanges){
		if(hasChanges){
			this.hasChanges = true;
			this.loadTableItems();
		}
		this.remove.setEnabled(getSelectedItem() != null);
		this.save.setEnabled(this.hasChanges);
	}

	
	private TableItem getSelectedItem(){
		TableItem item = null;		
		int itemSelected = table.getSelectionIndex();
		if(itemSelected >= 0){
			item = table.getItem(itemSelected);
		}		
		return item;
	}
	
	private KeyBinding getKeyBinding(KeyBinding kb){
		Iterator it = this.actionKeyBindings.iterator();
		while(it.hasNext()){
			ActionKeyBinding actionKeyBinding = (ActionKeyBinding)it.next();
			if(kb.equals(actionKeyBinding.getKeyBinding())){
				return actionKeyBinding.getKeyBinding();
			}
		}
		return null;
	}
	
	private void remove(KeyBinding kb){
		Iterator it = this.actionKeyBindings.iterator();
		while(it.hasNext()){
			ActionKeyBinding actionKeyBinding = (ActionKeyBinding)it.next();
			if(kb.equals(actionKeyBinding.getKeyBinding())){
				this.actionKeyBindings.remove(actionKeyBinding);
				break;
			}
		}		
	}
	
	private String getActionDescription(String actionName){
		return TuxGuitar.getProperty(actionName + "-description");
	}
	
	private void makeActionKeyBindings(){
		this.actionKeyBindings = new ArrayList();
		
		Iterator it = TuxGuitar.instance().getkeyBindingManager().getKeyBindings().iterator();
		while (it.hasNext()) {
			KeyBinding keyBinding = (KeyBinding)((KeyBinding) it.next()).clone();			
			String actionName = TuxGuitar.instance().getkeyBindingManager().getActionForKeyBinding(keyBinding).getName();
			
			this.actionKeyBindings.add(new ActionKeyBinding(actionName,keyBinding));
		}
	}
	
	private Map actionKeyBindingsToMap(){
		Map map = new HashMap();
		Iterator it = actionKeyBindings.iterator();
		while (it.hasNext()) {
			ActionKeyBinding actionkeyBinding = (ActionKeyBinding) it.next();			
			map.put(actionkeyBinding.getKeyBinding(),TuxGuitar.instance().getAction(actionkeyBinding.getActionName()));
		}
		return map;
	}
		
	
	private class KeySelector{				
		private KeyBinding keyBinding;
				
		public KeySelector(KeyBinding keyBinding){
			this.keyBinding = keyBinding;
		}
		
		public KeySelector(){
			this(null);
		}
		
		public KeyBinding select(Shell parent){			
			final Shell dialog = new Shell(parent,SWT.APPLICATION_MODAL | SWT.DIALOG_TRIM);
			dialog.setLayout(new FillLayout());
			
			Composite composite = new Composite(dialog,SWT.BORDER);
			composite.setLayout(new FillLayout());
			
			Label label = new Label(composite,SWT.NONE);
			label.setFont(new Font(dialog.getDisplay(),"Sans", 15, SWT.BOLD));
			label.setText(TuxGuitar.getProperty("key-bindings-editor-push-a-key"));
			
			composite.addKeyListener(new KeyAdapter() {			
				public void keyReleased(KeyEvent e) {					
					KeyBinding kb = new KeyBinding(e.keyCode,e.stateMask);
					if(KeyBindingReserveds.isReserved(kb)){
						String title = TuxGuitar.getProperty("key-bindings-editor-reserved-title");
						String message = TuxGuitar.getProperty("key-bindings-editor-reserved-message");
						new MessageDialog(title,message,SWT.NONE).show(dialog.getShell());
						return;
					}
					//verifico un existente
					KeyBinding oldKeyBinding = getKeyBinding(kb);									
					if(oldKeyBinding != null){
						if(keyBinding == null || !keyBinding.equals(oldKeyBinding)){
							if(!new ConfirmDialog(dialog.getShell(),TuxGuitar.getProperty("key-bindings-editor-override")).confirm()){
								return;
							}
							remove(kb);
						}
					}						
					
					if(keyBinding == null){
						keyBinding = new KeyBinding();
					}						
					keyBinding.setKey(kb.getKey());
					keyBinding.setMask(kb.getMask());						
					dialog.dispose();
				}
					
										
						
			});
			
            dialog.pack();
            dialog.open();

            int x = parent.getBounds().x + (parent.getBounds().width - dialog.getSize().x) / 2;
            int y = parent.getBounds().y + (parent.getBounds().height - dialog.getSize().y) / 2;
            dialog.setLocation(x, y);
            
    		while (!dialog.isDisposed()) {
    			if (!dialog.getDisplay().readAndDispatch()) {
    				dialog.getDisplay().sleep();
    			}
    		}		
            
            return this.keyBinding;
		}
		
		
		
		
	}
	
	
	private class ActionKeyBinding{
		private String actionName;
		private KeyBinding keyBinding;
		
		public ActionKeyBinding(String actionName,KeyBinding keyBinding){
			this.actionName = actionName;
			this.keyBinding = keyBinding;
		}

		public String getActionName() {
			return actionName;
		}

		public KeyBinding getKeyBinding() {
			return keyBinding;
		}
		
	}

}
