package org.herac.tuxguitar.song.managers;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.herac.tuxguitar.song.models.Measure;
import org.herac.tuxguitar.song.models.SongChannel;
import org.herac.tuxguitar.song.models.SongTrack;
import org.herac.tuxguitar.song.models.Tempo;
import org.herac.tuxguitar.song.models.TimeSignature;
import org.herac.tuxguitar.song.models.TrackColor;

public class SongTrackManager {
	private SongManager songManager;
	
	public SongTrackManager(SongManager songManager){
		this.songManager = songManager;
	}
	
	
	
    public int countMeasures(SongTrack track){
    	return (track.getMeasures().size());
    }
	
	   
    public Measure getFirstMeasure(SongTrack track){
        Measure firstMeasure = null;    
        for(int i = 0;i < track.getMeasures().size();i++){
            Measure currMeasure = (Measure)track.getMeasures().get(i);
            if(firstMeasure == null || (currMeasure.getStart() < firstMeasure.getStart())){
                firstMeasure = currMeasure;
            }
        }        
        return firstMeasure;
    }     

    public Measure getLastMeasure(SongTrack track){
        int lastIndex = track.getMeasures().size() - 1; 
        return (Measure)track.getMeasures().get(lastIndex);
    }      
    
    public Measure getPrevMeasure(SongTrack track,Measure measure){
        int prevIndex = measure.getNumber() - 1; 
        if(prevIndex > 0){
            return (Measure)track.getMeasures().get(prevIndex - 1);
        }
        return null;
    }      
    
    public Measure getNextMeasure(SongTrack track,Measure measure){
        int nextIndex = measure.getNumber(); 
        if(nextIndex < track.getMeasures().size()){
            return (Measure)track.getMeasures().get(nextIndex);
        }
        return null;
    }       
  
    public Measure getMeasureAt(SongTrack track,long start){
        Iterator it = track.getMeasures().iterator();
        while(it.hasNext()){
            Measure measure = (Measure)it.next();  
            long measureStart = measure.getStart();
            long measureLength = measure.getLength();
            if(start >= measureStart && start < measureStart + measureLength){
                return measure;
            }
        }     
        return null;
    }           
    
    public Measure getMeasure(SongTrack track,int number){
        for (int i = 0; i < track.getMeasures().size(); i++) {
            Measure measure = (Measure) track.getMeasures().get(i);            
            if(measure.getNumber() == number){
                return measure;
            }
        }     
        return null;
    }   
    
    /**
     * Retorna Todos los desde Start hasta el final del compas
     */
    public List getMeasuresBeforeEnd(SongTrack track,long fromStart) {
        List measures = new ArrayList();        
        Iterator it = track.getMeasures().iterator();
        while(it.hasNext()){           
            Measure currMeasure = (Measure)it.next();
            if (currMeasure.getStart() >= fromStart) {
                measures.add(currMeasure);
            }
        }
        return measures;
    }         
    
    /**
     * Retorna Todos los desde Start hasta el final del compas
     */
    public List getMeasuresBetween(SongTrack track,long p1,long p2) {
        List measures = new ArrayList();        
        Iterator it = track.getMeasures().iterator();
        while(it.hasNext()){           
            Measure currMeasure = (Measure)it.next();
            long start = currMeasure.getStart();
            if (start >= p1 && start <= p2) {
                measures.add(currMeasure);
            }
        }
        return measures;
    }          
    
    public void addNewMeasureBeforeEnd(SongTrack track){        
        addNewMeasureAfter(track,getLastMeasure(track));
    }    
    

    

    /**
     * Agrega un Compas
     */
    public void addNewMeasureAfter(SongTrack track,Measure measure){
    	long measureStart = measure.getStart();
        long measureLength = measure.getLength();
        List measuresBeforEnd = getMeasuresBeforeEnd(track,measureStart + 1);        
        Iterator it = measuresBeforEnd.iterator();
        while(it.hasNext()){
        	Measure currMeasure = (Measure)it.next();
        	moveMeasure(track,currMeasure,measureLength,1);
        }

        TimeSignature timeSignature = (TimeSignature)measure.getTimeSignature().clone();   
        Tempo tempo = (Tempo)measure.getTempo().clone(); 
        int number = (measure.getNumber() + 1);
        long start = measureStart + measureLength;
        boolean repeatStart = false;
        int nombreOfRepetitions = 0;    
        
        addMeasure(track,new Measure(number,start, new ArrayList(), new ArrayList(), timeSignature,tempo,repeatStart,nombreOfRepetitions));
    }
        
    
    public List copyMeasures(SongTrack track,long p1,long p2){
        List measures = new ArrayList();
        
        Iterator it = getMeasuresBetween(track,p1,p2).iterator();
        while(it.hasNext()){
            Measure measure = (Measure)it.next();
            measures.add(measure.clone());
        }
        return measures;
    }
    
    /**
     * Agrega un Compas
     */
    public void insertMeasures(SongTrack track,List measures,int fromNumber){
        long start = -1;
        long end = 0;
        Iterator it = null;
        
        if(!measures.isEmpty()){
            start = ((Measure)measures.get(0)).getStart(); 
            end = ((Measure)measures.get(measures.size() - 1)).getStart() + ((Measure)measures.get(measures.size() - 1)).getLength();
            
            it = getMeasuresBeforeEnd(track,start).iterator();
            while(it.hasNext()){
                Measure measure = (Measure)it.next();
                moveMeasure(track,measure,end - start,measures.size());            
            }

            it = measures.iterator();
            while(it.hasNext()){
                Measure measure = (Measure)it.next();
                measure.setNumber(fromNumber);
                addMeasure(track,(measure.getNumber() - 1),measure);
                fromNumber ++;
            }        
        }
        songManager.calculateMeasureStartWidthRepetitions();
    }    
        
    
    /**
     * Agrega un Compas
     */
    public void addMeasure(SongTrack track,Measure measure){
        track.getMeasures().add(measure);
    }    
    
    /**
     * Agrega un Compas
     */
    public void addMeasure(SongTrack track,int index,Measure measure){
        track.getMeasures().add(index,measure);
    }        
    
    public void removeMeasures(SongTrack track,long p1,long p2){
        List measures = new ArrayList();
        
        Iterator it = getMeasuresBetween(track,p1,p2).iterator();
        while(it.hasNext()){
            Measure measure = (Measure)it.next();
            removeMeasure(track,measure);
        }        
    }    
    
    public void removeMeasure(SongTrack track,long start){    
        removeMeasure(track,getMeasureAt(track,start));
    }
    
    public void removeMeasure(SongTrack track,Measure measure){        
        long start = measure.getStart();
        long length = measure.getLength();
        
        List nextMeasures = getMeasuresBeforeEnd(track,start + 1);
        Iterator it = nextMeasures.iterator();
        while(it.hasNext()){
            Measure currMeasure = (Measure)it.next();
            moveMeasure(track,currMeasure,-length,-1);            
        }                
        track.getMeasures().remove(measure);  
    }    
    
    public Measure replaceMeasure(SongTrack track,Measure newMeasure){    	
    	Measure measure = getMeasureAt(track,newMeasure.getStart());
    	measure.makeEqual(newMeasure);
    	return measure;
    }
    /**
     * Mueve el compas
     */
    public void moveMeasure(SongTrack track,Measure measure,long theMove,int numberMove){        
        
        songManager.getMeasureManager().moveAllNotes(measure,theMove);
        songManager.getMeasureManager().moveAllSilences(measure,theMove);
        measure.setNumber(measure.getNumber() + numberMove);
        measure.setStart(measure.getStart() + theMove);
    }
    
    
    
    public void changeTimeSignature(SongTrack track,long start,TimeSignature timeSignature,boolean toEnd){
        changeTimeSignature(track,getMeasureAt(track,start),timeSignature,toEnd);
    }
    
    /**
     * Cambia el Ritmo
     */
    public void changeTimeSignature(SongTrack track,Measure measure,TimeSignature timeSignature,boolean toEnd){        
        //asigno el nuevo ritmo
        measure.setTimeSignature((TimeSignature)timeSignature.clone());         

        long nextStart = measure.getStart() + measure.getLength();
        List measures = getMeasuresBeforeEnd(track,measure.getStart() + 1);
        Iterator it = measures.iterator();
        while(it.hasNext()){
            Measure nextMeasure = (Measure)it.next();
            
            long theMove = nextStart - nextMeasure.getStart();            
            moveMeasure(track,nextMeasure,theMove,0);
            
            if(toEnd){           
                nextMeasure.setTimeSignature((TimeSignature)timeSignature.clone());  
            }
            
            nextStart = nextMeasure.getStart() + nextMeasure.getLength();
        }
                  
    }

    public void changeTempo(SongTrack track,long start,Tempo tempo,boolean toEnd){
        changeTempo(track,getMeasureAt(track,start),tempo,toEnd);
    }    
    
    public void changeTempo(SongTrack track,Measure measure,Tempo tempo,boolean toEnd){        
        //asigno el nuevo tempo
        measure.setTempo((Tempo)tempo.clone());        

        if(toEnd){
            List measures = getMeasuresBeforeEnd(track,measure.getStart() + 1);
            Iterator it = measures.iterator();
            while(it.hasNext()){
                Measure nextMeasure = (Measure)it.next();
                nextMeasure.setTempo((Tempo)tempo.clone()); 
            }            
        }
        
    }    
    
    public void changeOpenRepeat(SongTrack track,long start){
        Measure measure = getMeasureAt(track,start);
        measure.setRepeatStart(!measure.isRepeatStart());
    }    

    public void changeCloseRepeat(SongTrack track,long start,int numberOfRepetitions){
        Measure measure = getMeasureAt(track,start);
        measure.setNumberOfRepetitions(numberOfRepetitions);
    }
    
    public void changeInfo(SongTrack track,String name,TrackColor color){         
        track.setName(name);
        track.setColor(color);
    }
    
    
    public void changeInstrumentStrings(SongTrack track,List strings){
        if(strings.size() < track.getStrings().size()){
            removeNotesAfterString(track,strings.size());            
        }                
        track.setStrings(strings);
    }
    
    public void removeNotesAfterString(SongTrack track,int string){
        Iterator it = track.getMeasures().iterator();
        while(it.hasNext()){
            Measure measure = (Measure)it.next();
            
            songManager.getMeasureManager().removeNotesAfterString(measure,string);
            //nextMeasure.removeNotesAfterString(string);
        }          
    }
    
    public void changeInstrument(SongTrack track,int instrument,boolean percusion){
    	track.getChannel().setInstrument((short)instrument);
        if(percusion){
        	track.getChannel().setChannel(SongChannel.DEFAULT_PERCUSION_CHANNEL);
        	track.getChannel().setEffectChannel(SongChannel.DEFAULT_PERCUSION_CHANNEL);        	
        	track.setStrings(SongManager.createPercusionStrings(track.getStrings().size()));
        }else{
        	if(track.getChannel().isPercusionChannel()){
        		SongChannel tempChannel = this.songManager.getFreeChannel((short)instrument,false);
        		track.getChannel().setChannel(tempChannel.getChannel());
        		track.getChannel().setEffectChannel(tempChannel.getEffectChannel()); 
        	}
        }
        this.songManager.updateChannel(track.getChannel());
    }
    
    
    public List getCloneMeasures(List measures) {
        List cloneMeasures = new ArrayList();

        Iterator it = measures.iterator();
        while (it.hasNext()) {
            Measure measure = (Measure) it.next();
            cloneMeasures.add(measure.clone());
        }
        return cloneMeasures;
    }
    
    
    /**
     * Retorna true si es el primer compas
     */         
    public boolean isFirstMeasure(Measure measure){
        return (measure.getNumber() == 1);
    }

    /**
     * Retorna true si es el ultimo compas
     */         
    public boolean isLastMeasure(SongTrack track,Measure measure){
        return (countMeasures(track) == measure.getNumber());
    }
}
