package org.herac.tuxguitar.song.managers;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.herac.tuxguitar.gui.tab.TablatureUtil;
import org.herac.tuxguitar.song.models.BendEffect;
import org.herac.tuxguitar.song.models.Component;
import org.herac.tuxguitar.song.models.Duration;
import org.herac.tuxguitar.song.models.Measure;
import org.herac.tuxguitar.song.models.Note;
import org.herac.tuxguitar.song.models.Silence;
import org.herac.tuxguitar.song.models.Tupleto;

public class MeasureManager {
	private SongManager songManager;
	
	public MeasureManager(SongManager songManager){
		this.songManager = songManager;
	}
	
    public void orderNotes(Measure measure){
        for(int i = 0;i < measure.getNotes().size();i++){
            Note minNote = null;
            for(int noteIdx = i;noteIdx < measure.getNotes().size();noteIdx++){
                Note note = (Note)measure.getNotes().get(noteIdx);
                if(minNote == null || note.getStart() < minNote.getStart()){
                    minNote = note;
                }
            }
            measure.getNotes().remove(minNote);
            measure.getNotes().add(i,minNote);
        }
    }
	
    public void addNote(Measure measure,Note note){                          
        //Verifico si entra en el compas
        if(canInsert(measure,note,true,false)){
            //Borro lo que haya en la misma posicion
            removeComponentsAt(measure,note.getStart(),note.getString(),false);            
            
            //Agrego la nota
            measure.addNote(note);            
            
            //trato de agregar un silencio similar al lado
            tryChangeSilenceAfter(measure,note);
        }                
    }
	
	
	public void removeNote(Measure measure,Note note){
		measure.removeNote(note);
	}

    /**
     * Agrega un silencio al compas
     */
    public void addSilence(Measure measure,Silence silence){                
        //Verifico si entra en el compas
        if(canInsert(measure,silence,true,false)){
            //Borro lo que haya en la misma posicion
            removeAllComponentsAt(measure,silence.getStart());             
            
            //Agrego el silencio
            measure.addSilence(silence);           
        }
    }  
	
	public void removeSilence(Measure measure,Silence silence){
		measure.removeSilence(silence);
	}
	
    /**
     * Elimina un silencio del compas.
     * si se asigna moveNextComponents = true. los componentes que le siguen 
     * se moveran para completar el espacio vacio que dejo el silencio
     */         
    public void removeSilence(Measure measure,Silence silence,boolean moveNextComponents){
        //TODO sacar el calculo de length con el nextComponent cuando en autosilences se agreguen tresillos
        List components = getComponents(measure);
    	Component component = getComponent(components,silence.getStart());
        Component nextComponent = getNextComponent(components,component);
        
        
        removeSilence(measure,silence);
        if(moveNextComponents){
            long start = silence.getStart();
            long length = silence.getDuration().getTime();
            
            if(nextComponent != null){
                length = nextComponent.getStart() - start;                
            }
            
            moveComponents(measure,start + length,-length);
        }
    }   
	
    public void removeNotesAfterString(Measure measure,int string){  
    	List notesToRemove = new ArrayList();
    	Iterator it = measure.getNotes().iterator();
        while(it.hasNext()){
            Note note = (Note)it.next();            
            if(note.getString() > string){
            	notesToRemove.add(note);            	
            }            
        } 
        it = notesToRemove.iterator();
        while(it.hasNext()){
            Note note = (Note)it.next();            
            removeNote(measure,note);        
        } 
    }
    
    /**
     * Retorna Todas las Notas en la posicion Start
     */
    public List getNotes(Measure measure,long start) {
        List notes = new ArrayList();        
        Iterator it = measure.getNotes().iterator();
        while(it.hasNext()){           
            Note note = (Note)it.next();
            if (note.getStart() == start) {
                notes.add(note);
            }
        }
        return notes;
    }    
    
    /**
     * Retorna las Nota en la posicion y cuerda
     */
    public Note getNote(Measure measure,long start,int string) {        
        Iterator it = measure.getNotes().iterator();
        while(it.hasNext()){           
            Note note = (Note)it.next();
            if (note.getStart() == start && note.getString() == string) {
                return note;
            }
        }
        return null;
    }    
    
    /**
     * Mueve todas las notas
     */
    public void moveAllNotes(Measure measure,long theMove){
    	moveComponents(measure.getNotes(),theMove);
    }  
    
    /**
     * Mueve todos los silencios
     */
    public void moveAllSilences(Measure measure,long theMove){
    	moveComponents(measure.getSilences(),theMove);
    }  
    
    /**
     * Mueve todos los componentes
     */
    public void moveAllComponents(Measure measure,long theMove){
    	moveComponents(getComponents(measure),theMove);
    } 
    
    
    /**
     * Mueve los componentes ubicados en start
     */    
    public boolean moveComponents(Measure measure,long start,long theMove){
        //obtengo los datos del compas
        long measureStart = measure.getStart();
        long measureLength = measure.getLength();
        
        //muevo los componentes
        List components = getComponentsBeforeEnd(getComponents(measure),start);
        moveComponents(components,theMove);
        
        //si el compas no quedo correctamente vuelvo a dejar todo como estaba
        Component first = getFirstComponent(getComponents(measure));
        while(first instanceof Silence){
            removeSilence(measure,(Silence)first);
            first = getNextComponent(getComponents(measure),first);
        }
        Component last = getLastComponent(getComponents(measure));
        while(last instanceof Silence){
            removeSilence(measure,(Silence)last);
            last = getPreviousComponent(getComponents(measure),last);
        }
        if(first != null && last != null){
            if(first.getStart() < measureStart || (last.getStart() + last.getDuration().getTime()) > (measureStart + measureLength)){
                moveComponents(components,-theMove);
                return false;
            }
        }
        return true;
    }       
    
    /**
     * Mueve los componentes
     */    
    private void moveComponents(List components,long theMove){
        Iterator it = components.iterator();
        while(it.hasNext()){
            Component component = (Component)it.next();
            moveComponent(component,theMove);             
        }       
    }
    
    /**
     * Mueve el componente
     */    
    private void moveComponent(Component component,long theMove){
        //obtengo el start viejo
        long start = component.getStart();
        
        //asigno el nuevo start
        component.setStart(start + theMove);                                   
    }
   
    /**
     * Retorna los silencios en la posicion start
     */
    public List getSilences(Measure measure,long start) {
        List silences = new ArrayList();
        for (int i = 0; i < measure.getSilences().size(); i++) {
            Silence silence = (Silence) measure.getSilences().get(i);            
            if (silence.getStart() == start) {
            	silences.add(silence);
            }
        }
        return silences;
    }  
    
    /**
     * Retorna el Siguiente Componente que sea de un silencio
     */
    public Silence getNextSilence(Measure measure,Component component) {
    	Silence nextSilence = null;
        for (int i = 0; i < measure.getSilences().size(); i++) {
        	Silence currSilence = (Silence) measure.getSilences().get(i);
            
            if (currSilence.getStart() > component.getStart()) {
                if (nextSilence == null) {
                	nextSilence = currSilence;
                } else if (currSilence.getStart() < nextSilence.getStart()) {
                	nextSilence = currSilence;
                } else if (currSilence.getStart() == nextSilence.getStart()
                        && currSilence.getDuration().getTime() <= nextSilence.getDuration().getTime()) {
                	nextSilence = currSilence;
                }
            }
            
        }
        return nextSilence;
    }  
    
    public List getComponents(Measure measure){
        List components = new ArrayList();
        components.addAll(measure.getNotes());
        components.addAll(measure.getSilences());
        return components;
    }
    
    /**
     * Retorna el Siguiente Componente
     */
    public Component getNextComponent(List components,Component component) {
        Component nextComponent = null;
        for (int noteIdx = 0; noteIdx < components.size(); noteIdx++) {
            Component currComponent = (Component) components.get(noteIdx);
            
            if (currComponent.getStart() > component.getStart()) {
                if (nextComponent == null) {
                    nextComponent = currComponent;
                } else if (currComponent.getStart() < nextComponent.getStart()) {
                    nextComponent = currComponent;
                } else if (currComponent.getStart() == nextComponent.getStart()
                        && currComponent.getDuration().getTime() <= nextComponent.getDuration().getTime()) {
                    nextComponent = currComponent;
                }
            }
        }
        return nextComponent;
    }
    
    /**
     * Retorna el Componente Anterior
     */
    public Component getPreviousComponent(List components,Component component) {
        Component prevComponent = null;
        for (int noteIdx = 0; noteIdx < components.size(); noteIdx++) {
            Component currComponent = (Component) components.get(noteIdx);

            if (currComponent.getStart() < component.getStart()) {
                if (prevComponent == null) {
                    prevComponent = currComponent;
                } else if (currComponent.getStart() > prevComponent.getStart()) {
                    prevComponent = currComponent;
                } else if (currComponent.getStart() == prevComponent.getStart()
                        && currComponent.getDuration().getTime() <= prevComponent.getDuration().getTime()) {
                    prevComponent = currComponent;
                }
            }
        }
        return prevComponent;
    }
    
    /**
     * Retorna el Primer Componente
     */
    public Component getFirstComponent(List components) {
        Component firstComponent = null;
        for (int i = 0; i < components.size(); i++) {
            Component currComponent = (Component) components.get(i);            
            if (firstComponent == null || currComponent.getStart() < firstComponent.getStart()) {
                firstComponent = currComponent;
            }
        }
        return firstComponent;
    }    
    
    /**
     * Retorna el Ultimo Componente
     */
    public Component getLastComponent(List components) {
        Component lastComponent = null;
        for (int i = 0; i < components.size(); i++) {
            Component currComponent = (Component) components.get(i);            
            if (lastComponent == null || lastComponent.getStart() < currComponent.getStart()) {
                lastComponent = currComponent;
            }
        }
        return lastComponent;
    }     
    
    /**
     * Retorna los componentes en la posicion start
     */
    public List getComponents(List components,long start) {
        List componentAtStart = new ArrayList();
        for (int i = 0; i < components.size(); i++) {
            Component currComponent = (Component) components.get(i);            
            if (currComponent.getStart() == start) {
            	componentAtStart.add(currComponent);
            }
        }
        return componentAtStart;
    }  
    
    /**
     * Retorna Un Componente en la posicion start
     */
    public Component getComponent(List components,long start) {
        Component component = null;
        for (int noteIdx = 0; noteIdx < components.size(); noteIdx++) {
            Component currComponent = (Component) components.get(noteIdx);            
            if (currComponent.getStart() == start) {
                component = currComponent;
                break;
            }
        }
        return component;
    }   
    
    /**
     * Retorna Todos los desde Start hasta el final del compas
     */
    public List getComponentsBeforeEnd(List components,long fromStart) {
        List componentBeforeEnd = new ArrayList();        
        Iterator it = components.iterator();
        while(it.hasNext()){           
            Component currComponent = (Component)it.next();
            if (currComponent.getStart() >= fromStart) {
            	componentBeforeEnd.add(currComponent);
            }
        }
        return componentBeforeEnd;
    } 
    
    /**
     * Elimina los Componentes que empiecen en Start y esten en la misma cuerda
     * Si hay un Silencio lo borra sin importar la cuerda
     */
    public void removeComponentsAt(Measure measure,long start,int string,boolean addSilence){
        if(string != -1){
        	List notes = getNotes(measure,start);
        	Iterator it = notes.iterator();
        	while(it.hasNext()){
        		Note note = (Note)it.next();
        		if(note.getString() == string){
        			removeNote(measure,note);
                    
        			//si era el unico componente agrego un silencio
        			if(addSilence && notes.size() == 1){
        				addSilence(measure,new Silence(note.getStart(),(Duration)note.getDuration().clone()));
        			}                
        		}
        	}    
        }
        List silences = getSilences(measure,start);
        
        Iterator it = silences.iterator();
        while(it.hasNext()){
            Silence silence = (Silence)it.next();
            removeSilence(measure,silence);                            
        }        
    }
    
    /**
     * Elimina los Componentes que empiecen en Start 
     */
    public void removeAllComponentsAt(Measure measure,long start){
    	List notes = getNotes(measure,start);
        Iterator it = notes.iterator();
        while(it.hasNext()){
            Note note = (Note)it.next();                
            removeNote(measure,note);                            
        }      	
    	
        List silences = getSilences(measure,start);
        it = silences.iterator();
        while(it.hasNext()){
            Silence silence = (Silence)it.next();            
            removeSilence(measure,silence);            
        }        
    }
    
    private void orderComponents(List components){
        for(int i = 0;i < components.size();i++){
            Component minComponent = null;
            for(int j = i;j < components.size();j++){
                Component component = (Component)components.get(j);
                if(minComponent == null || component.getStart() < minComponent.getStart()){
                    minComponent = component;
                }
            }
            components.remove(minComponent);
            components.add(i,minComponent);
        }
    }
    
    /**
     * Verifica si el componente se puede insertar en el compas.
     * si no puede, con la opcion removeSilences, verifica si el motivo por el
     * cual no entra es que lo siguen silencios. de ser asi los borra. 
     */
    public boolean canInsert(Measure measure,Component component,boolean removeSilences,boolean tryMove){        
        boolean canInsert = true;
        //List components = new ArrayList();
        //components.addAll(measure.getNotes());
        //components.addAll(measure.getSilences());
        List components = getComponents(measure);
        orderComponents(components);             
        
        //Verifico si hay lugar para meter la nota-----------------        
        Component nextComponent = getNextComponent(components,component);
        //si el componente es null, verifico el fin del compas
        long componentEnd = component.getStart() + component.getDuration().getTime();
        if(nextComponent == null){
            if(componentEnd > measure.getStart() + measure.getLength()){
                canInsert = false;
            }
        }else if(componentEnd > nextComponent.getStart()){
            canInsert = false;            
    	}
        
        //---Busca si hay espacio disponible de silencios entre el componente y el el que le sigue.. si encuentra lo borra
        if(removeSilences && !canInsert && nextComponent instanceof Silence){
            //Verifico si lo que sigue es un silencio. y lo borro     
            long nextComponentEnd = 0;
            List nextSilences = new ArrayList();
            while(nextComponent instanceof Silence){  
            	nextSilences.add(nextComponent);	
            	nextComponentEnd = nextComponent.getStart() + nextComponent.getDuration().getTime();                  
                nextComponent = getNextComponent(components,nextComponent);
            }
            if(nextComponent == null){
                nextComponentEnd = measure.getStart() + measure.getLength();
            }else if(nextComponent instanceof Note){
                nextComponentEnd = nextComponent.getStart();
            }
            if(componentEnd <= nextComponentEnd){
                while(!nextSilences.isEmpty()){
                    Silence currSilence = (Silence)nextSilences.get(0);
                    measure.removeSilence(currSilence);
                    nextSilences.remove(currSilence);
                }
                canInsert = true;                
            }             
        }
        
        
        
        
        
        
        
        
        //---Busca si hay espacio disponible de silencios entre el componente y el final.. si encuentra mueve todo
        if(!canInsert && removeSilences && tryMove){
            //nextComponent = getNextNoteComponent(component);
            nextComponent = getNextComponent(components,component);
            if(nextComponent != null){
                long requiredLength = (component.getDuration().getTime()  - (nextComponent.getStart() - component.getStart()));
                
                
                long nextSilenceLength = 0;
                List nextSilences = new ArrayList();
                Silence nextSilence = getNextSilence(measure,component);
                while(nextSilence != null){  
                	nextSilences.add(nextSilence);	
                	nextSilenceLength += nextSilence.getDuration().getTime();                  
                	nextSilence = getNextSilence(measure,nextSilence);
                }       
                if(requiredLength <= nextSilenceLength){
                    
                    //List components = getComponentsBeforeEnd(nextComponent.getStart());
                	components = getComponentsBeforeEnd(components,nextComponent.getStart());
                    while(!components.isEmpty()){
                        Component currComponent = (Component)components.get(0);
                        if(currComponent instanceof Silence){
                            Silence currSilence = (Silence)currComponent;                                                    
                            requiredLength -= currSilence.getDuration().getTime();                            
                            measure.removeSilence(currSilence);                                
                        }else if(requiredLength > 0){
                            moveComponent(currComponent,requiredLength);      
                        }
                        
                        
                        components.remove(0);                        
                    }
                    canInsert = true;     
                }
                
            }
        }
        //---FIN A PRUEVA--------------------------------------
        
        return canInsert;
    }           
    
    
    /**
     * Cambia la Duracion del componente.
     */
    public void changeDuration(Measure measure,Component component,Duration duration){
    	changeDuration(measure,component,duration,true);
    }
    
    /**
     * Cambia la Duracion del componente.
     */
    public void changeDuration(Measure measure,Component component,Duration duration,boolean tryMove){        
        //obtengo la duracion vieja
        Duration oldDuration = (Duration)component.getDuration().clone();
        
        //asigno la nueva duracion
        component.setDuration((Duration)duration.clone());
        
        //si no entra vuelvo a dejar la vieja
        //if(canInsert(measure,component,true,true)){            
        if(canInsert(measure,component,true,tryMove)){
            //se lo agrego a todas las notas en la posicion
            List components = getNotes(measure,component.getStart());
            Iterator it = components.iterator();
            while(it.hasNext()){
                Component currComponent = (Component)it.next();
                currComponent.setDuration((Duration)duration.clone());
            }            
            
            //trato de agregar un silencio similar al lado            
            boolean move = (component.getDuration().getTime() > oldDuration.getTime());
            tryChangeSilenceAfter(measure,component,move);
            
        }else{
            component.setDuration(oldDuration);
        }
        

    }
    
    public void tryChangeSilenceAfter(Measure measure,Component component){
    	tryChangeSilenceAfter(measure,component,true);
    }
    
    public void tryChangeSilenceAfter(Measure measure,Component component,boolean tryMove){
    	List components = getComponents(measure);
        autoCompleteSilences(measure,components);
        Component nextComponent = getNextComponent(getComponents(measure),component);
        
        long componentEnd = (component.getStart() + component.getDuration().getTime());
        long measureEnd = (measure.getStart() + measure.getLength());
        if(nextComponent instanceof Silence && componentEnd <= measureEnd){
            
            long theMove = (getRealStart(measure,componentEnd)) - getRealStart(measure,nextComponent.getStart());
        	//long theMove = (componentEnd - nextComponent.getStart());
            
            if((nextComponent.getStart() + theMove) < measureEnd && (nextComponent.getStart() + nextComponent.getDuration().getTime() + theMove) <= measureEnd){            
                moveComponent(nextComponent,theMove);                        
            	changeDuration(measure,nextComponent,(Duration)component.getDuration().clone(),tryMove);
            }
        }     
    }
    
    public void autoCompleteSilences(Measure measure){ 
    	autoCompleteSilences(measure,getComponents(measure));
    }
    
    /**
     * Calcula si hay espacios libres. y crea nuevos silencios
     */   
    public void autoCompleteSilences(Measure measure,List components){    

        long start = measure.getStart();
        long end = 0;
        long diff = 0;
        Component component = getFirstComponent(components);
        
        while (component != null) {
            end = component.getStart() + component.getDuration().getTime();  
            if(component.getStart() > start){
                diff = component.getStart() - start;
                if(diff > 0){                                                            
                    createSilences(measure,start,diff);
                }                
            }            
            start = end;       
            component = getNextComponent(components,component);            
        }        
        end = measure.getStart() + measure.getLength();
        diff = end - start;
        if(diff > 0){       
            createSilences(measure,start,diff);         
        }
    }    
    
    /**
     * Crea Silencios temporarios en base a length
     */    
    public void createSilences(Measure measure,long start,long length){
        List durations = TablatureUtil.createDurations(length);
        Iterator it = durations.iterator();
        while(it.hasNext()){
            Duration duration = (Duration)it.next();
            Silence silence = new Silence(start,duration);
            addSilence(measure,silence);
            start += duration.getTime();
        }
    }  
    
    
    public long getRealStart(Measure measure,long currStart){        
        long beatLength = TablatureUtil.getBeatLength(measure.getTimeSignature());        
        long start = currStart;
        
        boolean startBeat = (start % beatLength == 0); 
        if(!startBeat){

            
            Duration minDuration = new Duration(Duration.SIXTY_FOURTH,false,false,new Tupleto(3,2));
            for(int i = 0;i < minDuration.getTime();i++){
                start ++;    
                startBeat = (start % beatLength == 0);     
                if(startBeat){
                   break; 
                }
            }
            if(!startBeat){
                start = currStart;
            }
        }
        
        return start;
    }

    
    public boolean areInSameBeat(Measure measure,Component arg0,Component arg1){
        long measureEnd = measure.getStart() + measure.getLength();
        long beatLength = TablatureUtil.getBeatLength(measure.getTimeSignature());  
        long start1 = getRealStart(measure,arg0.getStart());        
        long start2 = getRealStart(measure,arg1.getStart());
        
        long currStart = measure.getStart();
        
        
        boolean finish = false;
        while(!finish){
            if(start1 >= currStart && start1 < currStart + beatLength && start2 >= currStart && start2 < currStart + beatLength){
                return true;
            }
            currStart += beatLength;
            if(currStart > measureEnd){
                finish = true;
            }
        }
        return false;          
    }
    
    
    
    /** 
     * Liga la nota
     */
    public void changeTieNote(Measure measure,long start,int string){
        Note note = getNote(measure,start,string);
        if(note != null){
            changeTieNote(note);
        }
    }
    
    /** 
     * Liga la nota
     */
    public void changeTieNote(Note note){        
        note.setTiedNote(!note.isTiedNote());
        note.getEffect().setDeadNote(false);
    }
    

    
    /** 
     * Agrega un vibrato
     */
    public void changeVibratoNote(Measure measure,long start,int string){
    	Note note = getNote(measure,start,string);
    	if(note != null){            
    		note.getEffect().setVibrato(!note.getEffect().isVibrato());
        }
    }    

    /** 
     * Agrega una nota muerta
     */
    public void changeDeadNote(Note note){
    	note.getEffect().set(note.getEffect().isVibrato(),null,!note.getEffect().isDeadNote(),false,false);
    	note.setTiedNote(false);        
    }
    
    /** 
     * Agrega un slide
     */
    public void changeSlideNote(Measure measure,long start,int string){
    	Note note = getNote(measure,start,string);
    	if(note != null){    
            note.getEffect().set(note.getEffect().isVibrato(),null,false,!note.getEffect().isSlide(),false);
        }
    }     
   
    
    /** 
     * Agrega un hammer
     */
    public void changeHammerNote(Measure measure,long start,int string){
    	Note note = getNote(measure,start,string);
    	if(note != null){ 
            note.getEffect().set(note.getEffect().isVibrato(),null,false,false,!note.getEffect().isHammer());            
        }
    }            
    
    /** 
     * Agrega un bend
     */
    public void changeBendNote(Measure measure,long start,int string,BendEffect bend){
    	Note note = getNote(measure,start,string);
    	if(note != null){ 
            note.getEffect().set(note.getEffect().isVibrato(),bend,false,false,false);            
        }
    }  
}
