package org.herac.tuxguitar.gui.printer;

import java.awt.print.Printable;
import java.net.URI;
import java.net.URISyntaxException;

import javax.print.Doc;
import javax.print.DocFlavor;
import javax.print.DocPrintJob;
import javax.print.PrintException;
import javax.print.PrintService;
import javax.print.PrintServiceLookup;
import javax.print.SimpleDoc;
import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.standard.Copies;
import javax.print.attribute.standard.Destination;
import javax.print.attribute.standard.PageRanges;

import org.eclipse.swt.SWTError;
import org.eclipse.swt.SWTException;
import org.eclipse.swt.graphics.Device;
import org.eclipse.swt.graphics.DeviceData;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.GCData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.internal.gtk.OS;
import org.eclipse.swt.printing.PrinterData;

public class Printer extends Device{	
	private PrinterData printerData;
	private DocPrintJob printJob;
	private String jobName;
	private int defaultFont;
	private Image image;
	private PrintDocument document;	
	/**
	 * Constructs a new printer representing the default printer.
	 * <p>
	 * You must dispose the printer when it is no longer required. 
	 * </p>
	 *
	 * @exception SWTError <ul>
	 *    <li>ERROR_NO_HANDLES - if there are no valid printers
	 * </ul>
	 *
	 * @see Device#dispose
	 */
	public Printer() {
		this(getDefaultPrinterData());
	}
	
	/**
	 * Constructs a new printer given a <code>PrinterData</code>
	 * object representing the desired printer.
	 * <p>
	 * You must dispose the printer when it is no longer required. 
	 * </p>
	 *
	 * @param data the printer data for the specified printer
	 *
	 * @exception IllegalArgumentException <ul>
	 *    <li>ERROR_INVALID_ARGUMENT - if the specified printer data does not represent a valid printer
	 * </ul>
	 * @exception SWTError <ul>
	 *    <li>ERROR_NO_HANDLES - if there are no valid printers
	 * </ul>
	 *
	 * @see Device#dispose
	 */
	public Printer(PrinterData data) {
		super (data);
		this.printerData = data;
		this.document = new PrintDocument();
		this.initializeSystemResources();
	}
	
	
	public Image getImage(){
		if(this.image == null){
			this.createImage(1024,768);
		}		
		return this.image;
	}
	
	private void createImage(int width,int height){
		if(this.image != null && !this.image.isDisposed()){
			this.image.dispose();
		}		
		this.image = new Image(this,width,height);
	}
	
	private void initializeSystemResources () {
		int shellHandle = OS.gtk_window_new (OS.GTK_WINDOW_TOPLEVEL);
		int style = OS.gtk_widget_get_style (shellHandle);	
		defaultFont = OS.pango_font_description_copy (OS.gtk_style_get_font_desc (style));
	}
	
	public Printer(DeviceData data) {
		super(data);
	}


	/**	 
	 * Invokes platform specific functionality to allocate a new GC handle.
	 * <p>
	 * <b>IMPORTANT:</b> This method is <em>not</em> part of the public
	 * API for <code>Printer</code>. It is marked public only so that it
	 * can be shared within the packages provided by SWT. It is not
	 * available on all platforms, and should never be called from
	 * application code.
	 * </p>
	 *
	 * @param data the platform specific GC data 
	 * @return the platform specific GC handle
	 */
	public int internal_new_GC(GCData data) {
		return getImage().internal_new_GC(data);
	}

	/**	 
	 * Invokes platform specific functionality to dispose a GC handle.
	 * <p>
	 * <b>IMPORTANT:</b> This method is <em>not</em> part of the public
	 * API for <code>Printer</code>. It is marked public only so that it
	 * can be shared within the packages provided by SWT. It is not
	 * available on all platforms, and should never be called from
	 * application code.
	 * </p>
	 *
	 * @param hDC the platform specific GC handle
	 * @param data the platform specific GC data 
	 */
	public void internal_dispose_GC(int handle, GCData data) {
		getImage().internal_dispose_GC(handle,data);
	}
	
	

	public Font getSystemFont () {
		checkDevice ();
		return Font.gtk_new (this, defaultFont);
	}
	
	

	/**
	 * Returns an array of <code>PrinterData</code> objects
	 * representing all available printers.
	 *
	 * @return the list of available printers
	 */
	public static PrinterData[] getPrinterList() {
		PrintService[] services = PrintServiceLookup.lookupPrintServices(null,null);		
		PrinterData[]	printerList = new PrinterData[services.length];
		for(int i = 0;i < printerList.length;i++){
			printerList[i] = toPrinterData(services[i]);
		}
		return printerList;
	}

	/**
	 * Returns a <code>PrinterData</code> object representing
	 * the default printer or <code>null</code> if there is no 
	 * printer available on the System.
	 *
	 * @return the default printer data or null
	 * 
	 * @since 2.1
	 */
	public static PrinterData getDefaultPrinterData() {
		return toPrinterData(PrintServiceLookup.lookupDefaultPrintService());
	}


	public static PrinterData toPrinterData(PrintService service){
		PrinterData data = new PrinterData();
		if (service != null) {
			data.name= service.getName();
			data.driver= "Virtual";
			data.printToFile= false;
		}		
		return data;
	}

	public PrintService getPrintService(){
		if(getPrinterData() != null && getPrinterData().name != null){
			PrintService[] services = PrintServiceLookup.lookupPrintServices(null,null);
			for(int i = 0;i < services.length;i++){
				if(services[i].getName().equals(getPrinterData().name)){
					return services[i];
				}
			}
				
		}
		if(this.printJob == null){
			PrintService service = PrintServiceLookup.lookupDefaultPrintService();
			if(service != null){
				return service;
			}
		}				
		return null;
	}
	

	/**
	 * Starts a print job and returns true if the job started successfully
	 * and false otherwise.
	 * <p>
	 * This must be the first method called to initiate a print job,
	 * followed by any number of startPage/endPage calls, followed by
	 * endJob. Calling startPage, endPage, or endJob before startJob
	 * will result in undefined behavior.
	 * </p>
	 * 
	 * @param jobName the name of the print job to start
	 * @return true if the job started successfully and false otherwise.
	 *
	 * @exception SWTException <ul>
	 *    <li>ERROR_DEVICE_DISPOSED - if the receiver has been disposed</li>
	 * </ul>
	 *
	 * @see #startPage
	 * @see #endPage
	 * @see #endJob
	 */
	public boolean startJob(String jobName) {		
		this.jobName = jobName;
		PrintService service = getPrintService();
		if(service != null){
			this.printJob = service.createPrintJob();	        
		}
		return (this.printJob != null);
	}

	/**
	 * Ends the current print job.
	 *
	 * @exception SWTException <ul>
	 *    <li>ERROR_DEVICE_DISPOSED - if the receiver has been disposed</li>
	 * </ul>
	 *
	 * @see #startJob
	 * @see #startPage
	 * @see #endPage
	 */
	public void endJob() {
		try {		
			if(this.printJob != null){
				Printable printable = new PrintableImpl(this.document);		
				DocFlavor flavor = DocFlavor.SERVICE_FORMATTED.PRINTABLE;		
				HashPrintRequestAttributeSet aset= new HashPrintRequestAttributeSet();				
				aset.add(new Copies(this.printerData.copyCount));
				if(this.printerData.scope == PrinterData.PAGE_RANGE){
					this.printerData.startPage = (this.printerData.startPage <= 0)?1:this.printerData.startPage;
					this.printerData.startPage = (this.printerData.startPage > this.document.count())?this.document.count():this.printerData.startPage;
					this.printerData.endPage = (this.printerData.endPage < this.printerData.startPage)?this.printerData.startPage:this.printerData.endPage;
					this.printerData.endPage = (this.printerData.endPage > this.document.count())?this.document.count():this.printerData.endPage;

					aset.add(new PageRanges(this.printerData.startPage,this.printerData.endPage));
				}
				if(this.printerData.printToFile && this.printerData.fileName != null && this.printerData.fileName.length() > 0){
					aset.add(new Destination(new URI(this.printerData.fileName)));
				}												
				Doc doc = new SimpleDoc(printable,flavor,null);
					
				this.printJob.print(doc,aset);
			}
		} catch (PrintException e) {
			e.printStackTrace();
		} catch (URISyntaxException e) {
			e.printStackTrace();
		}
		this.printJob = null;
	}

	/**
	 * Cancels a print job in progress. 
	 *
	 * @exception SWTException <ul>
	 *    <li>ERROR_DEVICE_DISPOSED - if the receiver has been disposed</li>
	 * </ul>
	 */
	public void cancelJob() {
		this.document.clear();
		this.printJob = null;
	}

	/**
	 * Starts a page and returns true if the page started successfully
	 * and false otherwise.
	 * <p>
	 * After calling startJob, this method may be called any number of times
	 * along with a matching endPage.
	 * </p>
	 * 
	 * @return true if the page started successfully and false otherwise.
	 *
	 * @exception SWTException <ul>
	 *    <li>ERROR_DEVICE_DISPOSED - if the receiver has been disposed</li>
	 * </ul>
	 *
	 * @see #endPage
	 * @see #startJob
	 * @see #endJob
	 */
	public boolean startPage() {		
		return (this.printJob != null);
	}

	/**
	 * Ends the current page.
	 *
	 * @exception SWTException <ul>
	 *    <li>ERROR_DEVICE_DISPOSED - if the receiver has been disposed</li>
	 * </ul>
	 *
	 * @see #startPage
	 * @see #startJob
	 * @see #endJob
	 */
	public void endPage() {
		this.document.makePage(getImage().getImageData());
	}

	/**
	 * Returns a point whose x coordinate is the horizontal
	 * dots per inch of the printer, and whose y coordinate
	 * is the vertical dots per inch of the printer.
	 *
	 * @return the horizontal and vertical DPI
	 *
	 * @exception SWTException <ul>
	 *    <li>ERROR_DEVICE_DISPOSED - if the receiver has been disposed</li>
	 * </ul>
	 */
	public Point getDPI() {
		return new Point(0, 0);
	}

	/**
	 * Returns a rectangle describing the receiver's size and location.
	 * For a printer, this is the size of a page, in pixels.
	 *
	 * @return the bounding rectangle
	 *
	 * @exception SWTException <ul>
	 *    <li>ERROR_DEVICE_DISPOSED - if the receiver has been disposed</li>
	 * </ul>
	 *
	 * @see #getClientArea
	 * @see #computeTrim
	 */
	public Rectangle getBounds() {
		return getImage().getBounds();
	}

	/**
	 * Returns a rectangle which describes the area of the
	 * receiver which is capable of displaying data.
	 * For a printer, this is the size of the printable area
	 * of a page, in pixels.
	 * 
	 * @return the client area
	 *
	 * @exception SWTException <ul>
	 *    <li>ERROR_DEVICE_DISPOSED - if the receiver has been disposed</li>
	 * </ul>
	 *
	 * @see #getBounds
	 * @see #computeTrim
	 */
	public Rectangle getClientArea() {		
		return getImage().getBounds();
	}

	/**
	 * Given a desired <em>client area</em> for the receiver
	 * (as described by the arguments), returns the bounding
	 * rectangle which would be required to produce that client
	 * area.
	 * <p>
	 * In other words, it returns a rectangle such that, if the
	 * receiver's bounds were set to that rectangle, the area
	 * of the receiver which is capable of displaying data
	 * (that is, not covered by the "trimmings") would be the
	 * rectangle described by the arguments (relative to the
	 * receiver's parent).
	 * </p>
	 * Note that there is no setBounds for a printer. This method
	 * is usually used by passing in the client area (the 'printable
	 * area') of the printer. It can also be useful to pass in 0, 0, 0, 0.
	 * 
	 * @param x the desired x coordinate of the client area
	 * @param y the desired y coordinate of the client area
	 * @param width the desired width of the client area
	 * @param height the desired height of the client area
	 * @return the required bounds to produce the given client area
	 *
	 * @exception SWTException <ul>
	 *    <li>ERROR_DEVICE_DISPOSED - if the receiver has been disposed</li>
	 * </ul>
	 *
	 * @see #getBounds
	 * @see #getClientArea
	 */
	public Rectangle computeTrim(int x, int y, int width, int height) {
		return new Rectangle(0,0,0,0);
	}

	/**
	 * Returns a <code>PrinterData</code> object representing the
	 * target printer for this print job.
	 * 
	 * @return a PrinterData object describing the receiver
	 */
	public PrinterData getPrinterData() {
		return this.printerData;
	}	
	
	
	
	public void dispose(){
		this.image.dispose();
		super.dispose();
	}
	
}
