# SBDumper - STDBuilder GAME.DAT Dumper
# Copyright (C) 2021 Remilia Scarlet <remilia@posteo.jp>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
require "libremiliacr/remiargparser"

module SBDumper
  VERSION = "0.1.0"

  record Entry, name : String, offset : Int32, size : Int32, checksum : Int32

  class DatFile
    getter entries = [] of Entry
    @numEntries = 0i32

    def initialize()
    end

    def load(src : IO, dontConvertJIS : Bool)
      # MLOV seems to be the 4-byte magic
      raise "Unknown file format" unless src.gets(4) == "MLOV"

      # Ignore four bytes
      src.read_bytes(Int32, IO::ByteFormat::LittleEndian)

      # Number of entries
      @numEntries = src.read_bytes(Int32, IO::ByteFormat::LittleEndian)

      # Something related to file size?
      src.read_bytes(Int32, IO::ByteFormat::LittleEndian)

      nameBuf = Bytes.new(16)

      # Read entry directory
      @numEntries.times do |_|
        chksum = src.read_bytes(Int32, IO::ByteFormat::LittleEndian)
        size = src.read_bytes(Int32, IO::ByteFormat::LittleEndian)
        src.read_bytes(Int32, IO::ByteFormat::LittleEndian) # Maybe size is 64-bit?
        offset = src.read_bytes(Int32, IO::ByteFormat::LittleEndian)

        # Skip flags
        8.times { |_| src.read_bytes(Int32, IO::ByteFormat::LittleEndian) }

        # Name seems to be null padded, and might be in Shift JIS
        src.read(nameBuf)
        firstNull = nameBuf.index('\0')
        name = if firstNull
                 if dontConvertJIS
                   String.new(nameBuf[0...firstNull])
                 else
                   String.new(nameBuf[0...firstNull], "SHIFT_JIS", :skip)
                 end
               else
                 if dontConvertJIS
                   String.new(nameBuf)
                 else
                   String.new(nameBuf, "SHIFT_JIS", :skip)
                 end
               end

        # Store the entry
        @entries << Entry.new(name.strip('\0'), offset, size, chksum)
      end
    end

    def dump(src : IO, outdir : String, quiet : Bool)
      total = 0

      entries.each do |ent|
        withSlashes = ent.name.gsub('¥', '/') # Yen signs are used for directory separators
        filename = Path[outdir, withSlashes]

        puts filename

        unless Dir.exists?(filename.dirname)
          puts "Creating #{filename.dirname}/"
          Dir.mkdir_p(filename.dirname)
        end

        STDOUT << "#{ent.name}... " unless quiet
        begin
          File.open(filename, "w") do |file|
            src.pos = ent.offset
            IO.copy(src, file, ent.size)
          end

          size = File.size(filename)
          total += size
          STDOUT << "done, #{size.humanize_bytes(format: Int::BinaryPrefixFormat::JEDEC)}\n" unless quiet
        rescue err : Exception
          STDOUT << "error! #{err.message || "unknown error"}\n"
        end
      end

      total
    end
  end

  class MainProg
    def initialize
      args = RemiArgParser::ArgParser.new("sbdumper", VERSION)
      args.addFlag("no-convert", 'c', help: "Don't convert filenames from Shift JIS to UTF-8")
      args.addString("file", 'f', help: "The filename to read")
      args.addString("outdir", 'o', help: "The directory to save files into")
      args.addFlag("list", 'l', help: "Don't dump anything, just list files")
      args.addFlag("quiet", 'q', help: "Print only minimal information")

      begin
        args.parse(ARGV)
      rescue err : RemiArgParser::ArgumentError
        STDERR << "#{err}\n"
        exit 1
      end

      abort("No file specified") unless args["file"].called

      dat = DatFile.new
      File.open(args["file"].str) { |file| dat.load(file, args["no-convert"].called) }

      if args["list"].called
        dat.entries.each do |ent|
          STDOUT << sprintf("%-7s - %s\n", ent.size.humanize_bytes(format: Int::BinaryPrefixFormat::JEDEC), ent.name)
        end
        puts "Total entries: #{dat.entries.size}"
      else
        abort("No output directory specified") unless args["outdir"].called

        File.open(args["file"].str) do |file|
          total = dat.dump(file, args["outdir"].str, args["quiet"].called)
          puts "Files dumped: #{dat.entries.size}"
          puts "Total size: #{total.humanize_bytes(format: Int::BinaryPrefixFormat::JEDEC)}"
        end
      end
    end
  end

  MainProg.new
end
