//! @file		pf_patrol.c
//! @brief		プラットフォーム(ラインセンサ)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "pf_gpio.h"
#include "pf_patrol.h"

//! @brief		ラインセンサ→GPIO IDテーブル
static const PF_GPIO_ID pf_patrol_to_gpio[PF_PATROL_ID_MAX] =
{
	PF_GPIO_ID_MAQUEEN_PATROL_L,			//!< PF_PATROL_ID_L
	PF_GPIO_ID_MAQUEEN_PATROL_R,			//!< PF_PATROL_ID_R
};

//! @brief		ラインセンサ情報
static PF_PATROL_COLOR pf_patrol_info[PF_PATROL_ID_MAX];

//! @brief		ラインセンサ初期化(単一のID)
//! @param		[in] id			ラインセンサのID
static void pf_patrol_init_id(PF_PATROL_ID id)
{
	// 現在データをWHITEに初期化
	pf_patrol_info[id] = PF_PATROL_COLOR_WHITE;
}

//! @brief		ラインセンサ初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_patrol_init(void)
{
	PF_PATROL_ID id;

	// オート変数初期化
	id = 0;

	// すべてのIDをループ
	for (id = 0; id < PF_PATROL_ID_MAX; id++)
	{
		// 1つのIDを初期化
		pf_patrol_init_id(id);
	}
}

//! @brief		ラインセンサ定期タスク(単一のID)
//! @param		[in] id			ラインセンサのID
static void pf_patrol_task_id(PF_PATROL_ID id)
{
	PF_GPIO_ID gpio;

	// オート変数初期化
	gpio = pf_patrol_to_gpio[id];

	// 現在のフォトリフレクタ情報を取得(GPIOからの入力が'L'で黒色)
	if (FALSE == pf_gpio_input(gpio))
	{
		// 黒色
		pf_patrol_info[id] = PF_PATROL_COLOR_BLACK;
	}
	else
	{
		// 白色
		pf_patrol_info[id] = PF_PATROL_COLOR_WHITE;
	}
}

//! @brief		ラインセンサ定期タスク
//! @remarks	プラットフォーム定期タスク(入力系)処理から呼び出すこと
void pf_patrol_task(void)
{
	PF_PATROL_ID id;

	// オート変数初期化
	id = 0;

	// すべてのIDをループ
	for (id = 0; id < PF_PATROL_ID_MAX; id++)
	{
		// 1つのIDを処理
		pf_patrol_task_id(id);
	}
}

//! @brief		ラインセンサの白黒状態を取得
//! @param		[in] id			ラインセンサのID
//! @return		色状態(PF_PATROL_COLOR_WHITE=白色/PF_PATROL_COLOR_BLACK=黒色)
PF_PATROL_COLOR pf_patrol_line(PF_PATROL_ID id)
{
	PF_PATROL_COLOR result;

	// オート変数初期化
	result = PF_PATROL_COLOR_WHITE;

	// パラメータチェック
	if (id < PF_PATROL_ID_MAX)
	{
		result = pf_patrol_info[id];
	}

	return result;
}
