//! @file		pf_music.c
//! @brief		プラットフォーム(音楽)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "nrf52833.h"
#include "nrf52833_bitfields.h"
#include "pf_gpio.h"
#include "pf_timer.h"
#include "pf_music.h"

//! @brief		コマンド定義
typedef enum PF_MUSIC_CMD_Tag
{
	PF_MUSIC_CMD_LEN1 = 0,					//!< 全音符(全休符)
	PF_MUSIC_CMD_LEN2P,						//!< 符点二分音符(符点二分休符)
	PF_MUSIC_CMD_LEN2,						//!< 二分音符(二分休符)
	PF_MUSIC_CMD_LEN4P,						//!< 符点四分音符(符点四分休符)
	PF_MUSIC_CMD_LEN4,						//!< 四分音符(四分休符)
	PF_MUSIC_CMD_LEN8P,						//!< 符点八分音符(符点八分休符)
	PF_MUSIC_CMD_LEN8,						//!< 八分音符(八分休符)
	PF_MUSIC_CMD_LEN16P,					//!< 符点十六分音符(符点十六分休符)
	PF_MUSIC_CMD_LEN3,						//!< 三連符(三連休符)
	PF_MUSIC_CMD_LEN16,						//!< 十六分音符(十六分休符)
	PF_MUSIC_CMD_C,							//!< C
	PF_MUSIC_CMD_CSHARP,					//!< C♯/D♭
	PF_MUSIC_CMD_D,							//!< D
	PF_MUSIC_CMD_DSHARP,					//!< D♯/E♭
	PF_MUSIC_CMD_E,							//!< E
	PF_MUSIC_CMD_F,							//!< F
	PF_MUSIC_CMD_FSHARP,					//!< F♯/G♭
	PF_MUSIC_CMD_G,							//!< G
	PF_MUSIC_CMD_GSHARP,					//!< G♯/A♭
	PF_MUSIC_CMD_A,							//!< A
	PF_MUSIC_CMD_ASHARP,					//!< A♯/B♭
	PF_MUSIC_CMD_B,							//!< B
	PF_MUSIC_CMD_REST,						//!< 休符
	PF_MUSIC_CMD_TIE,						//!< タイ
	PF_MUSIC_CMD_TEMPO,						//!< テンポ
	PF_MUSIC_CMD_QUANTIZE,					//!< クォンタイズ
	PF_MUSIC_CMD_OCTAVE_UP,					//!< 1オクターブ上げる
	PF_MUSIC_CMD_OCTAVE_DOWN,				//!< 1オクターブ下げる
	PF_MUSIC_CMD_LOOP_SET,					//!< ループ位置設定
	PF_MUSIC_CMD_LOOP_GO,					//!< ループ位置ジャンプ
} PF_MUSIC_CMD;

//! @brief		サンプル曲「春が来た」メロディー
const PF_MUSIC_CMD pf_music_melody[] =
{
	// 初回動作
	PF_MUSIC_CMD_TEMPO,
	(PF_MUSIC_CMD)108,
	PF_MUSIC_CMD_OCTAVE_UP,
	PF_MUSIC_CMD_OCTAVE_UP,
	PF_MUSIC_CMD_QUANTIZE,
	(PF_MUSIC_CMD)7,
	PF_MUSIC_CMD_LOOP_SET,

	// 第1小節
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_F,
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_A,

	// 第2小節
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_F,
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_OCTAVE_UP,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_OCTAVE_DOWN,

	// 第3小節
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_A,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_LEN4P,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_C,

	// 第4小節
	PF_MUSIC_CMD_LEN2P,
	PF_MUSIC_CMD_D,
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_REST,

	// 第5小節
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_A,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_G,

	// 第6小節
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_OCTAVE_UP,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_D,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_OCTAVE_DOWN,
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_A,
	PF_MUSIC_CMD_OCTAVE_UP,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_OCTAVE_DOWN,

	// 第7小節
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_OCTAVE_UP,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_LEN4P,
	PF_MUSIC_CMD_D,
	PF_MUSIC_CMD_OCTAVE_DOWN,
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_G,

	// 第8小節
	PF_MUSIC_CMD_LEN2P,
	PF_MUSIC_CMD_OCTAVE_UP,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_OCTAVE_DOWN,
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_REST,

	// ループ
	PF_MUSIC_CMD_LOOP_GO,
};

//! @brief		サンプル曲「春が来た」ベース
const PF_MUSIC_CMD pf_music_base[] =
{
	// 初回動作
	PF_MUSIC_CMD_OCTAVE_UP,
	PF_MUSIC_CMD_QUANTIZE,
	(PF_MUSIC_CMD)6,
	PF_MUSIC_CMD_LOOP_SET,

	// 第1小節
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_G,

	// 第2小節
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_G,

	// 第3小節
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_G,

	// 第4小節
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_OCTAVE_DOWN,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_OCTAVE_UP,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_F,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_D,
	PF_MUSIC_CMD_G,

	// 第5小節
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_G,

	// 第6小節
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_A,
	PF_MUSIC_CMD_F,
	PF_MUSIC_CMD_A,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_A,
	PF_MUSIC_CMD_F,
	PF_MUSIC_CMD_A,

	// 第7小節
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_OCTAVE_DOWN,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_OCTAVE_UP,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_OCTAVE_DOWN,
	PF_MUSIC_CMD_B,
	PF_MUSIC_CMD_OCTAVE_UP,
	PF_MUSIC_CMD_G,

	// 第8小節
	PF_MUSIC_CMD_LEN8,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_E,
	PF_MUSIC_CMD_G,
	PF_MUSIC_CMD_LEN4,
	PF_MUSIC_CMD_C,
	PF_MUSIC_CMD_REST,

	// ループ
	PF_MUSIC_CMD_LOOP_GO,
};

//! @brief		音階定義
typedef enum PF_MUSIC_NOTE_Tag
{
	PF_MUSIC_NOTE_C = 0,					//!< C
	PF_MUSIC_NOTE_CSHARP,					//!< C♯/D♭
	PF_MUSIC_NOTE_D,						//!< D
	PF_MUSIC_NOTE_DSHARP,					//!< D♯/E♭
	PF_MUSIC_NOTE_E,						//!< E
	PF_MUSIC_NOTE_F,						//!< F
	PF_MUSIC_NOTE_FSHARP,					//!< F♯/G♭
	PF_MUSIC_NOTE_G,						//!< G
	PF_MUSIC_NOTE_GSHARP,					//!< G♯/A♭
	PF_MUSIC_NOTE_A,						//!< A
	PF_MUSIC_NOTE_ASHARP,					//!< A♯/B♭
	PF_MUSIC_NOTE_B,						//!< B
	PF_MUSIC_NOTE_MAX,						//!< (音階の個数を表す)
} PF_MUSIC_NOTE;

//! @brief		音階→PWM周期テーブル(PWMクロック=4MHz)
const u4 pf_music_note_table[PF_MUSIC_NOTE_MAX] =
{
	30578,									//!< C
	28862,									//!< C♯/D♭
	27242,									//!< D
	25713,									//!< D♯/E♭
	24270,									//!< E
	22908,									//!< F
	21622,									//!< F♯/G♭
	20408,									//!< G
	19263,									//!< G♯/A♭
	18182,									//!< A
	17161,									//!< A♯/B♭
	16198,									//!< B
};

//! @brief		コマンド→クロックテーブル
const u4 pf_music_cmd_to_clock[10] =
{
	// PF_MUSIC_CMD_LEN1
	192,

	// PF_MUSIC_CMD_LEN2P
	144,

	// PF_MUSIC_CMD_LEN2
	96,

	// PF_MUSIC_CMD_LEN4P
	72,

	// PF_MUSIC_CMD_LEN4
	48,

	// PF_MUSIC_CMD_LEN8P
	36,

	// PF_MUSIC_CMD_LEN8
	24,

	// PF_MUSIC_CMD_LEN16P
	18,

	// PF_MUSIC_CMD_LEN3
	16,

	// PF_MUSIC_CMD_LEN16
	12
};

//! @brief		音楽デバイス→PWMテーブル
static NRF_PWM_Type* const pf_music_device_to_pwm[PF_MUSIC_DEVICE_MAX] =
{
	NRF_PWM0,								//!< 内蔵スピーカー
	NRF_PWM1,								//!< MAQUEENスピーカー
};

//! @brief		音楽デバイス→GPIOテーブル
static const PF_GPIO_ID pf_music_device_to_gpio[PF_MUSIC_DEVICE_MAX] =
{
	PF_GPIO_ID_INTERNAL_SPEAKER,			//!< 内蔵スピーカー
	PF_GPIO_ID_MAQUEEN_SPEAKER,				//!< MAQUEENスピーカー
};

//! @brief		演奏情報構造体
typedef struct PF_MUSIC_INFO_Tag
{
	BOOL				playing;			//!< 演奏中フラグ
	const PF_MUSIC_CMD	*ptr;				//!< データポインタ
	const PF_MUSIC_CMD	*loop;				//!< ループポインタ
	u4					clock;				//!< クロック数
	u4					length;				//!< 音長
	BOOL 				on;					//!< キーオンフラグ
	u4					off;				//!< キーオフカウンタ
	u4					octave;				//!< オクターブ(0=除算なし)
	u4					quantize;			//!< クォンタイズ
	PF_MUSIC_NOTE		note;				//!< 音階
	BOOL				tie;				//!< タイフラグ
	s4					countertop;			//!< PWM周波数
	BOOL				lfo_enable;			//!< LFO有効フラグ
	u4					lfo_delay;			//!< LFOディレイ
	u4					lfo_keyon;			//!< LFOディレイ(カレント)
	u4					lfo_cycle;			//!< LFOサイクル(三角波の片方向クロック)
	u4					lfo_current;		//!< LFOサイクル(カレント)
	s4					lfo_diff;			//!< LFO差分
	s4					lfo_signed;			//!< LFO差分(カレント)
	u2					duty;				//!< デューティー比
} PF_MUSIC_INFO;

//! @brief		演奏情報
static PF_MUSIC_INFO pf_music_info[PF_MUSIC_DEVICE_MAX];

//! @brief		PWM初期化
static void pf_music_init_pwm(void)
{
	NRF_PWM_Type *pwm;
	u4 loop;
	PF_GPIO_ID gpio;
	u4 port;
	u4 pin;

	// オート変数初期化
	pwm = NRF_PWM0;
	loop = 0;
	gpio = PF_GPIO_ID_INTERNAL_SPEAKER;
	port = 0;
	pin = 0;

	// 音楽デバイスだけループ
	for (loop = 0; loop < PF_MUSIC_DEVICE_MAX; loop++)
	{
		// PWMデバイス取得
		pwm = pf_music_device_to_pwm[loop];

		// PWM停止
		pwm->ENABLE = PWM_ENABLE_ENABLE_Disabled << PWM_ENABLE_ENABLE_Pos;

		// イベントのショートカット
		pwm->SHORTS = 0;

		// 割り込み有効
		pwm->INTEN = 0;

		// 波形モード
		pwm->MODE = PWM_MODE_UPDOWN_Up << PWM_MODE_UPDOWN_Pos;

		// デコーダーモード
		pwm->DECODER = (PWM_DECODER_LOAD_Common << PWM_DECODER_LOAD_Pos)
						| (PWM_DECODER_MODE_NextStep << PWM_DECODER_MODE_Pos);

		// プリスケーラー(4MHz)
		pwm->PRESCALER = PWM_PRESCALER_PRESCALER_DIV_4 << PWM_PRESCALER_PRESCALER_Pos;

		// PWM周期
		pwm->COUNTERTOP = pf_music_note_table[0];

		// ループ回数(RAMテーブルをループしない)
		pwm->LOOP = PWM_LOOP_CNT_Disabled << PWM_LOOP_CNT_Pos;

		// GPIO IDを取得
		gpio = pf_music_device_to_gpio[loop];

		// GPIOポートとGPIOピンを取得
		port = pf_gpio_get_port(gpio);
		pin = pf_gpio_get_pin(gpio);

		// PSEL.OUT[0]のみ使用する
		pwm->PSEL.OUT[0] = (pin << PWM_PSEL_OUT_PIN_Pos) | (port << PWM_PSEL_OUT_PORT_Pos)
						| (PWM_PSEL_OUT_CONNECT_Connected << PWM_PSEL_OUT_CONNECT_Pos);

		// PWM有効
		pwm->ENABLE = PWM_ENABLE_ENABLE_Enabled << PWM_ENABLE_ENABLE_Pos;
	}
}

//! @brief		キーオン
//! @param		[in] device		音楽デバイス
static void pf_music_on(u4 device)
{
	NRF_PWM_Type *pwm;
	u4 cycle;
	u4 octave;

	// オート変数初期化
	pwm = pf_music_device_to_pwm[device];
	cycle = (u4)pf_music_info[device].countertop;
	octave = pf_music_info[device].octave;

	// オクターブ演算
	cycle >>= octave;

	// PWM周期設定
	pwm->COUNTERTOP = (u2)cycle;

	// デューティー比(31.25%)設定値は逆の68.75%とする
	cycle *= (16 - 5);
	cycle >>= 4;
	pf_music_info[device].duty = (u2)cycle;

	// ポインタ
	pwm->SEQ[0].PTR = (u4)&pf_music_info[device].duty;

	// 個数
	pwm->SEQ[0].CNT = 1;

	// リフレッシュ
	pwm->SEQ[0].REFRESH = 0;

	// 遅延
	pwm->SEQ[0].ENDDELAY = 0;

	// シーケンススタート
	pwm->TASKS_SEQSTART[0] = PWM_TASKS_SEQSTART_TASKS_SEQSTART_Trigger
					<< PWM_TASKS_SEQSTART_TASKS_SEQSTART_Pos;
}

//! @brief		キーオフ
//! @param		[in] device		音楽デバイス
static void pf_music_off(u4 device)
{
	NRF_PWM_Type *pwm;

	// オート変数初期化
	pwm = pf_music_device_to_pwm[device];

	// PWM停止(DAC出力はLowレベル)
	pwm->TASKS_STOP = PWM_TASKS_STOP_TASKS_STOP_Trigger << PWM_TASKS_STOP_TASKS_STOP_Pos;
}

//! @brief		音符
//! @param		[in] device		音楽デバイス
//! @param		[in] note		音階
static void pf_music_note(u4 device, PF_MUSIC_NOTE note)
{
	u4 off;

	// オート変数初期化
	off = 0;

	// 音階セット
	pf_music_info[device].note = note;

	// タイの有無で分ける
	if (FALSE == pf_music_info[device].tie)
	{
		// クォンタイズを考慮してキーオフクロックを算出
		off = pf_music_info[device].length;
		off *= pf_music_info[device].quantize;
		off >>= 3;
		pf_music_info[device].off = off;
	}
	else
	{
		// タイの効力は1音きり(TIE+音符1+音符2のように構成する)
		pf_music_info[device].tie = FALSE;
		pf_music_info[device].off = 0;
	}

	// クロック
	pf_music_info[device].clock = pf_music_info[device].length;

	// PWM周波数設定
	pf_music_info[device].countertop = (s4)pf_music_note_table[note];

	// LFO初期化
	pf_music_info[device].lfo_keyon = pf_music_info[device].lfo_delay;
	pf_music_info[device].lfo_current = pf_music_info[device].lfo_cycle;
	pf_music_info[device].lfo_signed = pf_music_info[device].lfo_diff;

	// キーオン
	pf_music_on(device);

	// キーオン状態
	pf_music_info[device].on = TRUE;
}

//! @brief		休符
//! @param		[in] device		音楽デバイス
static void pf_music_rest(u4 device)
{
	// キーオン中なら、キーオフ
	if (TRUE == pf_music_info[device].on)
	{
		pf_music_off(device);

		// キーオフ状態
		pf_music_info[device].on = FALSE;
	}

	// キーオフカウンタとクロック
	pf_music_info[device].off = 0;
	pf_music_info[device].clock = pf_music_info[device].length;
}

//! @brief		タイ
//! @param		[in] device		音楽デバイス
static void pf_music_tie(u4 device)
{
	// フラグセットのみ(TIE+音符1+音符2のように構成する)
	pf_music_info[device].tie = TRUE;
}

//! @brief		テンポ
//! @param		[in] device		音楽デバイス
static void pf_music_tempo(u4 device)
{
	u4 cc;
	u4 bpm;
	PF_MUSIC_CMD cmd;

	// オート変数初期化
	cc = 1250000U;
	bpm = 0;
	cmd = PF_MUSIC_CMD_TEMPO;

	// データフェッチ
	cmd = *(pf_music_info[device].ptr);
	pf_music_info[device].ptr++;
	bpm = (u4)cmd;

	// 1拍=48クロックで算出する
	// 最適化前の式:60*1000*1000/(48*bpm)
	cc /= bpm;

	pf_timer_cc(PF_TIMER_ID_MUSIC, &cc);
}

//! @brief		クォンタイズ
//! @param		[in] device		音楽デバイス
static void pf_music_quantize(u4 device)
{
	u4 quantize;
	PF_MUSIC_CMD cmd;

	// オート変数初期化
	quantize = 0;
	cmd = PF_MUSIC_CMD_QUANTIZE;

	// データフェッチ
	cmd = *(pf_music_info[device].ptr);
	pf_music_info[device].ptr++;
	quantize = (u4)cmd;

	// セット
	pf_music_info[device].quantize = quantize;
}

//! @brief		オクターブアップ
//! @param		[in] device		音楽デバイス
static void pf_music_octave_up(u4 device)
{
	pf_music_info[device].octave++;
}

//! @brief		オクターブダウン
//! @param		[in] device		音楽デバイス
static void pf_music_octave_down(u4 device)
{
	pf_music_info[device].octave--;
}

//! @brief		ループ位置設定
//! @param		[in] device		音楽デバイス
static void pf_music_loop_set(u4 device)
{
	pf_music_info[device].loop = pf_music_info[device].ptr;
}

//! @brief		ループ位置ジャンプ
//! @param		[in] device		音楽デバイス
static void pf_music_loop_go(u4 device)
{
	pf_music_info[device].ptr = pf_music_info[device].loop;
}

//! @brief		コマンド解釈
//! @param		[in] device		音楽デバイス
static void pf_music_interpret(u4 device)
{
	PF_MUSIC_CMD cmd;

	// オート変数初期化
	cmd = PF_MUSIC_CMD_LOOP_GO;

	// コマンド取得
	cmd = *(pf_music_info[device].ptr);
	pf_music_info[device].ptr++;

	// コマンド別
	if (cmd < PF_MUSIC_CMD_C)
	{
		// 音長設定
		pf_music_info[device].length = pf_music_cmd_to_clock[cmd];
	}
	else
	{
		if (cmd < PF_MUSIC_CMD_REST)
		{
			// 音符
			pf_music_note(device, (PF_MUSIC_NOTE)(cmd - PF_MUSIC_CMD_C));
		}
		else
		{
			// コマンド別
			switch (cmd)
			{
			// 休符
			case PF_MUSIC_CMD_REST:
				pf_music_rest(device);
				break;

			// タイ
			case PF_MUSIC_CMD_TIE:
				pf_music_tie(device);
				break;

			// テンポ
			case PF_MUSIC_CMD_TEMPO:
				pf_music_tempo(device);
				break;

			// クォンタイズ
			case PF_MUSIC_CMD_QUANTIZE:
				pf_music_quantize(device);
				break;

			// オクターブアップ
			case PF_MUSIC_CMD_OCTAVE_UP:
				pf_music_octave_up(device);
				break;

			// オクターブダウン
			case PF_MUSIC_CMD_OCTAVE_DOWN:
				pf_music_octave_down(device);
				break;

			// ループ位置設定
			case PF_MUSIC_CMD_LOOP_SET:
				pf_music_loop_set(device);
				break;

			// ループ位置ジャンプ
			case PF_MUSIC_CMD_LOOP_GO:
				pf_music_loop_go(device);
				break;

			// その他(この1バイトは無視)
			default:
				break;
			}
		}
	}
}

//! @brief		LFO処理
//! @param		[in] device		音楽デバイス
static void pf_music_lfo(u4 device)
{
	// キーオン時かつLFO有効時に限る
	if ((TRUE == pf_music_info[device].on) && (TRUE == pf_music_info[device].lfo_enable))
	{
		// LFOディレイが有効であれば、デクリメント
		if (pf_music_info[device].lfo_keyon > 0)
		{
			pf_music_info[device].lfo_keyon--;
		}
		else
		{
			// 差分を加算
			pf_music_info[device].countertop += pf_music_info[device].lfo_signed;

			// 三角波カウンタをデクリメント
			pf_music_info[device].lfo_current--;

			// 0になったら、周期2倍と反転処理
			if (0 == pf_music_info[device].lfo_current)
			{
				pf_music_info[device].lfo_current = pf_music_info[device].lfo_cycle * 2;
				pf_music_info[device].lfo_signed = 0 - pf_music_info[device].lfo_signed;
			}

			// キーオン
			pf_music_on(device);
		}
	}
}

//! @brief		クロック処理
//! @param		[in] device		音楽デバイス
static void pf_music_clock(u4 device)
{
	// LFOは常に行う
	pf_music_lfo(device);

	// キーオフチェック
	if (pf_music_info[device].off > 0)
	{
		pf_music_info[device].off--;
		if (0 == pf_music_info[device].off)
		{
			// キーオン状態であれば、キーオフ
			if (TRUE == pf_music_info[device].on)
			{
				pf_music_off(device);
				pf_music_info[device].on = FALSE;
			}
		}
	}

	// クロックチェック
	if (pf_music_info[device].clock > 0)
	{
		pf_music_info[device].clock--;
		if (0 == pf_music_info[device].clock)
		{
			// コマンド解釈
			while (0 == pf_music_info[device].clock)
			{
				pf_music_interpret(device);
			}
		}
	}
}

//! @brief		Timerコールバック
//! @param		[in] eventbit	イベントビットパターン(bit0:クロックタイミング)
//! @attention	Timer割り込みコンテキストで実行される
static void pf_music_callback(u4 /*eventbit*/)
{
	u4 loop;

	// オート変数初期化
	loop = 0;

	// 音楽デバイスだけループ
	for (loop = 0; loop < PF_MUSIC_DEVICE_MAX; loop++)
	{
		// 演奏中の場合に限る
		if (TRUE == pf_music_info[loop].playing)
		{
			pf_music_clock(loop);
		}
	}
}

//! @brief		音楽演奏初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
//! @attention	GPIO初期化およびTimer初期化の後に呼び出すこと
void pf_music_init(void)
{
	// PWM初期化
	pf_music_init_pwm();

	// Timerコールバック関数設定
	pf_timer_callback(PF_TIMER_ID_MUSIC, pf_music_callback);
}

//! @brief		演奏開始のためのセットアップ
//! @param		[in] device		音楽デバイス
//! @param		[in] sequence	シーケンスデータ
static void pf_music_setup(u4 device, const PF_MUSIC_CMD *sequence)
{
	// 演奏中フラグ
	pf_music_info[device].playing = TRUE;

	// カレントポインタとループポインタ
	pf_music_info[device].ptr = sequence;
	pf_music_info[device].loop = sequence;

	// 初回クロック
	pf_music_info[device].clock = 1;

	// 音長(デフォルトで四分音符)
	pf_music_info[device].length = 48;

	// キーオン状態
	pf_music_info[device].on = FALSE;

	// キーオフするまでのカウンタ
	pf_music_info[device].off = 0;

	// オクターブ
	pf_music_info[device].octave = 0;

	// クォンタイズ(1～8)
	pf_music_info[device].quantize = 8;

	// タイフラグ
	pf_music_info[device].tie = FALSE;

	// LFO有効フラグ
	pf_music_info[device].lfo_enable = FALSE;

	// LFOディレイ
	pf_music_info[device].lfo_delay = 0;
	pf_music_info[device].lfo_keyon = 0;

	// LFOサイクル
	pf_music_info[device].lfo_cycle = 0;
	pf_music_info[device].lfo_current = 0;

	// LFO差分
	pf_music_info[device].lfo_diff = 0;
	pf_music_info[device].lfo_signed = 0;
}

//! @brief		音楽演奏開始
void pf_music_play(void)
{
	u4 loop;
	BOOL playing;

	// オート変数初期化
	loop = 0;
	playing = FALSE;

	// 音楽デバイスだけループ
	for (loop = 0; loop < PF_MUSIC_DEVICE_MAX; loop++)
	{
		// どちらかが演奏中なら、演奏中フラグを立てる
		if (TRUE == pf_music_info[loop].playing)
		{
			playing = TRUE;
		}
	}

	// ともに演奏停止中の場合
	if (FALSE == playing)
	{
		// 音楽デバイスのセットアップ
		pf_music_setup(0, pf_music_base);
		pf_music_setup(1, pf_music_melody);

		// メロディー側のみLFOを有効にする
		pf_music_info[1].lfo_enable = TRUE;
		pf_music_info[1].lfo_delay = 30;
		pf_music_info[1].lfo_cycle = 12;
		pf_music_info[1].lfo_diff = 14;

		// タイマ開始
		pf_timer_start(PF_TIMER_ID_MUSIC);
	}
}

//! @brief		音楽演奏開始
void pf_music_stop(void)
{
	u4 loop;
	BOOL playing;

	// オート変数初期化
	loop = 0;
	playing = FALSE;

	// 音楽デバイスだけループ
	for (loop = 0; loop < PF_MUSIC_DEVICE_MAX; loop++)
	{
		// どちらかが演奏中なら、演奏中フラグを立てる
		if (TRUE == pf_music_info[loop].playing)
		{
			playing = TRUE;
		}
	}

	// どちらかが演奏中の場合
	if (TRUE == playing)
	{
		// タイマ停止
		pf_timer_stop(PF_TIMER_ID_MUSIC);

		// 演奏停止
		for (loop = 0; loop < PF_MUSIC_DEVICE_MAX; loop++)
		{
			pf_music_info[loop].playing = FALSE;
			pf_music_off(loop);
		}
	}
}
