//! @file		pf_display.c
//! @brief		プラットフォーム(Display)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "pf_gpio.h"
#include "pf_interrupt.h"
#include "pf_timer.h"
#include "pf_display.h"

//! @brief		Display情報構造体
typedef struct PF_DISPLAY_INFO_Tag
{
	BOOL			active;											//!< アクティブフラグ
	BOOL			on;												//!< 表示ON/OFFフラグ
	BOOL			force;											//!< 強制固定フラグ
	u4				row;											//!< 現在の行(0～4)
	BOOL			cols[PF_DISPLAY_ROW_MAX][PF_DISPLAY_COL_MAX];	//!< 列の設定情報
} PF_DISPLAY_INFO;

//! @brief		Display情報
static PF_DISPLAY_INFO pf_display_info;

//! @brief		行→GPIO IDテーブル
static const PF_GPIO_ID pf_display_row_to_id[PF_DISPLAY_ROW_MAX] =
{
	PF_GPIO_ID_ROW1,						//!< ROW1
	PF_GPIO_ID_ROW2,						//!< ROW2
	PF_GPIO_ID_ROW3,						//!< ROW3
	PF_GPIO_ID_ROW4,						//!< ROW4
	PF_GPIO_ID_ROW5,						//!< ROW5
};

//! @brief		列→GPIO IDテーブル
static const PF_GPIO_ID pf_display_col_to_id[PF_DISPLAY_COL_MAX] =
{
	PF_GPIO_ID_COL1,						//!< COL1
	PF_GPIO_ID_COL2,						//!< COL2
	PF_GPIO_ID_COL3,						//!< COL3
	PF_GPIO_ID_COL4,						//!< COL4
	PF_GPIO_ID_COL5,						//!< COL5
};

//! @brief		micro:bit標準フォント(pendolino3) from codal-core(BitmapFont.cpp)
static const u1 pf_display_pendolino3[96][PF_DISPLAY_ROW_MAX * PF_DISPLAY_COL_MAX] =
{
	// 0x20( )
	{
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
	},

	// 0x21(!)
	{
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 1, 0, 0, 0,
	},

	// 0x22(")
	{
		0, 1, 0, 1, 0,
		0, 1, 0, 1, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
	},

	// 0x23(#)
	{
		0, 1, 0, 1, 0,
		1, 1, 1, 1, 1,
		0, 1, 0, 1, 0,
		1, 1, 1, 1, 1,
		0, 1, 0, 1, 0,
	},

	// 0x24($)
	{
		0, 1, 1, 1, 0,
		1, 1, 0, 0, 1,
		0, 1, 1, 1, 0,
		1, 0, 0, 1, 1,
		0, 1, 1, 1, 0,
	},

	// 0x25(%)
	{
		1, 1, 0, 0, 1,
		1, 0, 0, 1, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 1,
		1, 0, 0, 1, 1,
	},

	// 0x26(&)
	{
		0, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 0, 1,
	},

	// 0x27(')
	{
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
	},

	// 0x28(()
	{
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 0, 1, 0, 0,
	},

	// 0x29())
	{
		0, 1, 0, 0, 0,
		0, 0, 1, 0, 0,
		0, 0, 1, 0, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
	},

	// 0x2A(*)
	{
		0, 0, 0, 0, 0,
		0, 1, 0, 1, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 1, 0,
		0, 0, 0, 0, 0,
	},

	// 0x2B(+)
	{
		0, 0, 0, 0, 0,
		0, 0, 1, 0, 0,
		0, 1, 1, 1, 0,
		0, 0, 1, 0, 0,
		0, 0, 0, 0, 0,
	},

	// 0x2C(,)
	{
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
	},

	// 0x2D(-)
	{
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 1, 1, 1, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
	},

	// 0x2E(.)
	{
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 0, 0, 0, 0,
	},

	// 0x2F(/)
	{
		0, 0, 0, 0, 1,
		0, 0, 0, 1, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
		1, 0, 0, 0, 0,
	},

	// 0x30(0)
	{
		0, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
	},

	// 0x31(1)
	{
		0, 0, 1, 0, 0,
		0, 1, 1, 0, 0,
		0, 0, 1, 0, 0,
		0, 0, 1, 0, 0,
		0, 1, 1, 1, 0,
	},

	// 0x32(2)
	{
		1, 1, 1, 0, 0,
		0, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
		1, 0, 0, 0, 0,
		1, 1, 1, 1, 0,
	},

	// 0x33(3)
	{
		1, 1, 1, 1, 0,
		0, 0, 0, 1, 0,
		0, 0, 1, 0, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
	},

	// 0x34(4)
	{
		0, 0, 1, 1, 0,
		0, 1, 0, 1, 0,
		1, 0, 0, 1, 0,
		1, 1, 1, 1, 1,
		0, 0, 0, 1, 0,
	},

	// 0x35(5)
	{
		1, 1, 1, 1, 1,
		1, 0, 0, 0, 0,
		1, 1, 1, 1, 0,
		0, 0, 0, 0, 1,
		1, 1, 1, 1, 0,
	},

	// 0x36(6)
	{
		0, 0, 0, 1, 0,
		0, 0, 1, 0, 0,
		0, 1, 1, 1, 0,
		1, 0, 0, 0, 1,
		0, 1, 1, 1, 0,
	},

	// 0x37(7)
	{
		1, 1, 1, 1, 1,
		0, 0, 0, 1, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
		1, 0, 0, 0, 0,
	},

	// 0x38(8)
	{
		0, 1, 1, 1, 0,
		1, 0, 0, 0, 1,
		0, 1, 1, 1, 0,
		1, 0, 0, 0, 1,
		0, 1, 1, 1, 0,
	},

	// 0x39(9)
	{
		0, 1, 1, 1, 0,
		1, 0, 0, 0, 1,
		0, 1, 1, 1, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
	},

	// 0x3A(:)
	{
		0, 0, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 0, 0, 0, 0,
	},

	// 0x3B(;)
	{
		0, 0, 0, 0, 0,
		0, 0, 1, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
	},

	// 0x3C(<)
	{
		0, 0, 0, 1, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
		0, 0, 1, 0, 0,
		0, 0, 0, 1, 0,
	},

	// 0x3D(=)
	{
		0, 0, 0, 0, 0,
		0, 1, 1, 1, 0,
		0, 0, 0, 0, 0,
		0, 1, 1, 1, 0,
		0, 0, 0, 0, 0,
	},

	// 0x3E(>)
	{
		0, 1, 0, 0, 0,
		0, 0, 1, 0, 0,
		0, 0, 0, 1, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
	},

	// 0x3F(?)
	{
		0, 1, 1, 1, 0,
		1, 0, 0, 0, 1,
		0, 0, 1, 1, 0,
		0, 0, 0, 0, 0,
		0, 0, 1, 0, 0,
	},

	// 0x40(@)
	{
		0, 1, 1, 1, 0,
		1, 0, 0, 0, 1,
		1, 0, 1, 0, 1,
		1, 0, 0, 1, 1,
		0, 1, 1, 0, 0,
	},

	// 0x41(A)
	{
		0, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 1, 1, 1, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
	},

	// 0x42(B)
	{
		1, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 1, 1, 0, 0,
	},

	// 0x43(C)
	{
		0, 1, 1, 1, 0,
		1, 0, 0, 0, 0,
		1, 0, 0, 0, 0,
		1, 0, 0, 0, 0,
		0, 1, 1, 1, 0,
	},

	// 0x44(D)
	{
		1, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		1, 1, 1, 0, 0,
	},

	// 0x45(E)
	{
		1, 1, 1, 1, 0,
		1, 0, 0, 0, 0,
		1, 1, 1, 0, 0,
		1, 0, 0, 0, 0,
		1, 1, 1, 1, 0,
	},

	// 0x46(F)
	{
		1, 1, 1, 1, 0,
		1, 0, 0, 0, 0,
		1, 1, 1, 0, 0,
		1, 0, 0, 0, 0,
		1, 0, 0, 0, 0,
	},

	// 0x47(G)
	{
		0, 1, 1, 1, 0,
		1, 0, 0, 0, 0,
		1, 0, 0, 1, 1,
		1, 0, 0, 0, 1,
		0, 1, 1, 1, 0,
	},

	// 0x48(H)
	{
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		1, 1, 1, 1, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
	},

	// 0x49(I)
	{
		1, 1, 1, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		1, 1, 1, 0, 0,
	},

	// 0x4A(J)
	{
		1, 1, 1, 1, 1,
		0, 0, 0, 1, 0,
		0, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
	},

	// 0x4B(K)
	{
		1, 0, 0, 1, 0,
		1, 0, 1, 0, 0,
		1, 1, 0, 0, 0,
		1, 0, 1, 0, 0,
		1, 0, 0, 1, 0,
	},

	// 0x4C(L)
	{
		1, 0, 0, 0, 0,
		1, 0, 0, 0, 0,
		1, 0, 0, 0, 0,
		1, 0, 0, 0, 0,
		1, 1, 1, 1, 0,
	},

	// 0x4D(M)
	{
		1, 0, 0, 0, 1,
		1, 1, 0, 1, 1,
		1, 0, 1, 0, 1,
		1, 0, 0, 0, 1,
		1, 0, 0, 0, 1,
	},

	// 0x4E(N)
	{
		1, 0, 0, 0, 1,
		1, 1, 0, 0, 1,
		1, 0, 1, 0, 1,
		1, 0, 0, 1, 1,
		1, 0, 0, 0, 1,
	},

	// 0x4F(O)
	{
		0, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
	},

	// 0x50(P)
	{
		1, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 1, 1, 0, 0,
		1, 0, 0, 0, 0,
		1, 0, 0, 0, 0,
	},

	// 0x51(Q)
	{
		0, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
		0, 0, 1, 1, 0,
	},

	// 0x52(R)
	{
		1, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 0, 1,
	},

	// 0x53(S)
	{
		0, 1, 1, 1, 0,
		1, 0, 0, 0, 0,
		0, 1, 1, 0, 0,
		0, 0, 0, 1, 0,
		1, 1, 1, 0, 0,
	},

	// 0x54(T)
	{
		1, 1, 1, 1, 1,
		0, 0, 1, 0, 0,
		0, 0, 1, 0, 0,
		0, 0, 1, 0, 0,
		0, 0, 1, 0, 0,
	},

	// 0x55(U)
	{
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
	},

	// 0x56(V)
	{
		1, 0, 0, 0, 1,
		1, 0, 0, 0, 1,
		1, 0, 0, 0, 1,
		0, 1, 0, 1, 0,
		0, 0, 1, 0, 0,
	},

	// 0x57(W)
	{
		1, 0, 0, 0, 1,
		1, 0, 0, 0, 1,
		1, 0, 1, 0, 1,
		1, 1, 0, 1, 1,
		1, 0, 0, 0, 1,
	},

	// 0x58(X)
	{
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
	},

	// 0x59(Y)
	{
		1, 0, 0, 0, 1,
		0, 1, 0, 1, 0,
		0, 0, 1, 0, 0,
		0, 0, 1, 0, 0,
		0, 0, 1, 0, 0,
	},

	// 0x5A(Z)
	{
		1, 1, 1, 1, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
		1, 0, 0, 0, 0,
		1, 1, 1, 1, 0,
	},

	// 0x5B([)
	{
		0, 1, 1, 1, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 1, 1, 0,
	},

	// 0x5C(\)
	{
		1, 0, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 0, 1, 0, 0,
		0, 0, 0, 1, 0,
		0, 0, 0, 0, 1,
	},

	// 0x5D(])
	{
		0, 1, 1, 1, 0,
		0, 0, 0, 1, 0,
		0, 0, 0, 1, 0,
		0, 0, 0, 1, 0,
		0, 1, 1, 1, 0,
	},

	// 0x5E(^)
	{
		0, 0, 1, 0, 0,
		0, 1, 0, 1, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
	},

	// 0x5F(_)
	{
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		1, 1, 1, 1, 1,
	},

	// 0x60(`)
	{
		0, 1, 0, 0, 0,
		0, 0, 1, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
	},

	// 0x61(a)
	{
		0, 0, 0, 0, 0,
		0, 1, 1, 1, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 1, 1,
	},

	// 0x62(b)
	{
		1, 0, 0, 0, 0,
		1, 0, 0, 0, 0,
		1, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 1, 1, 0, 0,
	},

	// 0x63(c)
	{
		0, 0, 0, 0, 0,
		0, 1, 1, 1, 0,
		1, 0, 0, 0, 0,
		1, 0, 0, 0, 0,
		0, 1, 1, 1, 0,
	},

	// 0x64(d)
	{
		0, 0, 0, 1, 0,
		0, 0, 0, 1, 0,
		0, 1, 1, 1, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 1, 0,
	},

	// 0x65(e)
	{
		0, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 1, 1, 0, 0,
		1, 0, 0, 0, 0,
		0, 1, 1, 1, 0,
	},

	// 0x66(f)
	{
		0, 0, 1, 1, 0,
		0, 1, 0, 0, 0,
		1, 1, 1, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
	},

	// 0x67(g)
	{
		0, 1, 1, 1, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 1, 0,
		0, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
	},

	// 0x68(h)
	{
		1, 0, 0, 0, 0,
		1, 0, 0, 0, 0,
		1, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
	},

	// 0x69(i)
	{
		0, 1, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
	},

	// 0x6A(j)
	{
		0, 0, 0, 1, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 1, 0,
		0, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
	},

	// 0x6B(k)
	{
		1, 0, 0, 0, 0,
		1, 0, 1, 0, 0,
		1, 1, 0, 0, 0,
		1, 0, 1, 0, 0,
		1, 0, 0, 1, 0,
	},

	// 0x6C(l)
	{
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 0, 1, 1, 0,
	},

	// 0x6D(m)
	{
		0, 0, 0, 0, 0,
		1, 1, 0, 1, 1,
		1, 0, 1, 0, 1,
		1, 0, 0, 0, 1,
		1, 0, 0, 0, 1,
	},

	// 0x6E(n)
	{
		0, 0, 0, 0, 0,
		1, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
	},

	// 0x6F(o)
	{
		0, 0, 0, 0, 0,
		0, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
	},

	// 0x70(p)
	{
		0, 0, 0, 0, 0,
		1, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
		1, 1, 1, 0, 0,
		1, 0, 0, 0, 0,
	},

	// 0x71(q)
	{
		0, 0, 0, 0, 0,
		0, 1, 1, 1, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 1, 0,
		0, 0, 0, 1, 0,
	},

	// 0x72(r)
	{
		0, 0, 0, 0, 0,
		0, 1, 1, 1, 0,
		1, 0, 0, 0, 0,
		1, 0, 0, 0, 0,
		1, 0, 0, 0, 0,
	},

	// 0x73(s)
	{
		0, 0, 0, 0, 0,
		0, 0, 1, 1, 0,
		0, 1, 0, 0, 0,
		0, 0, 1, 0, 0,
		1, 1, 0, 0, 0,
	},

	// 0x74(t)
	{
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 1, 1, 0,
		0, 1, 0, 0, 0,
		0, 0, 1, 1, 1,
	},

	// 0x75(u)
	{
		0, 0, 0, 0, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 1, 1,
	},

	// 0x76(v)
	{
		0, 0, 0, 0, 0,
		1, 0, 0, 0, 1,
		1, 0, 0, 0, 1,
		0, 1, 0, 1, 0,
		0, 0, 1, 0, 0,
	},

	// 0x77(w)
	{
		0, 0, 0, 0, 0,
		1, 0, 0, 0, 1,
		1, 0, 0, 0, 1,
		1, 0, 1, 0, 1,
		1, 1, 0, 1, 1,
	},

	// 0x78(x)
	{
		0, 0, 0, 0, 0,
		1, 0, 0, 1, 0,
		0, 1, 1, 0, 0,
		0, 1, 1, 0, 0,
		1, 0, 0, 1, 0,
	},

	// 0x79(y)
	{
		0, 0, 0, 0, 0,
		1, 0, 0, 0, 1,
		0, 1, 0, 1, 0,
		0, 0, 1, 0, 0,
		1, 1, 0, 0, 0,
	},

	// 0x7A(z)
	{
		0, 0, 0, 0, 0,
		1, 1, 1, 1, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
		1, 1, 1, 1, 0,
	},

	// 0x7B({)
	{
		0, 0, 1, 1, 0,
		0, 0, 1, 0, 0,
		0, 1, 1, 0, 0,
		0, 0, 1, 0, 0,
		0, 0, 1, 1, 0,
	},

	// 0x7C(|)
	{
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
	},

	// 0x7D(})
	{
		1, 1, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 1, 1, 0, 0,
		0, 1, 0, 0, 0,
		1, 1, 0, 0, 0,
	},

	// 0x7E(~)
	{
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 1, 1, 0, 0,
		0, 0, 0, 1, 1,
		0, 0, 0, 0, 0,
	},

	// 0x7F(DEL)
	{
		0, 0, 0, 0, 0,
		1, 0, 1, 1, 1,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
	},
};

//! @brief		追加フォント(PF_DISPLAY_ID) from microPython(microbitconstimage.cpp)
static const u1 pf_display_font[PF_DISPLAY_ID_MAX][PF_DISPLAY_ROW_MAX * PF_DISPLAY_COL_MAX] =
{
	// PF_DISPLAY_ID_HEART
	{
		0, 1, 0, 1, 0,
		1, 1, 1, 1, 1,
		1, 1, 1, 1, 1,
		0, 1, 1, 1, 0,
		0, 0, 1, 0, 0,
	},

	// PF_DISPLAY_ID_HAPPY
	{
		0, 0, 0, 0, 0,
		0, 1, 0, 1, 0,
		0, 0, 0, 0, 0,
		1, 0, 0, 0, 1,
		0, 1, 1, 1, 0,
	},

	// PF_DISPLAY_ID_SMILE
	{
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 0,
		1, 0, 0, 0, 1,
		0, 1, 1, 1, 0,
	},

	// PF_DISPLAY_ID_SAD
	{
		0, 0, 0, 0, 0,
		0, 1, 0, 1, 0,
		0, 0, 0, 0, 0,
		0, 1, 1, 1, 0,
		1, 0, 0, 0, 1,
	},

	// PF_DISPLAY_ID_CONFUSED
	{
		0, 0, 0, 0, 0,
		0, 1, 0, 1, 0,
		0, 0, 0, 0, 0,
		0, 1, 0, 1, 0,
		1, 0, 1, 0, 1,
	},

	// PF_DISPALY_ID_ANGRY
	{
		1, 0, 0, 0, 1,
		0, 1, 0, 1, 0,
		0, 0, 0, 0, 0,
		1, 1, 1, 1, 1,
		1, 0, 1, 0, 1,
	},

	// PF_DISPLAY_ID_YES
	{
		0, 0, 0, 0, 0,
		0, 0, 0, 0, 1,
		0, 0, 0, 1, 0,
		1, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
	},

	// PF_DISPLAY_ID_NO
	{
		1, 0, 0, 0, 1,
		0, 1, 0, 1, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 1, 0,
		1, 0, 0, 0, 1,
	},

	// PF_DISPLAY_ID_ARROW_N
	{
		0, 0, 1, 0, 0,
		0, 1, 1, 1, 0,
		1, 0, 1, 0, 1,
		0, 0, 1, 0, 0,
		0, 0, 1, 0, 0,
	},

	// PF_DISPLAY_ID_ARROW_NE
	{
		0, 0, 1, 1, 1,
		0, 0, 0, 1, 1,
		0, 0, 1, 0, 1,
		0, 1, 0, 0, 0,
		1, 0, 0, 0, 0,
	},

	// PF_DISPLAY_ID_ARROW_E
	{
		0, 0, 1, 0, 0,
		0, 0, 0, 1, 0,
		1, 1, 1, 1, 1,
		0, 0, 0, 1, 0,
		0, 0, 1, 0, 0,
	},

	// PF_DISPLAY_ID_ARROW_SE
	{
		1, 0, 0, 0, 0,
		0, 1, 0, 0, 0,
		0, 0, 1, 0, 1,
		0, 0, 0, 1, 1,
		0, 0, 1, 1, 1,
	},

	// PF_DISPLAY_ID_ARROW_S
	{
		0, 0, 1, 0, 0,
		0, 0, 1, 0, 0,
		1, 0, 1, 0, 1,
		0, 1, 1, 1, 0,
		0, 0, 1, 0, 0,
	},

	// PF_DISPLAY_ID_ARROW_SW
	{
		0, 0, 0, 0, 1,
		0, 0, 0, 1, 0,
		1, 0, 1, 0, 0,
		1, 1, 0, 0, 0,
		1, 1, 1, 0, 0,
	},

	// PF_DISPLAY_ID_ARROW_W
	{
		0, 0, 1, 0, 0,
		0, 1, 0, 0, 0,
		1, 1, 1, 1, 1,
		0, 1, 0, 0, 0,
		0, 0, 1, 0, 0,
	},

	// PF_DISPLAY_ID_ARROW_NW
	{
		1, 1, 1, 0, 0,
		1, 1, 0, 0, 0,
		1, 0, 1, 0, 0,
		0, 0, 0, 1, 0,
		0, 0, 0, 0, 1,
	},

	// PF_DISPLAY_ID_TRIANGLE
	{
		0, 0, 0, 0, 0,
		0, 0, 1, 0, 0,
		0, 1, 0, 1, 0,
		1, 1, 1, 1, 1,
		0, 0, 0, 0, 0,
	},

	// PF_DISPLAY_ID_CHESSBOARD
	{
		0, 1, 0, 1, 0,
		1, 0, 1, 0, 1,
		0, 1, 0, 1, 0,
		1, 0, 1, 0, 1,
		0, 1, 0, 1, 0,
	},

	// PF_DISPLAY_ID_DIAMOND
	{
		0, 0, 1, 0, 0,
		0, 1, 0, 1, 0,
		1, 0, 0, 0, 1,
		0, 1, 0, 1, 0,
		0, 0, 1, 0, 0,
	},

	// PF_DISPLAY_ID_SQUARE
	{
		1, 1, 1, 1, 1,
		1, 0, 0, 0, 1,
		1, 0, 0, 0, 1,
		1, 0, 0, 0, 1,
		1, 1, 1, 1, 1,
	},

	// PF_DISPLAY_ID_MUSIC
	{
		0, 0, 1, 0, 0,
		0, 0, 1, 1, 0,
		0, 0, 1, 0, 1,
		1, 1, 1, 0, 0,
		1, 1, 1, 0, 0,
	},
};

//! @brief		Timerコールバック
//! @param		[in] eventbit	イベントビットパターン(bit0:消灯タイミング bit1:点灯タイミング)
//! @attention	Timer割り込みコンテキストで実行される
static void pf_display_callback(u4 eventbit)
{
	PF_GPIO_ID row;
	PF_GPIO_ID col;
	u4 loop;
	BOOL *data;

	// オート変数初期化
	row = pf_display_row_to_id[pf_display_info.row];
	col = PF_GPIO_ID_COL1;
	loop = 0;
	data = &(pf_display_info.cols[0][0]);

	// イベントビットによらず、常に直前の消灯を行う(アクティブフラグで確認)
	if (TRUE == pf_display_info.active)
	{
		// 消灯(列)
		for (loop = 0; loop < PF_DISPLAY_COL_MAX; loop++)
		{
			col = pf_display_col_to_id[loop];
			pf_gpio_output(col, TRUE);
		}

		// 消灯(行)
		pf_gpio_output(row, FALSE);

		// アクティブでない
		pf_display_info.active = FALSE;

		// 次の行へ移る(最終行の場合は最初の行に戻す)
		pf_display_info.row++;
		if (pf_display_info.row >= PF_DISPLAY_ROW_MAX)
		{
			pf_display_info.row = 0;
		}

		// row(GPIOピンのID)を再取得
		row = pf_display_row_to_id[pf_display_info.row];
	}

	// 点灯タイミングか
	if (0 != (eventbit & 2))
	{
		// 表示ON時のみ
		if (TRUE == pf_display_info.on)
		{
			// dataをポイント
			data = &(pf_display_info.cols[pf_display_info.row][0]);

			// 点灯(列)
			for (loop = 0; loop < PF_DISPLAY_COL_MAX; loop++)
			{
				col = pf_display_col_to_id[loop];

				// dataに対応した出力を行う(FALSEで点灯)
				pf_gpio_output(col, *data);
				data++;
			}

			// 点灯(行)
			pf_gpio_output(row, TRUE);

			// アクティブ
			pf_display_info.active = TRUE;
		}
	}
}

//! @brief		Display初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
//! @attention	GPIO初期化およびTimer初期化の後に呼び出すこと
void pf_display_init(void)
{
	u4 row;
	u4 col;

	// オート変数初期化
	row = 0;
	col = 0;

	// Display情報初期化
	pf_display_info.active = FALSE;
	pf_display_info.on = FALSE;
	pf_display_info.force = FALSE;
	pf_display_info.row = 0;

	// Display情報初期化(行)
	for (row = 0; row < PF_DISPLAY_ROW_MAX; row++)
	{
		// Display情報初期化(列)
		for (col = 0; col < PF_DISPLAY_COL_MAX; col++)
		{
			// TRUEで消灯、FALSEで点灯
			pf_display_info.cols[row][col] = TRUE;
		}
	}

	// Timerへコールバック関数を通知
	pf_timer_callback(PF_TIMER_ID_DISPLAY, pf_display_callback);

	// Timerスタート
	pf_timer_start(PF_TIMER_ID_DISPLAY);
}

//! @brief		Display表示(文字)
//! @param		[in] ch			ASCII文字(0x20～0x7F)
void pf_display_ch(char ch)
{
	const u1 *font;
	u4 offset;

	// オート変数初期化
	font = &pf_display_pendolino3[0][0];
	offset = 0;

	// 範囲内であれば、pendolino3フォントを使用する
	offset = (u4)ch;
	if ((offset >= 0x20) && (offset <= 0x7F))
	{
		offset -= 0x20;
		font = &pf_display_pendolino3[offset][0];
	}
	else
	{
		// 範囲外の場合、スペースを表示する
	}

	// Display表示(イメージ)と共通
	pf_display_image(font);
}

//! @brief		Display表示(ID)
//! @param		[in] id			イメージのID
void pf_display_id(PF_DISPLAY_ID id)
{
	const u1 *font;

	// オート変数初期化
	font = &pf_display_font[0][0];

	// 範囲内であれば、追加フォントを使用する
	if (id < PF_DISPLAY_ID_MAX)
	{
		font = &pf_display_font[id][0];
	}
	else
	{
		// 範囲外の場合、ハートを表示する
	}

	// Display表示(イメージ)と共通
	pf_display_image(font);
}

//! @brief		Display表示(イメージ)
//! @param		[in] image		イメージへのポインタ(0=OFF/1=ON)
void pf_display_image(const u1 *image)
{
	u4 enable;
	u4 row;
	u4 col;

	// オート変数初期化
	enable = 0;
	row = 0;
	col = 0;

	// パラメータチェック
	if (NULL != image)
	{
		// 強制フラグチェック
		if (FALSE == pf_display_info.force)
		{
			// 割り込み禁止
			enable = pf_interrupt_local_disable(PF_INTERRUPT_PRI_DISPLAY_TIMER);

			// ワークを設定
			pf_display_info.on = TRUE;

			// 行ループ
			for (row = 0; row < PF_DISPLAY_ROW_MAX; row++)
			{
				// 列ループ
				for (col = 0; col < PF_DISPLAY_COL_MAX; col++)
				{
					// 点灯の場合はFALSE、消灯の場合はTRUEをセット
					if (0 == *image)
					{
						pf_display_info.cols[row][col] = TRUE;
					}
					else
					{
						pf_display_info.cols[row][col] = FALSE;
					}
					image++;
				}
			}

			// 割り込み復元
			pf_interrupt_local_restore(PF_INTERRUPT_PRI_DISPLAY_TIMER, enable);
		}
	}
}

//! @brief		Display停止
void pf_display_off(void)
{
	u4 enable;

	// オート変数初期化
	enable = 0;

	// 強制フラグチェック
	if (FALSE == pf_display_info.force)
	{
		// 割り込み禁止
		enable = pf_interrupt_local_disable(PF_INTERRUPT_PRI_DISPLAY_TIMER);

		// ワークを設定
		pf_display_info.on = FALSE;

		// 割り込み復元
		pf_interrupt_local_restore(PF_INTERRUPT_PRI_DISPLAY_TIMER, enable);
	}
}

//! @brief		Display明るさ指定
//! @param		[in] brightness	明るさ(PF_DISPLAY_BRIGHTNESS_MIN～PF_DISPLAY_BRIGHTNESS_MAX)
//! @attention	範囲外の値は設定禁止
void pf_display_brightness(u4 brightness)
{
	u4 cc[2];

	// オート変数初期化
	cc[0] = MICROSEC_TO_TIMERCC(900);
	cc[1] = MICROSEC_TO_TIMERCC(1000);

	// パラメータチェック
	if ((brightness >= PF_DISPLAY_BRIGHTNESS_MIN) && (brightness <= PF_DISPLAY_BRIGHTNESS_MAX))
	{
		// 強制フラグチェック
		if (FALSE == pf_display_info.force)
		{
			// cc[1]は常に1000usとして、cc[0]までの間隔として40usのマージンを設ける
			cc[0] = MICROSEC_TO_TIMERCC(brightness * 10);

			// CC[i]レジスタを変更する
			pf_timer_cc(PF_TIMER_ID_DISPLAY, cc);
		}
	}
}

//! @brief		Display表示(電圧低下)
//! @attention	これ以降、Display系のAPIは使用できない
void pf_display_powerdown(void)
{
	// 強制フラグOFF
	pf_display_info.force = FALSE;

	// ID指定
	pf_display_id(PF_DISPLAY_ID_NO);

	// 明るさ設定(MINとMAXの中間)
	pf_display_brightness((PF_DISPLAY_BRIGHTNESS_MIN + PF_DISPLAY_BRIGHTNESS_MAX) / 2);

	// 強制フラグON
	pf_display_info.force = TRUE;
}
