//! @file		main.c
//! @brief		メインプログラム

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "pf_clock.h"
#include "pf_gpio.h"
#include "pf_systick.h"
#include "pf_timer.h"
#include "pf_uart.h"
#include "pf_display.h"
#include "pf_i2c.h"
#include "pf_power.h"
#include "pf_button.h"
#include "pf_led.h"
#include "pf_patrol.h"
#include "pf_motor.h"
#include "pf_music.h"

//! @brief		プラットフォーム初期化
//! @attention	初期化順序に注意する
static void pf_init(void)
{
	// Step1: クロックをHFXO(外部発振子)に切り替える
	pf_clock_init();

	// Step2: 低位モジュールを初期化する
	pf_gpio_init();
	pf_systick_init();
	pf_timer_init();

	// Step3: 中位モジュール(低位モジュールに依存するモジュール)を初期化する
	pf_uart_init();
	pf_display_init();
	pf_i2c_init();

	// Step4: 高位モジュール(低位モジュールまたは中位モジュールに依存するモジュール)を初期化する
	pf_power_init();
	pf_button_init();
	pf_led_init();
	pf_patrol_init();
	pf_motor_init();
	pf_music_init();
}

//! @brief		定期タスク(入力系)
static void pf_input_task(void)
{
	pf_power_task();
	pf_uart_task();
	pf_button_task();
	pf_patrol_task();
}

//! @brief		定期タスク(出力系)
static void pf_output_task(void)
{
	pf_led_task();
	pf_motor_task();
}

//! @brief		ボタン過去状態
static BOOL app_button_prev[2];

//! @brief		ボタンOFF→ONフラグ
static BOOL app_button_trigger[2];

//! @brief		ボタンOFF→ON判定
static void app_button_task(void)
{
	BOOL button[2];

	// トリガ初期化
	app_button_trigger[0] = FALSE;
	app_button_trigger[1] = FALSE;

	// ボタン取得
	button[0] = pf_button_get(PF_BUTTON_ID_BUTTON_A);
	button[1] = pf_button_get(PF_BUTTON_ID_BUTTON_B);

	// FALSE→TRUEへの変化でトリガON[0]
	if ((FALSE == app_button_prev[0]) && (TRUE == button[0]))
	{
		app_button_trigger[0] = TRUE;
	}

	// FALSE→TRUEへの変化でトリガON[1]
	if ((FALSE == app_button_prev[1]) && (TRUE == button[1]))
	{
		app_button_trigger[1] = TRUE;
	}

	// 現在のボタン状態を記憶
	app_button_prev[0] = button[0];
	app_button_prev[1] = button[1];
}

//! @brief		モード状態定義
typedef enum APP_MODE_ID_Tag
{
	APP_MODE_ID_IDLE,						//!< アイドル状態
	APP_MODE_ID_SQUARE,						//!< 矩形走行状態
	APP_MODE_ID_LINE,						//!< ライントレース状態
	APP_MODE_ID_MUSIC,						//!< 音楽演奏切り替え状態
	APP_MODE_ID_MAX,						//!< (IDの個数を表す)
} APP_MODE_ID;

//! @brief		モード状態遷移
static APP_MODE_ID app_mode;

//! @brief		走行状態遷移
static BOOL app_run;

//! @brief		音楽演奏状態遷移
static BOOL app_play;

//! @brief		走行状態ステップ
static u4 app_step;

//! @brief		走行方向
static PF_MOTOR_DIR app_dir;

//! @brief		アイドル時のディスプレイ個数
#define APP_DISPLAY_MAX				((u4)10U)

//! @brief		アイドル時のディスプレイテーブル
static const PF_DISPLAY_ID app_display_idle[10] =
{
	PF_DISPLAY_ID_HEART,					//!< ハート
	PF_DISPLAY_ID_HAPPY,					//!< HAPPY(嬉しい)
	PF_DISPLAY_ID_SMILE,					//!< SMILE(笑い)
	PF_DISPLAY_ID_SAD,						//!< SAD(悲しい)
	PF_DISPLAY_ID_CONFUSED,					//!< CONFUSED(混乱)
	PF_DISPLAY_ID_ANGRY,					//!< ANGRY(怒り)
	PF_DISPLAY_ID_YES,						//!< YES(チェックマーク)
	PF_DISPLAY_ID_NO,						//!< NO(×マーク)
	PF_DISPALY_ID_TRIANGLE,					//!< 三角形
	PF_DISPLAY_ID_CHESSBOARD,				//!< チェスボード(縞模様)
};

//! @brief		走行時のディスプレイテーブル
static const PF_DISPLAY_ID app_display_run[5] =
{
	PF_DISPLAY_ID_ARROW_N,					//!< 前進
	PF_DISPLAY_ID_ARROW_W,					//!< 左回転
	PF_DISPLAY_ID_ARROW_E,					//!< 右回転
	PF_DISPLAY_ID_ARROW_S,					//!< 後進
	PF_DISPLAY_ID_YES,						//!< 停止
};

//! @brief		音楽停止時のイメージ
static const u1 app_display_musicoff[PF_DISPLAY_ROW_MAX * PF_DISPLAY_COL_MAX] =
{
	0, 0, 1, 0, 0,
	0, 0, 1, 0, 0,
	0, 0, 1, 0, 0,
	1, 1, 1, 0, 0,
	1, 1, 1, 0, 0,
};

//! @brief		矩形走行単位構造体
typedef struct APP_SQUARE_UNIT_Tag
{
	PF_MOTOR_DIR	dir;					//!< 駆動方向
	u4				cycle;					//!< 動作周期
} APP_SQUARE_UNIT;

//! @brief		矩形走行テーブル
static const APP_SQUARE_UNIT app_square_table[9] =
{
	// 前進
	{
		PF_MOTOR_DIR_FORWARD,
		240,
	},

	// 右回転
	{
		PF_MOTOR_DIR_RIGHT,
		76,
	},

	// 前進
	{
		PF_MOTOR_DIR_FORWARD,
		240,
	},

	// 左回転
	{
		PF_MOTOR_DIR_LEFT,
		74,
	},

	// 後進
	{
		PF_MOTOR_DIR_BACKWARD,
		240,
	},

	// 左回転
	{
		PF_MOTOR_DIR_LEFT,
		72,
	},

	// 前進
	{
		PF_MOTOR_DIR_FORWARD,
		240,
	},

	// 右回転
	{
		PF_MOTOR_DIR_RIGHT,
		76,
	},

	// 終端
	{
		PF_MOTOR_DIR_STOP,
		0,
	},
};

//! @brief		モード状態遷移(Aボタンで遷移)
static void app_mode_task(void)
{
	// Aボタンで遷移
	if (TRUE == app_button_trigger[0])
	{
		app_mode++;
		if (app_mode >= APP_MODE_ID_MAX)
		{
			app_mode = 0;
		}

		// 常に停止状態、ステップ戻す
		app_run = FALSE;
		app_step = 0;
	}

	// Bボタンでトグル切り替え
	if (TRUE == app_button_trigger[1])
	{
		switch (app_mode)
		{
		// Bボタンでディスプレイ表示IDが変化する
		case APP_MODE_ID_IDLE:
			// ステップを上げる。APP_DISPLAY_MAXで1回転
			app_step++;
			if (app_step >= APP_DISPLAY_MAX)
			{
				app_step = 0;
			}
			break;

		// Bボタンで音楽演奏⇔音楽停止のトグル切り替え
		case APP_MODE_ID_MUSIC:
			if (FALSE == app_play)
			{
				app_play = TRUE;
			}
			else
			{
				app_play = FALSE;
			}
			break;

		// Bボタンで走行モード⇔停止モードのトグル切り替え
		default:
			if (FALSE == app_run)
			{
				app_run = TRUE;
			}
			else
			{
				app_run = FALSE;
			}

			// ステップを戻す
			app_step = 0;
			break;
		}
	}
}

//! @brief		ディスプレイ表示
static void app_display_task(void)
{
	// モードに応じたディスプレイ表示
	switch (app_mode)
	{
	// アイドル状態
	case APP_MODE_ID_IDLE:
		pf_display_id(app_display_idle[app_step]);
		break;

	// 音楽演奏切り替え状態
	case APP_MODE_ID_MUSIC:
		if (FALSE == app_play)
		{
			pf_display_image(app_display_musicoff);
		}
		else
		{
			pf_display_id(PF_DISPLAY_ID_MUSIC);
		}
		break;

	// 走行状態
	case APP_MODE_ID_SQUARE:
		if (FALSE == app_run)
		{
			pf_display_id(PF_DISPLAY_ID_SQUARE);
		}
		else
		{
			pf_display_id(app_display_run[app_dir]);
		}
		break;

	case APP_MODE_ID_LINE:
		if (FALSE == app_run)
		{
			pf_display_id(PF_DISPLAY_ID_DIAMOND);
		}
		else
		{
			pf_display_id(app_display_run[app_dir]);
		}
		break;

	default:
		break;
	}
}

//! @brief		LED表示
static void app_led_task(void)
{
	PF_PATROL_COLOR left;
	PF_PATROL_COLOR right;

	// 取得
	left = pf_patrol_line(PF_PATROL_ID_L);
	right = pf_patrol_line(PF_PATROL_ID_R);

	// stepが32の倍数(160ms)で点滅
	if (app_step & 0x0020)
	{
		// 常に消灯
		pf_led_ctrl(PF_LED_ID_L, FALSE);
		pf_led_ctrl(PF_LED_ID_R, FALSE);
	}
	else
	{
		// 白色で点灯(左)
		if (PF_PATROL_COLOR_WHITE == left)
		{
			pf_led_ctrl(PF_LED_ID_L, TRUE);
		}
		else
		{
			pf_led_ctrl(PF_LED_ID_L, FALSE);
		}

		// 白色で点灯(右)
		if (PF_PATROL_COLOR_WHITE == right)
		{
			pf_led_ctrl(PF_LED_ID_R, TRUE);
		}
		else
		{
			pf_led_ctrl(PF_LED_ID_R, FALSE);
		}
	}
}

//! @brief		走行(矩形)
static void app_run_square(void)
{
	const APP_SQUARE_UNIT *unit;
	u4 step;
	BOOL next;

	// オート変数初期化
	unit = &app_square_table[0];
	step = app_step;
	next = TRUE;

	// 現在のapp_stepより、適切なunitを割り出す
	while (step >= unit->cycle)
	{
		// unitを進める
		step -= unit->cycle;
		unit++;

		// unitが終端に来たら、巻き戻す
		if (0 == unit->cycle)
		{
			unit = &app_square_table[0];
			next = FALSE;
		}
	}

	// nextフラグでapp_stepを更新
	if (TRUE == next)
	{
		app_step++;
	}
	else
	{
		app_step = 0;
	}

	// 方向決定
	app_dir = unit->dir;
	pf_motor_drive(app_dir);
}

//! @brief		走行(ライントレース)
static void app_run_line(void)
{
	PF_PATROL_COLOR left;
	PF_PATROL_COLOR right;
	PF_MOTOR_DIR dir;

	// オート変数初期化
	left = PF_PATROL_COLOR_WHITE;
	right = PF_PATROL_COLOR_WHITE;
	dir = PF_MOTOR_DIR_BACKWARD;

	// 取得
	left = pf_patrol_line(PF_PATROL_ID_L);
	right = pf_patrol_line(PF_PATROL_ID_R);

	// 共に白色で前進
	if ((PF_PATROL_COLOR_WHITE == left) && (PF_PATROL_COLOR_WHITE == right))
	{
		dir = PF_MOTOR_DIR_FORWARD;
	}

	// 左が黒色なら右回転
	if ((PF_PATROL_COLOR_BLACK == left) && (PF_PATROL_COLOR_WHITE == right))
	{
		dir = PF_MOTOR_DIR_RIGHT;
	}

	// 右が黒色なら左回転
	if ((PF_PATROL_COLOR_WHITE == left) && (PF_PATROL_COLOR_BLACK == right))
	{
		dir = PF_MOTOR_DIR_LEFT;
	}

	// モータ速度(後進のみ遅い)
	if (PF_MOTOR_DIR_BACKWARD == dir)
	{
		pf_motor_speed(50);
	}
	else
	{
		pf_motor_speed(100);
	}

	// モータ設定
	app_dir = dir;
	pf_motor_drive(app_dir);

	// ステップ更新処理
	app_step++;
}

//! @brief		アプリケーションタスク
static void app_task(void)
{
	BOOL run;

	// オート変数初期化
	run = FALSE;

	// ボタン
	app_button_task();

	// モード切替
	app_mode_task();

	// ディスプレイ表示
	app_display_task();

	// LED表示
	app_led_task();

	// 走行(矩形)
	if ((APP_MODE_ID_SQUARE == app_mode) && (TRUE == app_run))
	{
		app_run_square();
		run = TRUE;
	}

	// 走行(ライントレース)
	if ((APP_MODE_ID_LINE == app_mode) && (TRUE == app_run))
	{
		app_run_line();
		run = TRUE;
	}

	// 走行状態以外なら停止
	if (FALSE == run)
	{
		pf_motor_drive(PF_MOTOR_DIR_STOP);
		pf_music_stop();
	}
	else
	{
		// 音楽演奏フラグに従い演奏
		if (TRUE == app_play)
		{
			pf_music_play();
		}
	}
}

//! @brief		メインプログラム
//! @remarks	マイコンリセット後、startup.cの_start()に制御が移り、その後main()が呼び出される
int main(void)
{
	// プラットフォーム初期化
	pf_init();

	// 無限ループ
	while (1)
	{
		// 制御周期まで待つ
		pf_systick_sync();

		// 定期タスク(入力系)
		pf_input_task();

		// 定期タスク(アプリケーション)
		app_task();

		// 定期タスク(出力系)
		pf_output_task();
	}
}
