//! @file		pf_i2c.h
//! @brief		プラットフォーム(I2C)ヘッダファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#ifndef PF_I2C_H
#define PF_I2C_H

#include "pf_types.h"

//! @brief		I2Cコールバック関数
//! @param		[in] status		通信ステータス(TRUE=成功/FALSE=失敗)
typedef void (*PF_I2C_CALLBACK)(BOOL status);

//! @brief		I2C通信先のID
typedef enum PF_I2C_ID_Tag
{
	PF_I2C_ID_ACCELEROMETER,				//!< 加速度センサ
	PF_I2C_ID_MAGNETOMETER,					//!< 磁気センサ
	PF_I2C_ID_MAQUEEN_MOTOR,				//!< MAQUEEN モータ
	PF_I2C_ID_MAX							//!< (IDの個数を表す)
} PF_I2C_ID;

//! @brief		I2Cエラー情報構造体
typedef struct PF_I2C_ERROR_Tag
{
	u4				overrun;				//!< オーバーラン発生回数(受信動作が不正)
	u4				anack;					//!< アドレスNACK発生回数
	u4				dnack;					//!< データNACK発生回数
} PF_I2C_ERROR;

//! @brief		I2C初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_i2c_init(void);

//! @brief		I2C送信
//! @param		[in] id			I2C通信先のID
//! @param		[in] buf		送信バッファへのポインタ
//! @param		[in] bytes		送信バイト数
//! @return		送信ステータス(TRUE=送信開始/FALSE=デバイスBUSY)
//! @attention	呼び出し側で送信バッファのライフタイムを保証すること
BOOL pf_i2c_send(PF_I2C_ID id, const u1 *buf, u4 bytes);

//! @brief		I2C受信
//! @param		[in] id			I2C通信先のID
//! @param		[in] reg		受信レジスタ
//! @param		[in] buf		受信バッファへのポインタ
//! @param		[in] bytes		受信バイト数
//! @return		受信ステータス(TRUE=受信開始/FALSE=デバイスBUSY)
BOOL pf_i2c_recv(PF_I2C_ID id, u1 reg, u1 *buf, u4 bytes);

//! @brief		I2Cエラー情報取得
//! @param		[in] id			I2C通信先のID
//! @param		[out] error		エラー情報構造体へのポインタ
//! @remarks	プラットフォーム内部のエラー情報はクリアされる
void pf_i2c_error(PF_I2C_ID id, PF_I2C_ERROR *error);

//! @brief		I2Cコールバック関数設定
//! @param		[in] id			I2C通信先のID
//! @param		[in] func		I2Cコールバック関数へのポインタ
//! @attention	I2Cコールバック関数は割り込みコンテキストで呼び出される
void pf_i2c_callback(PF_I2C_ID id, PF_I2C_CALLBACK func);

//! @brief		I2C割り込みハンドラ(チャネル0=内部I2Cバス)
//! @attention	データ競合(割り込み干渉)に注意する
void SPIM0_SPIS0_TWIM0_TWIS0_SPI0_TWI0_IRQHandler(void);

//! @brief		I2C割り込みハンドラ(チャネル1=外部I2Cバス)
//! @attention	データ競合(割り込み干渉)に注意する
void SPIM1_SPIS1_TWIM1_TWIS1_SPI1_TWI1_IRQHandler(void);

#endif  // PF_I2C_H
