package framework;

import flash.geom.Point;
import flash.errors.Error;
import framework.util.CsvLoader;
import flash.display.PixelSnapping;
import flash.display.Bitmap;
import flash.display.BitmapData;
import flash.display.Graphics;
import flash.display.Shape;
import flash.geom.ColorTransform;
import flash.geom.Matrix;
import flash.geom.Rectangle;
import flash.display.BitmapData;
import openfl.Assets;

import framework.util.Maths;

private class SpriteInfo {
    public var tex:String;
    public var name:String;
    public var x:Int;
    public var y:Int;
    public var w:Int;
    public var h:Int;
    public function new(tex:String, name:String, x:Int, y:Int, w:Int, h:Int) {
        this.tex = tex;
        this.name = name;
        this.x = x;
        this.y = y;
        this.w = w;
        this.h = h;
    }
    public function dump():Void {
        trace("tex:"+tex);
        trace("name:"+name);
        trace("x:"+x);
        trace("y:"+y);
        trace("w:"+w);
        trace("h:"+h);
    }
}

/**
 * 描画管理クラス
 * @author syun
 */
class GraphicManager {
    private static inline var ROOT_PATH = "assets/";
    /**
     * スクリーン
     */
    private var m_Graphics:Graphics;
    private var m_Screen:BitmapData;
    private var m_Capture:Bitmap; // キャプチャー画面
    private var m_Pool:Map<String, Bitmap> = new Map<String, Bitmap>();
    private var m_Sprits:Map<String, SpriteInfo> = new Map<String, SpriteInfo>();
    private var m_Rectangle:Rectangle; // 描画に使いまわすRectangle
    private var m_Point:Point;
    /**
     * 描画色
     */
    private var m_Color:Int = 0x000000;
    /**
     * 半透明
     */
    private var m_Alpha:Float = 1.0;
    /**
     * コンストラクタ
     * @param	screen
     */
    public function new(screen:BitmapData, g:Graphics) {
        m_Graphics = g;
        m_Screen = screen;
        var w = screen.width;
        var h = screen.height;
        m_Capture = new Bitmap(new BitmapData(w, h, false));
        m_Rectangle = new Rectangle();
        m_Point = new Point();
    }

    public function getCapture():Bitmap {
        return m_Capture;
    }
    /**
     * キャプチャーする
     */
    public function capture():Void {
        m_Capture.bitmapData.draw(m_Screen);
    }

    /**
     * 画像ファイルの読み込み
     * @param name キー
     * @param path ファイルパス
     **/
    public function load(name:String, path:String):Void {
        loadEx(name, ROOT_PATH+path);
    }
    public function loadEx(name:String, path:String):Void {
        var bmpData:BitmapData = Assets.getBitmapData(path);
        if(bmpData == null) {
            throw new Error("Error: File not found. " + path);
        }
        var bmp = new Bitmap(bmpData, PixelSnapping.AUTO, true);
        m_Pool.set(name, bmp);
    }
    public function get(name:String):Bitmap {
        return m_Pool.get(name);
    }
    public function loadSpriteFromCsv(tex:String, filepath:String): Void {
        var csv:CsvLoader = new CsvLoader();
        csv.load(filepath);
        for(id in csv.keys()) {
            var name = csv.getString(id, "name");
            var x = csv.getInt(id, "x");
            var y = csv.getInt(id, "x");
            var w = csv.getInt(id, "w");
            var h = csv.getInt(id, "h");
            addSprite(tex, name, x, y, w, h);
        }
    }
    public function addSprite(tex:String, name:String, x:Int, y:Int, w:Int, h:Int):Void {
        var spr:SpriteInfo = new SpriteInfo(tex, name, x, y, w, h);
        m_Sprits.set(name, spr);
    }
    public function drawSprite(key:String, x:Float, y:Float, rot:Float=0, sx:Float=1, sy:Float=1):Void {
        var spr:SpriteInfo = m_Sprits.get(key);
        var img:Bitmap = get(spr.tex);
        draw2(img, x, y, spr.x, spr.y, spr.w, spr.h, sx, sy);
    }
    public function drawSpriteEx(key:String, cx:Float, cy:Float, rot:Float=0, sx:Float=1, sy:Float=1):Void {
        var spr:SpriteInfo = m_Sprits.get(key);
        var x = cx - spr.w/2;
        var y = cy - spr.h/2;
        drawSprite(key, x, y, rot, sx, sy);
    }
    /**
     * キャプチャー画面の描画（拡大α）
     * @param	v 0～1のパラメータ
     */
    public function drawCaptureScale(v:Float):Void {
        var ct:ColorTransform = new ColorTransform(1, 1, 1, v);
        var m:Matrix = new Matrix();
        var s:Float = 1 + (1 - v) * 5;
        m.translate(-m_Capture.width / 2, -m_Capture.height / 2);
        m.scale(s, s);
        m.translate(m_Capture.width / 2, m_Capture.height / 2);
        m_Screen.draw(m_Capture, m, ct);
    }
    /**
     * キャプチャー画面の描画（分割して描画）
     * @param	v
     */
    public function drawCaptureDiv(v:Float):Void {
        var dSize = 64; // 分割サイズ
        var ofs = dSize * (1-v) / 2;
        var hSize:Int = cast(m_Capture.bitmapData.height/dSize, Int);
        var wSize:Int = cast(m_Capture.bitmapData.width/dSize, Int);
//        for (var j:int = 0; j < m_Capture.bitmapData.height/dSize; j++)
        for(j in 0...hSize) {
//            for (var i:int = 0; i < m_Capture.bitmapData.width/dSize; i++)
            for(i in 0...wSize) {
                var x:Float = i * dSize + ofs;
                var y:Float = j * dSize + ofs;
                draw2(m_Capture, x, y, x, y, dSize*v, dSize*v);
            }
        }
    }

    /**
     * 半透明の設定
     * @param	alpha
     */
    public function setAlpha(alpha:Float):Void {
        m_Alpha = alpha;
    }
    /**
     * 描画色の設定
     * @param	color 色
     * @param alpha 半透明値
     */
    public function setColor(color:Int = 0x000000, alpha:Float=1):Void {
        m_Color = color;
        m_Alpha = alpha;
    }

    /**
     * 塗りつぶしなしの矩形描画
     * @param	x
     * @param	y
     * @param	w
     * @param	h
     */
    public function drawRect(x:Float, y:Float, w:Float, h:Float):Void {
        var x1 = x;
        var y1 = y;
        var x2 = x + w;
        var y2 = y + h;
        if(m_Alpha == 1) {
            // 高速描画
            m_Rectangle.setTo(x1, y1, w, 1);
            m_Screen.fillRect(m_Rectangle, m_Color);
            m_Rectangle.setTo(x1, y1, 1, h);
            m_Screen.fillRect(m_Rectangle, m_Color);
            m_Rectangle.setTo(x1, y2, w, 1);
            m_Screen.fillRect(m_Rectangle, m_Color);
            m_Rectangle.setTo(x2, y1, 1, h);
            m_Screen.fillRect(m_Rectangle, m_Color);
            return;
        }

        drawLine(x1, y1, x2, y1);
        drawLine(x2, y1, x2, y2);
        drawLine(x1, y2, x2, y2);
        drawLine(x1, y1, x1, y2);
    }
    /**
     * 高速な矩形描画
     * @param	rect
     */
    public function fillRect2(rect:Rectangle):Void {
        m_Screen.fillRect(rect, m_Color);
    }
    /**
     * 矩形の描画（塗りつぶし）
     * @param	x
     * @param	y
     * @param	w
     * @param	h
     */
    public function fillRect(x:Float=0, y:Float=0, w:Float=-1, h:Float=-1, rot:Float=0):Void {
        if (w == -1 || h == -1) {
            w = Main.WIDTH;
            h = Main.HEIGHT;
        }

        if(rot == 0 && m_Alpha == 1) {
            // 回転なし・半島眼なしの場合は高速描画する
            var r:Rectangle = new Rectangle(x, y, w, h);
            m_Screen.fillRect(r, m_Color);
            return;
        }

        var s = new Shape();
        var m = new Matrix();
        s.graphics.beginFill(m_Color, m_Alpha);
        s.graphics.drawRect( -w / 2, -h / 2, w, h);
        s.graphics.endFill();
        m.rotate(Maths.deg2rad(rot));
        m.translate(x+w/2, y+h/2);
        m_Screen.draw(s, m);
    }
    /**
     * 矩形の描画（中心座標を指定）
     * @param	cx 中心座標X
     * @param	cy 中心座標Y
     * @param	w 幅
     * @param	h 高さ
     * @param rot 回転角度
     */
    public function fillRectEx(cx:Float, cy:Float, w:Float, h:Float, rot:Float=0):Void {
        var x = cx - w;
        var y = cy - h;
        fillRect(x, y, w * 2, h * 2, rot);
    }
    public function clearScreen(color:Int=0x000000):Void {
        m_Rectangle.setTo(0, 0, Application.Width(), Application.Height());
        m_Screen.fillRect(m_Rectangle, color);
    }

    /**
     * 線を引く
     * @param	x1
     * @param	y1
     * @param	x2
     * @param	y2
     */
    public function drawLine(x1:Float, y1:Float, x2:Float, y2:Float):Void {
        var rect:Rectangle;
        if (x1 == x2) {
            rect = new Rectangle(x1, y1, 1, y2 > y1 ? y2-y1 : y1 - y2);
            fillRect2(rect);
            return;
        }
        if(y1 == y2) {
            rect = new Rectangle(x1, y1, x2 > x1 ? x2 - x1 : x1 - x2, 1);
            fillRect2(rect);
            return;
        }
        var s:Shape = new Shape();
        s.graphics.lineStyle(1, m_Color, m_Alpha);
        s.graphics.moveTo(x1, y1);
        s.graphics.lineTo(x2, y2);
        m_Screen.draw(s);
    }

    public function drawCircle(cx:Float, cy:Float, r:Float, thick:Float=1):Void {
        var s:Shape = new Shape();
        s.graphics.lineStyle(thick, m_Color, m_Alpha);
        s.graphics.drawCircle(cx, cy, r);
        m_Screen.draw(s);
    }

    /**
     * 画像をスクロールする
     * @param	key
     * @param	dx
     * @param	dy
     */
    public function scrollImage(key:String, dx:Float, dy:Float):Void {
        //var img:Bitmap = ResourceImage.getValue(key);
        //img.scrollRect = new Rectangle(dx, dy, img.width-dx, img.height-dy);
    }

    public function draw(key:String, x:Float, y:Float, rot:Float=0, sx:Float=1, sy:Float=1):Void {
        var img:Bitmap = m_Pool.get(key);
        var m:Matrix = new Matrix();
        if (rot != 0) {
            // 中心を原点に持っていってから回転させる
            var w = img.width/2;
            var h = img.height/2;
            m.translate(-w, -h);
            m.rotate(Maths.deg2rad(-rot));
            m.translate(w, h);
        }
        m.translate(x, y);
        m_Screen.draw(img, m);
    }
    public function drawEx(key:String, cx:Float, cy:Float, rot:Float=0, sx:Float=1, sy:Float=1):Void {
        var img:Bitmap = m_Pool.get(key);
        var x = cx - img.width / 2;
        var y = cy - img.height / 2;
        draw(key, x, y, rot);
    }

    public function draw2(img:Bitmap, x:Float, y:Float, ox:Float=0, oy:Float=0, ow:Float=0, oh:Float=0, sx:Float=1, sy:Float=1):Void {
        var m:Matrix = new Matrix();
        if (ow == 0 || oh == 0) {
            ow = img.width;
            oh = img.height;
        }
        img.scrollRect = new Rectangle(ox, oy, ow, oh);

        if(sx == 1 && sy == 1) {
            // スケールなしの場合は高速描画
            m_Rectangle.setTo(ox, oy, ow, oh);
            m_Point.setTo(x, y);
            m_Screen.copyPixels(img.bitmapData, m_Rectangle, m_Point, null, null, true);
            return;
        }

        m.scale(sx, sy);
        m.translate(x, y);
        m_Screen.draw(img, m);
    }

    /**
     * 回転の連番画像を描画する
     * @param	key
     * @param	pattern
     * @param	cx
     * @param	cy
     * @param	deg
     */
    public function drawRot(key:String, pattern:Int, cx:Float, cy:Float, deg:Float):Void {
        var img:Bitmap = m_Pool.get(key);
        var rot:Int = Math.floor(deg);
        if (rot < 0) { rot = 360 + rot; }
        var idx = Math.floor(rot / (360 / pattern));
        var size = img.width / pattern;
        var ox = idx * size;
        //drawEx(key, cx, cy, ox, 0, size, size);
    }

    /**
     * フォントの描画
     * @param	f
     */
/*
    public function drawFont(f:FontObj):Void
    {
        m_Screen.draw(f.getTextField(), f.getMatrix());
    }
*/
}
