/*
 *  TLV - Trace Log Visualizer
 *
 *  Copyright (C) 2008-2010 by Nagoya Univ., JAPAN
 *
 *  上記著作権者は，以下の(1)〜(4)の条件を満たす場合に限り，本ソフトウェ
 *  ア（本ソフトウェアを改変したものを含む．以下同じ）を使用・複製・改
 *  変・再配布（以下，利用と呼ぶ）することを無償で許諾する．
 *  (1) 本ソフトウェアをソースコードの形で利用する場合には，上記の著作
 *      権表示，この利用条件および下記の無保証規定が，そのままの形でソー
 *      スコード中に含まれていること．
 *  (2) 本ソフトウェアを，ライブラリ形式など，他のソフトウェア開発に使
 *      用できる形で再配布する場合には，再配布に伴うドキュメント（利用
 *      者マニュアルなど）に，上記の著作権表示，この利用条件および下記
 *      の無保証規定を掲載すること．
 *  (3) 本ソフトウェアを，機器に組み込むなど，他のソフトウェア開発に使
 *      用できない形で再配布する場合には，次のいずれかの条件を満たすこ
 *      と．
 *    (a) 再配布に伴うドキュメント（利用者マニュアルなど）に，上記の著
 *        作権表示，この利用条件および下記の無保証規定を掲載すること．
 *    (b) 再配布の形態を，別に定める方法によって，TOPPERSプロジェクトに
 *        報告すること．
 *  (4) 本ソフトウェアの利用により直接的または間接的に生じるいかなる損
 *      害からも，上記著作権者およびTOPPERSプロジェクトを免責すること．
 *      また，本ソフトウェアのユーザまたはエンドユーザからのいかなる理
 *      由に基づく請求からも，上記著作権者およびTOPPERSプロジェクトを
 *      免責すること．
 *
 *  本ソフトウェアは，無保証で提供されているものである．上記著作権者お
 *  よびTOPPERSプロジェクトは，本ソフトウェアに関して，特定の使用目的
 *  に対する適合性も含めて，いかなる保証も行わない．また，本ソフトウェ
 *  アの利用により直接的または間接的に生じたいかなる損害に関しても，そ
 *  の責任を負わない．
 *
 *  @(#) $Id$
 */
using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using NU.OJL.MPRTOS.TLV.Base;

namespace NU.OJL.MPRTOS.TLV.Core
{
	/// <summary>
	/// 共通形式トレースログおよびマーカー等の情報を表すクラス
	/// </summary>
	public class TraceLogVisualizerData : IFileContextData
	{
		private bool _isDirty = false;
		private ResourceData _ResourceData;
		private TraceLogData _TraceLogData;
		private VisualizeData _VisualizeData;
		private SettingData _SettingData;
		private VisualizeShapeData _VisualizeShapeData;

		/// <summary>
		/// データが更新されたときに発生するイベント
		/// </summary>
		public event EventHandler<GeneralEventArgs<bool>> IsDirtyChanged = null;

		/// <summary>
		/// データが更新されているかどうか
		/// </summary>
		public bool IsDirty
		{
			get { return _isDirty; }
			set
			{
				if (_isDirty != value)
				{
					_isDirty = value;

					if (IsDirtyChanged != null)
						IsDirtyChanged(this, new GeneralEventArgs<bool>(_isDirty));
				}
			}
		}
		/// <summary>
		/// リソースデータ
		/// </summary>
		public ResourceData ResourceData { get { return _ResourceData; } private set { _ResourceData = value; } }
		/// <summary>
		/// トレースログのリスト
		/// </summary>
		public TraceLogData TraceLogData { get { return _TraceLogData; } private set { _TraceLogData = value; } }
		/// <summary>
		/// 可視化データ
		/// </summary>
		public VisualizeData VisualizeData { get { return _VisualizeData; } set { _VisualizeData = value; } }
		/// <summary>
		/// 設定データ
		/// </summary>
		public SettingData SettingData { get { return _SettingData; } set { _SettingData = value; } }

		/// <summary>
		/// 図形データ
		/// </summary>
		public VisualizeShapeData VisualizeShapeData { get { return _VisualizeShapeData; } set { _VisualizeShapeData = value; } }

		/// <summary>
		/// <c>CommonFormatTraceLog</c>のインスタンスを生成する
		/// </summary>
		public TraceLogVisualizerData()
		{
		}

		/// <summary>
		/// <c>CommonFormatTraceLog</c>のインスタンスを生成する
		/// </summary>
		/// <param name="resourceData">共通形式のリソースデータ</param>
		/// <param name="traceLogData">共通形式のトレースログデータ</param>
		public void Initialize(ResourceData resourceData, TraceLogData traceLogData, VisualizeData visualizeData, SettingData settingData, VisualizeShapeData shapesData)
		{
			ResourceData = resourceData;
			TraceLogData = traceLogData;
			VisualizeData = visualizeData;
			SettingData = settingData;
			VisualizeShapeData = shapesData;

			setVisualizeRuleToEvent();
		}

		private void setVisualizeRuleToEvent()
		{
			foreach (VisualizeRule rule in VisualizeData.VisualizeRules)
			{
				foreach (Event evnt in rule.Shapes)
				{
					evnt.SetVisualizeRuleName(rule.Name);
				}
			}
		}

		/// <summary>
		/// パスを指定してシリアライズ
		/// </summary>
		/// <param name="path">保存する先のパス</param>
		public void Serialize(string path)
		{
			// 一時ディレクトリ作成
			if (!Directory.Exists(ApplicationData.Setting.TemporaryDirectoryPath))
				Directory.CreateDirectory(ApplicationData.Setting.TemporaryDirectoryPath);

			string targetTmpDirPath = Path.Combine(ApplicationData.Setting.TemporaryDirectoryPath, "tlv_convertRuleTmp_" + DateTime.Now.Ticks.ToString() + @"\");
			Directory.CreateDirectory(targetTmpDirPath);

			IZip zip = ApplicationFactory.Zip;

			string name = Path.GetFileNameWithoutExtension(path);

			File.WriteAllText(targetTmpDirPath + name + "." + Properties.Resources.ResourceFileExtension, ResourceData.ToJson());
			File.WriteAllText(targetTmpDirPath + name + "." + Properties.Resources.TraceLogFileExtension, TraceLogData.TraceLogs.ToJson());
			File.WriteAllText(targetTmpDirPath + name + "." + Properties.Resources.VisualizeRuleFileExtension, VisualizeData.ToJson());
			File.WriteAllText(targetTmpDirPath + name + "." + Properties.Resources.SettingFileExtension, SettingData.ToJson());
			File.WriteAllText(targetTmpDirPath + name + "." + Properties.Resources.VisualizeShapesFileExtension, VisualizeShapeData.ToJson());
			zip.Compress(path, targetTmpDirPath);

			Directory.Delete(targetTmpDirPath, true);
		}

		/// <summary>
		/// パスを指定してデシリアライズ
		/// </summary>
		/// <param name="path">読み込むパス</param>
		public void Deserialize(string path)
		{
			// 一時ディレクトリ作成
			if (!Directory.Exists(ApplicationData.Setting.TemporaryDirectoryPath))
				Directory.CreateDirectory(ApplicationData.Setting.TemporaryDirectoryPath);

			string targetTmpDirPath = Path.Combine(ApplicationData.Setting.TemporaryDirectoryPath, "tlv_convertRuleTmp_" + DateTime.Now.Ticks.ToString() + @"\");
			Directory.CreateDirectory(targetTmpDirPath);

			IZip zip = ApplicationFactory.Zip;

			zip.Extract(path, targetTmpDirPath);

			string resFilePath = Directory.GetFiles(targetTmpDirPath, "*." + Properties.Resources.ResourceFileExtension)[0];
			string logFilePath = Directory.GetFiles(targetTmpDirPath, "*." + Properties.Resources.TraceLogFileExtension)[0];
			string vixFilePath = Directory.GetFiles(targetTmpDirPath, "*." + Properties.Resources.VisualizeRuleFileExtension)[0];
			string settingFilePath = Directory.GetFiles(targetTmpDirPath, "*." + Properties.Resources.SettingFileExtension)[0];
			string visualizeShapesPath = Directory.GetFiles(targetTmpDirPath, "*." + Properties.Resources.VisualizeShapesFileExtension)[0];

			ResourceData res = ApplicationFactory.JsonSerializer.Deserialize<ResourceData>(File.ReadAllText(resFilePath));
			TraceLogList log = ApplicationFactory.JsonSerializer.Deserialize<TraceLogList>(File.ReadAllText(logFilePath));
			VisualizeData viz = ApplicationFactory.JsonSerializer.Deserialize<VisualizeData>(File.ReadAllText(vixFilePath));
			SettingData setting = new SettingData(this);
			ApplicationFactory.JsonSerializer.Populate(File.ReadAllText(settingFilePath), setting);
			VisualizeShapeData shapes = ApplicationFactory.JsonSerializer.Deserialize<VisualizeShapeData>(File.ReadAllText(visualizeShapesPath));

			ResourceData = res;
			TraceLogData = new TraceLogData(log, res);
			VisualizeData = viz;
			SettingData = setting;
			VisualizeShapeData = shapes;

			setVisualizeRuleToEvent();

			foreach (KeyValuePair<string, TimeLineMarker> tlm in (ObservableDictionary<string, TimeLineMarker>)(SettingData.LocalSetting.TimeLineMarkerManager.Markers))
			{
				tlm.Value.Name = tlm.Key;
			}

			Directory.Delete(targetTmpDirPath, true);
		}
	}
}
