/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.util.HashSet;
import java.util.Set;

import net.morilib.range.Interval;
import net.morilib.range.Range;
import net.morilib.range.RangeAdder;
import net.morilib.util.IntervalMap;

public class NinaMap<T, V> {

	private IntervalMap<V> map = new IntervalMap<V>();
	private Set<V> eps = new HashSet<V>();

	NinaMap() {
	}

	void put(T key, V value) {
		if(key == null) {
			eps.add(value);
		} else {
			map.put(Interval.newPoint(key), value);
		}
	}

	void put(Range key, V value) {
		if(key == null || key.isEmpty()) {
			eps.add(value);
		} else {
			for(Interval a : key.intervals()) {
				map.put(a, value);
			}
		}
	}

	public Set<V> get(T key) {
		if(key != null) {
			return map.get(Interval.newPoint(key));
		} else {
			return eps;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.TinyMap#get(net.morilib.range.Interval)
	 */
	public Set<V> get(Range r) {
		Set<V> s = new HashSet<V>();

		if(r != null) {
			for(Interval v : r.intervals()) {
				s.addAll(map.get(v));
			}
			return s;
		} else {
			return eps;
		}
	}

	public Range getKeyRange() {
		RangeAdder a = new RangeAdder();

		for(Interval e : map.keySet()) {
			a.addInterval(e);
		}
		return a.toRange();
	}

	@SuppressWarnings("unchecked")
	public Set<T> getDiscreteKeys() {
		Set<T> r = new HashSet<T>();
		Object a, b;
		char ac;
		int ai;

		for(Interval e : map.keySet()) {
			a = e.getInfimumBound();
			b = e.getSupremumBound();
			if(a.equals(b)) {
				r.add((T)a);
			} else if(a instanceof Character) {
				ac = ((Character)a).charValue();
				for(int i = ac; e.contains(i); i++) {
					r.add((T)Integer.valueOf(i));
				}
			} else if(a instanceof Integer) {
				ai = ((Integer)a).intValue();
				for(int i = ai; e.contains(i); i++) {
					r.add((T)Integer.valueOf(i));
				}
			} else {
				throw new ClassCastException();
			}
		}
		return r;
	}

	public Set<V> valueSet() {
		Set<V> r = new HashSet<V>();

		for(Interval v : map.keySet()) {
			r.addAll(map.get(v));
		}
		return r;
	}

	public String toString() {
		StringBuffer b = new StringBuffer();
		String d = "";

		for(Interval v : map.keySet()) {
			b.append(d).append("<<").append(v);
			b.append(" , ").append(map.get(v)).append(">>");
			d = "\n";
		}
		return b.toString();
	}

}
