/*
 * Java Template Project
 * 
 * Copyright (C) 2006 Satoshi Nagashiba, All Rights Reserved.
 */
package org.jtp.web.util;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;

import javax.faces.application.Application;
import javax.faces.application.FacesMessage;
import javax.faces.application.FacesMessage.Severity;
import javax.faces.component.UIViewRoot;
import javax.faces.context.FacesContext;
import javax.faces.el.ValueBinding;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.resources.Message;
import org.apache.commons.resources.MessageList;
import org.apache.commons.resources.Messages;
import org.apache.commons.resources.Resources;
import org.apache.commons.resources.impl.BasicMessageList;
import org.apache.commons.resources.impl.ResourceBundleResources;
import org.omg.CORBA.portable.ApplicationException;

/**
 * メッセージ関連のユーティリティクラスです。
 * 
 * @author <a href="mailto:sato-vista@jade.plala.or.jp">Satoshi Nagashiba</a>
 */
public final class FacesUtils {

    public static String DEFAULT_MESSAGE_KEY = "error.unexpected";
    
    private static final Map _messagesCache = new HashMap();

    // --------------------------------------------------------- Constructor

    /**
     * コンストラクタ（外部からのインスタンス生成を許可しない）
     */
    private FacesUtils() {}

    /**
     * 
     * @return
     */
    public static Messages getMessages() {
		return getMessages(FacesContext.getCurrentInstance());
	}

    /**
     * 
     * @return
     */
	public static Locale getLocale() {
		return getLocale(FacesContext.getCurrentInstance());
	}

	/**
	 * 
	 * @param exception
	 */
	public static void addExceptionMessages(Exception exception) {
		addExceptionMessages(exception, DEFAULT_MESSAGE_KEY);
	}

	/**
	 * 
	 * @param exception
	 * @param defaultKey
	 */
	public static void addExceptionMessages(Exception exception, String defaultKey) {
		MessageList messageList = null;
		javax.faces.application.FacesMessage.Severity severity = FacesMessage.SEVERITY_FATAL;
		if (exception instanceof ApplicationException) {
			messageList = new BasicMessageList(exception.getMessage());
			severity = FacesMessage.SEVERITY_ERROR;
		}
		String detail = ExceptionUtils.getStackTrace(exception);
		FacesContext context = FacesContext.getCurrentInstance();
		Messages messages = getMessages();
		Locale locale = getLocale();
		if (messageList == null || messageList.isEmpty()) {
			String text = messages.getMessage(locale, defaultKey);
			FacesMessage message = new FacesMessage(severity, text, detail);
			context.addMessage(null, message);
		} else {
			FacesMessage message;
			for (Iterator itr = messageList.get(); itr.hasNext(); context.addMessage(null, message)) {
				Message element = (Message) itr.next();
				String key = element.getKey();
				Object values[] = element.getValues();
				String text = messages.getMessage(locale, key, values);
				message = new FacesMessage(severity, text, detail);
			}

		}
	}

	/**
	 * 
	 * @param key
	 * @return
	 */
	public static String formatMessage(String key) {
		return formatMessage(key, (Object[]) null);
	}

	/**
	 * 
	 * @param key
	 * @param value0
	 * @return
	 */
	public static String formatMessage(String key, Object value0) {
		return formatMessage(key, new Object[] { value0 });
	}

	/**
	 * 
	 * @param key
	 * @param values
	 * @return
	 */
	public static String formatMessage(String key, Object values[]) {
		if (values == null || values.length == 0) {
			return getMessages().getMessage(getLocale(), key);
		}
		else {
			return getMessages().getMessage(getLocale(), key, values);
		}
	}

	/**
	 * 
	 * @param message
	 * @return
	 */
	public static String formatMessage(Message message) {
		return formatMessage(message.getKey(), message.getValues());
	}

	/**
	 * 
	 * @param severity
	 * @param key
	 */
	public static void addMessage(Severity severity, String key) {
		addMessage(severity, key, (Object[]) null);
	}

	/**
	 * 
	 * @param severity
	 * @param key
	 * @param value0
	 */
	public static void addMessage(Severity severity, String key, Object value0) {
		addMessage(severity, key, new Object[] { value0 });
	}

	/**
	 * 
	 * @param severity
	 * @param key
	 * @param values
	 */
	public static void addMessage(Severity severity, String key, Object values[]) {
		String text = formatMessage(key, values);
		FacesContext context = FacesContext.getCurrentInstance();
		context.addMessage(null, new FacesMessage(severity, text, null));
	}

	/**
	 * 
	 * @param severity
	 * @param message
	 */
	public static void addMessage(Severity severity, Message message) {
		addMessage(severity, message.getKey(), message.getValues());
	}

	/**
	 * 
	 * @param severity
	 * @param clientId
	 * @param key
	 */
	public static void addMessage(Severity severity, String clientId, String key) {
		addMessage(severity, clientId, key, (Object[]) null);
	}

	/**
	 * 
	 * @param severity
	 * @param clientId
	 * @param key
	 * @param value0
	 */
	public static void addMessage(Severity severity, String clientId, String key, Object value0) {
		addMessage(severity, clientId, key, new Object[] { value0 });
	}

	/**
	 * 
	 * @param severity
	 * @param clientId
	 * @param key
	 * @param values
	 */
	public static void addMessage(Severity severity, String clientId, String key, Object values[]) {
		String text = formatMessage(key, values);
		FacesContext context = FacesContext.getCurrentInstance();
		context.addMessage(clientId, new FacesMessage(severity, text, null));
	}

	/**
	 * 
	 * @param severity
	 * @param clientId
	 * @param message
	 */
	public static void addMessage(Severity severity, String clientId, Message message) {
		addMessage(severity, clientId, message.getKey(), message.getValues());
	}

	/**
	 * 
	 * @param expression
	 * @return
	 */
	public static Object getBindingValue(String expression) {
		FacesContext context = FacesContext.getCurrentInstance();
		Application application = context.getApplication();
		ValueBinding valueBinding = application.createValueBinding(expression);
		return valueBinding.getValue(context);
	}

	/**
	 * 
	 * @param expression
	 * @param value
	 */
	public static void setBindingValue(String expression, Object value) {
		FacesContext context = FacesContext.getCurrentInstance();
		Application application = context.getApplication();
		ValueBinding valueBinding = application.createValueBinding(expression);
		valueBinding.setValue(context, value);
	}

	/**
	 * 
	 * @param context
	 * @return
	 */
    @SuppressWarnings("unchecked")
	private static Messages getMessages(FacesContext context) {
		String bundle = context.getApplication().getMessageBundle();
		Messages messages = (Messages) _messagesCache.get(bundle);
		if (messages == null) {
			Resources resources = new ResourceBundleResources(bundle, bundle);
			messages = new Messages(resources);
			_messagesCache.put(bundle, messages);
		}
		return messages;
	}

	/**
	 * 
	 * @param context
	 * @return
	 */
	private static Locale getLocale(FacesContext context) {
		UIViewRoot viewRoot = context.getViewRoot();
		return (viewRoot != null) ? viewRoot.getLocale() : context.getApplication().getDefaultLocale();
	}
}
