/* Copyright (c) 1991-2002 Doshita Lab. Speech Group, Kyoto University */
/* Copyright (c) 2000-2002 Speech and Acoustics Processing Lab., NAIST */
/*   All rights reserved   */

/* search_bestfirst_v2.c --- hypothesis handling and scoring functions
   for nextscan algorithm */

/* $Id: search_bestfirst_v2.c,v 1.7 2002/10/15 07:17:46 ri Exp $ */

/* by "fast" setting (default), search_bestfirst_v1.c is used for faster
   decoding.  Please specify option "--enable-setup=standard" or
   "--enable-strict-iwcd2" at "./configure" to activate this. */

#include <julius.h>

#ifdef PASS2_STRICT_IWCD

#undef TCD			/* define for CCD debug messages */

/**********************************************************************/
/************ $B2>@b%N!<%I$N4pK\A`:n(B                         ************/
/************ basic functions for hypothesis node handling ************/
/**********************************************************************/

/* node $B$r2rJ|(B */
/* free node */
void
free_node(NODE *node)
{
  if (node == NULL) return;
  free(node->g);
  free(node);
}

/* src $B$+$i(B dst $B$X%3%T!<(B */
/* copy node from 'src' to 'dst' */
NODE *
cpy_node(NODE *dst, NODE *src)
{
  
  dst->next = src->next;
  dst->prev = src->prev;
  memcpy(dst->g, src->g, sizeof(LOGPROB) * peseqlen);
  memcpy(dst->seq, src->seq, sizeof(WORD_ID) * MAXSEQNUM);
  dst->seqnum = src->seqnum;
  dst->score = src->score;
  dst->bestt = src->bestt;
  dst->estimated_next_t = src->estimated_next_t;
  dst->endflag = src->endflag;
#ifdef USE_DFA
  dst->state = src->state;
#endif
  dst->tre = src->tre;
  if (ccd_flag) {
    dst->last_ph = src->last_ph;
  }
#ifdef USE_NGRAM
  dst->totallscore = src->totallscore;
#endif
  return(dst);
}

/* $B?7$?$J2>@b%N!<%I$r3d$jIU$1(B */
/* allocate new node */
NODE *
newnode()
{
  NODE *tmp;
  int i;

  if ((tmp=(NODE *)mymalloc(sizeof(NODE)))==NULL) {
    j_error("can't malloc\n");
  }
  /*bzero(tmp,sizeof(NODE));*/
  tmp->next=NULL;
  tmp->prev=NULL;
  tmp->g = (LOGPROB *)mymalloc(sizeof(LOGPROB)*peseqlen);
  if (ccd_flag) {
    tmp->last_ph = NULL;
#ifdef USE_NGRAM
    tmp->totallscore = LOG_ZERO;
#endif
  }
  tmp->endflag = FALSE;
  tmp->seqnum = 0;
  for(i=0;i<peseqlen;i++) {
    tmp->g[i] = LOG_ZERO;
  }

  return(tmp);
}


/**********************************************************************/
/************ $BA08~$-%H%l%j%9E83+$HL`EY7W;;(B             ****************/
/************ expand trellis and update forward score *****************/
/**********************************************************************/

static LOGPROB **wordtrellis;	/* buffer for computing viterbi of a word */
static LOGPROB *g;
static HMM_Logical **phmmseq;
static int phmmlen_max;
static HMM_Logical *tailph;

/* $B#1C18lJ,$N%H%l%j%97W;;MQ$N%o!<%/%(%j%"(B wordtrellis[t][]$B$r3NJ](B */
/* allocate work area 'wordtrellis[t][]' for trellis computation of a word */
void
malloc_wordtrellis()
{
  int i;
  int maxwn;
  LOGPROB *p;

  maxwn = winfo->maxwn + 10;	/* CCD$B$K$h$kJQF0$r9MN8(B */
  wordtrellis = (LOGPROB **)mymalloc(sizeof(LOGPROB *) * peseqlen);
  p = (LOGPROB *)mymalloc(sizeof(LOGPROB) * peseqlen * maxwn);
  for (i=0;i<peseqlen;i++) {
    wordtrellis[i] = p + i * maxwn;
  }
  wordtrellis[0][maxwn] = 1.0;

  g = (LOGPROB *)mymalloc(sizeof(LOGPROB) * peseqlen);

  phmmlen_max = winfo->maxwlen + 2;
  phmmseq = (HMM_Logical **)mymalloc(sizeof(HMM_Logical *) * phmmlen_max);
}
/* $B>e5-$N%o!<%/%(%j%"(B wordtrellis[t][] $B$r2rJ|(B */
/* free the 'wordtrellis[t][]' */
void
free_wordtrellis()
{
  free(wordtrellis[0]);
  free(wordtrellis);
  free(g);
  free(phmmseq);
}



/**********************************************************************/
/************ $B2>@b$NA08~$-L`EY7W;;(B                  *******************/
/************ compute forward score of a hypothesis *******************/
/**********************************************************************/

/* --- $BBh(B2$B%Q%9C5:w(B(RL)$B$K$*$1$kC18l4V(Btriphone$B$N07$$(B ---

   search_bestfirst_v2.c $B$G$O!$%G%3!<%G%#%s%0$N@:EY$r=E;k$7$F(B,
   $B<!C18l$H$=$NA0$NC18l$K$*$1$kC18l4V2;AG%3%s%F%-%9%H$O(B
   $B2>@bE83+;~$K$9$Y$F07$o$l$k!%(B

   backscan $B$r9T$J$&(B search_bestfirst_v1.c $B$,2>@b$N(B POP $B;~$K9T$J$&$N$K(B
   $BHf$Y$F!$2>@b@8@.$N;~E@$G@53N$J%9%3%"$r7W;;$9$k$N$G%9%3%"@:EY$O9b$/(B
   $B$J$k!%$?$@$7@8@.$5$l$k$9$Y$F$N2>@b$KBP$7$F(B($B$?$H$(%9%?%C%/$KF~$i$J$$(B
   $B$h$&$J@\B3$G$"$C$F$b(B)$B%H%l%j%9$N:F7W;;$r9T$J$&$?$a!$7W;;NL$OA}Bg$9$k!%(B

   $B%G%U%)%k%H$O(B v1 $B$r;HMQ$9$k!%$3$N(B v2 $B$r;HMQ$9$k>l9g$O(B
   "--enable-setup=standard" $B$r;HMQ$9$k(B($B$"$k$$$O(B "--enable-strict-iwcd2")
*/

/* --- handling of cross-word triphone in 2nd pass search (in RL direction) ---

   Here in search_bestfirst_v2.c, all cross-word context dependencies between
   the next word and source hypothesis are computed when new hypotheses
   are expanded.

   As the precise cross-word triphone score is applied on hypothesis
   generation with no delay, more accurate score is obtained than
   the delayed backscan method in search_bestfirst_v1.c.
   On the other hand, the computational cost grows much by re-calculating
   forward score of cross-word triphones for all generated hypothethes.
*/

/* $BM?$($i$l$?2;AG$N$J$i$S(B phmmseq[0..phmmlen-1]$B$KBP$7$F(Bviterbi$B7W;;$r9T$&!%(B
   g[0..framelen-1] $B$N%9%3%"$r=i4|CM$H$7$F(B g_new[0..framelen-1]$B$K99?7CM$rBeF~!%(B
   $B:GDc(B least_frame $B$^$G$O(Bscan$B$9$k!%(B*/
/* Viterbi computation for the given phoneme sequence 'phmmseq[0..phmmlen-1]'
   with g[0..framelen-1] as initial values.  The results are stored in
   g_new[0..framelen-1].  Scan should not terminate at least it reaches
   'least_frame'. */
static void
do_viterbi(LOGPROB *g, LOGPROB *g_new, HMM_Logical **phmmseq, int phmmlen, HTK_Param *param, int framelen, int least_frame)
{
  HMM *whmm;			/* HMM */
  int wordhmmnum;		/* length of above */
  int startt;			/* scan start frame */
  LOGPROB tmpmax,tmpscore;	/* variables for Viterbi process */
  A_CELL *ac;
  int t,i;
  boolean node_exist_p;

#ifdef TCD
  j_printf("scan for:");
  for (i=0;i<phmmlen;i++) {
    j_printf(" %s", phmmseq[i]->name);
  }
  j_printf("\n");
#endif
  
  /* $BC18l(BHMM$B$r:n$k(B */
  /* make word HMM */
  whmm = new_make_word_hmm(hmminfo, phmmseq, phmmlen);
  wordhmmnum = whmm->len;
  if (wordhmmnum >= winfo->maxwn + 10) {
    j_error("scan_word: word too long\n");
  }

  /* scan$B3+;OE@$r8!:w(B -> startt$B$X(B*/
  /* search for the start frame -> set to startt */
  for(t = framelen-1; t >=0 ; t--) {
    if (
#ifdef SCAN_BEAM
	g[t] > framemaxscore[t] - scan_beam_thres &&
#endif
	g[t] > LOG_ZERO) {
      break;
    }
  }
  if (t < 0) {			/* no node has score > LOG_ZERO */
    /* reset all scores and end */
    for(t=0;t<framelen;t++) {
      g_new[t] = LOG_ZERO;
    }
    free_hmm(whmm);
    return;
  }
  startt = t;
  
  /* $B3+;OE@0J9_(B[startt+1..framelen-1] $B$N(B g_new[] $B$r%j%;%C%H(B */
  /* clear g_new[] for [startt+1..framelen-1] */
  for(t=framelen-1;t>startt;t--) g_new[t] = LOG_ZERO;

  /*****************/
  /* viterbi start */
  /*****************/
  
  /* $B;~4V(B [startt] $B>e$NCM$r=i4|2=(B */
  /* initialize scores on frame [startt] */
  for(i=0;i<wordhmmnum-1;i++) wordtrellis[startt][i] = LOG_ZERO;
  wordtrellis[startt][wordhmmnum-1] = g[startt] + outprob(startt, &(whmm->state[wordhmmnum-1]), param);
  g_new[startt] = LOG_ZERO;
  
  /* $B%a%$%s%k!<%W(B: startt $B$+$i;O$^$j(B 0 $B$K8~$+$C$F(B Viterbi $B7W;;(B */
  /* main loop: start from [startt], and compute Viterbi toward [0] */
  for(t=startt-1; t>=0; t--) {
    
    node_exist_p = FALSE;	/* TRUE if there is at least 1 survived node in this frame */

    /* $BC<$N%N!<%I(B [t][wordhmmnum-1]$B$O!$FbItA+0\(B $B$+(B g[]$B$N9b$$J}$K$J$k(B */
    /* the edge node [t][wordhmmnum-1] is either internal transitin or g[] */
    tmpscore = LOG_ZERO;
    for (ac=whmm->state[wordhmmnum-1].ac;ac;ac=ac->next) {
      if (tmpscore < wordtrellis[t+1][ac->arc] + ac->a)
	tmpscore = wordtrellis[t+1][ac->arc] + ac->a;
    }
    tmpmax = (g[t] > tmpscore) ? g[t] : tmpscore;

    /* $BC<$N%N!<%I$N%9%3%"%(%s%Y%m!<%W%A%'%C%/(B: $B0lDjI}30$J$iMn$H$9(B */
    /* check if the edge node is within score envelope */
    if (
#ifdef SCAN_BEAM
	tmpmax <= framemaxscore[t] - scan_beam_thres ||
#endif
	tmpmax <= LOG_ZERO
	) {
      wordtrellis[t][wordhmmnum-1] = LOG_ZERO;
    } else {
      node_exist_p = TRUE;
      wordtrellis[t][wordhmmnum-1] = tmpmax + outprob(t, &(whmm->state[wordhmmnum-1]), param);
    }

    /* node[wordhmmnum-2..0]$B$K$D$$$F%H%l%j%9$rE83+(B */
    /* expand trellis for node [t][wordhmmnum-2..0] */
    for(i=wordhmmnum-2;i>=0;i--) {
      
      /* $B:GL`%Q%9$H:GL`%9%3%"(B tmpmax $B$r8+$D$1$k(B */
      /* find most likely path and the max score 'tmpmax' */
      tmpmax = LOG_ZERO;
      for (ac=whmm->state[i].ac;ac;ac=ac->next) {
	tmpscore = wordtrellis[t+1][ac->arc] + ac->a;
	if (tmpmax < tmpscore) tmpmax = tmpscore;
      }
      
      /* $B%9%3%"%(%s%Y%m!<%W%A%'%C%/(B: $B0lDjI}30$J$iMn$H$9(B */
      /* check if score of this node is within the score envelope */
      if (
#ifdef SCAN_BEAM
	  tmpmax <= framemaxscore[t] - scan_beam_thres ||
#endif
	  tmpmax <= LOG_ZERO
	  ) {
	/* invalid node */
	wordtrellis[t][i] = LOG_ZERO;
      } else {
	/* survived node */
	node_exist_p = TRUE;
	wordtrellis[t][i] = tmpmax + outprob(t, &(whmm->state[i]), param);
      }
    } /* end of node loop */

    /* $B;~4V(B t $B$N(BViterbi$B7W;;=*N;!%?7$?$JA08~$-%9%3%"(B g_new[t] $B$r%;%C%H(B */
    /* Viterbi end for frame [t].  set the new forward score g_new[t] */
    g_new[t] = wordtrellis[t][0];

    /* $B;XDj$5$l$?(B least_frame $B$h$j@h$^$G(B t $B$,?J$s$G$*$j!$$+$D$3$N(B t $B$K$*$$$F(B
       $B%9%3%"%(%s%Y%m!<%W$K$h$C$F@8$-;D$C$?%N!<%I$,0l$D$bL5$+$C$?>l9g(B,
       $B$3$N%U%l!<%`$G7W;;$rBG$A@Z$j$=$l0J>e@h(B([0..t-1])$B$O7W;;$7$J$$(B */
    /* if frame 't' already reached the 'least_frame' and no node was
       survived in this frame (all nodes pruned by score envelope),
       terminate computation at this frame and do not computer further
       frame ([0..t-1]). */
    if (t < least_frame && (!node_exist_p)) {
      /* crear the rest scores */
      for (i=t-1;i>=0;i--) {
	g_new[i] = LOG_ZERO;
      }
      /* terminate loop */
      break;
    }
    
  } /* end of time loop */

  /* free work area */
  free_hmm(whmm);
}

/* next_word$BFb$G(B, now $B$N:G8e$N(B1$B2;AG(B(lastphone)$BJ,$KBP$7$F(Bviterbi$B7W;;$r(B
   $B9T$J$&!%(Bnow->g[] $B$r=i4|CM$H$7$F(B new->g[] $B$K99?7CM$rBeF~(B */
/* for next_word(), do Viterbi computation for the last phone ('lastphone'),
   with now->g[] as initial value.  Results are stored in new->g[] */
static void
do_viterbi_next_word(NODE *now, NODE *new, HMM_Logical *lastphone, HTK_Param *param)
{
  LOGPROB a_value;
  int t, n;

  /* $B$b$7E83+852>@b$N:G8e$NC18l$N2;AGD9$,(B 1 $B$G$"$l$P!$$=$N2;AG$O(B
     $BD>A0$N(B scan_word $B$G7W;;$5$l$F$$$J$$!%$3$N>l9g(B, now->g[] $B$K0JA0$N(B
     $B=i4|CM$,3JG<$5$l$F$$$k!%(B
     $B$b$72;AGD9$,#10J>e$G$"$l$P!$(Bnow->g[] $B$O$=$N<jA0$^$G7W;;$7$?>uBV(B
     $B$N%9%3%"$,F~$C$F$$$k$N$G(B,now->g[t] $B$+$i=i4|CM$r@_Dj$9$kI,MW$,$"$k(B */
  /* If the length of last word is 1, it means the last phone was not
     scanned in the last call of scan_word().  In this case, now->g[]
     keeps the previous initial value, so start viterbi with the old scores.
     If the length is more than 1, the now->g[] keeps the values of the
     scan result till the previous phone, so make initial value
     considering last transition probability. */
  if (winfo->wlen[now->seq[now->seqnum-1]] > 1) {
    n = hmm_logical_state_num(lastphone);
    a_value = (hmm_logical_trans(lastphone))->a[n-2][n-1];
    for(t=0; t<peseqlen-1; t++) g[t] = now->g[t+1] + a_value;
    g[peseqlen-1] = LOG_ZERO;
  } else {
    for(t=0; t<peseqlen; t++) g[t] = now->g[t];
  }
  
  /* do viterbi computation for the last phone */
  do_viterbi(g, new->g, &lastphone, 1, param, peseqlen, now->estimated_next_t);
}

/* $BJ82>@b$NA08~$-L`EY$r99?7$9$k(B
   $B:G8e$N(B1$BC18l$NA08~$-%H%l%j%9$r7W;;$9$k(B */
/* compute forward score by expanding trellis of the last word */
void
scan_word(NODE *now, HTK_Param *param)
{
  int   i,t;
  WORD_ID word;
  int phmmlen;
  
  /* ----------------------- prepare phoneme sequence ------------------ */
  /* triphone$B$J$i@hF,$N(B1$B2;AG$O$3$3$G$OBP>]30(B($B$"$H$G(Bnext_word$B$G$d$k(B) */
  /*             $BKvHx$N(B1$B2;AG$O%3%s%F%-%9%H$K$7$?$,$C$FCV49(B */
  /* with triphone, modify the tail phone of the last word according to the
     previous word, and do not compute the head phone here (that will be
     computed later in next_word() */
  word = now->seq[now->seqnum-1];
  
#ifdef TCD
    j_printf("w=");
    for(i=0;i<winfo->wlen[word];i++) {
      j_printf(" %s",(winfo->wseq[word][i])->name);
    }
    if (ccd_flag) {
      if (now->last_ph != NULL) {
	j_printf(" | %s", (now->last_ph)->name);
      }
    }
    j_printf("\n");
#endif /* TCD */
    
  if (ccd_flag) {
    
    /* the tail triphone of the last word varies by context */
    if (now->last_ph != NULL) {
      tailph = get_right_context_HMM(winfo->wseq[word][winfo->wlen[word]-1], now->last_ph->name, hmminfo);
      if (tailph == NULL) {
	/* fallback to the original bi/mono-phone */
	/* error if the original is pseudo phone (not explicitly defined
	   in hmmdefs/hmmlist) */
	/* exception: word with 1 phone (triphone may exist in the next expansion */
	if (winfo->wlen[word] > 1 && winfo->wseq[word][winfo->wlen[word]-1]->is_pseudo){
	  error_missing_right_triphone(winfo->wseq[word][winfo->wlen[word]-1], now->last_ph->name);
	}

	tailph = winfo->wseq[word][winfo->wlen[word]-1];
      }
    } else {
      tailph = winfo->wseq[word][winfo->wlen[word]-1];
    }
    /* $BD9$5(B1$B$NC18l$O<!$N(Bnextword$B$G$5$i$KJQ2=$9$k$N$G$3$3$G$O(Bscan$B$7$J$$(B */
    /* do not scan word if the length is 1, as it further varies in the
       following next_word() */
    if (winfo->wlen[word] == 1) {
      now->last_ph = tailph;
#ifdef TCD
      j_printf("suspended as %s\n", (now->last_ph)->name);
#endif
      return;
    }

    /* scan$BHO0O$N2;AGNs$r=`Hw(B */
    /* prepare HMM of the scan range */
    phmmlen = winfo->wlen[word] - 1;
    if (phmmlen > phmmlen_max) {
      j_error("short of phmmlen\n");
    }
    for (i=0;i<phmmlen-1;i++) {
      phmmseq[i] = winfo->wseq[word][i+1];
    }
    phmmseq[phmmlen-1] = tailph;
  } else {
    phmmlen = winfo->wlen[word]; 
    for (i=0;i<phmmlen;i++) phmmseq[i] = winfo->wseq[word][i];
  }

  /* $B85$N(Bg[]$B$r$$$C$?$sBTHr$7$F$*$/(B */
  /* temporally keeps the original g[] */
  for (t=0;t<peseqlen;t++) g[t] = now->g[t];

  /* viterbi$B$r<B9T$7$F(B g[] $B$+$i(B now->g[] $B$r99?7$9$k(B */
  /* do viterbi computation for phmmseq from g[] to now->g[] */
  do_viterbi(g, now->g, phmmseq, phmmlen, param, peseqlen, now->estimated_next_t);

  if (ccd_flag) {
    /* $B<!2s$N$?$a$K(B now->last_ph $B$r99?7(B */
    /* update 'now->last_ph' for future scan_word() */
    now->last_ph = winfo->wseq[word][0];
#ifdef TCD
    j_printf("last_ph = %s\n", (now->last_ph)->name);
#endif
  }
}

/**************************************************************************/
/*** $B?72>@b$NE83+$H%R%e!<%j%9%F%#%C%/$r7R$$$@A4BN%9%3%"$r7W;;(B           ***/
/*** expand new hypothesis and compute the total score (with heuristic) ***/
/**************************************************************************/

/* now $B$K<!C18l(B nword $B$r@\B3$7$F?7$7$$2>@b(B new $B$r@8@.$7!$$=$N(B
   ($B%R%e!<%j%9%F%#%C%/$r4^$`(B)$B2>@b%9%3%"$r5a$a$k(B */
/* generate a new hypothesis 'new' by connecting next word 'nword' to
   the source hypothesis 'now', and compute the total score of 'new' with
   heuristics */
static HMM_Logical *lastphone, *newphone;
static LOGPROB *g_src;
void
next_word(NODE *now, NODE *new, NEXTWORD *nword, HTK_Param *param, BACKTRELLIS *backtrellis)
{
  int   t;
  int lastword;
  int   i;
  LOGPROB a_value;
  int   startt;
  int word;
  LOGPROB totalscore, tmpp;
  TRELLIS_ATOM *tre;

  word = nword->id;
  lastword = now->seq[now->seqnum-1];

  /* lastphone ($BD>A0C18l$N@hF,2;AG(B) $B$r=`Hw(B */
  /* prepare lastphone (head phone of previous word) */
  if (ccd_flag) {
    /* $B:G=*2;AG(B triphone $B$r@\B3C18l$K2q$o$;$FJQ2=(B */
    /* modify triphone of last phone according to the next word */
    lastphone = get_left_context_HMM(now->last_ph, winfo->wseq[word][winfo->wlen[word]-1]->name, hmminfo);
    if (lastphone == NULL) {
      /* fallback to the original bi/mono-phone */
      /* error if the original is pseudo phone (not explicitly defined
	 in hmmdefs/hmmlist) */
      /* exception: word with 1 phone (triphone may exist in the next expansion */
      if (now->last_ph->is_pseudo){
	error_missing_left_triphone(now->last_ph, winfo->wseq[word][winfo->wlen[word]-1]->name);
      }
      lastphone = now->last_ph;
    }
  }

  /* newphone ($B@\B3C18l$NKvHx2;AG(B) $B$r=`Hw(B */
  /* prepare newphone (tail phone of next word) */
  if (ccd_flag) {
    newphone = get_right_context_HMM(winfo->wseq[word][winfo->wlen[word]-1], now->last_ph->name, hmminfo);
    if (newphone == NULL) {
      /* fallback to the original bi/mono-phone */
      /* error if the original is pseudo phone (not explicitly defined
	 in hmmdefs/hmmlist) */
      /* exception: word with 1 phone (triphone may exist in the next expansion */
      if (winfo->wlen[word] > 1 && winfo->wseq[word][winfo->wlen[word]-1]->is_pseudo){
	error_missing_right_triphone(winfo->wseq[word][winfo->wlen[word]-1], now->last_ph->name);
      }
      newphone = winfo->wseq[word][winfo->wlen[word]-1];
    }
  } else {
    newphone = winfo->wseq[word][winfo->wlen[word]-1];
  }
  
  /* $BC18lJB$S!"(BDFA$B>uBVHV9f!"8@8l%9%3%"$r(B new $B$X7Q>5!&99?7(B */
  /* inherit and update word sequence, DFA state and total LM score to 'new' */
  new->score = LOG_ZERO;
  for (i=0;i< now->seqnum;i++){
    new->seq[i] = now->seq[i];
  }
  new->seq[i] = word;
  new->seqnum = now->seqnum+1;
#ifdef USE_DFA
  new->state = nword->next_state;
#endif
#ifdef USE_NGRAM
  new->totallscore = now->totallscore + nword->lscore;
#endif  
  if (ccd_flag) {
    /* $B<!2>@b$NMzNr>pJs$H$7$FJ]B8(B */
    /* keep the lastphone for next scan_word() */
    new->last_ph = lastphone;
  }

  if (ccd_flag) {
    /* $B:G8e$N(B1$B2;AG(B(lastphone)$BJ,$r(Bscan$B$7!$99?7$7$?%9%3%"$r(B new $B$KJ]B8(B */
    /* scan the lastphone and set the updated score to new->g[] */
    do_viterbi_next_word(now, new, lastphone, param);
    g_src = new->g;
  } else {
    g_src = now->g;
  }
      
  /* $B<!2s$N(B scan_word $B$KHw$($F(B new->g[] $B$rJQ99$7$F$*$/(B */
  /* prepare new->g[] for next scan_word() */
  startt = peseqlen-2;
  i = hmm_logical_state_num(newphone);
  a_value = (hmm_logical_trans(newphone))->a[i-2][i-1];
  for(t=0; t <= startt; t++) {
    new->g[t] = g_src[t+1] + a_value
#ifdef USE_NGRAM
      + nword->lscore
#else
      + penalty2
#endif
      ;
  }

  /***************************************************************************/
  /* $BA08~$-(B($BBh#2%Q%9(B),$B8e$m8~$-(B($BBh#1%Q%9(B)$B%H%l%j%9$r@\B3$7:GL`@\B3E@$r8+$D$1$k(B */
  /* connect forward/backward trellis to look for the best connection time   */
  /***************************************************************************/
#ifdef WORD_GRAPH
  /*-----------------------------------------------------------------*/
  /* $BC18l%0%i%U%b!<%I$G$OF1$8%U%l!<%`$NF1$8C18l$G$bD>A0C18lKh$K%9%3(B
     $B%"$,0c$&!%(Bbestt $B$O(B $B8GDj(B */
  /* there are several trellis word with the same word ID in the same
     frame, with different word context in WORD_GRAPH mode.  And the
     best t is fixed (no re-esimation) */
  /*-----------------------------------------------------------------*/
  /* $B:GL`@\B3E@$O(B, $B:G8e$K;2>H$7$?(BTRELLIS_ATOM$B$N=*C<$K8GDj(B */
  /* the best connection point is fixed to the end time of last trellis word (no re-alignment */
  new->bestt = (nword->tre)->endtime;
  /* $B<!2s$NE83+C18l7hDj$N$?$a$N?dDj;OC<;~4V$O!$>e5-$N;OC<(B */
  /* the estimated next beginning frame for the next word expansion
     corresponds to the endtime above (this is also fixed) */
  new->estimated_next_t = (nword->tre)->begintime - 1;
  
  /* $B=PNO3NN($r(B tmpp $B$K7W;;(B */
  /* compute output prob to 'tmpp' */
  if (newphone->is_pseudo) {
    tmpp = outprob_cd(new->bestt, &(newphone->body.pseudo->stateset[newphone->body.pseudo->state_num-2]), param);
  } else {
    tmpp = outprob_state(new->bestt, newphone->body.defined->s[newphone->body.defined->state_num-2], param);
  }

  /* $B2>@bA4BN$N%9%3%"$r7W;;(B */
  new->score = new->g[new->bestt] + (nword->tre)->backscore + tmpp;
  
  /* $BE83+C18l$N%H%l%j%9>e$N>pJs$rEAHB(B */
  new->tre  =  nword->tre;
  
#else  /* not WORD_GRAPH */
  
  /*-----------------------------------------------------------------*/
  /* $BC18l%H%l%j%9$rC5$7$F(B, $B<!C18l$N:GL`@\B3E@$rH/8+$9$k(B */
  /* determine the best connection time of the new word, seeking the word
     trellis */
  /*-----------------------------------------------------------------*/

#ifdef USE_DFA
  if (!looktrellis_flag) {
    /* $B$9$Y$F$N%U%l!<%`$K$o$?$C$F:GL`$rC5$9(B */
    /* search for best trellis word throughout all frame */
    for(t = startt; t >= 0; t--) {
      tre = bt_binsearch_atom(backtrellis, t, (WORD_ID) word);
      if (tre == NULL) continue;
      if (newphone->is_pseudo) {
	tmpp = outprob_cd(t, &(newphone->body.pseudo->stateset[newphone->body.pseudo->state_num-2]), param);
      } else {
      tmpp = outprob_state(t, newphone->body.defined->s[newphone->body.defined->state_num-2], param);
      }
      totalscore = new->g[t] + tre->backscore + tmpp;
      if (new->score < totalscore) {
	new->score = totalscore;
	new->bestt = t;
	new->estimated_next_t = tre->begintime - 1;
      }
    }
  } else {
#endif /* USE_DFA */
    
  /* $B:G8e$K;2>H$7$?(BTRELLIS_ATOM$B$N=*C<;~4V$NA08e(B */
  /* new$B$N?dDj;~4V$O!$>e5-$G:NMQ$7$?(BTRELLIS_ATOM$B$N;OC<;~4V(B */

  /* $B$3$NE83+C18l$N%H%l%j%9>e$N=*C<;~4V$NA08e$N$_%9%-%c%s$9$k(B
     $BA08e$KO"B3$7$FB8:_$9$k%U%l!<%`$K$D$$$F$N$_7W;;(B */
  /* search for best trellis word only around the estimated time */
  /* 1. search forward */
  for(t = (nword->tre)->endtime; t >= 0; t--) {
    tre = bt_binsearch_atom(backtrellis, t, (WORD_ID) word);
    if (tre == NULL) break;	/* go to 2 if the trellis word disappear */
    if (newphone->is_pseudo) {
      tmpp = outprob_cd(t, &(newphone->body.pseudo->stateset[newphone->body.pseudo->state_num-2]), param);
    } else {
      tmpp = outprob_state(t, newphone->body.defined->s[newphone->body.defined->state_num-2], param);
    }
    totalscore = new->g[t] + tre->backscore + tmpp;
    if (new->score < totalscore) {
      new->score = totalscore;
      new->bestt = t;
      new->estimated_next_t = tre->begintime - 1;
    }
  }
  /* 2. search bckward */
  for(t = (nword->tre)->endtime + 1; t <= startt; t++) {
    tre = bt_binsearch_atom(backtrellis, t, (WORD_ID) word);
    if (tre == NULL) break;	/* end if the trellis word disapper */
    if (newphone->is_pseudo) {
      tmpp = outprob_cd(t, &(newphone->body.pseudo->stateset[newphone->body.pseudo->state_num-2]), param);
    } else {
      tmpp = outprob_state(t, newphone->body.defined->s[newphone->body.defined->state_num-2], param);
    }
    totalscore = new->g[t] + tre->backscore + tmpp;
    if (new->score < totalscore) {
      new->score = totalscore;
      new->bestt = t;
      new->estimated_next_t = tre->begintime - 1;
    }
  }

#ifdef USE_DFA
  }
#endif

#endif /* WORD_GRAPH */
}

/**********************************************************************/
/********** $B=i4|2>@b$N@8@.(B                 ****************************/
/********** generate an initial hypothesis ****************************/
/**********************************************************************/
void
start_word(
     NODE *new,
     NEXTWORD *nword,
     HTK_Param *param,
     BACKTRELLIS *backtrellis)
{
  HMM_Logical *newphone;
  WORD_ID word;
  TRELLIS_ATOM *tre = NULL;
  int t;
  LOGPROB tmpp;

  /* initialize data */
  word = nword->id;
  new->score = LOG_ZERO;
  new->seqnum = 1;
  new->seq[0] = word;

#ifdef USE_DFA
  new->state = nword->next_state;
#endif
#ifdef USE_NGRAM
  new->totallscore = nword->lscore;
#endif  

  /* cross-word triphone need not be handled on startup */
  newphone = winfo->wseq[word][winfo->wlen[word]-1];
  if (ccd_flag) {
    new->last_ph = NULL;
  }
  
#ifdef USE_NGRAM
  new->g[peseqlen-1] = nword->lscore;
#else  /* USE_DFA */
  new->g[peseqlen-1] = 0;
#endif
  
#ifdef WORD_GRAPH
  new->score = new->g[peseqlen-1] + (nword->tre)->backscore;
  new->tre  =  nword->tre;
  new->bestt = peseqlen-1;
  new->estimated_next_t = (nword->tre)->begintime - 1;
  
#else
  
  for (t=peseqlen-1; t>=0; t--) {
    tre = bt_binsearch_atom(backtrellis, t, word);
    if (tre != NULL) {
      new->bestt = t;
      if (newphone->is_pseudo) {
	tmpp = outprob_cd(peseqlen-1, &(newphone->body.pseudo->stateset[newphone->body.pseudo->state_num-2]), param);
      } else {
	tmpp = outprob_state(peseqlen-1, newphone->body.defined->s[newphone->body.defined->state_num-2], param);
      }
      new->score = new->g[peseqlen-1] + tre->backscore + tmpp;
      new->estimated_next_t = tre->begintime - 1;
      break;
    }
  }
  if (tre == NULL) {		/* no word in backtrellis */
    new->score = LOG_ZERO;
  }
#endif /* WORD_GRAPH */
}

/* $B:G=*2>@b$N%9%3%"7W;;(B */
/* generate final hypothesis from 'now' to 'new' */
void
last_next_word(NODE *now, NODE *new, HTK_Param *param)
{
  cpy_node(new, now);
  if (ccd_flag) {
    /* $B:G=*2;AGJ,$r(B viterbi $B$7$F:G=*%9%3%"$r@_Dj(B */
    /* scan the last phone and update the final score */
    do_viterbi_next_word(now, new, now->last_ph, param);
    new->score = new->g[0];
  } else {
    new->score = now->g[0];
  }
}

#endif /* PASS2_STRICT_IWCD */
