#ifdef WIN32
#include <windows.h>
#else
#include <unistd.h>
#endif

#include <stdio.h>
#include <GL/glu.h>
#include <GL/glut.h>

#include "DialogEdit.h"
#include "Document.h"
#include "Wfm.h"
#include "TransFace.h"
#include "Texture.h"
#include "ObjModel.h"
#include "ObjectModel.h"

DialogEdit::DialogEdit(Document *pDoc, int *w,int *h)
{
	m_pDoc = pDoc;
	m_pWidth = w;
	m_pHeight = h;
	
	// initialized region bit flag
	m_point.x = m_point.y = 0;					// mouse position
	m_bLButtonDown = m_bRButtonDown = m_bMButtonDown = false;	// button status bool
	m_bDocNew  = true;
	m_bDestroy = false;
	
	// set position
	m_trans[0] = m_trans[1] = m_trans[2] = m_rot[0] = m_rot[1] = m_rot[2] = 0.0;
	m_scale = 1.0;
	m_bEyeCLFlg = false;
}

DialogEdit::~DialogEdit()
{
}

void DialogEdit::redraw() 
{
	::glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
	::glClearColor( 0.0f, 0.0f, 0.0f, 0.0f );
	::glClearDepth( 1.0f );
	::glDepthFunc(GL_LEQUAL);
	::glEnable( GL_DEPTH_TEST );
	::glCullFace( GL_BACK );
	::glEnable( GL_CULL_FACE );
	
	SetupViewport();
	
	SetupViewingTransform();
	
	::glPushMatrix();
	DrawOpenGL();
	::glPopMatrix();
}

bool DialogEdit::SetupViewport()
{
	::glViewport(0, 0, w(), h());
	
	::glMatrixMode( GL_PROJECTION );
	::glLoadIdentity();
	
	SetupViewingFrustum(w(), h());
	
	::glMatrixMode( GL_MODELVIEW );
	::glLoadIdentity();
	
	return true;
}

bool DialogEdit::SetupViewingTransform()
{
	//++ Model View Matrix --
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	gluLookAt(0.0f, 0.0f, 5.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f);
	
	// Set texture environment
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
	
	//++ Texture Matrix --
	glMatrixMode(GL_TEXTURE);
	glLoadIdentity();
	glTranslatef(0.5f, 0.5f, 0.0f);
	glScalef(0.5f, 0.5f, 1.0f);
	
	// ++ for Model View Matrix --
	glMatrixMode(GL_MODELVIEW);
	
	return true;
}

bool DialogEdit::SetupViewingFrustum(int cx, int cy)
{
	if(cx <= cy) {
		glOrtho(-(GLfloat)cx/(GLfloat)cy, (GLfloat)cx/(GLfloat)cy, -1.0, 1.0, 0.01, 1000.0);
	} 
	else {
		glOrtho(-1.0f, 1.0f, 
			-(GLfloat)cy/(GLfloat)cx, (GLfloat)cy/(GLfloat)cx, 0.01, 1000.0);
	}

	glViewport(0, 0, cx, cy);
	return true;
}

bool DialogEdit::DrawOpenGL()
{	
	int i;
	if(!m_pDoc->IsModelOpened()) {
		return false;
	}
	
	GLfloat light_position[] = { 0.0, 0.0, 1.0, 1.0 };
	glLightfv(GL_LIGHT0, GL_POSITION, light_position);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, m_pDoc->GetTeethDifColor() );
	glLightf(GL_LIGHT0, GL_CONSTANT_ATTENUATION, 0.75);

	// prepare original and work wire-frame model
	Wfm *workWfm  = m_pDoc->GetWorkWfm();
	Wfm *tempWfm = m_pDoc->GetTempWfm();
	Texture *pTex = m_pDoc->GetTexture();
	Texture *eyeTex = m_pDoc->GetEyeTexture();
	ObjModel *obj = m_pDoc->GetEyeObject();
	Texture *mouthTex = m_pDoc->GetMouthwallTexture();

	// draw face region
	if(m_bEyeCLFlg && (m_pDoc->GetEyeTexture() != NULL)) {
		glNewList(EYE_RENEWAL_LIST, GL_COMPILE);

		glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

		// draw channel set
		int drawCh = eyeTex->alpha() ? GL_RGBA : GL_RGB;
		glTexImage2D(GL_TEXTURE_2D, 0, 3, 
			eyeTex->GetWidth(), eyeTex->GetHeight(), 0, drawCh, GL_UNSIGNED_BYTE, eyeTex->data());
		glEndList();
		m_bEyeCLFlg = false;
	}


	if(m_bDocNew) {
		glPolygonMode(GL_BACK, GL_FILL);

		m_pDoc->OpenObjFile();

		// Face Texture
		glNewList(TEXTUER_RENEWAL_LIST, GL_COMPILE);
		
		// draw channel set
		int drawCh = pTex->alpha() ? GL_RGBA : GL_RGB;
		glTexImage2D(GL_TEXTURE_2D, 0, 3, 
			pTex->GetWidth(), pTex->GetHeight(), 0, drawCh, GL_UNSIGNED_BYTE, pTex->data());
		
		glEndList();

		glPolygonMode(GL_FRONT, GL_FILL);
		// Mouthwall Texture
		glNewList(MOUTHWALL_RENEWAL_LIST, GL_COMPILE);
		
		glTexImage2D(GL_TEXTURE_2D, 0, 3, mouthTex->GetWidth(), 
			mouthTex->GetHeight(), 0, GL_RGBA, GL_UNSIGNED_BYTE, mouthTex->GetData());
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);		
		
		glEndList();

		m_bDocNew = false;
	}
	
	glPushMatrix();
	
	glTranslated(m_trans[0], m_trans[1], m_trans[2]);
	glRotated(m_rot[0], 1.0f, 0.0f, 0.0f);
	glRotated(m_rot[1], 0.0f, 1.0f, 0.0f);
	glRotated(m_rot[2], 0.0f, 0.0f, 1.0f);
	glScaled(m_scale, m_scale, m_scale);
	
	glCallList(TEXTUER_RENEWAL_LIST);
	
	// whole face
	glEnable(GL_TEXTURE_2D);	
	glColor4d(1.0, 1.0, 1.0, 1.0);
	
	for(i = 1; i <= tempWfm->trn(); i++) {
		if(tempWfm->S2Glinkgn(i) < 20 || tempWfm->S2Glinkgn(i) == 30 || tempWfm->S2Glinkgn(i) == 31) {
			glBegin(GL_POLYGON);
			//glBegin(GL_TRIANGLES);
			glTexCoord2dv(workWfm->SL2G1v(i));	
			glVertex3dv(tempWfm->SL2G1v(i));
			glTexCoord2dv(workWfm->SL2G2v(i));
			glVertex3dv(tempWfm->SL2G2v(i));
			glTexCoord2dv(workWfm->SL2G3v(i));
			glVertex3dv(tempWfm->SL2G3v(i));
			glEnd();
		}
	}

	for(i = 0; i < 40; i++) {		// infill the eye hole
		glBegin(GL_TRIANGLES);
		glTexCoord2dv(workWfm->Ggridv(Wfm::EyeHole[i][0].gn, Wfm::EyeHole[i][0].pn));
		glVertex3dv(workWfm->Ggridv(Wfm::EyeHole[i][0].gn, Wfm::EyeHole[i][0].pn));
		glTexCoord2dv(workWfm->Ggridv(Wfm::EyeHole[i][1].gn, Wfm::EyeHole[i][1].pn));
		glVertex3dv(workWfm->Ggridv(Wfm::EyeHole[i][1].gn, Wfm::EyeHole[i][1].pn));
		glTexCoord2dv(workWfm->Ggridv(Wfm::EyeHole[i][2].gn, Wfm::EyeHole[i][2].pn));
		glVertex3dv(workWfm->Ggridv(Wfm::EyeHole[i][2].gn, Wfm::EyeHole[i][2].pn));
		glEnd();
	}
	

	// redraw mouthwall and teeth
	ObjectModel *objTeethU = m_pDoc->GetUpperTeethObj();
	ObjectModel *objTeethL = m_pDoc->GetLowerTeethObj();
	ObjectModel *objMouth = m_pDoc->GetMoutthWallObj();

	if( objTeethU != NULL && objTeethL != NULL && objMouth != NULL ) {
		glColor4f(1.0, 1.0, 1.0, 1.0);
		glEnable(GL_LIGHTING);
		glEnable(GL_LIGHT0);
	    glDisable(GL_TEXTURE_2D);
		objTeethU->drawTeethModel(TEETH_UPPER);
	    objTeethL->drawTeethModel(TEETH_LOWER);
		glEnable(GL_TEXTURE_2D);
		glDisable(GL_LIGHTING);
		glDisable(GL_LIGHT0);
		glCallList(MOUTHWALL_RENEWAL_LIST);
		objMouth->drawMouthwallModel();
	}

	double *eye_r_t = m_pDoc->GetEyeRTrans();
	double *eye_l_t = m_pDoc->GetEyeLTrans();
	double &eye_r_s = m_pDoc->GetEyeRScale();
	double &eye_l_s = m_pDoc->GetEyeLScale();
	double *eye_local_r_t = m_pDoc->GetEyeRLocalTrans();
	double *eye_local_l_t = m_pDoc->GetEyeLLocalTrans();
	double *eye_local_r_r = m_pDoc->GetEyeRLocalRot();
	double *eye_local_l_r = m_pDoc->GetEyeLLocalRot();
	double &eye_local_r_s = m_pDoc->GetEyeRScale();
	double &eye_local_l_s = m_pDoc->GetEyeLScale();

	eye_r_s = 1.0 / fabs(obj->GetXMax() - obj->GetXMin()) * fabs(workWfm->Sgridx(123) - workWfm->Sgridx(117)) * 1.5;
	eye_l_s = eye_r_s;
	eye_r_t[0] = (workWfm->Sgridx(123) + workWfm->Sgridx(117)) / 2.0 + eye_local_r_t[0];
	eye_r_t[1] = workWfm->Sgridy(120) + eye_local_r_t[1];
	eye_r_t[2] = workWfm->Sgridz(120) - obj->GetZMax() * eye_r_s + eye_local_r_t[2]; 
	eye_l_t[0] = (workWfm->Sgridx(152) + workWfm->Sgridx(158)) / 2.0 + eye_local_l_t[0];
	eye_l_t[1] = workWfm->Sgridy(155) + eye_local_l_t[1];
	eye_l_t[2] = workWfm->Sgridz(155) - obj->GetZMax() * eye_r_s + eye_local_l_t[2];

	if(m_pDoc->GetEyeTexture() != NULL) {
		glCallList(EYE_RENEWAL_LIST);

		glPushMatrix();
		glTranslated(eye_r_t[0], eye_r_t[1], eye_r_t[2]);
		glScaled(eye_r_s, eye_r_s, eye_r_s);
		glRotated(eye_local_r_r[0], 1.0, 0.0, 0.0);
		glRotated(eye_local_r_r[1], 0.0, 1.0, 0.0);
		glRotated(eye_local_r_r[2], 0.0, 0.0, 1.0);
		obj->DrawObjModel();
		glPopMatrix();
	
		glPushMatrix();
		glTranslated(eye_l_t[0], eye_l_t[1], eye_l_t[2]);
		glScaled(eye_l_s, eye_l_s, eye_l_s);
		glRotated(eye_local_l_r[0], 1.0, 0.0, 0.0);
		glRotated(eye_local_l_r[1], 0.0, 1.0, 0.0);
		glRotated(eye_local_l_r[2], 0.0, 0.0, 1.0);
		obj->DrawObjModel();
		glPopMatrix();
	}

	glDisable(GL_TEXTURE_2D);
	
	glPopMatrix();
	
	return true;
}

void DialogEdit::SetupWindow()
{
	Texture *pTex = m_pDoc->GetTexture();
	int x = pTex->GetWidth() / 2;
	int y = pTex->GetHeight() / 2;
}

void DialogEdit::LeftMousePushEvent(int x, int y, int status)
{
	if(!m_pDoc->IsModelOpened()) return;
	
	m_point.x = x;
	m_point.y = y;
	
	m_bLButtonDown = true;
}

void DialogEdit::MidMousePushEvent(int x, int y, int status)
{
	if(!m_pDoc->IsModelOpened()) return;
	
	m_point.x = x;
	m_point.y = y;
	
	m_bMButtonDown = true;
}

void DialogEdit::RightMousePushEvent(int x, int y, int status)
{
	if(!m_pDoc->IsModelOpened()) return;
	
	m_point.x = x;
	m_point.y = y;
	
	m_bRButtonDown = true;
}

void DialogEdit::MouseDragEvent(int x, int y, int status)
{
	if(!m_pDoc->IsImageOpened()) return;
	
	if(m_bLButtonDown) {		
		double xr = m_rot[0];
		double yr = m_rot[1];
		double zr = m_rot[2];
		
		xr += (y - m_point.y) / 5.0;
		yr += (x - m_point.x) / 5.0;
		m_point.x = x;
		m_point.y = y;
		
		m_rot[0] = xr;
		m_rot[1] = yr;
		m_rot[2] = zr;	
	}
	else if(m_bMButtonDown) {
		double xt = m_trans[0];
		double yt = m_trans[1];
		double zt = m_trans[2];
		
		xt += (x - m_point.x) / (double)(w() / 2);
		yt -= (y - m_point.y) / (double)(h() / 2);
		m_point.x = x;
		m_point.y = y;
		
		m_trans[0] = xt;
		m_trans[1] = yt;
		m_trans[2] = zt;
	}
	else if(m_bRButtonDown) {
		m_scale -= (y - m_point.y) / ( 1.0 / m_scale * 500.0);
		m_point.x = x;
		m_point.y = y;
	}
}

void DialogEdit::LeftMouseReleaseEvent(int x, int y, int status)
{
	if(!m_pDoc->IsImageOpened()) return;
	m_bLButtonDown = false;
}

void DialogEdit::RightMouseReleaseEvent(int x, int y, int status)
{
	if(!m_pDoc->IsImageOpened()) return;
			m_bRButtonDown = false;	
}

void DialogEdit::MidMouseReleaseEvent(int x, int y, int status)
{
	if(!m_pDoc->IsImageOpened()) return;
			m_bMButtonDown = false;
}
