<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.parameters.datatypes
 */
/**
 * @file Base64BinaryParameter.php
 * @brief Base64BinaryParameter for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: Base64BinaryParameter.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * ValidatableSimpleParameter with normailizer.
 * treat param as base64Binary
 */

require_once("ficus/exception/ValidationException.php");
require_once("ficus/parameters/ValidatableSimpleParameter.php");

/**
 * @class Ficus_Base64BinaryParameter
 */
class Ficus_Base64BinaryParameter extends Ficus_ValidatableSimpleParameter
{
	const TYPENAME = "base64Binary";
	const PATTERN = '/^[a-z0-9+\/=+]+$/i';
	const LENGTH = "200";
	
	/**
 	 * construct parameter with name, value, and specific validator.
 	 * @param $name string name of this parameter.
 	 * @param $value string value of this parameter.
 	 * @param $validator array array of Ficus_Validator.
 	 */
 	public function __construct($name, $value = null, $validator = array()){
		parent::__construct($name, self::TYPENAME, $value);
		$this->setValidator($validator);
	}

    /**
	 * set value
	 * @param $value mixed value
	 */
	public function setValue($value){
		if(preg_match(self::PATTERN, trim(substr($value, 0, self::LENGTH))) &&
			strlen($value) % 4 == 0){
			$this->value = base64_decode($value);
		}else{
			$this->value = $value;
		}		
	}

	/**
	 * normalize parameter to base64Binary
	 * @param $param string decoded value
	 */
	public function normalize($param){
		return base64_encode($param);
	}
	
	/**
	 * check param that associated with simple parameter
	 * must be binary64 encoded string
	 * @param $parameter array data
	 * @return array validated param
     * @throw Ficus_ValidationException invalidate parameter.
	 */
	public function validate($parameter){
		$value = $parameter["value"];
		if(empty($value) ||
		   (preg_match(self::PATTERN, trim(substr($value, 0, self::LENGTH))) &&
			strlen($value) % 4 == 0)){
			$parameter["value"] = base64_decode($value);
			parent::validate($parameter);
		}else{
			throw new Ficus_ValidationException("failed base64 decoding {$this->name()}.");
		}
	}
}
?>
