<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.net
 */
/**
 * @file WSDL.php
 * @brief WSDL for php
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: WSDL.php 2 2007-07-11 10:37:48Z ishitoya $
 * 
 * WSDL for php
 */

require_once("ficus/lang/Assert.php");
require_once("ficus/net/URI.php");
require_once("ficus/parameters/ValidatableComplexParameter.php");
require_once("ficus/parameters/visitors/ParameterToWSDLConverter.php");
require_once("ficus/exception/IllegalArgumentException.php");
require_once("ficus/exception/URISyntaxException.php");

/**
 * @class Ficus_WSDL
 */
class Ficus_WSDL{
	//namespaces
	const XMLNS_WSDL = "http://schemas.xmlsoap.org/wsdl/";
	const XMLNS_XSD = "http://www.w3.org/2001/XMLSchema";
	const XMLNS_SOAP = "http://schemas.xmlsoap.org/wsdl/soap/";
	const XMLNS_SOAPENC = "http://schemas.xmlsoap.org/wsdl/soap/encoding/";
	//prefixes
	//tns for PEAR::SOAP
	const P_TYPENS = "tns:";
	const P_XSD = "xsd:";
	const P_SOAPENC = "SOAP-ENC:";
	//soap
	const SOAP_TRANSPORT = "http://schemas.xmlsoap.org/soap/http";
	const SOAP_ENCODING = "http://schemas.xmlsoap.org/soap/encoding/";
	const SOAP_ENCODED = "encoded";
	const SOAP_LITERAL = "literal";
	const SOAP_RPC = "rpc";
	const SOAP_DOCUMENT = "document";

	/**
	 * @var $dom DomDocument
	 */
	private $dom = null;

	/**
	 * @var $wsdl string WSDL string style
	 */
	private $wsdl = null;
	
	/**
	 * @var $endPoint string
	 */
	private $endPoint = null;
	
	/**
	 * @var $targetNS string target Namespace. In Soya usualy "urn:soya:module:..."
	 */
	private $targetNS = null;
	
	/**
	 * @var $portName string
	 */
	private $portName = null;

	/**
	 * @var $serviceName string "Service"
	 */
	private $serviceName = null;

	/**
	 * @var $bindingName string "Binding"
	 */
	private $bindingName = null;

	/**
	 * @var $operationName string
	 */
	private $operationName = null;

	/**
	 * @var $bindingStyle string style rpc/document default is rpc
	 */
	private $bindingStyle = null;

	/**
	 * @var $bindingUse string body encoded/literal default is encoded
	 */
	private $bindingUse = null;
	
	/**
	 * @var outputParameter Ficus_ValidatableComplexParameter output parameter
	 */
	private $outputParameter = null;
	
	/**
	 * $inputParameter Ficus_ValidatableComplexParameter input parameter
	 */
	private $inputParameter = null;

	/**
	 * Constructor
	 *
	 * In and out array format is like this
	 * array("HogeRequest" => array("HP" => "xsd:int", "MP" => "typens:MP"))
	 * array key will treat as name and array value will be type.
	 *
	 * @param $endPoint string endpoint of this service
	 * @param $targetNS string targetNameSpace
	 * @param $operationName string operationName
	 * @param $portName string portName will be prefix of binding and so on.
	 * @param $in Ficus_ValidatableComplexParameter input param
	 * @param $out Ficus_ValidatableComplexParameter output param
	 */
	function __construct($endPoint, $targetNS,
						 $operationName, $portName, $in, $out = null){
		$this->setEndPoint($endPoint);
		$this->setTargetNameSpace($targetNS);
		$this->setPortName($portName);
		$this->setOperationName($operationName);
		$this->setInput($in);
		$this->setOutput($out);
		$this->dom = new DOMDocument("1.0", "utf-8");
		$this->bindingStyle = self::SOAP_RPC;
		$this->bindingUse = self::SOAP_ENCODED;
	}

	/**
	 * set binding-style
	 * @param $style String bindingStyle
     * @throw Ficus_IllegalArgumentException no rpc or no document.
	 */
	public function setBindingStyle($style){
		$style = trim($style);
		if($style !== self::SOAP_RPC || $style !== self::SOAP_DOCUMENT){
			throw new Ficus_IllegalArgumentException(
				"Style must be rpc or document : param = $style");
		}
		$this->bindingStyle = $style;
	}

	/**
	 * set binding-use
	 * @param $use String bindingUse
     * @throw Ficus_IllegalArgumentException no encoded or no literal.
	 */
	public function setBindingUse($use){
		$use = trim($use);
		if($use !== self::SOAP_ENCODED || $use !== self::SOAP_LITERAL){
			throw new Ficus_IllegalArgumentException(
				"Use must be encoded or literal : param = $use");
		}
		$this->bindingUse = $use;
	}

	/**
	 * serialize WSDL
	 * @return string WSDL
	 */
	public function serialize(){
		if(empty($this->wsdl)){
			$this->createDefinitionElement();
		}
		return $this->wsdl;
	}

	/**
	 * create Definition Element
	 */
	private function createDefinitionElement(){
		//create definition element
		$root = $this->dom->createElementNS(self::XMLNS_WSDL, "definitions");
		$this->dom->appendChild($root);
		$root->setAttribute("name", $this->serviceName);
		$root->setAttribute("targetNamespace", $this->targetNS);
		//for fucking php DOM functions.
		$root->setAttribute("xmlns:xsd" , self::XMLNS_XSD);
		$root->setAttribute("xmlns:soap", self::XMLNS_SOAP);
		$root->setAttribute("xmlns:SOAP-ENC", self::XMLNS_SOAPENC);
		$root->setAttribute("xmlns:tns" , $this->targetNS);

		$converter = new Ficus_ParameterToWSDLConverter($this, $this->dom);
		$this->input->accept($converter);
		$this->output->accept($converter);
		$root->appendChild($converter->getConvertedTypes());
		foreach($converter->getConvertedMessages() as $message){
			$root->appendChild($message);
		}
		
		$root->appendChild($this->createPortTypeElement());
		$root->appendChild($this->createBindingElement());
		$root->appendChild($this->createServiceElement());

		$this->wsdl = $this->dom->saveXML();
	}
	/**
	 * create portType Element
	 * @return DomElemet
	 */
	private function createPortTypeElement(){
		$portType = $this->dom->createElement("portType");
		$portType->setAttribute("name", $this->portName);
		$operation = $this->dom->createElement("operation");
		$operation->setAttribute("name", $this->operationName);
		$input = $this->dom->createElement("input");
		$input->setAttribute(
			"message", self::P_TYPENS . $this->input->name());
		$output = $this->dom->createElement("output");
		$output->setAttribute(
			"message", self::P_TYPENS . $this->output->name());
		$operation->appendChild($input);
		$operation->appendChild($output);
		$portType->appendChild($operation);
		return $portType;
	}
	
	/**
	 * create Binding Element
	 * @return DomElements 
	 */
	private function createBindingElement(){
		$binding = $this->dom->createElement("binding");
		$binding->setAttribute("name", $this->bindingName);
		$binding->setAttribute("type", self::P_TYPENS . $this->portName);
		$soap_binding = $this->dom->createElementNS(
			self::XMLNS_SOAP, "soap:binding");
		$soap_binding->setAttribute("style", $this->bindingStyle);
		$soap_binding->setAttribute("transport", self::SOAP_TRANSPORT);
		$operation = $this->dom->createElement("operation");
		$operation->setAttribute("name", $this->operationName);
		$operation->setAttribute("soapAction", $this->operationName);
		$soap_operation = $this->dom->createElementNS(
			self::XMLNS_SOAP, "soap:operation");
		$soap_operation->setAttribute("soapAction", $this->operationName);
		$input = $this->dom->createElement("input");
		$input_soap_body = $this->dom->createElementNS(
			self::XMLNS_SOAP, "soap:body");
		$input_soap_body->setAttribute("use", self::SOAP_ENCODED);
		$input_soap_body->setAttribute("namespace", $this->targetNS);
		$input_soap_body->setAttribute("encodingStyle", self::SOAP_ENCODING);
		$output = $this->dom->createElement("output");
		$output_soap_body = $this->dom->createElementNS(
			self::XMLNS_SOAP, "soap:body");
		$output_soap_body->setAttribute("use", self::SOAP_ENCODED);
		$output_soap_body->setAttribute("namespace", $this->targetNS);
		$output_soap_body->setAttribute("encodingStyle", self::SOAP_ENCODING);
		$binding->appendChild($soap_binding);
		$input->appendChild($input_soap_body);
		$output->appendChild($output_soap_body);
		$operation->appendChild($soap_operation);
		$operation->appendChild($input);
		$operation->appendChild($output);
		$binding->appendChild($operation);
		return $binding;
	}

	/**
	 * create Service Element
	 * @return DomElements
	 */
	private function createServiceElement(){
		$service = $this->dom->createElement("service");
		$service->setAttribute("name", $this->serviceName);
		$port = $this->dom->createElement("port");
		$port->setAttribute("name", $this->portName);
		$port->setAttribute("binding", self::P_TYPENS . $this->bindingName);
		$address = $this->dom->createElementNS(
			self::XMLNS_SOAP, "soap:address");
		$address->setAttribute("location", $this->endPoint);
		$port->appendChild($address);
		$service->appendChild($port);
		return $service;
	}		
	
	/**
	 * validate() and set endPoint
	 * @param $endPoint string endPoint URI
     * @throw Ficus_URISyntaxException invalid.
	 */
	private function setEndPoint($endPoint){
        try {
            Ficus_URIBuilder::create($endPoint);
        } catch (Ficus_URISyntaxException $e) {
			throw new Ficus_URISyntaxException(
				"End point URI must be valid! : $endPoint");
		}
		$this->endPoint = $endPoint;
	}

	/**
	 * validate() and set targetNS
	 * @param $targetNS string targetNS URI
     * @throw Ficus_URISyntaxException invalid.
	 */
	private function setTargetNameSpace($targetNS){
        try {
            Ficus_URIBuilder::create($targetNS);
        } catch (Ficus_URISyntaxException $e) {
			throw new Ficus_URISyntaxException(
				"Target Namespace URI is invalid : $targetNS");
		}
		$this->targetNS = $targetNS;
	}

	/**
	 * get TargetNS
	 * @return string targetNS
	 */
	public function targetNameSpace(){
		return $this->targetNS;
	}

	/**
	 * validate() and set portName
	 * @param $portName string set port name
     * @throw Ficus_IllegalArgumentException invalid port name.
	 */
	private function setPortName($portName){
		if(empty($portName)){
			throw new Ficus_IllegalArgumentException(
				"Port Name is Invalid : $portName");
		}
		$this->portName    = $portName . "PortType";
		$this->serviceName = $portName . "Service";
		$this->bindingName = $portName . "Binding";
	}

	/**
	 * validate() and set operationName
	 * @param $operationName string set operation name
     * @throw Ficus_IllegalArgumentException invalid operation name.
	 */
	private function setOperationName($operationName){
		if(empty($operationName)){
			throw new Ficus_IllegalArgumentException(
				"Operation Name is Invalid : $operationName");
		}
		$this->operationName = $operationName;
	}

	/**
	 * validate() and set Input parameter
	 * @param $input Ficus_ValidatableComplexParameter input array
	 */
	private function setInput($input){
        Ficus_Assert::typeHinting('Ficus_ValidatableComplexParameter', $input);
        $this->input = $input;
	}

	/**
	 * validate() and set Output parameter
	 * @param $output Ficus_ValidatableComplexParameter output array
	 */
	private function setOutput($output){
        Ficus_Assert::typeHinting('Ficus_ValidatableComplexParameter', $output);
        $this->output = $output;
	}
}
?>
