<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.db.s2dao.models.annotation
 */
/**
 * @file EntityAnnotationReader.php
 * @brief EntityAnnotationReader.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: S2DaoEntityAnnotationReader.php 2 2007-07-11 10:37:48Z ishitoya $
 *
 */

require_once("ficus/beans/Bean.php");
require_once("ficus/exception/PropertyNotFoundException.php");
require_once("ficus/db/s2dao/models/S2DaoModelConstants.php");

/**
 * @class Ficus_S2DaoEntityAnnotationReader
 */
class Ficus_S2DaoEntityAnnotationReader implements Ficus_S2DaoModelConstants{
    /**
     * target annotations
     */
    protected static $targets = array(self::ANNOTATION_FORM,
                                      self::ANNOTATION_VIEW,
                                      self::ANNOTATION_EXPAND,
                                      self::ANNOTATION_TYPE,
                                      self::ANNOTATION_DATATYPE,
                                      self::ANNOTATION_LABEL,
                                      self::ANNOTATION_REMARK,
                                      self::ANNOTATION_VALIDATOR,
                                      self::ANNOTATION_COLUMN);
    /**
     * @var $annotations array of annotations
     */
    protected $annotations = array();

    /**
     * @var $target ReflectionClass target entity
     */
    protected $target = null;

    /**
     * gettarget
     * @return ReflectionClass
     */
    public function getTarget(){
        return $this->target;
    }
        
    /**
     * constructor
     * @param $entity Ficus_S2DaoEntity target entity
     */
    public function __construct($entity){
        $this->target = $entity;
        $class = new ReflectionClass($entity);
        $annotation =
            Ficus_BeanConstantAnnotationAccessor::getAccessor($entity);
        $properties = $class->getDefaultProperties();
        $properties = array_keys($properties);
        foreach($properties as $property){
            foreach(self::$targets as $target){
                $this->setAnnotation($annotation, $property, $target);
            }
        }
    }

    /**
     * search annotation on property
     * @param $annotation Ficus_BeanConstantAnnotationAccessor accessor
     * @param $property string property name
     * @param $type string type of annotation
     */
    protected function setAnnotation($annotation, $property, $type){
        $name = "{$property}_$type";
        if($annotation->has($name)){
            $this->annotations[$property][$type] = $annotation->get($name);
        }
    }

    /**
     * get array of properties
     * @return array of property
     */
    public function properties(){
        return array_keys($this->annotations);
    }

    /**
     * check has anotation on property
     * @param $property string property name
     * @param $annotation string annotation name
     * @return boolean true if exists
     */
    public function has($property, $annotation){
        if(isset($this->annotations[$property][$annotation])){
            return true;
        }
        return false;
    }

    /**
     * get annotation on property 
     * @param $property string property name
     * @param $annotation string annotation name
     * @return mixed annotation
     */
    public function get($property, $annotation){
        if(isset($this->annotations[$property][$annotation]) == false){
            throw new Ficus_ConstantNotFoundException("annotation $annotation is not defined on $property");
        }
        return $this->annotations[$property][$annotation];
    }

    /**
     * sort properties
     * @param $order array of property order
     */
    public function sort($order){
        $order = array_keys($order);
        $temp = array();
        foreach($order as $key => $property){
            if(isset($this->annotations[$property])){
                $temp[$property] = $this->annotations[$property];
            }
        }
        foreach($this->annotations as $key => $property){
            if(isset($temp[$key])){
                continue;
            }
            $temp[$key] = $property;
        }
        $this->annotations = $temp;
    }
    
    const METHOD_SIGNATURE = '/^(has)?(.*?)?(view|form|expand|type|datatype|label|remark|validator|column)?$/i';
    /**
     * call method
     * @param $name string name of method
     * @param $arguments array of argument
     */
    public function __call($name, $arguments){
        if(preg_match(self::METHOD_SIGNATURE, $name, $regs)){
            if(empty($regs[2])){
                $property = $arguments[0];
            }else{
                $property = strtolower($regs[2][0]) . substr($regs[2], 1);
            }
            $type = strtoupper($regs[3]);
            if($regs[1] == "has"){
                return $this->has($property, $type);
            }
            return $this->get($property, $type); 
        }else{
            throw new Ficus_MethodNotFoundException("method $name is not found");
        }
    }
}
?>
