<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.beans
 */
/**
 * @file Bean.php
 * @brief This interface is Bean.
 * @author <a href="mailto:kent@guarana.cc">ISHITOYA Kentaro</a>
 * @version $Id: Bean.php 2 2007-07-11 10:37:48Z ishitoya $
 *
 * Abstract Bean.
 * __set and __get mediates $bean->setData() of $bean->getData()
 * on Java or other languge, you must implement
 * $bean->set("Data", $object) or $bean->get("Data") to mediate beans method.
 */

require_once "ficus/lang/Serializable.php";
require_once "ficus/net/URI.php";
require_once "ficus/exception/MethodNotFoundException.php";
require_once "ficus/exception/IllegalArgumentException.php";
require_once "ficus/exception/PropertyNotFoundException.php";
require_once "ficus/exception/NotReadyException.php";
require_once "ficus/beans/BeanComponentFactory.php";
require_once "ficus/beans/annotation/BeanAnnotationAccessor.php";

/**
 * @class Ficus_Bean
 */
abstract class Ficus_Bean implements Ficus_Serializable{
    const METHOD_SIGNATURE = '/^(set|get|add|delete|shift|push|pop|insert|numOf|count|clear|has|isEmpty|is|enable|disable)([a-zA-Z][a-zA-Z0-9_]*)/';
    const METHOD_SIGNATURE_SUFFIX = '/^([a-zA-Z][a-zA-Z0-9_]*?)(EqualsTo)/';
    const BEAN_METHOD_SIGNATURE = '/^get([a-zA-Z][a-zA-Z0-9_]*)Bean/';
    
    /**
     * serialize bean
     * @param $type string type of serializer
     * @return string serialized string
     */
    public function serialize($type = null){
        if(is_null($type)){
            $reflection = new ReflectionClass($this);
            $props = $reflection->getDefaultProperties();
            foreach($props as $key => $value){
                $props[$key] = $this->$key;
            }
            return serialize($props);
        }

        if($type instanceof Ficus_BeanSerializer){
            $serializer = $type;
        }else{
            $factory =
              Ficus_BeanComponentFactory::getComponent("BeanSerializerFactory");
            $serializer = $factory->create($type);
        }
        if(func_num_args() > 1){
            $arguments = func_get_args();
            array_shift($arguments);
            return $serializer->serialize($this, $arguments);
        }else{
            return $serializer->serialize($this);
        }
    }

    public function unserialize($serialized){
        $unserialized = unserialize($serialized);
        foreach($unserialized as $key => $value){
            $this->$key = $value;
        }
    }


    /**
     * deserialize bean
     * @param $type string type of deserializer
     */
    public function deserialize($data, $deserializer=null){
        if(($deserializer instanceof Ficus_BeanDeserializer) == false){
            $factory =
              Ficus_BeanComponentFactory::getComponent(
                      "BeanDeserializerFactory");
            $deserializer = $factory->create($data);
        }
        return $deserializer->deserialize($this, $data);
    }

    /**
     * create clone
     * @return Ficus_Bean cloned object
     */
    public function createClone(){
        $clone = clone $this;
        return $clone;
    }

    /**
     * set value to specified property
     * @param $name string property name
     * @param $value mixed value to set
     */
    public function set($name, $value){
        if($this->has($name) == false){
            throw new Ficus_PropertyNotFoundException("property $name is not found");
        }
        $this->{$name} = $value;
    }
    
    /**
     * get value
     * @return mixed value
     */
    public function get($name, $index = null){
        if($this->has($name) == false){
            throw new Ficus_PropertyNotFoundException("property $name is not found");
        }
        $method = "get" . ucfirst($name);
        if(method_exists($this, $method)){
            return $this->{$method}();
        }
        if(is_null($index)){
            return $this->{$name};
        }else{
            return $this->{$name}[$index];
        }
    }

    /**
     * has property
     * @param $name string property name
     * @return boolean true if property exists
     */
    public function has($name){
        return property_exists($this, $name);
    }
    

    /**
     * other functions
     */
    public function __call($name, $arguments){
        return $this->__onCall($name, $arguments);
    }

    /**
     * handle call
     */
    protected function __onCall($name, $arguments){
        $class = new ReflectionClass($this);
        if($class->hasProperty($name)){
            $property = $class->getProperty($name);
            if($property->isStatic()){
                $values = $class->getStaticProperties();
                $value = $values[$name];
            }else{
                $value = $this->{$name};
            }
            if(isset($arguments[0])){
                return $value[$arguments[0]];
            }else{
                return $value;
            }
        }

        if(preg_match(self::METHOD_SIGNATURE, $name, $matches)){
            $operator = $matches[1];
            $property = strtolower($matches[2][0]) . substr($matches[2], 1);

            switch($operator){
            case "has" :
                return $this->has($property);
            }

            if($this->has($property)){
                switch($operator){
                case "set" :
                    $this->{$property} = $arguments[0];
                    return;
                case "get" :
                    if(isset($arguments[0])){
                        if(is_null($arguments[0])){
                            return null;
                        }
                        if(isset($this->{$property}[$arguments[0]])){
                            return $this->{$property}[$arguments[0]];
                        }else{
                            return null;
                        }
                    }
                    return $this->{$property};
                case "add" :
                case "push" :
                    return array_push($this->{$property}, $arguments[0]);
                case "shift" :
                    return array_shift($this->{$property});
                case "pop" :
                    return array_pop($this->{$property});
                case "delete" :
                    if(isset($arguments[0])){
                        if((is_numeric($arguments[0]) ||
                            is_string($arguments[0])) &&
                           isset($this->{$property}[$arguments[0]])){
                            $temp = $this->{$property}[$arguments[0]];
                            unset($this->{$property}[$arguments[0]]);
                            return $temp;
                        }else{
                            foreach($this->{$property} as $key => $item){
                                if($item == $arguments[0]){
                                    unset($this->{$property}[$key]);
                                    return $item;
                                }
                            }
                            return;
                        }
                    }
                    break;
                case "insert" :
                    if(isset($arguments[0])){
                        $this->{$property}[$arguments[0]] = $arguments[1];
                        return;
                    }
                    break;
                case "numOf" :
                case "count" :
                    return count($this->{$property});
                case "clear" :
                    if(is_array($this->{$property})){
                        $this->{$property} = array();
                    }else{
                        $this->{$property} = null;
                    }
                    return;
                case "isEmpty" :
                    return empty($this->{$property}); 
                case "is" :
                    return (boolean)$this->{$property};
                case "enable" :
                    $this->{$property} = true;
                    return;
                case "disable" :
                     $this->{$property} = false;
                    return;
               }
            }
        }
        if(preg_match(self::METHOD_SIGNATURE_SUFFIX, $name, $matches)){
            $operator = $matches[2];
            $property = strtolower($matches[1][0]) . substr($matches[1], 1);
            if($this->has($property)){
                switch($operator){
                case "EqualsTo" :
                    return Ficus_Types::equalsByToString($arguments[0],
                                                        $this->{$property});
                }
            }
        }
        if(method_exists($this, $name)){
            call_user_func_array(array($this, $name), $arguments);
        }else if(isset($property) &&
                 $this->has($property) == false){
            throw new Ficus_PropertyNotFoundException("property $property is not found.");
        }else{
            throw new Ficus_MethodNotFoundException("method $name is not found.");
        }
    }
}
