<?php 
/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4: */
/*
 * Copyright 2004-2007 Project Guarana Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @package ficus.XML
 */
/**
 * @file NamespacePrefixes.php
 * @brief NamespacePrefixes.
 * @author <a href="mailto:sumi@wakhok.ac.jp">SUMI Masafumi</a>
 * @version $Id: NamespacePrefixes.php 2 2007-07-11 10:37:48Z ishitoya $
 *
 */
require_once("ficus/XML/NamespacePrefixAlreadyExistException.php");
require_once("ficus/lang/Types.php");
/**
 * @class Ficus_NamespacePrefixes
 */
final class Ficus_NamespacePrefixes implements IteratorAggregate
{
    /**
     * namespaces.
     */
    private $namespaces;

    /**
     * auto prefix number.
     */
    private $prefixNumber;

    /**
     * constructor.
     */
    public function __construct() {
        $this->namespaces = array();
        $this->prefixNumber = 1;
    }

    /**
     * add namespace.
     *
     * You can not set a prefix that already exist on a namespace.
     * But you can set different prefix on same namespace.
     *
     * @param $qName string namespace uri.
     * @param $prefix string namespace prefix.
     */
    public function addNamespaceOf($qName, $prefix = null) {
        Ficus_Assert::typeHinting("Ficus_QName", $qName);
        $this->addNamespace($qName->uri(), $prefix);
    }

    /**
     * add namespace.
     *
     * You can not set a prefix that already exist on a namespace.
     * But you can set different prefix on same namespace.
     * When a prefix is not specified, It will automaticaly search same
     * namespace prefix. And if it is a new namespace, sets new prefix.
     *
     * @param $namespace string namespace uri.
     * @param $prefix string namespace prefix.
     * @throw Ficus_NamespacePrefixAlreadyExistException namespace prefix already exist.
     */
    public function addNamespace($namespace, $prefix = null) {
        if ($prefix == null) {
            $alreadyPrefix = array_search($namespace, $this->namespaces);
            if ($alreadyPrefix !== FALSE) {
                $prefix = $alreadyPrefix;
            } else {
                $prefix = $this->newPrefix();
            }
        }

        $already = $this->getNamespace($prefix);
        if ($already == $namespace) {
            return;
        }

        if ($already !== NULL) {
            throw new Ficus_NamespacePrefixAlreadyExistException(
                "already prefix: " + $prefix
                + ", existing namespace: " + $this->namespaces[$prefix]
                + ", tried to add namespace : " + $namespaces
            );
        }

        $this->namespaces[$prefix] = $namespace;
    }

    /**
     * get namespace.
     *
     * @param $prefix string namespace prefix.
     * @return namespace or NULL if not found.
     */
     public function getNamespace($prefix) {
        if(isset($this->namespaces[$prefix])){
         return $this->namespaces[$prefix];
     }
        return null;
    }

    /**
     * get prefix.
     *
     * @param $namespace string namespace uri.
     * @return array of namespace prefix or FALSE if not found.
     */
    public function getPrefix($namespace) {
        $prefix = array_search($namespace, $this->namespaces);
        return $prefix !== FALSE ? $prefix : NULL;
    }

    /**
     * get prefixed qname.
     *
     * @param $qname string namespace uri.
     * @return array of namespace prefix or FALSE if not found.
     */
    public function getPrefixedQName($qname) {
        return $this->getPrefix($qname->uri()) . ":" . $qname->localPart();
    }

    /**
     * get prefix.
     *
     * @param $namespace string namespace uri.
     * @return array of namespace prefix or FALSE if not found.
     */
    public function getPrefixes($namespace) {
        return array_keys($this->namespaces, $namespace);
    }

    /**
     * contains namespace.
     *
     * @param $namespace string namespace uri.
     */
    public function contains($namespace) {
        return array_search($namespace, $this->namespaces) !== FALSE;
    }

    /**
     * contains namespace prefix.
     *
     * @param $prefix string namespace prefix.
     */
    public function containsPrefix($prefix) {
        return array_key_exists($prefix, $this->namespaces);
    }

    /**
     * create new prefix.
     *
     * @return prefix string.
     */
    private function newPrefix() {
        do {
            $newPrefix = sprintf("ns%d", $this->prefixNumber++);
        } while (array_key_exists($newPrefix, $this->namespaces));
        return $newPrefix;
    }

    /**
     * extract namespaces in qnames.
     *
     * @param $qnames array array of Ficus_PrefixedQName.
     */
    public function extractNamespaces($qnames) {
        foreach ($qnames as $qname) {
            if ($qname instanceof Ficus_PrefixedQName) {
                $this->addNamespace($qname->uri(), $qname->prefix());
            } else {
                $this->addNamespace($qname->uri(), $this->newPrefix());
            }
        }
    }

    /**
     * add prefix to array of QName.
     *
     * @param $qnames array qnames.
     * @return array prefixed qnames.
     */
    public function addPrefixToQNames($qnames) {
        $prefixedQNames = array();
        foreach ($qnames as $qname) {
            array_push($prefixedQNames, $this->addPrefixToQName($qname));
        }
        return $prefixedQNames;
    }

    /**
     * add prefix to QName.
     *
     * @param $qname string qname.
     */
    public function addPrefixToQName($qname) {
        if (($prefix = $this->getPrefix($qname->uri())) == NULL) {
            $prefix = $this->newPrefix();
        }
        return new Ficus_PrefixedQName($qname->uri(), $qname->localPart(), $prefix);
    }

    /**
     * get iterator.
     *
     * return ArrayIterator of namespaces iterator.
     */
    public function getIterator() {
        $ao = new ArrayObject($this->namespaces);
        return $ao->getIterator();
}

    /**
     * remove namespace repetition.
     *
     * return Ficus_NamespacePrefixes namespace repetition was removed.
     */
    public function pack() {
        $packedNamespaces = new Ficus_NamespacePrefixes();
        foreach ($this as $prefix => $uri) {
            if (!$packedNamespaces->contains($uri)) {
                $newPrefix = $prefix;
                while ($packedNamespaces->containsPrefix($newPrefix)) {
                    $newPrefix = sprintf("ns%d", $packedNamespaces->prefixNumber++);
                }
                $packedNamespaces->addNamespace($uri, $newPrefix);
            }
        }
        return $packedNamespaces;
    }
}
?>
