/*
 * Copyright 2000-2001,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.jetspeed.test;

// Junit imports
import junit.awtui.TestRunner;
import junit.framework.Test;
import junit.framework.TestSuite;

// Cactus imports
import org.apache.cactus.ServletTestCase;
import org.apache.cactus.WebRequest;
import org.apache.cactus.WebResponse;

// Jetspeed imports
import org.apache.jetspeed.capability.CapabilityMap;
import org.apache.jetspeed.om.profile.Profile;
import org.apache.jetspeed.om.profile.ProfileLocator;
import org.apache.jetspeed.om.profile.PSMLDocument;
import org.apache.jetspeed.services.Profiler;
import org.apache.jetspeed.services.rundata.JetspeedRunData;
import org.apache.jetspeed.test.TurbineTestUtilities;

// Turbine imports
import org.apache.turbine.services.pull.TurbinePull;
import org.apache.turbine.services.velocity.TurbineVelocity;
import org.apache.turbine.util.RunData;
import org.apache.turbine.util.RunDataFactory;

import org.apache.jetspeed.om.security.JetspeedUser;

// Velocity
import org.apache.velocity.context.Context;

/**
 *
 * @author <a href="paulsp@apache.org">Paul Spencer</a>
 * @version $Id $
 */

public class TestBasicSanity extends ServletTestCase
{
    private static String TEST_ANON_USER_NAME = "";
    private static String TEST_CONTEXT = null;
    private static String TEST_DEFAULT_PAGE = "default";
    private static String TEST_HOST = "localhost";
    private static String TEST_SERVLET = "/portal";
    private static String TEST_GROUP = "apache";
    private static String TEST_PAGE = "news";
    private static String TEST_USER = "turbine";
    private static String TEST_USER_PASSWORD ="turbine";
    
    private RunData rundata = null;

    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestBasicSanity(String name)
    {
        super( name );
    }

    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[])
    {
        TestRunner.main( new String[] { TestBasicSanity.class.getName() } );
    }
    
    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite()
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestBasicSanity.class );
    }
    
    /**
     * Sets up the test case.
     *
     */
    protected void setUp () throws Exception 
    {
    }
    
    /**
     *  Test: DefaultURL
     *  With the default URL "/"
     *    1) A page is generated 
     *    2) The user is anonymous
     *    3) Group and Role are not set
     */
    public void beginDefaultURL(WebRequest theRequest)
    {
        System.out.println("URL = " + theRequest.getURL());
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET, "", null); 
        System.out.println("post set URL = " + theRequest.getURL());
    }

    /**
     * Execute the test
     *
     * @throws Exception
     */
    public void testDefaultURL() throws Exception
    {
        // Create the RunData object to be used during testing.        
        rundata = RunDataFactory.getRunData ( request, response, config );
        assertNotNull( "Got rundata", rundata);
        TurbineTestUtilities.setupRunData(rundata);

        // Verify we have a user
        JetspeedUser user = (JetspeedUser)rundata.getUser();
        assertNotNull( "Got user", user);

        // Verify we have a CapabilityMap
        CapabilityMap cm = ((JetspeedRunData)rundata).getCapability();
        assertNotNull( "Got Capability", cm);

        // Verify we have a profile
        Profile profile = Profiler.getProfile(rundata);
        assertNotNull( "Got profile from Profiler", profile);
        
        // Verify the profile location information in the profile
        if (profile instanceof ProfileLocator)
        {
            ProfileLocator profileLocator = (ProfileLocator) profile;
            assertTrue("Verify the 'anonymous' is set", profileLocator.getAnonymous());
            assertNull("Verify the group is null", profileLocator.getGroup());
            assertNull("Verify the role is null", profileLocator.getRole());
        } else
        {
            assertTrue( "profile does not implement ProfileLocator", false);
        }
        

        // Verify we have a Document
        PSMLDocument psmlDoc = profile.getDocument();
        assertNotNull( "Got psmlDocument", psmlDoc);

        System.out.println("DocumentName = " + profile.getDocument().getName()); 

        // Get and populate the context
        Context context = TurbineVelocity.getContext(rundata);
        assertNotNull( "Got context", context);
        TurbinePull.populateContext( context, rundata);

        // Verify tool are in the context
        assertNotNull( "Got jlink from context", context.get("jlink"));

        // Generatate and output thre page
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);

        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }

    public void endDefaultURL(org.apache.cactus.WebResponse theResponse)
    {
        System.out.println("text length = " + theResponse.getText().length());
//        System.out.println("text length = " + theResponse.getText());
    }

    /**
     *  Test: GroupURL
     *  With the default URL "/group/apache"
     *    1) A page is generated 
     *    2) The user is anonymous
     *    3) Group is set to "apache"
     *    4) Role is not set
     */
    public void beginGroupUrl(WebRequest theRequest)
    {
        System.out.println("URL = " + theRequest.getURL());
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET
          , "/group/" + TEST_GROUP , null); 
        System.out.println("post set URL = " + theRequest.getURL());
    }

    /**
     * Test the Group link
     * @throws Exception
     */
    public void testGroupUrl() throws Exception
    {
        // Create the RunData object to be used during testing.        
        rundata = RunDataFactory.getRunData ( request, response, config );
        assertNotNull( "Got rundata", rundata);

        TurbineTestUtilities.setupRunData(rundata);

        // Verify we have a profile
        Profile profile = Profiler.getProfile(rundata);
        assertNotNull( "Got profile from Profiler", profile);
        
        // Verify the profile location information in the profile
        if (profile instanceof ProfileLocator)
        {
            // FIXME: Need to verify 'anonymous' and group name
            ProfileLocator profileLocator = (ProfileLocator) profile;
//            assertTrue("Verify the 'anonymous' is set", profileLocator.getAnonymous());
            assertNotNull("Verify the group is not null", profileLocator.getGroup());
            assertNull("Verify the role is null", profileLocator.getRole());
        } else
        {
            assertTrue( "profile does not implement ProfileLocator", false);
        }
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);

        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }
    
    public void endGroupURL(WebResponse theResponse)
    {
        System.out.println("text length = " + theResponse.getText().length());
//        System.out.println("text length = " + theResponse.getText());
    }

    /**
     *  Test: PageURL
     *  With the page URL "/page/apache"
     *    1) A page is generated 
     *    2) The user is anonymous
     *    3) Group is set to "apache"
     *    4) Role is not set
     */
    public void beginPageUrl(WebRequest theRequest)
    {
        System.out.println("URL = " + theRequest.getURL());
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET
          , "/page/" + TEST_PAGE , null); 
        System.out.println("post set URL = " + theRequest.getURL());
    }

    /**
     * Test the PageURL
     *
     * @throws Exception
     */
    public void testPageUrl() throws Exception
    {
        // Create the RunData object to be used during testing.        
        rundata = RunDataFactory.getRunData ( request, response, config );
        assertNotNull( "Got rundata", rundata);

        TurbineTestUtilities.setupRunData(rundata);

        // Verify we have a profile
        Profile profile = Profiler.getProfile(rundata);
        assertNotNull( "Got profile from Profiler", profile);
        
        // Verify the profile location information in the profile
        if (profile instanceof ProfileLocator)
        {
            ProfileLocator profileLocator = (ProfileLocator) profile;
            // FIXME: Need to verify 'anonymous' and page name
            assertTrue("Verify the 'anonymous' is set", profileLocator.getAnonymous());
            assertNull("Verify the group is null", profileLocator.getGroup());
            assertNull("Verify the role is null", profileLocator.getRole());
            assertEquals("Verify the page name", profileLocator.getName(), TEST_PAGE + ".psml");
        } else
        {
            assertTrue( "profile does not implement ProfileLocator", false);
        }
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);

        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }
    
    public void endPageURL(WebResponse theResponse)
    {
        System.out.println("text length = " + theResponse.getText().length());
//        System.out.println("text length = " + theResponse.getText());
    }
    /**
     *  Test: PageURL
     *  With the page URL "/page/apache"
     *    1) A page is generated 
     *    2) The user is anonymous
     *    3) Group is set to "apache"
     *    4) Role is not set
     */
    public void beginUserPageUrl(WebRequest theRequest)
    {
        System.out.println("URL = " + theRequest.getURL());
        theRequest.setURL(TEST_HOST, TEST_CONTEXT, TEST_SERVLET
          , "/page/" + TEST_DEFAULT_PAGE ,"action=JLoginUser&username=turbine&password=turbine"); 
        System.out.println("post set URL = " + theRequest.getURL());
    }

    /**
     * Test the PageURL
     *
     * @throws Exception
     */
    public void testUserPageUrl() throws Exception
    {
        // Create the RunData object to be used during testing.        
        rundata = RunDataFactory.getRunData ( request, response, config );
        assertNotNull( "Got rundata", rundata);

        TurbineTestUtilities.setupRunData(rundata);

        // Verify we have a profile
        Profile profile = Profiler.getProfile(rundata);
        assertNotNull( "Got profile from Profiler", profile);
        
        // Verify the profile location information in the profile
        if (profile instanceof ProfileLocator)
        {
            ProfileLocator profileLocator = (ProfileLocator) profile;
            // FIXME: Need to verify 'anonymous' and page name
            assertTrue("Verify the 'anonymous' is not", !profileLocator.getAnonymous());
            assertNull("Verify the group is null", profileLocator.getGroup());
            assertNull("Verify the role is null", profileLocator.getRole());
            assertNotNull("Verify the user is not null", profileLocator.getUser());
            assertTrue("Verify the user is logged in", profileLocator.getUser().hasLoggedIn());
            assertEquals("Verify the user's username", profileLocator.getUser().getUserName(),TEST_USER);
            assertEquals("Verify the page name", profileLocator.getName(), TEST_DEFAULT_PAGE + ".psml");
        } else
        {
            assertTrue( "profile does not implement ProfileLocator", false);
        }
        TurbineTestUtilities.generatePage(rundata);
        TurbineTestUtilities.outputPage(rundata);

        // Return the used RunData to the factory for recycling.
        RunDataFactory.putRunData(rundata);
    }
    
    public void endUserPageURL(WebResponse theResponse)
    {
        System.out.println("text length = " + theResponse.getText().length());
//        System.out.println("text length = " + theResponse.getText());
    }
}
