/**
 * Copyright (c) 2006, yher.net
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice, 
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * * Neither the name of the nor the names of its contributors may be used to endorse or 
 *   promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 */
package net.yher.workstyle;

import java.util.List;

import junit.framework.TestCase;
import net.yher.commons.collections.CollectionUtils;
import net.yher.workstyle.TaskQuery;

public class TaskQueryTest extends TestCase {
	TaskQuery query;

	protected void setUp() throws Exception {
		super.setUp();
		query = new TaskQuery();
	}

	/*
	 * Test method for 'net.yher.workstyle.TaskQuery.hasKeyword()'
	 */
	public void testHasKeyword_keywordが１文字以上あればtrueを返す() {
		query.setKeyword("a");
		assertTrue(query.hasKeyword());
	}
	public void testHasKeyword_keywordが空文字またはnullの場合はfalseを返す() {
		query.setKeyword("");
		assertFalse(query.hasKeyword());
		query.setKeyword(null);
		assertFalse(query.hasKeyword());
	}

	/*
	 * Test method for 'net.yher.workstyle.TaskQuery.hasStatus()'
	 */
	public void testHasStatus_statusが１つ以上あればtrue() {
		query.setStatus(new int[]{1});
		assertTrue(query.hasStatus());
	}
	public void testHasStatus_statusが空ならfalse() {
		query.setStatus(new int[]{});
		assertFalse(query.hasStatus());
	}
	public void testHasStatus_statusがnullならfalse() {
		query.setStatus(null);
		assertFalse(query.hasStatus());
	}

	/*
	 * Test method for 'net.yher.workstyle.TaskQuery.hasStatus(int)'
	 */
	public void testHasStatus_指定のstatusを持っていればtrue() {
		query.setStatus(new int[]{1});
		assertTrue(query.hasStatus(1));
	}
	public void testHasStatus_指定のstatusを持っていなければfalse() {
		query.setStatus(new int[]{2});
		assertFalse(query.hasStatus(1));

		query.setStatus(new int[]{});
		assertFalse(query.hasStatus(1));

		query.setStatus(null);
		assertFalse(query.hasStatus(1));
	}

	/*
	 * Test method for 'net.yher.workstyle.TaskQuery.hasTag()'
	 */
	public void testHasTag_tagを１つ以上持っていればtrue() {
		query.setTag(new String[]{"tag"});
		assertTrue(query.hasTag());
	}
	public void testHasTag_tagが空もしくはnullならfalse() {
		query.setTag(new String[]{});
		assertFalse(query.hasTag());

		query.setTag(null);
		assertFalse(query.hasTag());
	}

	/*
	 * Test method for 'net.yher.workstyle.TaskQuery.hasTag(String)'
	 */
	public void testHasTagString_指定のtagを持っていればtrue() {
		query.setTag(new String[]{"tag"});
		assertTrue(query.hasTag("tag"));
	}
	public void testHasTagString_指定のtagを持っていなければfalse() {
		query.setTag(new String[]{"hoge"});
		assertFalse(query.hasTag("tag"));
	}
	public void testHasTagString_tagが空もしくはnullならfalse() {
		query.setTag(new String[]{""});
		assertFalse(query.hasTag("tag"));

		query.setTag(null);
		assertFalse(query.hasTag("tag"));
	}
	public void testHasTagString_指定のtagが空もしくはnullならfalse() {
		query.setTag(new String[]{"hoge"});
		assertFalse(query.hasTag(""));
		assertFalse(query.hasTag(null));
	}

	/*
	 * Test method for 'net.yher.workstyle.TaskQuery.hasOrder(int)'
	 */
	public void testHasOrder_指定のorderを持っていればtrue() {
		query.setOrder(TaskQuery.ORDER_BY_UPDATE_DATE_ASC);
		assertTrue(query.hasOrder(TaskQuery.ORDER_BY_UPDATE_DATE_ASC));
	}
	public void testHasOrder_指定のorderを持っていなければfalse() {
		query.setOrder(TaskQuery.ORDER_BY_UPDATE_DATE_DESC);
		assertFalse(query.hasOrder(TaskQuery.ORDER_BY_UPDATE_DATE_ASC));
	}

	/*
	 * Test method for 'net.yher.workstyle.TaskQuery.getKeywordAsList()'
	 */
	public void testGetKeywordAsList_空白区切りの文字列をリストにして返す() {
		query.setKeyword("a b c d");
		List<String> list = query.getKeywordAsList();
		
		assertEquals(4, list.size());
		assertEquals("a,b,c,d", CollectionUtils.toCsv(list));
	}
	public void testGetKeywordAsList_空文字もしくはnullの場合は空のリストを返す() {
		query.setKeyword("");
		assertEquals(0, query.getKeywordAsList().size());

		query.setKeyword(null);
		assertEquals(0, query.getKeywordAsList().size());
	}
}
