/**
 * Copyright (c) 2006, yher.net
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice, 
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * * Neither the name of the nor the names of its contributors may be used to endorse or 
 *   promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 */
package net.yher.commons.lang;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * 複数リソースファイルをまとめて扱うResourceBundle.
 * メインのリソースファイルに、includeプロパティとして指定されたリソースファイルを全て
 * まとめて扱う。
 * @author hisaboh
 */
public class ExtendResourceBundle {
	private static final String PROP_INCLUDE = "include";
	private List<ResourceBundle> resourceBundleList;
	
	public static ExtendResourceBundle getBundle(String baseName) {
		return getBundle(baseName, Locale.getDefault());
	}
	public static ExtendResourceBundle getBundle(String baseName, Locale locale) {
		ExtendResourceBundle resource = new ExtendResourceBundle();
		resource.addResourceBundle(ResourceBundle.getBundle(baseName, locale));
		if (resource.getKeyList().contains(PROP_INCLUDE)) {
			String include[] = resource.getStringArray(PROP_INCLUDE);
			for (int i=0; i<include.length; i++) {
				resource.addResourceBundle(ResourceBundle.getBundle(include[i], locale));
			}
		}
		return resource;
	}
	/**
	 * コンストラクタ
	 */
	private ExtendResourceBundle() {
		resourceBundleList = new ArrayList<ResourceBundle>();
	}
	private void addResourceBundle(ResourceBundle resourceBundle) {
		this.resourceBundleList.add(resourceBundle);
	}
	
	private List<String> getKeyList() {
		ArrayList<String> keyList = new ArrayList<String>();
		Enumeration<String> e = null;
		for (ResourceBundle resource : resourceBundleList) {
			e = resource.getKeys();
			while (e.hasMoreElements()) {
				keyList.add(e.nextElement());
			}
		}
		return keyList;
	}
	
	public Iterator<String> getKeys() {
		return getKeyList().iterator();
	}
	public String getString(String key) {
		for (ResourceBundle resource : resourceBundleList) {
			try {
				return resource.getString(key);
			} catch (MissingResourceException e) {
			}
		}
		throw new MissingResourceException("Can't find resource", this.getClass().getName(), key);
	}
	public String[] getStringArray(String key) {
		String val = getString(key);
		if (key == null || key.length() == 0) return new String[0];
		String list[] = val.split(",");
		for (int i=0; i<list.length; i++)
			list[i] = list[i].trim();
		return list;
	}
	public float getFloat(String key) {
		return Float.parseFloat(getString(key));
	}
	public double getDouble(String key) {
		return Double.parseDouble(getString(key));
	}
	public int getInt(String key) {
		return Integer.parseInt(getString(key));
	}
	public int getInt(String key, int defaultValue) {
		try {
			return Integer.parseInt(getString(key));
		} catch (Exception e) {
			return defaultValue;
		}
	}
}
