# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001-2004 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2004, 2005 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import codecs
import os
import re
import sys
import urllib
import time

from xml.dom import pulldom

import ninix.home

if 'DISPLAY' in os.environ:
    import gtk
    import gobject
    import ninix.pix

# 注意:
# - このURLを本ゴーストマネージャクローン以外の用途で使う場合は,
#   「できるだけ事前に」AQRS氏に連絡をお願いします.
#   (「何かゴーストマネージャ」のページ: http://www.aqrs.jp/ngm/)
# - アクセスには帯域を多く使用しますので,
#   必ず日時指定の差分アクセスをし余分な負荷をかけないようお願いします.
#   (差分アクセスの方法については本プログラムの実装が参考になると思います.)
MASTERLIST_URL = 'http://www.aqrs.jp/cgi-bin/ghostdb/request2.cgi'

# 10000以上のIDを持つデータは仮登録
ID_LIMIT = 10000

ELEMENTS = ['Name', 'SakuraName', 'KeroName', 'GhostType',
            'HPUrl', 'HPTitle', 'Author', 'PublicUrl', 'ArchiveUrl',
            'ArchiveSize', 'ArchiveTime', 'Version', 'AIName',
            'SurfaceSetFlg', 'KisekaeFlg', 'AliasName',
            'SakuraSurfaceList', 'KeroSurfaceList', 'DeleteFlg',
            'NetworkUpdateTime', 'AcceptName', 'NetworkUpdateFlg',
            'SakuraPreviewMD5', 'KeroPreviewMD5', 'ArchiveMD5',
            'InstallDir', 'ArchiveName', 'UpdateUrl',
            'AnalysisError', 'InstallCount']


class Catalog_xml:

    #public methods
    def __init__(self, datadir):
        self.data = {}
        self.url = {}
        self.cgi = {}
        self.datadir = datadir
        if not os.path.exists(self.datadir):
            os.makedirs(self.datadir)
        self.last_update = '1970-01-01 00:00:00'
        self.load_MasterList()

    # data handling functions
    def get(self, entry, key):
        if key in entry:
            return entry[key]
        else:
            return None

    # updates etc
    def network_update(self):
        last_update = self.last_update
        self.last_update = time.strftime('%Y-%m-%d %H:%M:%S')
        if self.cgi:
            priority = self.cgi.keys()
            priority.sort()
            url = self.cgi[priority[-1]]
        else:
            url = MASTERLIST_URL
        try:
            f = urllib.urlopen(url, urllib.urlencode(
                {'time': '"%s"' % last_update, 'charset': 'EUC-JP'}))
        except:
            return ## FIXME
        self.import_from_fileobj(f)
        self.save_MasterList()
        f.close()

    # private methods
    def load_MasterList(self):
        try:
            f = open(os.path.join(self.datadir, 'MasterList.xml'), 'r')
            self.import_from_fileobj(f)
            f.close()
        except IOError:
            return

    def save_MasterList(self):
        f = open(os.path.join(self.datadir, 'MasterList.xml'), 'w')
        self.export_to_fileobj(f)
        f.close()

    def get_encoding(self, line):
        m = re.compile('<\?xml version="1.0" encoding="(.+)" \?>').search(line)
        return m.group(1)

    def create_entry(self, node):
        entry = {}
        for key, text in node:
            assert key in ELEMENTS
            entry[key] = text
        return entry

    def import_from_fileobj(self, fileobj):
        line0 = fileobj.readline()
        encoding = self.get_encoding(line0)
        try:
            codecs.lookup(encoding)
        except:
            sys.stderr.write('Unsupported encoding %s' % repr(encoding))
            sys.exit(-1)
        nest = 0
        new_entry = {}
        set_id = None
        node = []
        for line in fileobj:
            assert nest >= 0
            line = unicode(line, encoding, 'ignore').encode('utf-8')
            if not line:
                continue
            m = re.compile('<GhostList>').search(line)
            if m:
                nest += 1
                continue
            m = re.compile('<FileSet ID="?([0-9]+)"?>').search(line)
            if m:
                nest += 1
                set_id = int(m.group(1))
                continue
            m = re.compile('</FileSet>').search(line)
            if m:
                nest -= 1
                new_entry[set_id] = self.create_entry(node)
                node = []
                continue
            m = re.compile('<RequestCgiUrl Priority="?([0-9]+)"?>(.+)</RequestCgiUrl>').search(line)
            if m:
                g = m.groups()
                priority = int(g[0])
                url = g[1]
                if priority in self.cgi:
                    self.cgi[priority].append(url)
                else:
                    self.cgi[priority] = [url]
                continue
            m = re.compile('<(.+)>(.+)</(.+)>').search(line)
            if m:
                g = m.groups()
                if set_id is not None:
                    key = g[0]
                    text = g[1]
                    text = text.replace('&apos;', '\'')                
                    text = text.replace('&quot;', '"')
                    text = text.replace('&gt;', '>')
                    text = text.replace('&lt;', '<')
                    text = text.replace('&amp;', '&')
                    node.append([key, text])
                else:
                    key = g[0]
                    text = g[1]
                    assert key in ['LastUpdate', 'NGMVersion',
                                   'SakuraPreviewBaseUrl',
                                   'KeroPreviewBaseUrl',
                                   'ArcMD5BaseUrl', 'NGMUpdateBaseUrl']
                    if key == 'LastUpdate':
                        self.last_update = text
                    elif key == 'NGMVersion':
                        version = float(text)
                        if version < 0.51:
                            return
                        else:
                            self.version = version
                    elif key in ['SakuraPreviewBaseUrl', 'KeroPreviewBaseUrl',
                                  'ArcMD5BaseUrl', 'NGMUpdateBaseUrl']:
                        self.url[key] = text
        self.data.update(new_entry)        

    def dump_entry(self, entry, fileobj):
        for key in ELEMENTS:
            if key in entry and entry[key] is not None:
                text = entry[key]
                text = text.replace('&', '&amp;')
                text = text.replace('<', '&lt;')
                text = text.replace('>', '&gt;')
                text = text.replace('"', '&quot;')
                text = text.replace('\'', '&apos;')
                text = text.encode('EUC-JP')
            else:
                text = '' # fileobj.write('  <%s/>\n' % key)
            fileobj.write('  <%s>%s</%s>\n' % (key, text, key))

    def export_to_fileobj(self, fileobj):
        fileobj.write('<?xml version="1.0" encoding="EUC-JP" ?>\n')
        fileobj.write('<GhostList>\n')
        fileobj.write('<LastUpdate>%s</LastUpdate>\n' % self.last_update)
        for key in ['SakuraPreviewBaseUrl', 'KeroPreviewBaseUrl',
                    'ArcMD5BaseUrl', 'NGMUpdateBaseUrl']:
            if key in self.url:
                fileobj.write('<%s>%s</%s>\n' % (key, self.url[key], key))
            else:
                fileobj.write('<%s></%s>\n' % (key, key)) # fileobj.write('<%s/>\n' % key)
        fileobj.write('<NGMVersion>%s</NGMVersion>\n' % self.version)
        key_list = self.cgi.keys()
        key_list.sort()
        key_list.reverse()
        for priority in key_list:
            for url in self.cgi[priority]:
                fileobj.write(
                    '<RequestCgiUrl Priority="%d">%s</RequestCgiUrl>\n' % \
                    (priority, url))
        ids = self.data.keys()
        ids.sort()
        for set_id in ids:
            fileobj.write('<FileSet ID="%s">\n' % set_id)
            entry = self.data[set_id]
            self.dump_entry(entry, fileobj)
            fileobj.write('</FileSet>\n')
        fileobj.write('</GhostList>\n')

class Catalog(Catalog_xml):

    TYPE = ['Sakura', 'Kero']

    def image_filename(self, set_id, side):
        p = '%s_%d.png' % (self.TYPE[side], set_id)
        d = os.path.join(self.datadir, p)
        if os.path.exists(d):
            return d
        return None

    def retrieve_image(self, set_id, side):
        p = '%s_%d.png' % (self.TYPE[side], set_id)
        d = os.path.join(self.datadir, p)
        if not os.path.exists(d):
            if side == 0:
                url = self.url['SakuraPreviewBaseUrl']
            else:
                url = self.url['KeroPreviewBaseUrl']
            try:
                urllib.urlretrieve(''.join((url, p)), d)
            except:
                return ## FIXME

class InstallationManager(list):

    def __init__(self, ngm):
        self.master = ngm

    def set_current_ghost(self, set_id):
        self.current_ghost = set_id

    def install_ghost(self, dialog):
        a = self.master.ui.catalog.get(self.current_ghost, 'ArchiveUrl')
        os.system("%s '%s'" % (self.master.ninix_install_cmd, a))

class SearchDialog:

    def __init__(self, ui):
        self.ui = ui
        self.dialog = gtk.Dialog()
        self.dialog.connect('delete_event', self.cancel)
        self.dialog.set_modal(True)
        self.dialog.set_position(gtk.WIN_POS_CENTER)
        label = gtk.Label(_('Search for'))
        self.dialog.vbox.pack_start(label)
        self.pattern_entry = gtk.Entry()
        self.pattern_entry.set_size_request(300, -1)
        self.dialog.vbox.pack_start(self.pattern_entry)
        self.dialog.vbox.show_all()
        self.ok_button = gtk.Button(_('OK'))
        self.ok_button.connect('clicked', self.ok)
        self.dialog.action_area.pack_start(self.ok_button)
        button = gtk.Button(_('Cancel'))
        button.connect('clicked', self.cancel)
        self.dialog.action_area.pack_start(button)
        self.dialog.action_area.show_all()

    def set_pattern(self, text):
        self.pattern_entry.set_text(text)
                
    def get_pattern(self):
        return self.pattern_entry.get_text()
    
    def hide(self):
        self.dialog.hide()

    def show(self, default=None):
        if default:
            self.set_pattern(default)
        self.dialog.show()

    def ok(self, widget, event=None):
        word = unicode(self.get_pattern(), 'utf-8')
        self.ui.search(word)
        self.hide()
        return True

    def cancel(self, widget, event=None):
        self.hide()
        return True

class UI:
    def __init__(self, ngm):
        self.ngm = ngm
        self.ui_info = '''
        <ui>
          <menubar name='MenuBar'>
            <menu action='FileMenu'>
              <menuitem action='Search'/>
              <menuitem action='Search Forward'/>
              <separator/>
              <menuitem action='Settings'/>
              <separator/>
              <menuitem action='DB Network Update'/>
              <separator/>
              <menuitem action='Close'/>
            </menu>
            <menu action='ViewMenu'>
              <menuitem action='Mask'/>
              <menuitem action='Reset to Default'/>
              <menuitem action='Show All'/>
            </menu>
            <menu action='ArchiveMenu'>
            </menu>
            <menu action='HelpMenu'>
            </menu>
          </menubar>
        </ui>'''
        self.entries = (
            ( 'FileMenu', None, _('_File') ),
            ( 'ViewMenu', None, _('_View') ),
            ( 'ArchiveMenu', None, _('_Archive') ),
            ( 'HelpMenu', None, _('_Help') ),
            ( 'Search', None,                  # name, stock id
              _('Search(_F)'), '<control>F',   # label, accelerator
              'Search',                        # tooltip
              lambda *a: self.open_search_dialog() ),
            ( 'Search Forward', None,
              _('Search Forward(_S)'), 'F3',
              None,
              lambda *a: self.search_forward() ),
            ( 'Settings', None,
              _('Settings(_O)'), None,
              None,
              lambda *a: self.ngm.open_preference_dialog() ),
            ( 'DB Network Update', None,
              _('DB Network Update(_N)'), None,
              None,
              lambda *a: self.network_update() ),
            ( 'Close', None,
              _('Close(_X)'), None,
              None,
              lambda *a: self.close() ),
            ( 'Mask', None,
              _('Mask(_M)'), None,
              None,
              lambda *a: self.ngm.open_mask_dialog() ),
            ( 'Reset to Default', None,
              _('Reset to Default(_Y)'), None,
              None,
              lambda *a: self.ngm.reset_to_default() ),
            ( 'Show All', None,
              _('Show All(_Z)'), None,
              None,
              lambda *a: self.ngm.show_all() ),
            )
        self.opened = 0
        self.textview = [None, None]
        self.darea = [None, None]
        self.info = None
        self.button = {}
        self.url = {}
        self.search_word = ''
        self.search_dialog = SearchDialog(self)
        self.create_dialog()

    def create_dialog(self):
        self.window = gtk.Window()
        self.window.set_title(_('Ghost Manager'))
        self.window.set_resizable(False)
        self.window.connect('delete_event', lambda *a: self.close())
        self.window.set_position(gtk.WIN_POS_CENTER)
        self.window.set_gravity(gtk.gdk.GRAVITY_CENTER)
        actions = gtk.ActionGroup('Actions')
        actions.add_actions(self.entries)
        ui = gtk.UIManager()
        ui.insert_action_group(actions, 0)
        self.window.add_accel_group(ui.get_accel_group())
        try:
            mergeid = ui.add_ui_from_string(self.ui_info)
        except gobject.GError, msg:
            print 'building menus failed: %s' % msg
        vbox = gtk.VBox(False, 0)
        self.window.add(vbox)
        vbox.show()
        vbox.pack_start(ui.get_widget('/MenuBar'), False, False, 0)
        separator = gtk.HSeparator()
        vbox.pack_start(separator, False, True, 0)
        separator.show()
        hbox = gtk.HBox(False, 0)
        vbox.pack_start(hbox, False, True, 10)
        hbox.show()
        self.surface_area_sakura = self.create_surface_area(0)
        hbox.pack_start(self.surface_area_sakura, False, True, 10)
        self.surface_area_kero = self.create_surface_area(1)
        hbox.pack_start(self.surface_area_kero, False, True, 10)
        self.info_area = self.create_info_area()
        hbox.pack_start(self.info_area, False, True, 10)
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_SPREAD)
        vbox.pack_start(box, False, True, 4)
        box.show()
        button = gtk.Button(_('Previous'))
        button.connect('clicked', lambda b, w=self: w.show_previous())
        box.add(button)
        button.show()
        self.button['previous'] = button
        button = gtk.Button(_('Next'))
        button.connect('clicked', lambda b, w=self: w.show_next())
        box.add(button)
        button.show()
        self.button['next'] = button
        self.statusbar = gtk.Statusbar()
        vbox.pack_start(self.statusbar, False, True, 0)
        self.statusbar.show()

    def network_update(self):
        self.ngm.network_update()
        self.update()

    def open_search_dialog(self):
        self.search_dialog.show(default=self.search_word)

    def search(self, word):
        if word:
            self.search_word = word
            if self.ngm.search(word):
                self.update()
            else:
                pass ## FIXME

    def search_forward(self):
        if self.search_word:
            if self.ngm.search_forward(self.search_word):
                self.update()
            else:
                pass ## FIXME

    def show_next(self):
        self.ngm.next()
        self.update()

    def show_previous(self):
        self.ngm.previous()
        self.update()

    def create_surface_area(self, side):
        assert side in [0, 1]
        vbox = gtk.VBox(False, 0)
        vbox.show()
        textview = gtk.TextView()
        textview.set_editable(False)
        textview.set_size_request(128, 16)
        vbox.pack_start(textview, False, True, 0)
        textview.show()
        self.textview[side] = textview
        darea = gtk.DrawingArea()
        vbox.pack_start(darea, False, True, 0)
        darea.show()
        self.darea[side] = darea
        return vbox

    def update_surface_area(self):
        for side in [0, 1]:
            if side == 0:
                name = self.ngm.get('SakuraName')
            else:
                name = self.ngm.get('KeroName')
            textbuffer = self.textview[side].get_buffer()
            textbuffer.set_text(name)
            filename = self.ngm.get_image_filename(side)
            darea = self.darea[side]
            darea.realize()
            if filename is not None:
                try:
                    pixbuf = ninix.pix.create_pixbuf_from_file(filename)
                    image, mask = pixbuf.render_pixmap_and_mask(255)
                except:
                    darea.window.set_back_pixmap(None, False)
                else:
                    w, h = image.get_size()
                    darea.set_size_request(w, h)
                    darea.window.set_back_pixmap(image, False)
                    darea.window.shape_combine_mask(mask, 0, 0)
            else:                
                darea.window.set_back_pixmap(None, False)
            darea.queue_draw()

    def create_info_area(self):
        vbox = gtk.VBox(False, 0)
        vbox.show()
        hbox = gtk.HBox(False, 0)
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_SPREAD)
        box.show()
        button = gtk.Button(_('Install'))
        button.connect('clicked', lambda b, w=self: w.ngm.install_current())
        box.add(button)
        button.show()
        self.button['install'] = button
        button = gtk.Button(_('Update'))
        button.connect('clicked', lambda b, w=self: w.ngm.update_current())
        box.add(button)
        button.show()
        self.button['update'] = button
        hbox.pack_start(box, True, True, 10)
        vbox2 = gtk.VBox(False, 0)
        hbox.pack_start(vbox2, False, True, 0)
        vbox2.show()
        button = gtk.Button('') # with GtkLabel
        button.set_relief(gtk.RELIEF_NONE)
        self.url['HP'] = [None, button.get_child()]
        vbox2.pack_start(button, False, True, 0)
        button.connect('clicked',
                       lambda b, w=self: w.launch_browser(self.url['HP'][0]))
        button.show()
        button = gtk.Button('')
        button.set_relief(gtk.RELIEF_NONE)
        button.set_use_underline(True)
        self.url['Public'] = [None, button.get_child()]
        vbox2.pack_start(button, False, True, 0)
        button.connect(
            'clicked',
            lambda b, w=self: w.launch_browser(self.url['Public'][0]))
        button.show()
        vbox.pack_start(hbox, False, True, 0)
        hbox.show()
        textview = gtk.TextView()
        textview.set_editable(False)
        textview.set_size_request(256, 144)
        vbox.pack_start(textview, False, True, 0)
        textview.show()
        self.info = textview
        return vbox

    def launch_browser(self, url):
        command = self.ngm.app.get_browser()
        if '%s' not in command:
            sys.stderr.write('cannot execute command (%s missing)\n')
            return
        os.system(' '.join((command.replace('%s', url, 1), '&')))

    def update_info_area(self):
        info_list = [(_('Author:'), 'Author'),
                     (_('ArchiveTime:'), 'ArchiveTime'),
                     (_('ArchiveSize:'), 'ArchiveSize'),
                     (_('NetworkUpdateTime:'), 'NetworkUpdateTime'),
                     (_('Version:'), 'Version'),
                     (_('AIName:'), 'AIName')]
        text = ''
        text = ''.join((text, self.ngm.get('Name'), '\n'))
        for item in info_list:
            text = ''.join((text, item[0], self.ngm.get(item[1]), '\n'))
        text = ''.join((text,
                        self.ngm.get('SakuraName'),
                        _('SurfaceList:'), self.ngm.get('SakuraSurfaceList'),
                        '\n'))
        text = ''.join((text,
                        self.ngm.get('KeroName'),
                        _('SurfaceList:'), self.ngm.get('KeroSurfaceList'),
                        '\n'))
        textbuffer = self.info.get_buffer()
        textbuffer.set_text(text)
        url = self.ngm.get('HPUrl')
        text = self.ngm.get('HPTitle')
        self.url['HP'][0] = url
        label = self.url['HP'][1]
        label.set_markup('<span foreground="blue">%s</span>' % text)
        url = self.ngm.get('PublicUrl')
        text = ''.join((self.ngm.get('Name'), _(' Web Page')))
        self.url['Public'][0] = url
        label = self.url['Public'][1]
        label.set_markup('<span foreground="blue">%s</span>' % text)
        
        target_dir = os.path.join(self.ngm.home_dir, 'ghost',
                                  self.ngm.get('InstallDir'))
        if not os.path.isdir(target_dir) and \
           self.ngm.get('ArchiveUrl') != 'No data':
            self.button['install'].set_sensitive(True)
        else:
            self.button['install'].set_sensitive(False)
        if os.path.isdir(target_dir) and \
           self.ngm.get('GhostType') == 'ゴースト' and \
           self.ngm.get('UpdateUrl') != 'No data':
            self.button['update'].set_sensitive(True)
        else:
            self.button['update'].set_sensitive(False)

    def update(self):
        self.update_surface_area()
        self.update_info_area()
        if self.ngm.exist_next():
            self.button['next'].set_sensitive(True)
        else:
            self.button['next'].set_sensitive(False)
        if self.ngm.exist_previous():
            self.button['previous'].set_sensitive(True)
        else:
            self.button['previous'].set_sensitive(False)

    def show(self):
        if self.opened:
            return
        self.update()
        self.window.show()
        self.opened = 1

    def close(self):
        self.window.hide()
        self.opened = 0

class NGM:

    def __init__(self, app):
        self.current = 0
        self.app = app
        self.opened = 0
        self.home_dir = ninix.home.get_ninix_home()
        self.catalog = Catalog(os.path.join(self.home_dir, 'ngm/data'))
        self.ui = UI(self)

    def get(self, element):
        if self.current in self.catalog.data:
            entry = self.catalog.data[self.current]
            text = self.catalog.get(entry, element)
            if text is not None:
                return text
        return 'No data'

    def get_image_filename(self, side):
        return self.catalog.image_filename(self.current, side)

    def search(self, word, set_id=0):
        while set_id < ID_LIMIT:
            if set_id in self.catalog.data:
                entry = self.catalog.data[set_id]
                for element in ['Name', 'SakuraName', 'KeroName',
                                'Author', 'HPTitle']:
                    text = self.catalog.get(entry, element)
                    if text and word in text:
                        self.current = set_id
                        return True
            set_id += 1
        return False

    def search_forward(self, word):
        return self.search(word, set_id=self.current + 1)

    def open_preference_dialog(self):
        pass

    def network_update(self):
        self.catalog.network_update()
        for set_id in self.catalog.data:
            for side in [0, 1]:
                self.catalog.retrieve_image(set_id, side)

    def open_mask_dialog(self):
        pass

    def reset_to_default(self):
        pass

    def show_all(self):
        pass

    def next(self):
        next = self.current + 1
        if next < ID_LIMIT and next in self.catalog.data:
            self.current = next

    def exist_next(self):
        next = self.current + 1
        if next < ID_LIMIT and next in self.catalog.data:
            return True
        else:
            return False

    def previous(self):
        previous = self.current - 1
        assert previous < ID_LIMIT
        if previous in self.catalog.data:
            self.current = previous

    def exist_previous(self):
        previous = self.current - 1
        assert previous < ID_LIMIT
        if previous in self.catalog.data:
            return True
        else:
            return False

    def show_dialog(self):
        self.ui.show()

    def install_current(self): ## FIXME
        pass

    def update_current(self): ## FIXME
        item = self.app.find_ghost_by_name(self.get('Name'), 0)
        ghost = self.app.select_ghost_from_list(item)
        if not ghost.is_running():
            self.app.start_sakura(item, init=1)
        ghost.update()


if __name__ == '__main__':
    pass
