#!/usr/bin/env python
# -*- coding: ascii -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2006 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003-2005 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#
# $Id: main.py,v 1.135 2006/03/25 08:07:09 shy Exp $
#

import getopt
import os
import signal
import socket
import sys
import random
import gettext
import traceback
import cStringIO
import fcntl

gettext.install('ninix')

if 'DISPLAY' in os.environ:
    if 'gtk' not in sys.modules:
        try:
            import pygtk
            pygtk.require('2.0')
        except ImportError:
            pass
    import gtk
    import gobject
    import pango
    import ninix.pix

import ninix.home
import ninix.prefs
import ninix.sakura
import ninix.ghost
import ninix.surface
import ninix.balloon
import ninix.sstp
import ninix.dll
import ninix.communicate
import ninix.ngm

import ninix.nekodorif
import ninix.kinoko

USAGE = '''\
Usage: ninix [options]
Options:
  -H, --homedir DIR   ninix home directory (default: ~/.ninix)
  -U, --userdir DIR   user data directory (default: ninix home directory)
  --sstp-port NUM     additional port for listening SSTP requests
  -h, --help          show this message
'''

def usage():
    sys.stderr.write(USAGE)
    sys.exit(1)

def _traceback(exception_type, value, tb):
    response_id = 1
    dialog = gtk.MessageDialog(None, 0, gtk.MESSAGE_ERROR, gtk.BUTTONS_NONE,
                               _('A ninix-aya error has been detected.'))
    dialog.set_title(_('Bug Detected'))
    dialog.set_position(gtk.WIN_POS_CENTER)
    dialog.set_gravity(gtk.gdk.GRAVITY_CENTER)
    button = dialog.add_button(_('Show Details'), response_id)
    dialog.add_button(gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE)
    textview = gtk.TextView()
    textview.set_editable(False)
    width = gtk.gdk.screen_width() / 2
    height = gtk.gdk.screen_height() / 4
    textview.set_size_request(width, height)
    textview.show()
    sw = gtk.ScrolledWindow()
    sw.show()
    sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.add(textview)
    frame = gtk.Frame()
    frame.set_shadow_type(gtk.SHADOW_IN)
    frame.add(sw)
    frame.set_border_width(7)
    dialog.vbox.add(frame)
    stringio = cStringIO.StringIO()
    traceback.print_exception(exception_type, value, tb, None, stringio)
    textbuffer = textview.get_buffer()
    textbuffer.set_text(stringio.getvalue())
    while 1:
        if dialog.run() == response_id:
            frame.show()
            button.set_sensitive(0)
        else: # close button
            break
    dialog.destroy()
    sys.exit(1)

if 'DISPLAY' in os.environ:
    if 'gtk' in sys.modules:
        sys.excepthook = _traceback

def main():
    if gtk.pygtk_version < (2,8,0):
        print "PyGtk 2.8.0 or later required"
        raise SystemExit
    # reopen stderr to avoid IOError
    sys.stderr = os.fdopen(sys.stderr.fileno(), 'w')
    # check if X is running
    if 'DISPLAY' not in os.environ:
        sys.stderr.write('Error: cannot open display (abort)\n')
        sys.exit(1)
    # parse command line arguments
    try:
        options, rest = getopt.getopt(sys.argv[1:], 'H:U:R:ph',
            ['homedir=', 'userdir=', 'sstp-port=', 'debug=', 'help'])
    except getopt.error, e:
        sys.stderr.write('Error: %s\n' % str(e))
        usage()
    if rest:
        usage()
    home_dir = os.environ.get('NINIX_HOME')
    user_dir = os.environ.get('NINIX_USER')
    sstp_port = [9801, 11000]
    debug = 0
    # check environment variables
    val = os.environ.get('NINIX_SSTP_PORT')
    if val:
        port = []
        for num in val.split(','):
            try:
                num = int(num)
                if num < 1024:
                    raise ValueError
                port.append(num)
            except ValueError:
                sys.stderr.write('Invalid NINIX_SSTP_PORT number (ignored)\n')
                break
        else:
            sstp_port.extend(port)
    # parse command line options
    for opt, val in options:
        if opt in ['-H', '--homedir']:
            home_dir = val
        elif opt in ['-U', '--userdir']:
            user_dir = val
        elif opt == '--sstp-port':
            num = int(val)
            if num < 1024:
                sys.stderr.write('Invalid --sstp-port number (ignored)\n')
            else:
                sstp_port.append(num)
        elif opt == '--debug':
            debug = int(val)
        else:
            usage()
    if home_dir:
        if not os.path.isabs(os.path.expanduser(home_dir)):
            home_dir = os.path.join(os.getcwd(), home_dir)
        ninix.home.NINIX_HOME = home_dir
    if user_dir:
        if not os.path.isabs(os.path.expanduser(user_dir)):
            user_dir = os.path.join(os.getcwd(), user_dir)
        ninix.home.NINIX_USER = user_dir
    config = ninix.home.load_config()
    if config is None:
        sys.stderr.write('Home directory not found (abort)\n')
        sys.exit(1)
    # aquire Inter Process Mutex (not Global Mutex)
    lockfile = open(os.path.join(ninix.home.get_ninix_home(), '.lock'), 'w')
    try:
        fcntl.flock(lockfile, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except:
        print 'ninix-aya is already running'
        sys.exit(1)
    # check if at least one set of ghost, surface, and balloon
    ghosts, shells, balloons, plugins, nekoninni, katochan, kinoko = config ## FIXME
    if not ghosts:
        sys.stderr.write('Error: no ghost found\n')
        sys.exit(1)
    elif not shells and \
         not filter(lambda ghost: ghost[3], ghosts):
        sys.stderr.write('Error: no shell found\n')
        sys.exit(1)
    elif not balloons:
        sys.stderr.write('Error: no balloon found\n')
        sys.exit(1)
    for desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name in ghosts:
        print 'ghost:', desc.get('name', '<none>').encode('utf-8', 'ignore')
        for name, surface_dir, surface_desc, surface_alias, surface in \
                surface_set:
            if name:
                print '\tsurface:', name.encode('utf-8', 'ignore'),
            if surface_alias is not None:
                print '(w/alias)',
            print
        if balloon:
            print '\tballoon:', \
                  balloon[0].get('name', '<none>').encode('utf-8', 'ignore')
    for shell_name, surface_set, balloon in shells:
        print 'shell:', (shell_name or '<none>').encode('utf-8', 'ignore')
        for name, surface_dir, surface_desc, surface_alias, surface in \
                surface_set:
            if name:
                print '\tsurface:', name.encode('utf-8', 'ignore'),
            if surface_alias is not None:
                print '(w/alias)',
            print
        if balloon:
            print '\tballoon:', \
                  balloon[0].get('name', '<none>').encode('utf-8', 'ignore')
    for desc, balloon in balloons:
        print 'balloon:', desc.get('name', '<none>').encode('utf-8', 'ignore')
    for plugin_name, plugin_dir, startup, menu_items in plugins:
        print 'plugin:', plugin_name.encode('utf-8', 'ignore')
    # read gtkrc
    gtk.rc_parse(ninix.home.get_gtkrc())
    # start
    sys.stdout.write('loading...')
    sys.stdout.flush()
    app = Application(config, sstp_port, debug)
    sys.stdout.write('done.\n')
    app.run()
    try:
        fcntl.flock(lockfile, fcntl.LOCK_UN)
    except:
        pass
    lockfile.close()

def get_default_surface_scale():
    return ninix.surface.range_scale[0][1]

def get_default_script_speed():
    i = len(ninix.sakura.range_script_speed) / 2
    return ninix.sakura.range_script_speed[i][1]


class Application:

    PREFS_SAKURA_NAME       = 'sakura_name'
    PREFS_SAKURA_SURFACE    = 'sakura_surface'
    PREFS_DEFAULT_BALLOON   = 'default_balloon'
    PREFS_IGNORE_DEFAULT    = 'ignore_default'
    PREFS_SCRIPT_SPEED      = 'script_speed'
    PREFS_SURFACE_SCALE     = 'surface_scale'
    PREFS_BALLOON_SCALLING  = 'balloon_scalling'
    PREFS_EVENT_KILL_LIST   = 'event_kill_list'
    PREFS_MOUSE_BUTTON1     = 'mouse_button1'
    PREFS_MOUSE_BUTTON3     = 'mouse_button3'
    PREFS_BROWSER           = 'browser'
    PREFS_HELPER_PATTERN    = 'helper_%d_pattern'
    PREFS_HELPER_COMMAND    = 'helper_%d_command'
    PREFS_TOP_MARGIN        = 'top_margin'
    PREFS_BOTTOM_MARGIN     = 'bottom_margin'
    PREFS_BALLOON_FONTS     = 'balloon_fonts'
    PREFS_ALLOWEMBRYO       = 'allowembryo'
    PREFS_CHECK_COLLISION   = 'check_collision'
    PREFS_USE_PNA           = 'use_pna'
    PREFS_SINK_AFTER_TALK   = 'sink_after_talk'
    PREFS_SURFACE_ALPHA     = 'surface_alpha'
    PREFS_BALLOON_ALPHA     = 'balloon_alpha'
    PREFS_RAISE_BEFORE_TALK = 'raise_before_talk'

    def __init__(self, config, sstp_port=[9801, 11000], debug=0):
        self.ghosts, self.shells, self.balloons, self.plugins, \
                     self.nekoninni, self.katochan, self.kinoko = config
        self.sstp_port = sstp_port
        self.debug = debug
        self.plugin_pids = []
        self.sstp_servers = []
        self.__sstp_queue = []
        self.__sstp_flag = 0
        self.__current_sender = None
        # create preference dialog
        self.pref_dialog = PreferenceDialog(self)
        # create usage dialog
        self.usage_dialog = UsageDialog()
        self.__communicate = ninix.communicate.Communicate()
        # create ghost manager
        self.__ngm = ninix.ngm.NGM(self)

    def enqueue_script_if_ghost(self, if_ghost, script, sender, handle,
                                address, show_sstp_marker,
                                use_translator, entry_db):
        self.__sstp_queue.append(
            (if_ghost, script, sender, handle, address, show_sstp_marker,
             use_translator, entry_db))

    def set_sstp_flag(self, sender):
        self.__sstp_flag = 1
        self.__current_sender = sender

    def reset_sstp_flag(self):
        self.__sstp_flag = 0
        self.__current_sender = None        

    def handle_sstp_queue(self):
        if self.__sstp_flag or not self.__sstp_queue:
            return
        if_ghost = self.__sstp_queue[0][0]
        ghost = self.get_ghost(if_ghost)
        if ghost is not None:
            self.select_current_sakura(if_ghost)
        elif not self.get_allowembryo(): # Refuse
            self.__sstp_queue.pop(0)
            return
        if_ghost, script, sender, handle, address, show_sstp_marker, \
                  use_translator, entry_db = self.__sstp_queue.pop(0)
        self.set_sstp_flag(sender)
        ghost = self.ghost_list[self.current_sakura[1]]['instance']
        ghost.enter_temp_mode()
        sakura = self.get_current_sakura()
        sakura.enqueue_script(
            script, sender, handle,
            address, show_sstp_marker, use_translator, entry_db)

    def receive_sstp_request(self):
        try:
            for sstp_server in self.sstp_servers:
                sstp_server.handle_request()
        except socket.error, (code, message):
            sys.stderr.write('socket.error: %s (%d)\n' % (message, code))
        except ValueError: # may happen when ninix is terminated
            return

    def create_ghost(self, data, debug):
        default_path = os.path.join(os.environ['PYTHONPATH'], 'ninix/dll')
        ghost = ninix.ghost.Ghost(self, data, default_path,
                                  self.__communicate, debug)
        ghost.set_top_margin(self.get_top_margin())
        ghost.set_bottom_margin(self.get_bottom_margin())
        ghost.set_balloon_fonts(self.get_balloon_fonts())
        ghost.set_mouse_button1(self.get_mouse_button1())
        ghost.set_mouse_button3(self.get_mouse_button3())
        ghost.set_event_kill_list(self.get_event_kill_list())
        ghost.set_browser(self.get_browser())
        ghost.set_helpers(self.get_helpers())
        if self.get_check_collision(): ## FIXME
            ghost.set_debug(self.debug | 4096)
        else:
            ghost.set_debug((self.debug & 4096) ^ self.debug)
        return ghost

    def get_working_ghost(self, cantalk=0):
        working_list = [item['instance'] for item in self.ghost_list \
                        if item is not None and \
                           item['instance'] is not None and \
                           item['instance'].is_running()]
        if cantalk:
            working_list = [ghost for ghost in working_list if ghost.cantalk()]
        return working_list

    def set_allowembryo(self, flag):
        self.prefs[self.PREFS_ALLOWEMBRYO] = str(flag)

    def get_allowembryo(self):
        return self.prefs.getint(self.PREFS_ALLOWEMBRYO)

    def set_balloon_scalling(self, flag):
        self.prefs[self.PREFS_BALLOON_SCALLING] = str(flag)

    def get_balloon_scalling(self):
        return self.prefs.getint(self.PREFS_BALLOON_SCALLING)

    def set_surface_scale(self, scale):
        self.prefs[self.PREFS_SURFACE_SCALE] = str(scale)

    def get_surface_scale(self):
        return self.prefs.getint(self.PREFS_SURFACE_SCALE,
                                 get_default_surface_scale())

    def set_script_speed(self, speed):
        self.prefs[self.PREFS_SCRIPT_SPEED] = str(speed)

    def get_script_speed(self):
        return self.prefs.getint(self.PREFS_SCRIPT_SPEED,
                                 get_default_script_speed())

    def set_top_margin(self, margin):
        self.prefs[self.PREFS_TOP_MARGIN] = str(margin)
        for ghost in self.get_working_ghost():
            ghost.set_top_margin(margin)

    def get_top_margin(self):
        return self.prefs.getint(self.PREFS_TOP_MARGIN, 0)

    def set_bottom_margin(self, margin):
        self.prefs[self.PREFS_BOTTOM_MARGIN] = str(margin)
        for ghost in self.get_working_ghost():
            ghost.set_bottom_margin(margin)

    def get_bottom_margin(self):
        return self.prefs.getint(self.PREFS_BOTTOM_MARGIN, 0)

    def set_default_balloon(self, name):
        if self.find_balloon_by_name(name) is not None:
            self.prefs[self.PREFS_DEFAULT_BALLOON] = name

    def get_default_balloon(self):
        return self.prefs.get(self.PREFS_DEFAULT_BALLOON)

    def set_ignore_default(self, flag):
        self.prefs[self.PREFS_IGNORE_DEFAULT] = flag

    def get_ignore_default(self):
        return self.prefs.getint(self.PREFS_IGNORE_DEFAULT, 0)

    def set_balloon_fonts(self, fonts):
        self.prefs[self.PREFS_BALLOON_FONTS] = fonts
        for ghost in self.get_working_ghost():
            ghost.set_balloon_fonts(fonts)

    def get_balloon_fonts(self):
        return self.prefs.get(self.PREFS_BALLOON_FONTS, 'Sans')

    def set_mouse_button1(self, button1):
        self.prefs[self.PREFS_MOUSE_BUTTON1] = button1
        for ghost in self.get_working_ghost():
            ghost.set_mouse_button1(button1)

    def get_mouse_button1(self):
        return self.prefs.get(self.PREFS_MOUSE_BUTTON1,
                              ninix.sakura.BUTTON1_RAISE)

    def set_mouse_button3(self, button3):
        self.prefs[self.PREFS_MOUSE_BUTTON3] = button3
        for ghost in self.get_working_ghost():
            ghost.set_mouse_button3(button3)

    def get_mouse_button3(self):
        return self.prefs.get(self.PREFS_MOUSE_BUTTON3,
                              ninix.sakura.BUTTON3_CLOSE)

    def set_event_kill_list(self, event_list):
        self.prefs[self.PREFS_EVENT_KILL_LIST] = ' '.join(event_list)
        for ghost in self.get_working_ghost():
            ghost.set_event_kill_list(event_list)

    def get_browser(self):
        return self.prefs.get(self.PREFS_BROWSER,
                              ninix.sakura.DEFAULT_BROWSER)

    def set_browser(self, command):
        self.prefs[self.PREFS_BROWSER] = command
        for ghost in self.get_working_ghost():
            ghost.set_browser(command)

    def get_helpers(self):
        helper_list = []
        while 1:
            pattern = self.prefs.get(self.PREFS_HELPER_PATTERN % len(helper_list))
            command = self.prefs.get(self.PREFS_HELPER_COMMAND % len(helper_list))
            if not pattern or not command:
                break
            helper_list.append((pattern, command))
        return helper_list

    def set_helpers(self, helper_list):
        for i in range(len(helper_list)):
            pattern, command = helper_list[i]
            self.prefs[self.PREFS_HELPER_PATTERN % i] = pattern
            self.prefs[self.PREFS_HELPER_COMMAND % i] = command
        for ghost in self.get_working_ghost():
            ghost.set_helpers(helper_list)

    def get_event_kill_list(self):
        event_list = self.prefs.get(self.PREFS_EVENT_KILL_LIST)
        if event_list:
            event_list = event_list.split()
        else:
            event_list = []
        return event_list

    def set_check_collision(self, flag):
        self.prefs[self.PREFS_CHECK_COLLISION] = flag
        for ghost in self.get_working_ghost():
            if flag:
                ghost.set_debug(self.debug | 4096)
            else:
                ghost.set_debug((self.debug & 4096) ^ self.debug)

    def get_check_collision(self):
        return self.prefs.getint(self.PREFS_CHECK_COLLISION, 0)

    def set_use_pna(self, flag):
        self.prefs[self.PREFS_USE_PNA] = str(flag)
        for ghost in self.get_working_ghost():
            ghost.set_use_pna(flag)

    def get_use_pna(self):
        return self.prefs.getint(self.PREFS_USE_PNA)

    def set_sink_after_talk(self, flag):
        self.prefs[self.PREFS_SINK_AFTER_TALK] = str(flag)
        for ghost in self.get_working_ghost():
            ghost.set_sink_after_talk(flag)

    def get_sink_after_talk(self):
        return self.prefs.getint(self.PREFS_SINK_AFTER_TALK)

    def set_raise_before_talk(self, flag):
        self.prefs[self.PREFS_RAISE_BEFORE_TALK] = str(flag)
        for ghost in self.get_working_ghost():
            ghost.set_raise_before_talk(flag)

    def get_raise_before_talk(self):
        return self.prefs.getint(self.PREFS_RAISE_BEFORE_TALK)

    def set_surface_alpha(self, alpha):
        self.prefs[self.PREFS_SURFACE_ALPHA] = str(alpha)
        for ghost in self.get_working_ghost():
            ghost.set_surface_alpha(alpha)

    def get_surface_alpha(self):
        return self.prefs.getint(self.PREFS_SURFACE_ALPHA)

    def set_balloon_alpha(self, alpha):
        self.prefs[self.PREFS_BALLOON_ALPHA] = str(alpha)
        for ghost in self.get_working_ghost():
            ghost.set_balloon_alpha(alpha)

    def get_balloon_alpha(self):
        return self.prefs.getint(self.PREFS_BALLOON_ALPHA)

    def get_ghost_list(self):
        return self.ghost_list

    def range_ghosts(self):
        index_list = []
        for i in range(len(self.ghosts)):
            if self.ghosts[i] is None:
                continue
            else:
                index_list.append(i)
        return index_list

    def __create_ghost_list_item(self, i):
        if i >= len(self.ghosts):
            return None
        if self.ghosts[i] is None:
            return None
        set_type = 'g'
        desc, shiori_dir, use_makoto, surface_set, balloon, prefix, \
              shiori_dll, shiori_name = self.ghosts[i]
        icon = desc.get('icon', None)
        if icon is not None:
            icon_path = os.path.join(shiori_dir, icon)
            if not os.path.exists(icon_path):
                icon_path = None
        else:
            icon_path = None
        name = desc.get('name',
                        ''.join((unicode(_('Ghost'), 'utf-8'),
                                 '#%d' % (i + 1))))
        shell_list = []
        for j in range(len(surface_set)):
            value = (set_type, i, j)
            shell_name = surface_set[j][0]
            shell_list.append((shell_name, value))
        item = {}
        item['name'] = name
        item['icon'] = icon_path
        item['shell'] = shell_list
        item['instance'] = None
        return item

    def update_ghost_list(self, i, ghost):
        self.ghost_list[i] = self.__create_ghost_list_item(i)
        if self.ghost_list[i] is not None:
            self.ghost_list[i]['instance'] = ghost
        else:
            pass ## FIXME: ghost instance should be deleted

    def create_ghost_list(self):
        ghost_list = []
        for i in range(len(self.ghosts)):
            item = self.__create_ghost_list_item(i)
            ghost_list.append(item)
        self.ghost_list = ghost_list

    def get_shell_list(self):
        set_type = 's'
        set_list = []
        for i in range(len(self.shells)):
            name, surface_set, balloon = self.shells[i]
            if name is None:
                name = unicode(''.join((_('Shell'), '#%d' % (i + 1))), 'utf-8')
            shell_list = []
            for j in range(len(surface_set)):
                value = (set_type, i, j)
                shell_name = surface_set[j][0]
                shell_list.append((shell_name, value))
            item = {}
            item['name'] = name
            item['icon'] = None
            item['shell'] = shell_list
            set_list.append(item)
        return set_list

    def get_balloon_list(self):
        balloon_list = []
        for i in range(len(self.balloons)):
            desc, balloon = self.balloons[i]
            name = desc.get('name',
                            ''.join((unicode(_('Balloon'), 'utf-8'),
                                     '#%d' % (i + 1))))
            item = {}
            item['name'] = name
            item['icon'] = None
            balloon_list.append(item)
        return balloon_list

    def get_plugin_list(self):
        plugin_list = []
        for i in range(len(self.plugins)):
            plugin_name, plugin_dir, startup, menu_items = self.plugins[i]
            if not menu_items:
                continue
            item = {}
            item['name'] = plugin_name
            item['icon'] = None
            item_list = []
            for j in range(len(menu_items)):
                label, argv = menu_items[j]
                value = (i, j)
                item_list.append((label, value))
            item['items'] = item_list
            plugin_list.append(item)
        return plugin_list

    def get_nekodorif_list(self):
        nekodorif_list = []
        for i in range(len(self.nekoninni)):
            nekoninni_name, nekoninni_dir = self.nekoninni[i]
            if not nekoninni_name:
                continue
            item = {}
            item['name'] = nekoninni_name
            item['dir'] = nekoninni_dir
            nekodorif_list.append(item)
        return nekodorif_list

    def get_kinoko_list(self):
        return self.kinoko

    def load(self):
        self.create_ghost_list()
        # load user preferences
        self.prefs = ninix.prefs.load_prefs()
        # choose default ghost/shell
        name = self.prefs.get(self.PREFS_SAKURA_NAME)
        surface = self.prefs.getint(self.PREFS_SAKURA_SURFACE, 0)
        default_sakura = self.find_ghost_by_name(name, surface) or \
                         self.find_shell_by_name(name, surface) or \
                         self.choose_default_sakura()
        # load ghost/shell
        self.current_sakura = default_sakura
        for i in range(len(self.ghosts)):
            ghost = self.create_ghost(self.ghosts[i], self.debug)
            self.ghost_list[i]['instance'] = ghost
        ##names = self.get_ghost_names()
        ##for i in range(len(names)):
        ##    print 'GHOST(%d): %s' % (i, names[i].encode('utf-8', 'ignore'))
        self.start_sakura(self.current_sakura, init=1)

    def find_ghost_by_name(self, name, surface):
        for i in range(len(self.ghosts)):
            if self.ghosts[i] is None:
                continue
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, \
                  shiori_dll, shiori_name = self.ghosts[i]
            try:
                if desc.get('name') == name:
                    if not surface_set or surface < len(surface_set):
                        return ('g', i, surface)
            except: # old preferences(EUC-JP)
                pass
        return None

    def find_shell_by_name(self, name, surface):
        for i in range(len(self.shells)):
            shell_name, surface_set, balloon = self.shells[i]
            try:
                if shell_name == name and surface < len(surface_set):
                    return ('s', i, surface)
            except: # old preferences(EUC-JP)
                pass
        return None

    def choose_default_sakura(self):
        for i in range(len(self.ghosts)):
            if self.ghosts[i] is None:
                continue
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, \
                  shiori_dll, shiori_name = self.ghosts[i]
            if surface_set:
                return ('g', i, 0)
        return ('s', 0, 0)

    def find_balloon_by_name(self, name):
        for i in range(len(self.balloons)):
            desc, balloon = self.balloons[i]
            try:
                if desc.get('name') == name:
                    return i
                if balloon['balloon_dir'][0] == name.encode('utf-8').lower(): # XXX
                    return i
            except: # old preferences(EUC-JP)
                pass
        return None

    def run(self): ## FIXME
        self.load()
        # set SIGCHLD handler
        signal.signal(signal.SIGCHLD, self.terminate_plugin)
        # start SSTP server
        for port in self.sstp_port:
            try:
                server = ninix.sstp.SSTPServer(('', port), self)
            except socket.error, (code, message):
                sys.stderr.write('Port %d: %s (ignored)\n' % (port, message))
                continue
            self.sstp_servers.append(server)
            print 'Serving SSTP on port', port
        # start plugins
        try:
            os.setpgid(0, 0)
        except OSError:
            pass
        for plugin_name, plugin_dir, startup, menu_items in self.plugins:
            if startup is not None:
                self.exec_plugin(plugin_dir, startup)
        self.set_timeout()
        gtk.main()

    def get_current_sakura(self):
        return self.ghost_list[self.current_sakura[1]]['instance'].get_sakura() ## FIXME

    def check_request_queue(self, sender):
        count = 0
        for request in self.__sstp_queue:
            if request[2].split(' / ')[0] == sender.split(' / ')[0]:
                count += 1
        if self.__sstp_flag and \
           self.__current_sender.split(' / ')[0] == sender.split(' / ')[0]:
            count += 1
        return str(count), str(len(self.__sstp_queue))

    def get_ghost_names(self): ## FIXME
        name_list = []
        for item in self.ghost_list:
            ghost = item['instance']
            name = ghost.get_selfname()
            name_list.append(name)
        return name_list

    def get_ghost(self, if_ghost):
        for item in self.ghost_list:
            if item is None:
                continue
            ghost = item['instance']
            assert ghost is not None
            if ghost.get_selfname() == if_ghost:
                return ghost
        else:
            return None

    def if_ghost(self, if_ghost):
        for item in self.ghost_list:
            if item is None:
                continue
            ghost = item['instance']
            assert ghost is not None
            if ghost.get_selfname() == if_ghost:
                return 1
        else:
            return 0

    def select_current_sakura(self, ifghost=None):
        if ifghost is not None:
            for item in self.ghost_list:
                if item is None:
                    continue
                ghost = item['instance']
                assert ghost is not None
                names = '%s,%s' % (ghost.get_selfname(),
                                   ghost.get_keroname())
                name =  '%s' % ghost.get_selfname()
                if ifghost in [name, names]:
                    if not ghost.is_running():
                        self.current_sakura = item['shell'][0][1] ## FIXME
                        self.start_sakura(self.current_sakura, init=1, temp=1) ## FIXME
                    else:
                        self.current_sakura = ghost.current
                    break
                else:
                    pass
        else:
            working_list = self.get_working_ghost(cantalk=1)
            if working_list:
                self.current_sakura = random.choice(working_list).current

    def remove_ghost(self, ghost): ## FIXME
        if not self.get_working_ghost():
            self.quit()
        elif self.current_sakura == ghost.current:
            self.select_current_sakura()

    def quit(self):
        gobject.source_remove(self.timeout_id)
        self.usage_dialog.close()
        for server in self.sstp_servers:
            server.close()
        self.save_preferences()
        gtk.main_quit()
        # stop plugins
        signal.signal(signal.SIGHUP, signal.SIG_IGN)
        os.kill(0, signal.SIGHUP)

    def save_preferences(self): ## FIXME
        # last ghost/shell
        set_type, i, j = self.current_sakura
        if set_type == 'g':
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, \
                  shiori_dll, shiori_name = self.ghosts[i]
            name = desc.get('name', '')
        else:
            name, surface_set, balloon = self.shells[i]
        self.prefs[self.PREFS_SAKURA_NAME] = name
        self.prefs[self.PREFS_SAKURA_SURFACE] = str(j)
        # save preferences
        try:
            self.prefs.save()
        except IOError:
            sys.stderr.write('Cannot write preferences to file (ignored).\n')

    def select_ghost(self, ghost, sequential, event=1):
        if len(self.range_ghosts()) < 2:
            return
        set_type, i, j = ghost.current
        if set_type == 's':
            i = 0
        if sequential: ## FIXME
            i += 1
            if i == len(self.ghosts): ## FIXME
                i = 0
        else:
            index_list = self.range_ghosts()
            index_list.remove(i)
            i = random.choice(index_list)
        self.change_sakura(ghost, ('g', i, 0), 'automatic', event)

    def select_ghost_by_name(self, ghost, name, event=1):
        item = self.find_ghost_by_name(name, 0)
        if item is None:
            return
        self.change_sakura(ghost, item, 'automatic', event)

    def select_sakura(self, event, args):
        ghost, item = args
        if ghost.busy():
            gtk.gdk.beep()
            return
        self.change_sakura(ghost, item, 'manual')

    def change_sakura(self, ghost, item, method, event=1, vanished=0):
        set_type, i, j = item
        assert self.ghosts[i] is not None
        if ghost.current == item: ## FIXME: needs reloading?
            ghost.notify_ghost_changed(ghost.get_selfname(), 0, None)
            return
        if set_type == 'g':
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, \
                  shiori_dll, shiori_name = self.ghosts[i]
            if surface_set:
                name, surface_dir, surface_desc, surface_alias, surface = \
                      surface_set[j]
            else:
                shell_name, surface_set, balloon = self.shells[0]
                name, surface_dir, surface_desc, surface_alias, surface = \
                      surface_set[0]
        elif set_type == 's':
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, \
                  shiori_dll, shiori_name = self.ghosts[0] ## FIXME: this should be default
            shell_name, surface_set, shell_balloon = self.shells[i]
            name, surface_dir, surface_desc, surface_alias, surface = \
                  surface_set[j]
        def proc(self=self, item=item):
            self.stop_sakura(ghost, self.start_sakura, item, ghost.current) ## FIXME
        if vanished:
            ghost.finalize()
            self.start_sakura(item, ghost.current, vanished)
            self.remove_ghost(ghost)
        elif not event:
            proc()
        elif ghost.current[0] == set_type and ghost.current[1] == i:
            assert ghost.current[2] != j
            ghost.notify_shell_changing(name, surface_dir, proc)
        else:
            name = surface_desc.get('sakura.name', desc.get('sakura.name'))
            ghost.notify_ghost_changing(name, method, proc)

    def stop_sakura(self, ghost, starter=None, *args): ## FIXME
        ghost.finalize()
        if starter is not None:
            starter(*args)
        self.remove_ghost(ghost)

    def request_shell(self, index):
        return self.shells[index]

    def request_balloon(self, index):
        return self.balloons[index]

    def select_ghost_from_list(self, item):
        set_type, i, j = item
        assert self.ghosts[i] is not None
        assert self.ghost_list[i]['instance'] is not None
        ghost = self.ghost_list[i]['instance']
        return ghost

    def start_sakura(self, item, prev=None, vanished=0, init=0, temp=0):
        set_type, i, j = item
        ghost = self.select_ghost_from_list(item)
        if prev is not None:
            assert self.ghosts[prev[1]] is not None
            assert self.ghost_list[prev[1]]['instance'] is not None
        if init:
            ghost_changed = 0
        else:
            assert prev is not None ## FIXME
            if prev[0] == set_type and prev[1] == i:
                ghost_changed = 0
            else:
                ghost_changed = 1
        if ghost_changed:
            name = self.ghost_list[prev[1]]['instance'].get_selfname()
        else:
            name = None
        ghost.set_use_pna(self.get_use_pna())
        ghost.set_sink_after_talk(self.get_sink_after_talk())
        ghost.set_raise_before_talk(self.get_raise_before_talk())
        ghost.set_surface_scale(self.get_surface_scale())
        ghost.set_balloon_scalling(self.get_balloon_scalling())
        ghost.set_surface_alpha(self.get_surface_alpha())
        ghost.set_balloon_alpha(self.get_balloon_alpha())
        ghost.set_script_speed(self.get_script_speed())
        ghost.start(item, init, temp, vanished, ghost_changed,
                    self.get_ignore_default(), name)

    def start_sakura_cb(self, event, item):
        self.start_sakura(item, init=1)

    def get_balloon_description(self, name):
        number = self.find_balloon_by_name(name)
        if number is None:
            ##print 'Balloon %s not found.' % name
            default_balloon = self.get_default_balloon()
            number = self.find_balloon_by_name(default_balloon) or 0
        desc, balloon = self.balloons[number]
        return desc, balloon

    def reload(self, ghost):
        if ghost.busy():
            gtk.gdk.beep()
            return
        def proc(self=self):
            self.stop_sakura(ghost, self.reload_current_sakura, ghost)
        ghost.notify_ninix_reloading(proc)

    def reload_current_sakura(self, ghost):
        self.save_preferences()
        set_type, i, j = ghost.current
        if set_type == 's':
            i = 0
        home_dir = ninix.home.get_ninix_home()
        ghost_dir = ghost.get_prefix()
        ghost_conf, shell_conf = ninix.home.search_ghosts(home_dir,
                                                          [ghost_dir])
        if ghost_conf:
            self.ghosts[i] = ghost_conf[0]
        else:
            self.ghosts[i] = None
            self.shells.append(shell_conf[0]) ## FIXME
        self.update_ghost_list(i, ghost)
        item = (set_type, i, j)
        self.start_sakura(item, item)

    def vanish_sakura(self, ghost):
        # remove ghost
        prefix = ghost.get_prefix()
        for filename in os.listdir(prefix):
            if filename != 'HISTORY':
                command = ''.join(('rm -rf ', os.path.join(prefix, filename)))
                print command
                if os.system(command):
                    print '***FAILED***'
        # select another ghost
        set_type, i, j = ghost.current
        if set_type == 's':
            i = 0
        index_list = self.range_ghosts()
        index_list.remove(i)
        next_i = random.choice(index_list)
        assert i != next_i
        if self.current_sakura == ghost.current:
            self.current_sakura = ('g', next_i, 0)
        # reload and start the new ghost
        self.ghosts[i] = None
        self.update_ghost_list(i, ghost)
        self.change_sakura(ghost, ('g', next_i, 0), 'automatic', vanished=1)

    def select_plugin(self, event, item):
        i, j = item
        plugin_name, plugin_dir, startup, menu_items = self.plugins[i]
        label, argv = menu_items[j]
        self.exec_plugin(plugin_dir, argv)

    def select_nekodorif(self, event, item): ## FIXME
        target, nekodorif_dir = item
        ninix.nekodorif.Nekoninni().load(nekodorif_dir, self.katochan, target)

    def select_kinoko(self, event, item): ## FIXME
        target, data = item
        ninix.kinoko.Kinoko(self.kinoko).load(data, target)

    def terminate_plugin(self, signum, frame):
        for pid in self.plugin_pids[:]:
            try:
                (pid, status) = os.waitpid(pid, os.WNOHANG)
            except OSError:
                ##print 'Process %d not found.' % pid
                self.plugin_pids.remove(pid)
            if pid > 0:
                ##print 'Process %d terminated.' % pid
                self.plugin_pids.remove(pid)
        ##print 'Running subprocesses:', self.plugin_pids

    def get_sstp_port(self):
        if not self.sstp_servers:
            return None
        return self.sstp_servers[0].server_address[1]

    def exec_plugin(self, plugin_dir, argv):
        ##print 'exec_plugin:', ' '.join(argv)
        if not os.path.exists(argv[0]):
            return
        port = self.get_sstp_port()
        if port is None:
            port = 'none'
        environ = os.environ.copy()
        environ['NINIX_PID'] = str(os.getpid())
        environ['NINIX_SSTP_PORT'] = str(port)
        environ['NINIX_PLUGIN_DIR'] = plugin_dir
        try:
            pid = os.fork()
        except OSError:
            sys.stderr.write('Error: %s failed (ignored)\n' % argv[0])
            return
        if pid == 0:
            os.chdir(plugin_dir)
            try:
                os.execve(argv[0], argv, environ)
            except OSError:
                sys.stderr.write('Error: %s failed (abort)\n' % argv[0])
                os._exit(1)
        self.plugin_pids.append(pid)

    def open_ghost_manager(self):
        self.__ngm.show_dialog()

    def edit_preferences(self):
        self.saved_prefs = [
            self.get_event_kill_list(),
            self.get_balloon_fonts(),
            self.get_mouse_button1(),
            self.get_mouse_button3(),
            self.get_browser(),
            self.get_helpers(),
            self.get_top_margin(),
            self.get_bottom_margin(),
            self.get_default_balloon(),
            self.get_ignore_default(),
            self.get_surface_scale(),
            self.get_script_speed(),
            self.get_balloon_scalling(),
            self.get_allowembryo(),
            self.get_check_collision(),
            self.get_use_pna(),
            self.get_sink_after_talk(),
            self.get_surface_alpha(),
            self.get_balloon_alpha(),
            self.get_raise_before_talk(),
            ]
        self.pref_dialog.set_event_kill_list(self.saved_prefs[0])
        self.pref_dialog.set_balloon_fonts(self.saved_prefs[1])
        self.pref_dialog.set_mouse_button1(self.saved_prefs[2])
        self.pref_dialog.set_mouse_button3(self.saved_prefs[3])
        self.pref_dialog.set_browser(self.saved_prefs[4])
        self.pref_dialog.set_helpers(self.saved_prefs[5])
        self.pref_dialog.set_top_margin(self.saved_prefs[6])
        self.pref_dialog.set_bottom_margin(self.saved_prefs[7])
        self.pref_dialog.set_default_balloon(self.saved_prefs[8])
        self.pref_dialog.set_ignore_default(self.saved_prefs[9])
        self.pref_dialog.set_surface_scale(self.saved_prefs[10])
        self.pref_dialog.set_script_speed(self.saved_prefs[11])
        self.pref_dialog.set_balloon_scalling(self.saved_prefs[12])
        self.pref_dialog.set_allowembryo(self.saved_prefs[13])
        self.pref_dialog.set_check_collision(self.saved_prefs[14])
        self.pref_dialog.set_use_pna(self.saved_prefs[15])
        self.pref_dialog.set_sink_after_talk(self.saved_prefs[16])
        self.pref_dialog.set_surface_alpha(self.saved_prefs[17])
        self.pref_dialog.set_balloon_alpha(self.saved_prefs[18])
        self.pref_dialog.set_raise_before_talk(self.saved_prefs[19])
        self.pref_dialog.show()

    def notify_preferences_changed(self, done):
        self.set_event_kill_list(self.pref_dialog.get_event_kill_list())
        self.set_balloon_fonts(self.pref_dialog.get_balloon_fonts())
        self.set_mouse_button1(self.pref_dialog.get_mouse_button1())
        self.set_mouse_button3(self.pref_dialog.get_mouse_button3())
        self.set_browser(self.pref_dialog.get_browser())
        self.set_helpers(self.pref_dialog.get_helpers())
        self.set_top_margin(self.pref_dialog.get_top_margin())
        self.set_bottom_margin(self.pref_dialog.get_bottom_margin())
        self.set_default_balloon(self.pref_dialog.get_default_balloon())
        self.set_ignore_default(self.pref_dialog.get_ignore_default())
        self.set_surface_scale(self.pref_dialog.get_surface_scale())
        self.set_script_speed(self.pref_dialog.get_script_speed())
        self.set_balloon_scalling(self.pref_dialog.get_balloon_scalling())
        self.set_allowembryo(self.pref_dialog.get_allowembryo())
        self.set_check_collision(self.pref_dialog.get_check_collision())
        self.set_use_pna(self.pref_dialog.get_use_pna())
        self.set_sink_after_talk(self.pref_dialog.get_sink_after_talk())
        self.set_surface_alpha(self.pref_dialog.get_surface_alpha())
        self.set_balloon_alpha(self.pref_dialog.get_balloon_alpha())
        self.set_raise_before_talk(self.pref_dialog.get_raise_before_talk())

    def notify_preferences_reverted(self):
        self.set_event_kill_list(self.saved_prefs[0])
        self.set_balloon_fonts(self.saved_prefs[1])
        self.set_mouse_button1(self.saved_prefs[2])
        self.set_mouse_button3(self.saved_prefs[3])
        self.set_browser(self.saved_prefs[4])
        self.set_helpers(self.saved_prefs[5])
        self.set_top_margin(self.saved_prefs[6])
        self.set_bottom_margin(self.saved_prefs[7])
        self.set_default_balloon(self.saved_prefs[8])
        self.set_ignore_default(self.saved_prefs[9])
        self.set_surface_scale(self.saved_prefs[10])
        self.set_script_speed(self.saved_prefs[11])
        self.set_balloon_scalling(self.saved_prefs[12])
        self.set_allowembryo(self.saved_prefs[13])
        self.set_check_collision(self.saved_prefs[14])
        self.set_use_pna(self.saved_prefs[15])
        self.set_sink_after_talk(self.saved_prefs[16])
        self.set_surface_alpha(self.saved_prefs[17])
        self.set_balloon_alpha(self.saved_prefs[18])
        self.set_raise_before_talk(self.saved_prefs[19])

    def show_usage(self):
        for ghost in self.get_working_ghost():
            ghost.save_history()
        history = {}
        for i in range(len(self.ghosts)):
            if self.ghosts[i] is None:
                continue
            desc = self.ghosts[i][0]
            name = desc.get('name',
                            ''.join((unicode(_('Ghost'), 'utf-8'),
                                     '#%d' % (i + 1))))
            ghost_time = 0
            prefix = self.ghosts[i][5]
            path = os.path.join(prefix, 'HISTORY')
            if os.path.exists(path):
                try:
                    f = open(path, 'r')
                except IOError, (code, message):
                    sys.stderr.write('cannot read %s\n' % path)
                else:
                    for line in f:
                        if ',' not in line:
                            continue
                        key, value = [x.strip() for x in line.split(',', 1)]
                        if key == 'time':
                            try:
                                ghost_time = int(value)
                            except:
                                pass
                f.close()
            ai_list = []
            dirlist = os.listdir(os.path.join(prefix, 'shell'))
            for subdir in dirlist:
                path = os.path.join(prefix, 'shell', subdir, 'ai.png')
                if os.path.exists(path):
                    ai_list.append(path)
            history[name] = (ghost_time, ai_list)
        self.usage_dialog.open(history)

    def set_timeout(self):
        self.timeout_id = gobject.timeout_add(100, self.do_idle_tasks) # 100ms

    def do_idle_tasks(self):
        self.handle_sstp_queue()
        self.receive_sstp_request()
        self.set_timeout()

class PreferenceDialog:

    def __init__(self, app):
        self.app = app
        self.window = gtk.Dialog()
        self.window.set_title('Preferences')
        self.window.connect('delete_event', self.cancel)
        self.notebook = gtk.Notebook()
        self.notebook.set_tab_pos(gtk.POS_TOP)
        self.window.vbox.pack_start(self.notebook)
        self.notebook.show()
        for name, constructor in [
            (_('Event'),           self.make_page_events),
            (_('Font'),            self.make_page_fonts),
            (_('Mouse'),           self.make_page_mouse),
            (_('Browser'),         self.make_page_browser),
            (_('Helper'),          self.make_page_helper),
            (_('Surface&Balloon'), self.make_page_surface_n_balloon),
            (_('Misc'),            self.make_page_misc),
            (_('Debug'),           self.make_page_debug),
            ]:
            self.notebook.append_page(constructor(),
                                      gtk.Label(unicode(name, 'utf-8')))
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button('OK')
        button.connect('clicked', self.ok)
        box.add(button)
        button.show()
        button = gtk.Button('Apply')
        button.connect('clicked', self.apply)
        box.add(button)
        button.show()
        button = gtk.Button('Cancel')
        button.connect('clicked', self.cancel)
        box.add(button)
        button.show()
        self.rule_editor = RuleEditor(self.window)

    def ok(self, widget):
        self.hide()
        self.app.notify_preferences_changed(1)

    def apply(self, widget):
        self.app.notify_preferences_changed(0)

    def cancel(self, widget, event=None):
        self.app.notify_preferences_reverted()
        self.hide()
        return True

    def show(self):
        self.window.show()

    def hide(self):
        self.window.hide()

    def make_page_events(self):
        frame = gtk.Frame(unicode(_('Event(s) to be ignored'), 'utf-8'))
        frame.set_size_request(480, 300)
        frame.set_border_width(5)
        frame.show()
        box = gtk.VBox()
        frame.add(box)
        box.show()
        table = gtk.Table(8, 2, True)
        table.set_row_spacings(3)
        table.set_col_spacings(5)
        table.set_border_width(5)
        box.pack_start(table, False)
        table.show()
        events = [
            'OnBoot',          'OnClose',
            'OnGhostChanging', 'OnGhostChanged',
            'OnShellChanging', 'OnShellChanged',
            'OnSurfaceChange', 'OnSurfaceRestore',
            'OnMinuteChange',  'OnSecondChange',
            'OnMouseClick',    'OnMouseDoubleClick',
            'OnMouseMove',     'OnMouseWheel',
            'OnKeyPress',
            ]
        self.event_kill_list = {}
        for i in range(len(events)):
            y, x = divmod(i, 2)
            button = gtk.CheckButton(events[i])
            table.attach(button, x, x + 1, y, y + 1)
            button.show()
            self.event_kill_list[events[i]] = button
        return frame

    def make_page_fonts(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        # font
        frame = gtk.Frame(unicode(_('Font(s) for balloons'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame)
        frame.show()
        self.fontsel = gtk.FontSelection()
        self.fontsel.show()
        frame.add(self.fontsel)
        page.show()
        return page

    def make_page_mouse(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        frame = gtk.Frame(unicode(_('Left button'), 'utf-8'))
        frame.set_size_request(400, 125)
        frame.show()
        page.pack_start(frame)
        box = gtk.VBox(spacing=5)
        frame.add(box)
        box.show()
        box.set_border_width(5)
        button1 = gtk.RadioButton(None,
                                  unicode(_('delete balloon(s)'), 'utf-8'))
        box.pack_start(button1, False)
        button1.show()
        button2 = gtk.RadioButton(button1,
                                  unicode(_('raise all windows'), 'utf-8'))
        box.pack_start(button2, False)
        button2.show()
        button3 = gtk.RadioButton(button1,
                                  unicode(_('lower all windows'), 'utf-8'))
        box.pack_start(button3, False)
        button3.show()
        self.mouse_button1 = {
            ninix.sakura.BUTTON1_CLOSE: button1,
            ninix.sakura.BUTTON1_RAISE: button2,
            ninix.sakura.BUTTON1_LOWER: button3,
            }
        frame = gtk.Frame(unicode(_('Right button'), 'utf-8'))
        frame.set_size_request(400, 125)
        frame.show()
        page.pack_start(frame)
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button1 = gtk.RadioButton(None,
                                  unicode(_('delete balloon(s)'), 'utf-8'))
        box.pack_start(button1, False)
        button1.show()
        button2 = gtk.RadioButton(button1,
                                  unicode(_('raise all windows'), 'utf-8'))
        box.pack_start(button2, False)
        button2.show()
        button3 = gtk.RadioButton(button1,
                                  unicode(_('lower all windows'), 'utf-8'))
        box.pack_start(button3, False)
        button3.show()
        self.mouse_button3 = {
            ninix.sakura.BUTTON3_CLOSE: button1,
            ninix.sakura.BUTTON3_RAISE: button2,
            ninix.sakura.BUTTON3_LOWER: button3,
            }
        page.show()
        return page

    def make_page_browser(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        # browser
        frame = gtk.Frame(unicode(_('Browser'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame)
        frame.show()
        box = gtk.VBox(spacing=2)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        self.browser = gtk.Entry()
        box.pack_start(self.browser, False)
        self.browser.show()
        # help messages
        frame = gtk.Frame()
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=2)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        label = gtk.Label(unicode(_('- %s in this command line will be replaced with the URL'), 'utf-8'))
        label.set_alignment(0, -1)
        box.pack_start(label, False)
        label.show()
        label = gtk.Label(unicode(_('- trailing & is not required.(automagically added)'), 'utf-8'))
        label.set_alignment(0, -1)
        box.pack_start(label, False)
        label.show()
        page.show()
        return page

    def make_page_helper(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        frame = gtk.Frame(unicode(_('Application'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        swin = gtk.ScrolledWindow()
        swin.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        box.pack_start(swin)
        swin.show()
        lstore = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING)
        self.helpers = gtk.TreeView(lstore)
        column = gtk.TreeViewColumn('Pattern', gtk.CellRendererText(), text=0)
        self.helpers.append_column(column)
        column = gtk.TreeViewColumn('Command', gtk.CellRendererText(), text=1)
        self.helpers.append_column(column)
        swin.add(self.helpers)
        self.helpers.show()
        bbox = gtk.HButtonBox()
        bbox.set_spacing(10)
        bbox.set_layout(gtk.BUTTONBOX_EDGE)
        bbox.set_border_width(5)
        box.pack_start(bbox, False)
        bbox.show()
        button = gtk.Button('New')
        button.connect('clicked', self.rule_new)
        bbox.pack_start(button)
        button.show()
        button = gtk.Button('Edit')
        button.connect('clicked', self.rule_edit)
        bbox.pack_start(button)
        button.show()
        button = gtk.Button('Delete')
        button.connect('clicked', self.rule_delete)
        bbox.pack_start(button)
        button.show()
        button = gtk.Button('Up')
        button.connect('clicked', self.rule_up)
        bbox.pack_start(button)
        button.show()
        button = gtk.Button('Down')
        button.connect('clicked', self.rule_down)
        bbox.pack_start(button)
        button.show()
        frame = gtk.Frame()
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=2)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        label = gtk.Label(unicode(_('- %s in this command line will be replaced with the filename'), 'utf-8'))
        label.set_alignment(0, -1)
        box.pack_start(label, False)
        label.show()
        label = gtk.Label(unicode(_('- trailing & is not required.(automagically added)'), 'utf-8'))
        label.set_alignment(0, -1)
        box.pack_start(label, False)
        label.show()
        page.show()
        return page

    def make_page_surface_n_balloon(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_('Surface Scaling'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        hbox = gtk.HBox(spacing=5)
        box.pack_start(hbox, False)
        hbox.show()
        label = gtk.Label(unicode(_('Default Setting'), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.surface_scale_combo = gtk.combo_box_new_text()
        for label, value in ninix.surface.range_scale:
            self.surface_scale_combo.append_text(label)
        hbox.pack_start(self.surface_scale_combo, False)
        self.surface_scale_combo.show()
        button = gtk.CheckButton(unicode(_('Scale Balloon'), 'utf-8'))
        self.balloon_scalling_button = button
        box.pack_start(button, False)
        button.show()
        frame = gtk.Frame(unicode(_('Default Balloon'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        scrolled = gtk.ScrolledWindow()
        scrolled.set_policy(gtk.POLICY_NEVER, gtk.POLICY_ALWAYS)
        scrolled.set_shadow_type(gtk.SHADOW_ETCHED_IN)
        box.pack_start(scrolled, True)
        scrolled.show()
        model = gtk.ListStore(gobject.TYPE_STRING)
        for desc, balloon in self.app.balloons:
            name = desc.get('name', '')
            listiter = model.append()
            model.set_value(listiter, 0, name)
        treeview = gtk.TreeView(model)
        column = gtk.TreeViewColumn(
            _('Balloon Name'), gtk.CellRendererText(), text=0)
        treeview.append_column(column)
        treeview.get_selection().set_mode(gtk.SELECTION_SINGLE)
        self.balloon_treeview = treeview
        scrolled.add(treeview)
        treeview.show()
        button = gtk.CheckButton(unicode(_('Always Use This Balloon'), 'utf-8'))
        self.ignore_button = button
        box.pack_start(button, False)
        button.show()
        frame = gtk.Frame(unicode(_('Translucency'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button = gtk.CheckButton(unicode(_('Use PNA file'), 'utf-8'))
        self.use_pna_button = button
        box.pack_start(button, False)
        button.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_("Surfaces' alpha channel"), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.surface_alpha_adjustment = gtk.Adjustment(1.0, 0.1, 1.0, 0.1, 0.5)
        button = gtk.SpinButton(self.surface_alpha_adjustment, 0.2, 1)
        hbox.pack_start(button, False)
        button.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_("Balloons' alpha channel"), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.balloon_alpha_adjustment = gtk.Adjustment(1.0, 0.1, 1.0, 0.1, 0.5)
        button = gtk.SpinButton(self.balloon_alpha_adjustment, 0.2, 1)
        hbox.pack_start(button, False)
        button.show()
        return page        

    def make_page_misc(self):
        scrn_h = gtk.gdk.screen_height()
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_('SSTP Setting'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        button = gtk.CheckButton(unicode(_('Allowembryo'), 'utf-8'))
        self.allowembryo_button = button
        frame.add(button)
        button.show()
        frame = gtk.Frame(unicode(_('Script Wait'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        hbox = gtk.HBox(spacing=5)
        frame.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_('Default Setting'), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.script_speed_combo = gtk.combo_box_new_text()
        for label, value in ninix.sakura.range_script_speed:
            self.script_speed_combo.append_text(label)
        hbox.pack_start(self.script_speed_combo, False)
        self.script_speed_combo.show()
        frame = gtk.Frame(unicode(_('Top & Bottom Margin'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_('Top Margin'), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.top_adjustment = gtk.Adjustment(0, 0, scrn_h / 4, 1)
        button = gtk.SpinButton(self.top_adjustment)
        hbox.pack_start(button, False)
        button.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_('Bottom Margin'), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.bottom_adjustment = gtk.Adjustment(0, 0, scrn_h / 4, 1)
        button = gtk.SpinButton(self.bottom_adjustment)
        hbox.pack_start(button, False)
        button.show()
        frame = gtk.Frame(unicode(_('Raise & Lower'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button = gtk.CheckButton(unicode(_('Sink after Talk'), 'utf-8'))
        self.sink_after_talk_button = button
        box.pack_start(button, False)
        button.show()
        button = gtk.CheckButton(unicode(_('Raise before Talk'), 'utf-8'))
        self.raise_before_talk_button = button
        box.pack_start(button, False)
        button.show()
        return page

    def make_page_debug(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_('Surface Debugging'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        button = gtk.CheckButton(unicode(_('Display Collision Area'), 'utf-8'))
        self.check_collision_button = button
        frame.add(button)
        button.show()
        return page

    def rule_new(self, widget):
        self.rule_editor.set_pattern('')
        self.rule_editor.set_command('')
        if self.rule_editor.run('New rule...'):
            pattern = self.rule_editor.get_pattern()
            command = self.rule_editor.get_command()
            lstore = self.helpers.get_model()
            listiter = lstore.append()
            lstore.set(listiter, 0, pattern, 1, command)

    def rule_edit(self, widget):
        selection = self.helpers.get_selection()
        lstore, listiter = selection.get_selected()
        if not listiter:
            return
        pattern = lstore.get_value(listiter, 0)
        command = lstore.get_value(listiter, 1)
        self.rule_editor.set_pattern(pattern)
        self.rule_editor.set_command(command)
        if self.rule_editor.run('Edit rule...'):
            pattern = self.rule_editor.get_pattern()
            command = self.rule_editor.get_command()
            lstore.set(listiter, 0, pattern, 1, command)

    def rule_delete(self, widget):
        selection = self.helpers.get_selection()
        lstore, listiter = selection.get_selected()
        if not listiter:
            return
        lstore.remove(listiter)

    def rule_up(self, widget):
        selection = self.helpers.get_selection()
        lstore, listiter = selection.get_selected()
        if not listiter:
            return
        path = lstore.get_path(listiter)
        if path[0] == 0:
            return
        listiter_ = lstore.get_iter(path[0] - 1)
        if listiter_:
            lstore.swap(listiter_, listiter)

    def rule_down(self, widget):
        selection = self.helpers.get_selection()
        lstore, listiter = selection.get_selected()
        if not listiter:
            return
        listiter_ = lstore.iter_next(listiter)
        if listiter_:
            lstore.swap(listiter_, listiter)

    def set_event_kill_list(self, kill_list):
        for name in self.event_kill_list:
            if name == 'OnFirstBoot':
                continue
            if name in kill_list:
                self.event_kill_list[name].set_active(True)
            else:
                self.event_kill_list[name].set_active(False)

    def get_event_kill_list(self):
        buf = []
        for name in self.event_kill_list:
            if self.event_kill_list[name].get_active():
                buf.append(name)
                if name == 'OnBoot':
                    buf.append('OnFirstBoot')
        return buf

    def set_balloon_fonts(self, name):
        self.fontsel.set_font_name(name)

    def get_balloon_fonts(self):
        name = self.fontsel.get_font_name()
        return name

    def set_mouse_button1(self, name):
        self.mouse_button1[name].set_active(True)

    def get_mouse_button1(self):
        for name in self.mouse_button1:
            if self.mouse_button1[name].get_active():
                return name
        return None # should not reach here

    def set_mouse_button3(self, name):
        self.mouse_button3[name].set_active(True)

    def get_mouse_button3(self):
        for name in self.mouse_button3:
            if self.mouse_button3[name].get_active():
                return name
        return None # should not reach here

    def set_browser(self, command):
        self.browser.set_text(command)

    def get_browser(self):
        return self.browser.get_text()

    def set_helpers(self, helper_list):
        lstore = self.helpers.get_model()
        lstore.clear()
        for pattern, command in helper_list:
            listiter = lstore.append()
            lstore.set(listiter, 0, pattern, 1, command)

    def get_helpers(self):
        helper_list = []
        lstore = self.helpers.get_model()
        listiter = lstore.get_iter_first()
        while listiter:
            pattern = lstore.get_value(listiter, 0)
            command = lstore.get_value(listiter, 1)
            helper_list.append((pattern, command))
            listiter = lstore.iter_next(listiter)
        return helper_list

    def set_balloon_scalling(self, flag):
        if flag:
            self.balloon_scalling_button.set_active(True)
        else:
            self.balloon_scalling_button.set_active(False)

    def get_balloon_scalling(self):
        if self.balloon_scalling_button.get_active():
            return 1
        else:
            return 0

    def set_script_speed(self, speed):
        index = 0
        for label, value in ninix.sakura.range_script_speed:
            if speed == value:
                self.script_speed_combo.set_active(index)
                break
            index += 1

    def get_script_speed(self):
        index = self.script_speed_combo.get_active()
        label, value = ninix.sakura.range_script_speed[index]
        return int(value)

    def set_surface_scale(self, scale):
        index = 0
        for label, value in ninix.surface.range_scale:
            if scale == value:
                self.surface_scale_combo.set_active(index)
                break
            index += 1

    def get_surface_scale(self):
        index = self.surface_scale_combo.get_active()
        label, value = ninix.surface.range_scale[index]
        return int(value)

    def set_top_margin(self, margin):
        self.top_adjustment.set_value(margin)

    def get_top_margin(self):
        margin = self.top_adjustment.get_value()
        return int(margin)

    def set_bottom_margin(self, margin):
        self.bottom_adjustment.set_value(margin)

    def get_bottom_margin(self):
        margin = self.bottom_adjustment.get_value()
        return int(margin)

    def set_default_balloon(self, name):
        model = self.balloon_treeview.get_model()
        listiter = model.get_iter_first()
        while listiter is not None:
            value = model.get_value(listiter, 0)
            if value == name or name is None:
                self.balloon_treeview.get_selection().select_iter(listiter)
                break
            listiter = model.iter_next(listiter) 
        else:
            listiter = model.get_iter_first()
            assert listiter is not None
            self.balloon_treeview.get_selection().select_iter(listiter)

    def get_default_balloon(self):
        name = None
        selected = self.balloon_treeview.get_selection().get_selected()
        if selected:
            model, listiter = selected
            name = model.get_value(listiter, 0)
        return name

    def set_ignore_default(self, flag):
        if flag:
            self.ignore_button.set_active(True)
        else:
            self.ignore_button.set_active(False)

    def get_ignore_default(self):
        if self.ignore_button.get_active():
            return 1
        else:
            return 0

    def set_check_collision(self, flag):
        if flag:
            self.check_collision_button.set_active(True)
        else:
            self.check_collision_button.set_active(False)

    def get_check_collision(self):
        if self.check_collision_button.get_active():
            return 1
        else:
            return 0

    def set_allowembryo(self, flag):
        if flag:
            self.allowembryo_button.set_active(True)
        else:
            self.allowembryo_button.set_active(False)

    def get_allowembryo(self):
        if self.allowembryo_button.get_active():
            return 1
        else:
            return 0

    def set_use_pna(self, flag):
        if flag:
            self.use_pna_button.set_active(True)
        else:
            self.use_pna_button.set_active(False)

    def get_use_pna(self):
        if self.use_pna_button.get_active():
            return 1
        else:
            return 0

    def set_sink_after_talk(self, flag):
        if flag:
            self.sink_after_talk_button.set_active(True)
        else:
            self.sink_after_talk_button.set_active(False)

    def get_sink_after_talk(self):
        if self.sink_after_talk_button.get_active():
            return 1
        else:
            return 0

    def set_raise_before_talk(self, flag):
        if flag:
            self.raise_before_talk_button.set_active(True)
        else:
            self.raise_before_talk_button.set_active(False)

    def get_raise_before_talk(self):
        if self.raise_before_talk_button.get_active():
            return 1
        else:
            return 0

    def set_surface_alpha(self, alpha):
        if alpha is None:
            alpha = 1.0
        self.surface_alpha_adjustment.set_value(alpha)

    def get_surface_alpha(self):
        alpha = self.surface_alpha_adjustment.get_value()
        return float(alpha)

    def set_balloon_alpha(self, alpha):
        if alpha is None:
            alpha = 1.0
        self.balloon_alpha_adjustment.set_value(alpha)

    def get_balloon_alpha(self):
        alpha = self.balloon_alpha_adjustment.get_value()
        return float(alpha)

class RuleEditor:

    def __init__(self, master=None):
        self.dialog = gtk.Dialog()
        self.dialog.connect('delete_event', self.cancel)
        self.dialog.set_modal(True)
        self.dialog.set_position(gtk.WIN_POS_CENTER)
        if master is not None:
            self.dialog.set_transient_for(master)
        # entries
        table = gtk.Table(2, 2)
        table.set_row_spacings(5)
        table.set_col_spacings(5)
        table.set_border_width(10)
        self.dialog.vbox.pack_start(table)
        label = gtk.Label('Pattern')
        table.attach(label, 0, 1, 0, 1, xoptions=gtk.FILL)
        self.pattern_entry = gtk.Entry()
        self.pattern_entry.set_size_request(300, -1)
        self.pattern_entry.connect('changed', self.changed)
        table.attach(self.pattern_entry, 1, 2, 0, 1)
        label = gtk.Label('Command')
        table.attach(label, 0, 1, 1, 2, xoptions=gtk.FILL)
        self.command_entry = gtk.Entry()
        self.command_entry.set_size_request(300, -1)
        self.command_entry.connect('changed', self.changed)
        table.attach(self.command_entry, 1, 2, 1, 2)
        self.dialog.vbox.show_all()
        # buttons
        self.ok_button = gtk.Button('OK')
        self.ok_button.connect('clicked', self.ok)
        self.dialog.action_area.pack_start(self.ok_button)
        button = gtk.Button('Cancel')
        button.connect('clicked', self.cancel)
        self.dialog.action_area.pack_start(button)
        self.dialog.action_area.show_all()

    def set_pattern(self, text):
        self.pattern_entry.set_text(text)

    def get_pattern(self):
        return self.pattern_entry.get_text()

    def set_command(self, text):
        self.command_entry.set_text(text)

    def get_command(self):
        return self.command_entry.get_text()

    def changed(self, widget, event=None):
        if self.pattern_entry.get_text() and \
           self.command_entry.get_text():
            self.ok_button.set_sensitive(True)
        else:
            self.ok_button.set_sensitive(False)

    def run(self, title):
        self.dialog.set_title(title)
        self.dialog.show()
        gtk.main()
        return self.done

    def hide(self):
        self.dialog.hide()
        gtk.main_quit()

    def ok(self, widget, event=None):
        self.done = 1
        self.hide()
        return True

    def cancel(self, widget, event=None):
        self.done = 0
        self.hide()
        return True

class UsageDialog:

    def __init__(self):
        self.window = gtk.Dialog()
        self.window.set_title('Usage')
        self.window.connect('delete_event', self.close)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.size = (550, 330)
        self.darea.set_size_request(*self.size)
        self.darea.connect('configure_event', self.configure)
        self.darea.connect('expose_event', self.redraw)
        self.window.vbox.pack_start(self.darea)
        self.darea.show()
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button('Close')
        button.connect('clicked', self.close)
        box.add(button)
        button.show()
        self.opened = 0

    def open(self, history):
        if self.opened:
            return
        self.history = history
        self.items = []
        for name, (clock, path) in self.history.iteritems():
            self.items.append((name, clock, path))
        self.items[:] = [(x[1], x) for x in self.items]
        self.items.sort()
        self.items[:] = [x for x_1, x in self.items]
        self.items.reverse()
        ai_list = self.items[0][2]
        if ai_list:
            path = random.choice(ai_list)
            assert os.path.exists(path)
            self.pixbuf = ninix.pix.create_pixbuf_from_file(path, is_pnr=0)
            self.pixbuf.saturate_and_pixelate(self.pixbuf, 1.0, True)
        else:
            self.pixbuf = None
        self.window.show()
        self.opened = 1

    def close(self, widget=None, event=None):
        self.window.hide()
        self.opened = 0
        return True

    def configure(self, darea, event):
        x, y, w, h = darea.get_allocation()
        self.size = (w, h)

    def redraw(self, darea, event):
        if not self.items:
            return # should not reach here
        total = float(0)
        for name, clock, path in self.items:
            total += clock
        # prepare gc
        layout = pango.Layout(darea.get_pango_context())
        font_desc = pango.FontDescription()
        font_desc.set_size(9 * pango.SCALE)
        font_desc.set_family('Sans') # FIXME
        layout.set_font_description(font_desc)
        white_gc = darea.get_style().white_gc
        black_gc = darea.get_style().black_gc
        cmap = darea.get_colormap()
        gray_gc = darea.window.new_gc()
        gray_gc.foreground = cmap.alloc_color('#cfcfcf')
        # redraw graph
        w, h = self.size
        darea.window.draw_rectangle(white_gc, True, 0, 0, w, h)
        # ai.png
        if self.pixbuf:
            darea.window.draw_pixbuf(None, self.pixbuf, 0, 0, 16, 32, -1, -1)
        w3 = w4 = 0
        rows = []
        for name, clock, path in self.items[:14]:
            layout.set_text(name)
            name_w, name_h = layout.get_pixel_size()
            rate = '%.1f%%' % (clock / total * 100)
            layout.set_text(rate)
            rate_w, rate_h = layout.get_pixel_size()
            w3 = max(rate_w, w3)
            time = '%d:%02d' % divmod(clock / 60, 60)
            layout.set_text(time)
            time_w, time_h = layout.get_pixel_size()
            w4 = max(time_w, w4)
            rows.append((clock, name, name_w, name_h, rate, rate_w, rate_h,
                         time, time_w, time_h))
        w1 = 280
        w2 = w - w1 - w3 - w4 - 70
        x = 20
        y = 15
        x += w1 + 10
        label = 'name'
        layout.set_text(label)
        label_name_w, label_name_h = layout.get_pixel_size()
        darea.window.draw_layout(gray_gc, x, y, layout)
        x = x + w2 + 10
        label = 'rate'
        layout.set_text(label)
        label_rate_w, label_rate_h = layout.get_pixel_size()
        darea.window.draw_layout(gray_gc, x + w3 - label_rate_w, y, layout)
        x += w3 + 10
        label = 'time'
        layout.set_text(label)
        label_time_w, label_time_h = layout.get_pixel_size()
        darea.window.draw_layout(gray_gc, x + w4 - label_time_w, y, layout)
        y += max([label_name_h, label_rate_h, label_time_h]) + 4
        for clock, name, name_w, name_h, rate, rate_w, rate_h, time, time_w, \
                time_h  in rows:
            x = 20
            bw = int(clock / total * w1)
            bh = max([name_h, rate_h, time_h]) - 1
            darea.window.draw_rectangle(gray_gc,  False, x + 1, y + 1, bw, bh)
            darea.window.draw_rectangle(white_gc, False, x, y, bw, bh)
            darea.window.draw_rectangle(black_gc, False, x, y, bw, bh)
            x += w1 + 10
            layout.set_text(name)
            end = len(name)
            while end > 0:
                w, h = layout.get_pixel_size()
                if w > 168:
                    end -= 1
                    layout.set_text(''.join((name[:end], u'...')))
                else:
                    break
            darea.window.draw_layout(black_gc, x, y, layout)
            x += w2 + 10
            layout.set_text(rate)
            darea.window.draw_layout(black_gc, x + w3 - rate_w, y, layout)
            x += w3 + 10
            layout.set_text(time)
            darea.window.draw_layout(black_gc, x + w4 - time_w, y, layout)
            y += max([name_h, rate_h, time_h]) + 4


if __name__ == '__main__':
    main()
