# -*- coding: ascii -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2006 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import os

if 'DISPLAY' in os.environ:
    import gtk
    import gobject
    import pango
    import cairo

import pix


class Balloon:

    def __init__(self, sakura, debug=0):
        self.sakura = sakura
        self.debug = debug
        self.synchronized = []
        self.__scalling = 0
        self.__scale = 100 # %
        self.__use_pna = 0
        self.__alpha_channel = None
        self.font_name = 'Sans'
        self.window = []
        # create communicatebox
        self.communicatebox = CommunicateBox(sakura, debug)
        # create teachbox
        self.teachbox = TeachBox(sakura, debug)
        # create inputbox
        self.inputbox = InputBox(sakura, debug)

    def get_text_count(self, side):
        if len(self.window) > side:
            return self.window[side].get_text_count()
        else:
            return 0

    def get_window(self, side):
        if len(self.window) > side:
            return self.window[side].window ## FIXME
        else:
            return None

    def redraw_arrow(self, side, direction):
        if len(self.window) > side:
            if direction == 0:
                self.window[side].redraw_arrow0()
            elif direction == 1:
                self.window[side].redraw_arrow1()

    def reset_text_count(self, side):
        if len(self.window) > side:
            self.window[side].reset_text_count()

    def set_use_pna(self, flag):
        if flag:
            self.__use_pna = 1
        else:
            self.__use_pna = 0
        for balloon_window in self.window:
            balloon_window.set_use_pna(flag)
            balloon_window.reset_balloon()

    def set_alpha_channel(self, alpha):
        if not 0.1 <= alpha <= 1.0 or alpha is None:
            alpha = 1.0
        self.__alpha_channel = alpha
        for balloon_window in self.window:
            balloon_window.set_alpha_channel(self.__alpha_channel)

    def set_scale(self, scale):
        self.__scale = scale # %
        for balloon_window in self.window:
            balloon_window.set_scale(scale)
            balloon_window.reset_balloon()

    def get_scalling(self):
        return self.__scalling

    def set_scalling(self, flag):
        self.__scalling = flag
        for balloon_window in self.window:
            balloon_window.set_scalling(flag)

    def create_gtk_window(self, title):
        window = gtk.Window()
        pix.set_rgba_colormap(window)
        window.set_title(title)
        window.set_decorated(False)
        window.set_resizable(False)
        window.set_skip_pager_hint(False)
        window.set_skip_taskbar_hint(True)
        window.set_focus_on_map(False)
        window.font_name = self.font_name
        window.connect('delete_event', self.delete)
        window.realize()
        return window

    def delete(self, window, event):
        self.sakura.ghost.finalize() ## FIXME
        return False

    def finalize(self):
        for balloon_window in self.window:
            balloon_window.destroy()
        self.window = []
        self.communicatebox.destroy()
        self.teachbox.destroy()
        self.inputbox.destroy()

    def new(self, desc, balloon):
        self.desc = desc
        balloon0 = {}
        balloon1 = {}
        communicate0 = None
        communicate1 = None
        communicate2 = None
        communicate3 = None
        for key, value in balloon.iteritems():
            if key in ['arrow0', 'arrow1']:
                balloon0[key] = value
                balloon1[key] = value
            elif key == 'sstp':
                balloon0[key] = value  # sstp marker
            elif key.startswith('s'):
                balloon0[key] = value  # Sakura
            elif key.startswith('k'):
                balloon1[key] = value  # Unyuu
            elif key == 'c0':
                communicate0 = value # send box
            elif key == 'c1':
                communicate1 = value # communicate box
            elif key == 'c2':
                communicate2 = value # teach box
            elif key == 'c3':
                communicate3 = value # input box
        self.balloon1 = balloon1 ## FIXME
        # create balloon windows
        for balloon_window in self.window:
            balloon_window.destroy()
        self.window = []
        for name, side, id_format, balloon in [('sakura', 0, 's%d', balloon0),
                                               ('kero', 1, 'k%d', balloon1)]:
            gtk_window = self.create_gtk_window(''.join(('balloon.', name)))
            balloon_window = BalloonWindow(
                gtk_window, side, self.sakura, desc, balloon,
                id_format, self.__use_pna, self.__alpha_channel, self.debug)
            self.window.append(balloon_window)
        for balloon_window in self.window:
            balloon_window.set_scalling(self.__scalling)
            balloon_window.set_scale(self.__scale)
        # configure communicatebox
        self.communicatebox.new(desc, communicate1)
        # configure teachbox
        self.teachbox.new(desc, communicate2)
        # configure inputbox
        self.inputbox.new(desc, communicate3)

    def add_window(self, side):
        assert side >= 2 and len(self.window) == side ## FIXME
        gtk_window = self.create_gtk_window('balloon.char%d' % side)
        id_format = 'k%d'
        balloon = self.balloon1 ## FIXME
        balloon_window = BalloonWindow(
            gtk_window, side, self.sakura, self.desc, balloon,
            id_format, self.__use_pna, self.__alpha_channel, self.debug)
        self.window.append(balloon_window)
        balloon_window.set_scale(self.__scale)
        balloon_window.set_balloon(0) ## FIXME

    def get_balloon_fonts(self):
        return self.font_name

    def set_balloon_fonts(self, font_name):
        if self.font_name == font_name:
            return
        self.font_name = font_name
        for window in self.window:
            window.window.font_name = font_name
            window.update_gc()
            window.redraw()

    def get_balloon_name(self):
        return self.desc.get('name', '')

    def get_balloon_size(self, side):
        if len(self.window) > side:
            return self.window[side].get_balloon_size()
        else:
            return (0, 0)

    def reset_balloon_all(self):
        for side in range(len(self.window)):
            self.window[side].reset_balloon()

    def reset_balloon(self, side):
        if len(self.window) > side:
            self.window[side].reset_balloon()

    def set_balloon_default(self):
        for side in range(len(self.window)):
            self.window[side].set_balloon(0) ## FIXME

    def set_balloon(self, side, num):
        if len(self.window) > side:
            self.window[side].set_balloon(num)

    def set_direction(self, side, dir):
        if len(self.window) > side:
            self.window[side].set_direction(dir)

    def set_position(self, side, x, y):
        if len(self.window) > side:
            self.window[side].set_position(x, y)

    def get_position(self, side):
        if len(self.window) > side:
            return self.window[side].get_position()
        else:
            return (0, 0)

    def is_shown(self, side):
        if len(self.window) > side:
            if self.window[side].is_shown():
                return 1
            else:
                return 0
        else:
            return 0

    def show(self, side):
        if len(self.window) > side:
            self.window[side].show()

    def hide_all(self):
        for side in range(len(self.window)):
            self.window[side].hide()

    def hide(self, side):
        if len(self.window) > side:
            self.window[side].hide()

    def raise_all(self):
        for side in range(len(self.window)):
            self.window[side].raise_()

    def raise_(self, side):
        if len(self.window) > side:
            self.window[side].raise_()

    def lower_all(self):
        for side in range(len(self.window)):
            self.window[side].lower()

    def lower(self, side):
        if len(self.window) > side:
            self.window[side].lower()

    def synchronize(self, list):
        self.synchronized = list

    def clear_text_all(self):
        for side in range(len(self.window)):
            self.clear_text(side)

    def clear_text(self, side):
        if self.synchronized:
            for side in self.synchronized:
                if len(self.window) > side:
                    self.window[side].clear_text()
        else:
            if len(self.window) > side:
                self.window[side].clear_text()

    def append_text(self, side, text):
        if self.synchronized:
            for side in self.synchronized:
                if len(self.window) > side:
                    self.window[side].append_text(text)
        else:
            if len(self.window) > side:
                self.window[side].append_text(text)

    def append_sstp_marker(self, side):
        if len(self.window) > side:
            self.window[side].append_sstp_marker()

    def append_link(self, side, label, value, newline_required=0):
        if self.synchronized:
            for side in self.synchronized:
                if len(self.window) > side:
                    self.window[side].append_link(label, value,
                                                  newline_required)
        else:
            if len(self.window) > side:
                self.window[side].append_link(label, value, newline_required)

    def append_image(self, side, path, x, y):
        if len(self.window) > side:
            self.window[side].append_image(path, x, y)

    def show_sstp_message(self, message, sender):
        self.window[0].show_sstp_message(message, sender)

    def hide_sstp_message(self):
        self.window[0].hide_sstp_message()

    def show_communicatebox(self):
        self.communicatebox.show()

    def show_teachbox(self):
        self.teachbox.show()

    def show_inputbox(self, symbol, limittime, default):
        self.inputbox.set_symbol(symbol)
        self.inputbox.set_limittime(limittime)
        self.inputbox.show(default)

class BalloonWindow:

    def __init__(self, window, side, sakura, desc, balloon,
                 id_format, use_pna, alpha, debug):
        self.window = window
        self.side = side
        self.sakura = sakura
        self.desc = desc
        self.balloon = balloon
        self.balloon_id = None
        self.id_format = id_format
        self.num = 0
        self.debug = debug
        self.__shown = 0
        self.sstp_marker = []
        self.sstp_region = None
        self.sstp_message = None
        self.images = []
        self.width = 0
        self.height = 0
        self.__scalling = 0
        self.__scale = 100 # %
        self.__use_pna = use_pna
        self.__alpha_channel = alpha
        self.text_count = 0
        # load files
        self.pixbuf = {}
        for key, (path, config) in balloon.iteritems():
            try:
                pixbuf = pix.create_pixbuf_from_file(path)
            except:
                continue
            self.pixbuf[key] = (pixbuf,
                                (pixbuf.get_width(), pixbuf.get_height()))
        # create drawing area
        self.darea = gtk.DrawingArea()
        self.darea.show()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK|
                              gtk.gdk.BUTTON_PRESS_MASK|
                              gtk.gdk.POINTER_MOTION_MASK|
                              gtk.gdk.POINTER_MOTION_HINT_MASK|
                              gtk.gdk.SCROLL_MASK)
        self.callbacks = []
        for signal, func in [('expose_event',        self.redraw),
                             ('button_press_event',  self.button_press),
                             ('motion_notify_event', self.motion_notify),
                             ('scroll_event',        self.scroll)]:
            self.callbacks.append(self.darea.connect(signal, func))
        self.window.add(self.darea)
        self.darea.realize()
        self.darea.window.set_back_pixmap(None, False)
        mask_r = desc.getint('maskcolor.r', 128)
        mask_g = desc.getint('maskcolor.g', 128)
        mask_b = desc.getint('maskcolor.b', 128)
        self.cursor_color = '#%02x%02x%02x' % (mask_r, mask_g, mask_b)
        text_r = desc.getint(['font.color.r', 'fontcolor.r'], 0)
        text_g = desc.getint(['font.color.g', 'fontcolor.g'], 0)
        text_b = desc.getint(['font.color.b', 'fontcolor.b'], 0)
        self.text_normal_color = '#%02x%02x%02x' % (text_r, text_g, text_b)
        if desc.getint('maskmethod') == 1:
            text_r = 255 - text_r
            text_g = 255 - text_g
            text_b = 255 - text_b
        self.text_active_color = '#%02x%02x%02x' % (text_r, text_g, text_b)
        # initialize
        self.direction = min(side, 1) ## kluge: multi chractor
        self.set_position(0, 0)
        self.clear_text()

    def reset_pixbuf_cache(self):
        self.pixbuf = {}
        for key, (path, config) in self.balloon.iteritems():
            try:
                pixbuf = pix.create_pixbuf_from_file(path,
                                                     use_pna=self.__use_pna)
            except:
                continue
            self.pixbuf[key] = (pixbuf,
                                (pixbuf.get_width(), pixbuf.get_height()))

    def set_use_pna(self, flag):
        if flag:
            self.__use_pna = 1
        else:
            self.__use_pna = 0
        self.reset_pixbuf_cache()

    def set_alpha_channel(self, value):
        if self.__alpha_channel == value:
            return
        self.__alpha_channel = value
        self.redraw()

    def get_scale(self):
        if self.__scalling:
            return self.__scale
        else:
            return 100 # %

    def set_scale(self, scale):
        self.__scale = scale # %

    def set_scalling(self, flag):
        self.__scalling = flag

    def update_gc(self):
        # colors
        cmap = self.darea.get_colormap()
        self.cursor_gc = self.darea.window.new_gc()
        self.cursor_gc.foreground = cmap.alloc_color(self.cursor_color)
        normal_gc = self.darea.window.new_gc()
        normal_gc.foreground = cmap.alloc_color(self.text_normal_color)
        active_gc = self.darea.window.new_gc()
        active_gc.foreground = cmap.alloc_color(self.text_active_color)
        self.text_gc = {}
        self.text_gc[gtk.STATE_NORMAL] = normal_gc
        self.text_gc[gtk.STATE_ACTIVE] = active_gc
        # arrow positions
        self.arrow = []
        w, h = self.pixbuf[self.balloon_id][1]
        x = self.config_adjust('arrow0.x', w, -10)
        y = self.config_adjust('arrow0.y', h,  10)
        self.arrow.append((x, y))
        x = self.config_adjust('arrow1.x', w, -10)
        y = self.config_adjust('arrow1.y', h, -20)
        self.arrow.append((x, y))
        # sstp marker position
        if self.side == 0:
            self.sstp = []
            x = self.config_adjust('sstpmarker.x', w,  30)
            y = self.config_adjust('sstpmarker.y', h, -20)
            self.sstp.append((x, y)) # sstp marker
            x = self.config_adjust('sstpmessage.x', w,  50)
            y = self.config_adjust('sstpmessage.y', h, -20)
            self.sstp.append((x, y)) # sstp message
        # arrow pixmaps and masks
        x, y = self.arrow[0]
        pixbuf = self.pixbuf['arrow0'][0]
        scale = self.get_scale()
        w = max(1, int(pixbuf.get_width() * scale / 100))
        h = max(1, int(pixbuf.get_height() * scale / 100))
        pixmap, mask = pixbuf.scale_simple(
            w, h, gtk.gdk.INTERP_BILINEAR).render_pixmap_and_mask(255)
        self.arrow0_pixmap = pixmap, mask, (w, h)
        self.arrow0_gc = self.new_mask_gc(mask, x, y)
        x, y = self.arrow[1]
        pixbuf = self.pixbuf['arrow1'][0]
        w = max(1, int(pixbuf.get_width() * scale / 100))
        h = max(1, int(pixbuf.get_height() * scale / 100))
        pixmap, mask = pixbuf.scale_simple(
            w, h, gtk.gdk.INTERP_BILINEAR).render_pixmap_and_mask(255)
        self.arrow1_pixmap = pixmap, mask, (w, h)
        self.arrow1_gc = self.new_mask_gc(mask, x, y)
        # sstp marker pixmap and mask
        if self.side == 0 and 'sstp' in self.pixbuf:
            x, y = self.sstp[0]
            pixbuf = self.pixbuf['sstp'][0]
            w = max(1, int(pixbuf.get_width() * scale / 100))
            h = max(1, int(pixbuf.get_height() * scale / 100))
            pixmap, mask = pixbuf.scale_simple(
                w, h, gtk.gdk.INTERP_BILINEAR).render_pixmap_and_mask(255)
            self.sstp_pixmap = pixmap, mask, (w, h)
            self.sstp_gc = self.new_mask_gc(mask, x, y)
        # font
        default_size = 12 # for Windows environment
        size = self.desc.getint(['font.height', 'font.size'], default_size)
        self.layout = pango.Layout(self.darea.get_pango_context())
        self.font_desc = pango.FontDescription(self.window.font_name)
        pango_size = self.font_desc.get_size()
        if pango_size == 0:
            pango_size = size * 3 / 4 # convert from Windows to GTK+
            pango_size *= pango.SCALE
        self.font_desc.set_size(pango_size * scale / 100)
        self.layout.set_font_description(self.font_desc)
        self.layout.set_wrap(pango.WRAP_CHAR)
        w, h = self.layout.get_pixel_size()
        self.font_height = h
        self.line_space = 1
        self.layout.set_spacing(self.line_space)
        # font for sstp message
        if self.side == 0:
            default_size = 10 # for Windows environment
            size = self.desc.getint('sstpmessage.font.height', default_size)
            self.sstp_layout = pango.Layout(self.darea.get_pango_context())
            self.sstp_font_desc = pango.FontDescription(self.window.font_name)
            pango_size = self.sstp_font_desc.get_size()
            if pango_size == 0:
                pango_size = size * 3 / 4 # convert from Windows to GTK+
            self.sstp_font_desc.set_size(pango_size * scale / 100)
            self.sstp_layout.set_font_description(self.sstp_font_desc)
            self.sstp_layout.set_wrap(pango.WRAP_CHAR)
            w, h = self.sstp_layout.get_pixel_size()
            sstp_font_height = h
        else:
            sstp_font_height = 0
        # font metrics
        origin_x = self.config_getint(
            'origin.x',
            self.config_getint('zeropoint.x',
                               self.config_getint('validrect.left', 14)))
        origin_y = self.config_getint(
            'origin.y',
            self.config_getint('zeropoint.y',
                               self.config_getint('validrect.top', 14)))
        wpx = self.config_getint(
            'wordwrappoint.x',
            self.config_getint('validrect.right', -14))
        if wpx > 0:
            line_width = wpx - origin_x
        elif wpx < 0:
            line_width = self.width - origin_x + wpx
        else:
            line_width = self.width - origin_x * 2
        wpy = self.config_getint('validrect.bottom', -14)
        if wpy > 0:
            text_height = min(wpy, self.height) - origin_y
        elif wpy < 0:
            text_height = self.height - origin_y + wpy
        else:
            text_height = self.height - origin_y * 2
        line_height = self.font_height + self.line_space
        self.lines = text_height / line_height
        self.line_regions = []
        y = origin_y
        for i in range(self.lines + 1):
            self.line_regions.append((origin_x, y, line_width, line_height))
            y = y + line_height
        self.line_width = line_width
        # sstp message region
        if self.side == 0:
            x, y = self.sstp[1]
            w = line_width + origin_x - x
            h = sstp_font_height
            self.sstp_region = (x, y, w, h)

    def update_line_regions(self, offset, new_y):
        origin_y = self.config_getint(
            'origin.y',
            self.config_getint('zeropoint.y',
                               self.config_getint('validrect.top', 14)))
        wpy = self.config_getint('validrect.bottom', -14)
        if wpy > 0:
            text_height = min(wpy, self.height) - origin_y
        elif wpy < 0:
            text_height = self.height - origin_y + wpy
        else:
            text_height = self.height - origin_y * 2
        line_height = self.font_height + self.line_space
        origin_x, y, line_width, line_height = self.line_regions[offset]
        self.lines = offset + (text_height - new_y) / line_height
        y = new_y
        for i in range(offset, len(self.line_regions)):
            self.line_regions[i] = (origin_x, y, line_width, line_height)
            y += line_height
        for i in range(len(self.line_regions), self.lines + 1):
            self.line_regions.append((origin_x, y, line_width, line_height))
            y += line_height

    def new_mask_gc(self, mask, x, y):
        mask_gc = self.darea.window.new_gc()
        mask_gc.set_clip_mask(mask)
        mask_gc.set_clip_origin(x, y)
        return mask_gc

    def get_balloon_size(self):
        return (self.width, self.height)

    def reset_balloon(self, reset_position=1):
        self.set_balloon(self.num, reset_position)

    def set_balloon(self, num, reset_position=1):
        self.num = num
        balloon_id = self.id_format % (num * 2 + self.direction)
        if balloon_id not in self.pixbuf:
            balloon_id = self.id_format % (0 + self.direction)
        self.balloon_id = balloon_id
        # change pixmap and window position
        x, y = self.position
        pixbuf, (w, h) = self.pixbuf[balloon_id]
        scale = self.get_scale()
        self.width  = max(1, int(w * scale / 100))
        self.height = max(1, int(h * scale / 100))
        self.darea.set_size_request(self.width, self.height)
        self.balloon_pixbuf = pixbuf.scale_simple(
            self.width, self.height, gtk.gdk.INTERP_BILINEAR)
        pixmap, mask = self.balloon_pixbuf.render_pixmap_and_mask(255)
        self.window.shape_combine_mask(mask, 0, 0)
        if self.__shown:
            self.update_gc()
        if reset_position:
            self.sakura.ghost.position_balloons() ## FIXME

    def set_direction(self, dir):
        if self.direction != dir:
            self.direction = dir # 0: left, 1: right
            self.reset_balloon(reset_position=0)

    def config_adjust(self, name, base, default_value):
        path, config = self.balloon[self.balloon_id]
        value = config.adjust(name, base)
        if value is None:
            value = self.desc.adjust(name, base)
            if value is None:
                if default_value < 0:
                    value = base + default_value
                else:
                    value = default_value
        return int(value * self.get_scale() / 100)

    def config_getint(self, name, default_value):
        path, config = self.balloon[self.balloon_id]
        value = config.getint(name)
        if value is None:
            value = self.desc.getint(name)
            if value is None:
                value = default_value
        return int(value * self.get_scale() / 100)

    def __move(self):
        x, y = self.get_position()
        self.window.move(x, y)

    def set_position(self, x, y):
        self.position = (x, y)
        if self.__shown:
            self.__move()

    def get_position(self):
        return self.position

    def destroy(self, finalize=0):
        for tag in self.callbacks:
            self.darea.disconnect(tag)
        self.window.remove(self.darea)
        self.darea.destroy()
        self.window.destroy()

    def is_shown(self):
        if self.__shown:
            return 1
        else:
            return 0

    def show(self):
        if not self.__shown:
            self.window.show()
            # make sure window is in its position
            self.__move()
            self.update_gc()
            self.raise_()
            self.__shown = 1

    def hide(self):
        if self.__shown:
            self.window.hide()
            self.__shown = 0
            self.images = []

    def raise_(self):
        if self.__shown:
            self.window.window.raise_()

    def lower(self):
        if self.__shown:
            self.window.window.lower()

    def show_sstp_message(self, message, sender):
        if self.sstp_region is None:
            self.show()
        self.sstp_message = '%s (%s)' % (message, sender)
        x, y, w, h = self.sstp_region
        self.sstp_layout.set_text(self.sstp_message)
        message_width, message_height = self.sstp_layout.get_pixel_size()
        if message_width > w:
            self.sstp_message = u'... (%s)' % sender
            i = 0
            while 1:
                i += 1
                s = '%s... (%s)' % (message[:i], sender)
                self.sstp_layout.set_text(s)
                message_width, message_height = \
                               self.sstp_layout.get_pixel_size()
                if message_width > w:
                    break
                self.sstp_message = s
        self.redraw_sstp_message()

    def hide_sstp_message(self):
        self.sstp_message = None
        self.redraw_sstp_message()

    def redraw_sstp_message(self):
        if not self.__shown:
            return
        # draw/erase sstp marker
        if 'sstp' in self.pixbuf:
            x, y = self.sstp[0]
            w, h = self.sstp_pixmap[2]
            if self.sstp_message:
                pixmap = self.sstp_pixmap[0]
                self.darea.window.draw_drawable(
                    self.sstp_gc, pixmap, 0, 0, x, y, w, h)
            else:
                self.redraw_area(x, y, w, h)
        # draw/erase sstp message
        x, y, w, h = self.sstp_region
        self.redraw_area(x, y, w, h)
        if self.sstp_message is None:
            return
        self.sstp_layout.set_text(self.sstp_message)
        self.darea.window.draw_layout(self.text_gc[gtk.STATE_NORMAL],
                                      x, y, self.sstp_layout)

    def redraw_arrow0(self, clear=1):
        x, y = self.arrow[0]
        pixmap, mask, (w, h) = self.arrow0_pixmap
        if self.lineno > 0:
            self.darea.window.draw_drawable(
                self.arrow0_gc, pixmap, 0, 0, x, y, w, h)
        elif clear:
            self.redraw_area(x, y, w, h)

    def redraw_arrow1(self, clear=1):
        x, y = self.arrow[1]
        pixmap, mask, (w, h) = self.arrow1_pixmap
        if self.lineno + self.lines < len(self.text_buffer) or \
           self.sakura.script_mode == self.sakura.PAUSE_MODE:
            self.darea.window.draw_drawable(
                self.arrow1_gc, pixmap, 0, 0, x, y, w, h)
        elif clear:
            self.redraw_area(x, y, w, h)

    def redraw_area(self, x, y ,w ,h):
        cr = self.darea.window.cairo_create()
        cr.rectangle(x, y, w, h)
        cr.clip()
        cr.set_source_pixbuf(self.balloon_pixbuf, 0, 0)
        cr.paint_with_alpha(self.__alpha_channel)

    def redraw(self, darea=None, event=None):
        if not self.__shown:
            return True
        if darea is None:
            darea = self.darea
        self.update_gc()
        cr = darea.window.cairo_create()
        cr.set_source_rgba(1.0, 1.0, 1.0, 0.0)
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.paint()
        cr.set_source_pixbuf(self.balloon_pixbuf, 0, 0)
        cr.paint_with_alpha(self.__alpha_channel)
        # draw foreground pixmap
        for i in range(len(self.images)):
            pixbuf, (w, h), (x, y) = self.images[i]
            scale = self.get_scale()
            w = max(1, int(w * scale / 100))
            h = max(1, int(h * scale / 100))
            if x == 'centerx':
                bw, bh = self.get_balloon_size()
                x = (bw - w) / 2
            else:
                try:
                    x = int(x)
                except:
                    continue
            if y == 'centery':
                bw, bh = self.get_balloon_size()
                y = (bh - h) / 2
            else:
                try:
                    y = int(y)
                except:
                    continue
            pixmap, mask = pixbuf.scale_simple(
                w, h, gtk.gdk.INTERP_BILINEAR).render_pixmap_and_mask(255)
            gc = pixmap.new_gc()
            gc.set_clip_mask(mask)
            gc.set_clip_origin(x, y)
            darea.window.draw_drawable(gc, pixmap, 0, 0, x, y, w, h)
        # draw text
        i = self.lineno
        j = len(self.text_buffer)
        line = 0
        while line < self.lines:
            if i >= j:
                break
            x, y, w, h = self.line_regions[line]
            if self.text_buffer[i].endswith('\n[half]'):
                new_y = int(y + (self.font_height + self.line_space) / 2)
                self.update_line_regions(line + 1, new_y)
                self.layout.set_text(self.text_buffer[i][:-7])
                darea.window.draw_layout(self.text_gc[gtk.STATE_NORMAL],
                                         x, y, self.layout)
            else:
                self.layout.set_text(self.text_buffer[i])
                darea.window.draw_layout(self.text_gc[gtk.STATE_NORMAL],
                                         x, y, self.layout)
            for l, c in self.sstp_marker:
                if l == i:
                    pixmap, mask, (mw, mh) = self.sstp_pixmap
                    self.layout.set_text(self.text_buffer[i][:c])
                    text_w, text_h = self.layout.get_pixel_size()
                    mx = x + text_w
                    my = y + (self.font_height + self.line_space) / 2
                    my = my - mh / 2
                    gc = self.new_mask_gc(mask, mx, my)
                    self.darea.window.draw_drawable(
                        gc, pixmap, 0, 0, mx, my, mw, mh)
            i += 1
            line += 1
        if self.side == 0 and self.sstp_message:
            self.redraw_sstp_message()
        if self.selection is not None:
            self.update_link_region(darea, self.selection)
        self.redraw_arrow0(clear=0)
        self.redraw_arrow1(clear=0)
        return False

    def update_link_region(self, darea, index, clear=0):
        sl = self.link_buffer[index][0]
        el = self.link_buffer[index][2]
        if self.lineno <= sl <= self.lineno + self.lines:
            sn = self.link_buffer[index][1]
            en = self.link_buffer[index][3]
            for n in range(sl, el + 1):
                if n - self.lineno >= len(self.line_regions):
                    break
                x, y, w, h = self.line_regions[n - self.lineno]
                if sl == el:
                    self.layout.set_text(self.text_buffer[n][:sn])
                    text_w, text_h =  self.layout.get_pixel_size()
                    x += text_w
                    self.layout.set_text(self.text_buffer[n][sn:en])
                    text_w, text_h =  self.layout.get_pixel_size()
                    w = text_w
                    start = sn
                    end = en
                elif n == sl:
                    self.layout.set_text(self.text_buffer[n][:sn])
                    text_w, text_h =  self.layout.get_pixel_size()
                    x += text_w
                    self.layout.set_text(self.text_buffer[n][sn:])
                    text_w, text_h =  self.layout.get_pixel_size()
                    w = text_w
                    start = sn
                    end = len(self.text_buffer[n])
                elif n == el:
                    self.layout.set_text(self.text_buffer[n][:en])
                    text_w, text_h =  self.layout.get_pixel_size()
                    w = text_w
                    start = 0
                    end = en
                else:
                    self.layout.set_text(self.text_buffer[n])
                    text_w, text_h =  self.layout.get_pixel_size()
                    w = text_w
                    start = 0
                    end = len(self.text_buffer[n])
                self.layout.set_text(self.text_buffer[n][start:end])
                self.redraw_area(x, y, w, h)
                if not clear:
                    darea.window.draw_rectangle(self.cursor_gc, True,
                                                x, y, w, h)
                    darea.window.draw_layout(self.text_gc[gtk.STATE_ACTIVE],
                                             x, y, self.layout)
                else:
                    darea.window.draw_layout(self.text_gc[gtk.STATE_NORMAL],
                                             x, y, self.layout)

    def check_link_region(self, px, py, clear=0):
        new_selection = None
        for i in range(len(self.link_buffer)):
            sl = self.link_buffer[i][0]
            el = self.link_buffer[i][2]
            if self.lineno <= sl <= self.lineno + self.lines:
                sn = self.link_buffer[i][1]
                en = self.link_buffer[i][3]
                for n in range(sl,el + 1):
                    if n - self.lineno >= len(self.line_regions):
                        break
                    x, y, w, h = self.line_regions[n - self.lineno]
                    if n == sl:
                        self.layout.set_text(self.text_buffer[n][:sn])
                        text_w, text_h =  self.layout.get_pixel_size()
                        x += text_w
                    if n == sl and n == el:
                        self.layout.set_text(self.text_buffer[n][sn:en])
                    elif n == el:
                        self.layout.set_text(self.text_buffer[n][:en])
                    else:
                        self.layout.set_text(self.text_buffer[n])
                    text_w, text_h =  self.layout.get_pixel_size()
                    w = text_w
                    if x <= px < x + w and y <= py < y + h:
                        new_selection = i
                        break
        if new_selection is not None:
            if self.selection != new_selection:
                sl, sn, el, en, link_id, raw_text, text = \
                    self.link_buffer[new_selection]
                self.sakura.notify_event(
                    'OnChoiceEnter', raw_text, link_id, self.selection)
        else:
            if self.selection is not None:
                self.sakura.notify_event('OnChoiceEnter')
        if new_selection == self.selection:
            return 0
        else:
            if clear and self.selection is not None:
                self.update_link_region(self.darea, self.selection, clear=1)
            self.selection = new_selection
            return 1

    def motion_notify(self, darea, event):
        if event.is_hint:
            x, y, state = self.darea.window.get_pointer()
        else:
            x, y, state = event.x, event.y, event.state
        if not self.link_buffer:
            return True
        px = int(x)
        py = int(y)
        if self.check_link_region(px, py, clear=1):
            if self.selection is not None:
                self.update_link_region(darea, self.selection)
        return True

    def scroll(self, darea, event):
        px = int(event.x)
        py = int(event.y)
        if event.direction == gtk.gdk.SCROLL_UP:
            if self.lineno > 0:
                self.lineno = max(self.lineno - 2, 0)
                self.check_link_region(px, py)
                self.redraw()
        elif event.direction == gtk.gdk.SCROLL_DOWN:
            if self.lineno + self.lines < len(self.text_buffer):
                self.lineno = min(self.lineno + 2,
                                  len(self.text_buffer) - self.lines)
                self.check_link_region(px, py)
                self.redraw()
        return True

    def button_press(self, darea, event):
        self.sakura.reset_idle_time()
        if event.type == gtk.gdk.BUTTON_PRESS:
            click = 1
        else:
            click = 2
        if self.sakura.script_mode == self.sakura.PAUSE_MODE:
            self.sakura.notify_balloon_click(event.button, click, self.side)
            return True
        # arrows
        px = int(event.x)
        py = int(event.y)
        # up arrow
        w, h = self.arrow0_pixmap[2]
        x, y = self.arrow[0]
        if x <= px <= x + w and y <= py <= y + h:
            if self.lineno > 0:
                self.lineno = max(self.lineno - 2, 0)
                self.redraw()
            return True
        # down arrow
        w, h = self.arrow1_pixmap[2]
        x, y = self.arrow[1]
        if x <= px <= x + w and y <= py <= y + h:
            if self.lineno + self.lines < len(self.text_buffer):
                self.lineno = min(self.lineno + 2,
                                  len(self.text_buffer) - self.lines)
                self.redraw()
            return True
        # links
        if self.selection is not None:
            sl, sn, el, en, link_id, raw_text, text = \
                self.link_buffer[self.selection]
            self.sakura.notify_link_selection(link_id, raw_text, self.selection)
            return True
        # balloon's background
        self.sakura.notify_balloon_click(event.button, click, self.side)
        return True

    def clear_text(self):
        self.selection = None
        self.lineno = 0
        self.text_buffer = []
        self.link_buffer = []
        self.newline_required = 0
        self.images = []
        self.sstp_marker = []
        self.redraw()

    def get_text_count(self):
        return self.text_count

    def reset_text_count(self):
        self.text_count = 0

    def append_text(self, text):
        if not self.text_buffer or self.newline_required:
            s = ''
            column = 0
            self.newline_required = 0
        else:
            s = self.text_buffer.pop(-1)
            column = len(s)
        i = len(s)
        text = ''.join((s, text))
        j = len(text)
        self.text_count += j
        p = 0
        while 1:
            if i >= j:
                self.text_buffer.append(text[p:i])
                self.draw_last_line(column)
                break
            if text[i] == '\n':
                if j >= i + 7 and text[i:i + 7] == '\n[half]':
                    self.text_buffer.append(''.join((text[p:i], '\n[half]')))
                    p = i = i + 7
                else:
                    self.text_buffer.append(text[p:i])
                    p = i = i + 1
                self.draw_last_line(column)
                column = 0
                continue
            n = i + 1
            if not self.__shown:
                self.show()
            self.layout.set_text(text[p:n])
            text_width, text_height =  self.layout.get_pixel_size()
            if text_width > self.line_width:
                self.text_buffer.append(text[p:i])
                self.draw_last_line(column)
                column = 0
                p = i
            i = n

    def append_sstp_marker(self):
        if 'sstp' not in self.pixbuf:
            return
        if not self.text_buffer:
            line = 0
            offset = 0
        else:
            line = len(self.text_buffer) - 1
            offset = len(self.text_buffer[-1])
        if self.newline_required:
            line = line + 1
            offset = 0
        self.sstp_marker.append((line, offset))
        w, h = self.pixbuf['sstp'][1]
        i = 1
        while 1:
            space = u'\u3000' * i ## FIXME
            self.layout.set_text(space)
            text_w, text_h = self.layout.get_pixel_size()
            if text_w > w:
                break
            else:
                i += 1
        self.append_text(space)
        self.draw_last_line(offset)

    def append_link(self, link_id, text, newline_required=0):
        if not text:
            return
        self.show()
        raw_text = text
        if not self.text_buffer or newline_required:
            offset = ''
            sl = len(self.text_buffer)
            sn = 0
        else:
            offset = self.text_buffer[-1]
            sl = len(self.text_buffer) - 1
            sn = len(self.text_buffer[-1])
        start = i = 0
        line_number = 0
        while 1:
            n = i + 1
            self.layout.set_text(''.join((offset, text[start:n])))
            text_w, text_h = self.layout.get_pixel_size()
            if text_w > self.line_width:
                if line_number == 0 and \
                   not newline_required and self.text_buffer:
                    self.text_buffer[-1] = ''.join(
                        (self.text_buffer[-1], text[start:i]))
                    self.draw_last_line()
                else:
                    self.text_buffer.append(text[start:i])
                line_number = line_number + 1
                start = i
                offset = ''
            i = n
            if i >= len(text):
                if line_number == 0 and \
                   not newline_required and self.text_buffer:
                    self.text_buffer[-1] = ''.join(
                        (self.text_buffer[-1], text[start:i]))
                else:
                    self.text_buffer.append(text[start:i])
                break
        el = sl + line_number
        en = len(self.text_buffer[-1])
        self.link_buffer.append((sl, sn, el, en, link_id, raw_text, text))
        self.draw_last_line()
        self.newline_required = newline_required

    def append_image(self, path, x, y):
        try:
            pixbuf = pix.create_pixbuf_from_file(path)
        except:
            return
        self.show()
        w = pixbuf.get_width()
        h = pixbuf.get_height()
        self.images.append((pixbuf, (w, h), (x, y)))
        self.redraw()

    def draw_last_line(self, column=0):
        if not self.__shown:
            return
        line = len(self.text_buffer) - 1
        if self.lineno <= line < self.lineno + self.lines:
            x, y, w, h = self.line_regions[line - self.lineno]
            if self.text_buffer[line].endswith('\n[half]'):
                offset = line - self.lineno + 1
                new_y = int(y + (self.font_height + self.line_space) / 2)
                self.update_line_regions(offset, new_y)
            else:
                self.darea.queue_draw_area(x, y, w, h)
            for l, c in self.sstp_marker:
                if l == line:
                    pixmap, mask, (mw, mh) = self.sstp_pixmap
                    self.layout.set_text(self.text_buffer[l][:c])
                    text_w, text_h = self.layout.get_pixel_size()
                    mx = x + text_w
                    my = y + (self.font_height + self.line_space) / 2
                    my = my - mh / 2
                    gc = self.new_mask_gc(mask, mx, my)
                    self.darea.window.draw_drawable(
                        gc, pixmap, 0, 0, mx, my, mw, mh)
        else:
            self.redraw()
            while line >= self.lineno + self.lines:
                self.lineno += 1
                self.redraw()

class CommunicateWindow:

    NAME = ''
    ENTRY = ''

    def __init__(self, sakura, debug):
        self.sakura = sakura
        self.debug = debug
        self.window = None
        self.dragged = 0

    def new(self, desc, balloon):
        if self.window:
            self.window.destroy()
        self.window = gtk.Window()
        self.window.set_title('communicate')
        self.window.set_decorated(False)
        self.window.set_resizable(False)
        self.window.connect('delete_event',         self.delete)
        self.window.connect('key_press_event',      self.key_press)
        self.window.connect('button_press_event',   self.button_press)
        self.window.set_events(gtk.gdk.BUTTON_PRESS_MASK)
        self.window.set_modal(True)
        self.window.set_position(gtk.WIN_POS_CENTER)
        self.window.realize()
        w = desc.getint('communicatebox.width', 250)
        h = desc.getint('communicatebox.height', -1)
        self.entry = gtk.Entry()
        self.entry.connect('activate', self.activate)
        self.entry.set_size_request(w, h)
        self.entry.show()
        image, mask = None, None
        if balloon:
            path, config = balloon
            # load pixmap
            try:
                image, mask = pix.create_pixmap_from_file(path)
            except:
                image, mask = None, None
        if image is not None:
            gtk_image = gtk.Image()
            gtk_image.set_alignment(0, 0)
            gtk_image.set_padding(0, 0)
            gtk_image.set_from_pixmap(image, mask)
            gtk_image.show()
            x = desc.getint('communicatebox.x', 10)
            y = desc.getint('communicatebox.y', 20)
            fixed = gtk.Fixed()
            fixed.put(gtk_image, 0, 0)
            fixed.put(self.entry, x, y)
            fixed.show()
            self.window.add(fixed)
            self.window.shape_combine_mask(mask, 0, 0)
        else:
            box = gtk.HBox(spacing=10)
            box.set_border_width(10)
            if self.ENTRY:
                label = gtk.Label(self.ENTRY)
                box.pack_start(label, False)
                label.show()
            box.pack_start(self.entry)
            self.window.add(box)
            box.show()

    def destroy(self):
        if self.window:
            self.window.destroy()
            self.window = None

    def delete(self, widget, event):
        self.window.hide()
        self.cancel()
        return True

    def key_press(self, widget, event):
        if event.keyval == gtk.keysyms.Escape:
            self.window.hide()
            self.cancel()
            return True
        return False

    def button_press(self, widget, event):
        if event.button in [1, 2]:
            self.window.begin_move_drag(
                event.button, int(event.x_root), int(event.y_root),
                gtk.get_current_event_time())
        return True

    def activate(self, widget):
        self.window.hide()
        self.enter()
        return True

    def show(self, default=''):
        self.entry.set_text(default)
        self.window.show()

    def enter(self):
        pass

    def cancel(self):
        pass

class CommunicateBox(CommunicateWindow):

    NAME = 'communicatebox'
    ENTRY = 'Communicate'

    def new(self, desc, balloon):
        CommunicateWindow.new(self, desc, balloon)
        self.window.set_modal(False)

    def delete(self, widget, event):
        self.window.hide()
        self.cancel()
        self.sakura.user_interaction = 0
        return True

    def key_press(self, widget, event):
        if event.keyval == gtk.keysyms.Escape:
            self.window.hide()
            self.cancel()
            self.sakura.user_interaction = 0
            return True
        return False

    def activate(self, widget):
        self.enter()
        self.entry.set_text('')
        return True

    def enter(self):
        self.send(self.entry.get_text())

    def cancel(self):
        self.send(None)

    def send(self, data):
        if data:
            data = unicode(data, 'utf-8')
        self.sakura.notify_user_communicate(data)

class TeachBox(CommunicateWindow):

    NAME = 'teachbox'
    ENTRY = 'Teach'

    def enter(self):
        self.send(self.entry.get_text())

    def cancel(self):
        self.send(None)

    def send(self, data):
        if data:
            data = unicode(data, 'utf-8')
        self.sakura.notify_user_teach(data)

class InputBox(CommunicateWindow):

    NAME = 'inputbox'
    ENTRY = 'Input'

    def new(self, desc, balloon):
        CommunicateWindow.new(self, desc, balloon)
        self.symbol = None
        self.limittime = -1

    def set_symbol(self, symbol):
        self.symbol = symbol
        
    def set_limittime(self, limittime):
        try:
            limittime = int(limittime)
        except ValueError:
            limittime = -1
        self.limittime = limittime

    def show(self, default):
        if default is not None:
            try:
                text = unicode(default).encode('utf-8')
            except:
                text = ''
        else:
            text = ''
        if int(self.limittime) < 0:
            self.timeout_id = None
        else:
            self.timeout_id = gobject.timeout_add(self.limittime, self.timeout)
        CommunicateWindow.show(self, text)

    def timeout(self):
        self.window.hide()
        self.send('timeout')

    def enter(self):
        self.send(self.entry.get_text())

    def cancel(self):
        self.send(None)

    def send(self, data):
        if self.timeout_id is not None:
            gobject.source_remove(self.timeout_id)            
        if data:
            data = unicode(data, 'utf-8')
        self.sakura.notify_user_input(self.symbol, data)


def test():
    pass

if __name__ == '__main__':
    test()
