/**********************************************************************
 * Copyright (C) 2015-2016 NTT DATA Corporation
 * This program is free software; you can redistribute it and/or
 * Modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation, version 2.
 * 
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *********************************************************************/

package com.clustercontrol.xcloud.ui.views;

import static com.clustercontrol.xcloud.common.CloudConstants.bundle_messages;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.activation.DataHandler;

import org.apache.log4j.Logger;
import org.eclipse.jface.layout.TreeColumnLayout;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;

import com.clustercontrol.ws.xcloud.BillingResult;
import com.clustercontrol.ws.xcloud.DataPoint;
import com.clustercontrol.ws.xcloud.FacilityBilling;
import com.clustercontrol.ws.xcloud.ResourceBilling;
import com.clustercontrol.xcloud.common.CloudStringConstants;


/**
 */
public class BillingDetailsView extends AbstractCloudViewPart implements CloudStringConstants {
	public static final String Id = BillingDetailsView.class.getName();
	
	private static Pattern pattern = Pattern.compile("^(\\d*\\.\\d*[1-9]|\\d*)0*$");

	public interface DataHolder {
		String getTitle();
		int getYear();
		int getMonth();
		com.clustercontrol.ws.xcloud.BillingResult getData();
		DataHandler getDataHandler();
	}
	
	public interface DataProvider {
		DataHolder getData(int year, int month);
	}

	public static class State {
		public int year;
		public int month;
		public DataHolder dataHolder;
		public BillingResult result;
		public BillingDetailRenderer renderer;
	}
	
	
	public static final int Prop_billing_detail_target = 0;
	
	private DataProvider dataProvider;
	private Label lblHeader;
	private Label lblFooter;
	private Composite base;
	private StackLayout layout;

	private Map<Object, State> rendererMap = new HashMap<>();
	private State currentState;
	private State nullState;
	
	private ISelectionProvider provider = new ISelectionProvider() {
		private List<ISelectionChangedListener> listeners = new ArrayList<ISelectionChangedListener>();

		ISelection theSelection = StructuredSelection.EMPTY;

		public void addSelectionChangedListener(ISelectionChangedListener listener) {
			listeners.add(listener);
		}

		public ISelection getSelection() {
			return theSelection;
		}

		public void removeSelectionChangedListener(ISelectionChangedListener listener) {
			listeners.remove(listener);
		}

		public void setSelection(ISelection selection) {
			theSelection = selection;
			SelectionChangedEvent e = new SelectionChangedEvent(this, selection);
			for (ISelectionChangedListener l: listeners) {
				l.selectionChanged(e);
			}
		}
	};
	
	public BillingDetailsView() {
 		super();
	}
	
	@Override
	protected void internalCreatePartControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);

		GridLayout gl_composite = new GridLayout(1, true);
		gl_composite.horizontalSpacing = 0;
		gl_composite.marginHeight = 0;
		gl_composite.marginWidth = 0;
		gl_composite.verticalSpacing = 0;
		composite.setLayout(gl_composite);
		
		lblHeader = new Label(composite, SWT.NONE);
		lblHeader.setSize(lblHeader.getSize().x, 80);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		lblHeader.setLayoutData(gridData);
		
		base = new Composite(composite, SWT.NONE);
		layout = new StackLayout();
		base.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		base.setLayoutData(gridData);

		lblFooter = new Label(composite, SWT.NONE);
		lblFooter.setAlignment(SWT.RIGHT);
		lblFooter.setSize(lblFooter.getSize().x, 80);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		lblFooter.setLayoutData(gridData);
		
		nullState = new State();
		nullState.renderer = new BillingDetailRenderer(base);
		
		currentState = nullState;
		updateControls(nullState);
		
		getSite().setSelectionProvider(provider);
	}
	
	@Override
	public void setFocus() {
		base.setFocus();
	}

	public void update() {
		if (currentState == nullState) return;

		// TODO
		// あとで更新実装しないと。
		
//		if (currentRenderer != null) {
//			currentRenderer.dispose();
//			currentRenderer = null;
//		}
//
//		BillingResult result = null;
//		if (currentDataHolder != null) {
//			result = currentDataHolder.getData();
//		}
	}
	
	private static class BillingDetailRenderer {
		private enum ViewColumn{
			facility_id(
				bundle_messages.getString("word.facility_id"),
				new ColumnPixelData(260, true, true),
				new ColumnLabelProvider(){
					@Override
					public String getText(Object element) {
						if(element instanceof FacilityBillingWrapper){
							FacilityBillingWrapper billing = (FacilityBillingWrapper)element;
							if ("others".equals(billing.getData().getFacilityId())) {
								return bundle_messages.getString("word.billing_others");
							}
							else {
								return billing.getData().getFacilityId();
							}
						}
						return null;
					}
				}
			),
			facility_name(
				bundle_messages.getString("word.facility_name"),
				new ColumnPixelData(160, true, true),
				new ColumnLabelProvider(){
					@Override
					public String getText(Object element) {
						if(element instanceof FacilityBillingWrapper){
							FacilityBillingWrapper billing = (FacilityBillingWrapper)element;
							if ("others".equals(billing.getData().getFacilityId())) {
								return "-";
							}
							else {
								return billing.getData().getFacilityName();
							}
						}
						return null;
					}
				}
			),
			account_resource(
				strCloudScopeId,
				new ColumnPixelData(100, true, true),
				new ColumnLabelProvider(){
					@Override
					public String getText(Object element) {
						if(element instanceof ResourceBillingWrapper){
							ResourceBillingWrapper billing = (ResourceBillingWrapper)element;
							return billing.getData().getCloudScopeId();
						}
						return null;
					}
				}
			),
			resource_id(
				bundle_messages.getString("word.billing_resource_id"),
				new ColumnPixelData(160, true, true),
				new ColumnLabelProvider(){
					@Override
					public String getText(Object element) {
						if(element instanceof ResourceBillingWrapper){
							ResourceBillingWrapper billing = (ResourceBillingWrapper)element;
							return billing.getData().getResourceId();
						}
						return null;
					}
				}
			),
			category(
				bundle_messages.getString("word.billing_category"),
				new ColumnPixelData(220, true, true),
				new ColumnLabelProvider(){
					@Override
					public String getText(Object element) {
						if(element instanceof ResourceBillingWrapper){
							ResourceBillingWrapper billing = (ResourceBillingWrapper)element;
							if (billing.getData().getCategoryDetail() != null) {
								return billing.getData().getCategory() + "(" + billing.getData().getCategoryDetail() + ")";
							}
							else {
								return billing.getData().getCategory();
							}
						}
						return null;
					}
				}
			),
			display_name(
				bundle_messages.getString("word.billing_display_name"),
				new ColumnPixelData(200, true, true),
				new ColumnLabelProvider(){
					@Override
					public String getText(Object element) {
						if(element instanceof ResourceBillingWrapper){
							ResourceBillingWrapper billing = (ResourceBillingWrapper)element;
							return billing.getData().getDisplayName();
						}
						return null;
					}
				}
			);

			private String label;
			private ColumnLabelProvider provider;
			private ColumnPixelData pixelData;
			
			ViewColumn(String label, ColumnPixelData pixelData, ColumnLabelProvider provider){
				this.label = label;
				this.pixelData = pixelData;
				this.provider = provider;
			}

			public String getLabel() {
				return label;
			}

			public ColumnPixelData getPixelData() {
				return pixelData;
			}

			public ColumnLabelProvider getProvider() {
				return provider;
			}
		}

		private static class TreeContentProvider implements ITreeContentProvider{
			public Object[] getChildren(Object parentElement) {
				if(parentElement instanceof FacilityBillingWrapper){
					FacilityBillingWrapper facility = (FacilityBillingWrapper)parentElement;
					return facility.getChildren().toArray();
				}
				return null;
			}

			public Object getParent(Object element) {
				if (element instanceof BillingResultWrapper) {
					return null;
				}
				else if (element instanceof FacilityBillingWrapper) {
					return ((FacilityBillingWrapper)element).getParent();
				}
				else if (element instanceof ResourceBillingWrapper) {
					return ((ResourceBillingWrapper)element).getParent();
				}
				return null;
			}

			public boolean hasChildren(Object element) {
				if (element instanceof BillingResultWrapper) {
					return !((BillingResultWrapper)element).getChildren().isEmpty();
				}
				else if (element instanceof FacilityBillingWrapper) {
					return !((FacilityBillingWrapper)element).getChildren().isEmpty();
				}
				return false;
			}

			public Object[] getElements(Object inputElement) {
				if (inputElement instanceof BillingResultWrapper) {
					return ((BillingResultWrapper)inputElement).getChildren().toArray();
				}
				return null;
			}

			public void dispose() {
			}

			public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			}
		}
		
		private static class CostLabelProvider extends ColumnLabelProvider {
			private int day;
			
			public CostLabelProvider(int day) {
				this.day = day;
			}
			
			@Override
			public String getText(Object element) {
				if(element instanceof FacilityBillingWrapper){
					FacilityBillingWrapper billing = (FacilityBillingWrapper)element;
					DataPoint p = billing.getPrice(day);
					
					if (p != null) {
						// とりあえず、AWS の 課金の csv が、8 桁になっています。
						String value = new BigDecimal(p.getPrice()).setScale(8, BigDecimal.ROUND_UP).toPlainString();
						Matcher m = pattern.matcher(value);
						return m.matches() ? m.group(1): ("0.00000000".equals(value) ? "0": value);
					}
					else {
						return "-";
					}
//					return p != null ? Double.toString(p.getPrice()): "-";
				}
				else if(element instanceof ResourceBillingWrapper){
					ResourceBillingWrapper billing = (ResourceBillingWrapper)element;
					DataPoint p = billing.getPrice(day);

					if (p != null) {
						// とりあえず、AWS の 課金の csv が、8 桁になっています。
						String value = new BigDecimal(p.getPrice()).setScale(8, BigDecimal.ROUND_UP).toPlainString();
						Matcher m = pattern.matcher(value);
						return m.matches() ? m.group(1): ("0.00000000".equals(value) ? "0": value);
					}
					else {
						return "-";
					}
//					return p != null ? Double.toString(p.getPrice()): "-";
				}
				return null;
			}
		}

		private static class BillingResultWrapper {
			private BillingResult data;
			private List<FacilityBillingWrapper> children;
			
			public BillingResultWrapper(BillingResult data) {
				this.data = data;
			}

//			public BillingResult getData() {
//				return data;
//			}

			public List<FacilityBillingWrapper> getChildren() {
				if (children == null) {
					children = new ArrayList<>();
					for (int i = data.getFacilities().size() - 1; i >= 0; --i) {
						if ("etc".equals(data.getFacilities().get(i).getFacilityId())) {
							children.add(new FacilityBillingWrapper(data, data.getFacilities().get(i)));
						}
						else {
							children.add(0, new FacilityBillingWrapper(data, data.getFacilities().get(i)));
						}
					}
				}
				return children;
			}
		}

		private static class FacilityBillingWrapper {
			private BillingResult parent;
			private FacilityBilling data;
			private List<ResourceBillingWrapper> children;
			private Map<Integer, DataPoint> pricesMap;
			
			public FacilityBillingWrapper(BillingResult parent, FacilityBilling data) {
				this.parent = parent;
				this.data = data;
			}
			
			public BillingResult getParent() {
				return parent;
			}

			public FacilityBilling getData() {
				return data;
			}

			public List<ResourceBillingWrapper> getChildren() {
				if (children == null) {
					int etcIndex = 0;
					children = new ArrayList<>();
					for (ResourceBilling r: data.getResources()) {
						if ("etc".equals(r.getCategory())) {
							children.add(new ResourceBillingWrapper(data, r));
						}
						else {
							children.add(etcIndex++, new ResourceBillingWrapper(data, r));
						}
					}
				}
				return children;
			}

			public DataPoint getPrice(int day) {
				if (pricesMap == null) {
					pricesMap = new HashMap<>();
					for (DataPoint price: data.getTotalsPerDate()) {
						pricesMap.put(price.getDay(), price);
					}
				}
				return pricesMap.get(day);
			}
		}
		
		private static class ResourceBillingWrapper {
			private FacilityBilling parent;
			private ResourceBilling data;
			private Map<Integer, DataPoint> pricesMap;
			
			public ResourceBillingWrapper(FacilityBilling parent, ResourceBilling data) {
				this.parent = parent;
				this.data = data;
			}
			
			public FacilityBilling getParent() {
				return parent;
			}

			public ResourceBilling getData() {
				return data;
			}
			
			public DataPoint getPrice(int day) {
				if (pricesMap == null) {
					pricesMap = new HashMap<>();
					for (DataPoint price: data.getPrices()) {
						pricesMap.put(price.getDay(), price);
					}
				}
				return pricesMap.get(day);
			}
		}
		
		private TreeViewer viwer;
		private Composite control;
		
		public BillingDetailRenderer(Composite base) {
			control = new Composite(base, SWT.NONE);

			TreeColumnLayout tcl = new TreeColumnLayout();
			control.setLayout(tcl);
			
			viwer = new TreeViewer(control, SWT.BORDER);
			Tree tree = viwer.getTree();
			tree.setLinesVisible(true);
			tree.setHeaderVisible(true);
			tree.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
			
			for (ViewColumn column: ViewColumn.values()) {
				TreeViewerColumn treeViewerColumn = new TreeViewerColumn(viwer, SWT.NONE);
				TreeColumn trclmnAaa = treeViewerColumn.getColumn();
				trclmnAaa.setText(column.getLabel());
				treeViewerColumn.setLabelProvider(column.getProvider());
				tcl.setColumnData(trclmnAaa, column.getPixelData());
			}
			base.layout();
		}
		
		public BillingDetailRenderer(Composite base, BillingResult result) {
			assert result != null;

			control = new Composite(base, SWT.NONE);

			TreeColumnLayout tcl = new TreeColumnLayout();
			control.setLayout(tcl);
			
			viwer = new TreeViewer(control, SWT.BORDER);
			Tree tree = viwer.getTree();
			tree.setLinesVisible(true);
			tree.setHeaderVisible(true);
			tree.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1));
			
			for (ViewColumn column: ViewColumn.values()) {
				TreeViewerColumn treeViewerColumn = new TreeViewerColumn(viwer, SWT.NONE);
				TreeColumn trclmnAaa = treeViewerColumn.getColumn();
				trclmnAaa.setText(column.getLabel());
				treeViewerColumn.setLabelProvider(column.getProvider());
				tcl.setColumnData(trclmnAaa, column.getPixelData());
			}
			
			Calendar cal = Calendar.getInstance();
			cal.set(result.getTargetYear(), result.getTargetMonth() - 1, 1);
			
			int maxDay = cal.getActualMaximum(Calendar.DAY_OF_MONTH);
			for (int i = 1; i <= maxDay; ++i) {
				TreeViewerColumn treeViewerColumn = new TreeViewerColumn(viwer, SWT.NONE);
				TreeColumn trclmnAaa = treeViewerColumn.getColumn();
				trclmnAaa.setText(Integer.toString(i));
				trclmnAaa.setAlignment(SWT.RIGHT);
				treeViewerColumn.setLabelProvider(new CostLabelProvider(i));
				tcl.setColumnData(trclmnAaa, new ColumnPixelData(80, true, true));
			}
			
			viwer.setContentProvider(new TreeContentProvider());
			viwer.setInput(new BillingResultWrapper(result));
		}
		
		public Control getControl() {
			return control;
		}
		
		public void dispose() {
			control.dispose();
		}
	}
	
	Logger getLogger() {
		return Logger.getLogger(BillingDetailsView.class);
	}
	
	public void update(DataProvider provider, int year, int month) {
		assert provider != null;
		
		setState(nullState);

		// 全ての情報を破棄。
		for (State s: rendererMap.values()) {
			if (s.renderer != nullState.renderer) {
				s.renderer.dispose();
			}
		}
		rendererMap.clear();
		
		this.dataProvider = provider;
		
		update(year, month);
	}
	
	public State getCurrentState() {
		return currentState == nullState ? null: currentState;
	}
	
	public void update(int year, int month) {
		Object key = Arrays.asList(year, month);
		State s = rendererMap.get(key);
		if (s == null) {
			s = new State();
			s.year = year;
			s.month = month;
			s.dataHolder = dataProvider.getData(year, month);
			s.result = s.dataHolder.getData();
			if (s.result == null) {
				s.renderer = nullState.renderer;
			}
			else {
				s.renderer = new BillingDetailRenderer(base, s.result);
			}
			rendererMap.put(key, s);
		}
		setState(s);
	}

	private void setState(State s) {
		if (s != currentState) {
			currentState = s;
			updateControls(currentState);
			
			firePropertyChange(Prop_billing_detail_target);
		}
	}

	private void updateControls(State s) {
		layout.topControl = s.renderer.getControl();

		if (s.result != null) {
			switch (s.dataHolder.getData().getType()) {
			case CLOUD_SCOPE:
				lblHeader.setText(String.format(bundle_messages.getString("word.billing_date_format_cloudscope"), s.year, s.month, s.result.getTargetId()));
				break;
			case FACILITY:
				lblHeader.setText(String.format(bundle_messages.getString("word.billing_date_format_facility"), s.year, s.month, s.result.getTargetId()));
				break;
			}
			lblFooter.setText(bundle_messages.getString("word.billing_unit") + " " + bundle_messages.getString("caption.title_separator") + " " + s.result.getUnit());
		}
		else {
			lblHeader.setText(bundle_messages.getString("word.billing_display_date") + " " + bundle_messages.getString("caption.title_separator") + " ");
			lblFooter.setText(bundle_messages.getString("word.billing_unit") + " " + bundle_messages.getString("caption.title_separator") + "    ");
		}

		base.layout();
	}
	
	public void nextMonth() {
		if (currentState == nullState) return;
		
		Calendar calendar = Calendar.getInstance();
		calendar.set(currentState.year, currentState.month - 1, 1);
		calendar.add(Calendar.MONTH, 1);
		
		update(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH) + 1);
	}
	
	public void previousMonth() {
		if (currentState == nullState) return;
		
		Calendar calendar = Calendar.getInstance();
		calendar.set(currentState.year, currentState.month - 1, 1);
		calendar.add(Calendar.MONTH, -1);
		
		update(calendar.get(Calendar.YEAR), calendar.get(Calendar.MONTH) + 1);
	}

	@Override
	protected StructuredViewer getViewer() {
		return currentState.renderer.viwer;
	}

	@Override
	public String getId() {
		return Id;
	}
}
