/*
Copyright (C) 2015 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.xcloud.ui.views;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.layout.TreeColumnLayout;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;

import com.clustercontrol.bean.FacilityImageConstant;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.util.FacilityTreeCache;
import com.clustercontrol.util.Messages;
import com.clustercontrol.xcloud.Activator;
import com.clustercontrol.xcloud.extensions.IModelContentProvider;
import com.clustercontrol.xcloud.extensions.ModelContentProviderExtension;
import com.clustercontrol.xcloud.model.base.ElementBaseModeWatch;
import com.clustercontrol.xcloud.model.base.IElement;
import com.clustercontrol.xcloud.model.cloud.ICloudScope;
import com.clustercontrol.xcloud.model.cloud.IHinemosManager;
import com.clustercontrol.xcloud.model.cloud.ILocation;
import com.clustercontrol.xcloud.model.repository.ICloudRepository;
import com.clustercontrol.xcloud.model.repository.ICloudScopeRootScope;
import com.clustercontrol.xcloud.model.repository.ICloudScopeScope;
import com.clustercontrol.xcloud.model.repository.IFacility;
import com.clustercontrol.xcloud.model.repository.IInstanceNode;
import com.clustercontrol.xcloud.model.repository.ILocationScope;
import com.clustercontrol.xcloud.model.repository.INode;
import com.clustercontrol.xcloud.model.repository.IScope;
import com.clustercontrol.xcloud.plugin.CloudOptionSourceProvider;
import com.clustercontrol.xcloud.util.CollectionComparator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;


public class RepositoryView extends AbstractCloudViewPart {
	public static final String Id = "com.clustercontrol.xcloud.ui.views.RepositoryView";
	
	protected Runnable refreshTask;

	protected IHinemosManager currentHinemosManager = null;
	protected ICloudScope currentScope = null;
	protected ILocation currentLocation = null;
	protected boolean initialized;

	protected class CloudRepositories {
		private List<ICloudRepository> repositories = new ArrayList<>();

		public List<ICloudRepository> getCloudRepositories() {
			return repositories;
		}

		public void update(boolean initialize) {
			List<IHinemosManager> managers = Activator.getDefault().getHinemosManagers();

			List<ICloudRepository> newRepositories = new ArrayList<>();
			for (IHinemosManager m: managers) {
				try {
					if (!initialize || (initialize && !m.isInitialized()))
						m.update();
					newRepositories.add(m.getCloudRepository());
				} catch (Exception e) {
					Logger.getLogger(this.getClass()).warn(e.getMessage(), e);
				}
			}
			
			CollectionComparator.compareCollection(repositories, newRepositories, new CollectionComparator.Comparator<ICloudRepository, ICloudRepository>() {
				@Override
				public boolean match(ICloudRepository o1, ICloudRepository o2) {
					return o1.getHinemosManager().getManagerName().equals(o2.getHinemosManager().getManagerName());
				}
				@Override
				public void afterO1(ICloudRepository o1) {
					o1.getHinemosManager().getModelWatch().removeWatcher(o1, watcher);
					repositories.remove(o1);
				}
				@Override
				public void afterO2(ICloudRepository o2) {
					o2.getHinemosManager().getModelWatch().addWatcher(o2, watcher);
					repositories.add(o2);
				}
			});
			
			if (refreshTask == null) {
				refreshTask = new Runnable() {
					@Override
					public void run() {
						try {
							refresh();
						} finally {
							refreshTask = null;
						}
					}
				};
				Display.getCurrent().asyncExec(refreshTask);
			}
		}
	}

	private class FacilityRootUpdateService {
		private com.clustercontrol.composite.FacilityTreeComposite cacheComposite;

		public FacilityRootUpdateService() {
			cacheComposite = new com.clustercontrol.composite.FacilityTreeComposite(rootCompo, SWT.None, null, null, false) {
				@Override
				public void update() {
					rootCompo.getDisplay().asyncExec(new Runnable() {
						@Override
						public void run() {
							RepositoryView.this.update();
						}
					});
				}
				@Override
				public boolean isDisposed () {
					return false;
				}
				@Override
				protected void checkWidget() {
				}
			};
			cacheComposite.dispose();
			FacilityTreeCache.addComposite(cacheComposite);
		}

		public void dispose() {
			FacilityTreeCache.delComposite(cacheComposite);
		}
	}

	private class FacilityTreeContentProvider implements ITreeContentProvider{
		public Object[] getChildren(Object element) {
			if (element instanceof ICloudRepository) {
				return ((ICloudRepository)element).getRootScopes();
			} if (element instanceof ICloudScopeRootScope) {
				ICloudScopeRootScope facility = (ICloudScopeRootScope)element;
				return facility.getFacilities();
			} else if (element instanceof IScope) {
				IScope scope = (IScope)element;
				String platformId = ((IScope)element).getCloudScopeScope().getCloudScope().getPlatformId();

				IModelContentProvider provider = ModelContentProviderExtension.getModelContentProvider(platformId);
				if (provider != null)
					return provider.getChildren(scope, (Object[])scope.getFacilities());

				return scope.getFacilities();
			}
			return null;
		}

		public Object getParent(Object element) {
			return null;
		}

		public boolean hasChildren(Object element) {
			if (element instanceof ICloudRepository) {
				return ((ICloudRepository)element).getRootScopes().length != 0;
			} else if (element instanceof ICloudScopeRootScope) {
				return ((ICloudScopeRootScope)element).getFacilities().length != 0;
			} else if (element instanceof IScope){
				IScope scope = (IScope)element;
				String platformId = ((IScope)element).getCloudScopeScope().getCloudScope().getPlatformId();

				IModelContentProvider provider = ModelContentProviderExtension.getModelContentProvider(platformId);
				if (provider != null)
					return provider.getChildren(scope, scope.getFacilities()).length != 0;

				return scope.getFacilities().length != 0;
			}
			return false;	
		}

		public Object[] getElements(Object inputElement) {
			if (inputElement instanceof CloudRepositories) {
				return ((CloudRepositories)inputElement).getCloudRepositories().toArray();
			}
			return new Object[]{};
		}

		public void dispose() {
		}

		@Override
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
	}

	private class FacilityLabelProvider extends LabelProvider {
		@Override
		public Image getImage(Object element) {
			if (element instanceof ICloudRepository) {
				return FacilityImageConstant.typeToImage(FacilityConstant.TYPE_COMPOSITE);
			} else if (element instanceof ICloudScopeRootScope) {
				return FacilityImageConstant.typeToImage(FacilityConstant.TYPE_SCOPE);
			} else if (element instanceof ICloudScopeScope) {
				Image defaultImage = Activator.getDefault().getImageRegistry().get("cloudscope");
				String platformId = ((IFacility)element).getCloudScopeScope().getCloudScope().getPlatformId();
				IModelContentProvider provider = ModelContentProviderExtension.getModelContentProvider(platformId);
				if (provider != null)
					return provider.getImage(element, defaultImage);
				return defaultImage;
			} else if (element instanceof ILocationScope) {
				Image defaultImage = Activator.getDefault().getImageRegistry().get("location");
				String platformId = ((IFacility)element).getCloudScopeScope().getCloudScope().getPlatformId();
				IModelContentProvider provider = ModelContentProviderExtension.getModelContentProvider(platformId);
				if (provider != null)
					return provider.getImage(element, defaultImage);
				return defaultImage;
			} else if (element instanceof IScope) {
				Image defaultImage = FacilityImageConstant.typeToImage(FacilityConstant.TYPE_SCOPE);
				String platformId = ((IFacility)element).getCloudScopeScope().getCloudScope().getPlatformId();
				IModelContentProvider provider = ModelContentProviderExtension.getModelContentProvider(platformId);
				if (provider != null)
					return provider.getImage(element, defaultImage);
				return defaultImage;
			} else if (element instanceof IInstanceNode) {
				IInstanceNode instanceNode = (IInstanceNode)element;

				Image defaultImage = Activator.getDefault().getImageRegistry().get("instance");
				if (!instanceNode.getInstance().getStatus().equals("running")) {
					defaultImage = Activator.getDefault().getImageRegistry().get("instance2");
				}
				
				String platformId = instanceNode.getCloudScopeScope().getCloudScope().getPlatformId();
				IModelContentProvider provider = ModelContentProviderExtension.getModelContentProvider(platformId);
				if (provider != null)
					return provider.getImage(element, defaultImage);
				return defaultImage;
			} else if (element instanceof INode) {
				Image defaultImage = FacilityImageConstant.typeToImage(FacilityConstant.TYPE_NODE);
				String platformId = ((IFacility)element).getCloudScopeScope().getCloudScope().getPlatformId();
				IModelContentProvider provider = ModelContentProviderExtension.getModelContentProvider(platformId);
				if (provider != null)
					return provider.getImage(element, defaultImage);
				return defaultImage;
			} else {
				return FacilityImageConstant.typeToImage(FacilityConstant.TYPE_NODE);
			}
		}
		@Override
		public String getText(Object element) {
			if (element instanceof ICloudRepository) {
				return ((ICloudRepository)element).getHinemosManager().getManagerName();
			} else if (element instanceof ICloudScopeRootScope) {
				IFacility facility = (IFacility)element;
				return facility.getName() + "(" + facility.getFacilityId() + ")";
			} else if (element instanceof IFacility) {
				IFacility facility = (IFacility)element;
				String platformId = facility.getCloudScopeScope().getCloudScope().getPlatformId();
				IModelContentProvider provider = ModelContentProviderExtension.getModelContentProvider(platformId);
				if (provider != null)
					return provider.getText(element, facility.getName());
				else
					return facility.getName() + "(" + facility.getFacilityId() + ")";
			}
			return element.toString();
		}
	}

	private class TreeViewerComparator extends ViewerComparator {

		/**
		 * Set sorting key by element type
		 * @param element
		 * @return
		 */
		private String getSortingKey(Object element){
			String key = null;
			if (element instanceof ICloudRepository)
				key = ((ICloudRepository)element).getHinemosManager().getManagerName();
			else if(element instanceof IFacility)
				key = ((IFacility)element).getName();
			return (null==key)? "" : key;
		}

		@Override
		public int compare(Viewer viewer, Object e1, Object e2) {
			return getSortingKey(e1).compareTo(getSortingKey(e2));
		}
	}

	protected ElementBaseModeWatch.AnyPropertyWatcher watcher = new ElementBaseModeWatch.AnyPropertyWatcher() {
		@Override
		public void elementAdded(ElementAddedEvent event) {
			refreshView();
		}

		@Override
		public void elementRemoved(ElementRemovedEvent event) {
			refreshView();
		}

		@Override
		public void propertyChanged(ValueChangedEvent event) {
			refreshView();
		}

		@Override
		public void unwatched(IElement owning, IElement owned) {
			refreshView();
		}
		
		public void refreshView() {
			if (refreshTask == null) {
				refreshTask = new Runnable() {
					@Override
					public void run() {
						try {
							refresh();
						} finally {
							refreshTask = null;
						}
					}
				};
				Display.getCurrent().asyncExec(refreshTask);
			}
		}
	};

	private FacilityRootUpdateService service;

	private CloudRepositories cloudRepositories = new CloudRepositories();
	private TreeViewer treeViewer;
	private Composite rootCompo;
	
	private List<List<String>> initPaths = Collections.emptyList();
	
	public RepositoryView() {
		super();
	}

	@Override
	protected void internalCreatePartControl(Composite parent) {
		rootCompo = new Composite(parent, SWT.NONE);
		rootCompo.setLayout(new FillLayout(SWT.HORIZONTAL));

		Composite composite_1 = new Composite(rootCompo, SWT.NONE);
		TreeColumnLayout tcl_composite = new TreeColumnLayout();
		composite_1.setLayout(tcl_composite);

		treeViewer = new TreeViewer(composite_1, SWT.MULTI | SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);

		treeViewer.setContentProvider(new FacilityTreeContentProvider());
		treeViewer.setLabelProvider(new FacilityLabelProvider());
		treeViewer.setInput(cloudRepositories);
		treeViewer.setComparator(new TreeViewerComparator());

		this.getSite().setSelectionProvider(treeViewer);

	    treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {	
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				IStructuredSelection sselection = (IStructuredSelection)event.getSelection();
				if (sselection.isEmpty()) {
					CloudOptionSourceProvider.setActiveCloudScopeToProvider(null);
					return;
				}
				
				currentHinemosManager = null;
				currentScope = null;
				currentLocation = null;
				
				Object selected = sselection.getFirstElement();
				if (selected instanceof IElement) {
					IElement scope = (IElement)selected;
					currentHinemosManager = (IHinemosManager)scope.getAdapter(IHinemosManager.class);
					currentScope = (ICloudScope)scope.getAdapter(ICloudScope.class);
					currentLocation = (ILocation)scope.getAdapter(ILocation.class);
				}
				
				CloudOptionSourceProvider.setActiveHinemosManagerToProvider(currentHinemosManager);
				CloudOptionSourceProvider.setActiveCloudScopeToProvider(currentScope);
				CloudOptionSourceProvider.setActiveLocationToProvider(currentLocation);
			}
		});
		
		try {
			cloudRepositories.update(true);
			
			List<TreePath> treePaths = new ArrayList<>();
			for (List<String> path: initPaths) {
				String manager = path.get(0);
				ICloudRepository repository = null;
				List<Object> segments = new ArrayList<>();
				for (ICloudRepository r: cloudRepositories.getCloudRepositories()) {
					if (r.getHinemosManager().getUrl().equals(manager)) {
						segments.add(r);
						repository = r;
						break;
					}
				}
				
				if (repository == null)
					continue;
				
				IFacility[] facilities = repository.getRootScopes();
				loop_end:
				for (int i = 1; i < path.size(); ++i) {
					String facilityId = path.get(i);
					for (IFacility f: facilities) {
						if (f.getFacilityId().equals(facilityId)) {
							segments.add(f);
							if (f instanceof IScope) {
								facilities = ((IScope)f).getFacilities();
								break;
							} else {
								break loop_end;
							}
						}
					}
				}
				treePaths.add(new TreePath(segments.toArray()));
			}
			treeViewer.setExpandedTreePaths(treePaths.toArray(new TreePath[treePaths.size()]));
		} catch(Exception e) {
			Logger logger = Logger.getLogger(this.getClass());
			logger.error(e.getMessage(), e);

			String message = e.getMessage();
			if (message == null) {
				ByteArrayOutputStream bos = new ByteArrayOutputStream();
				PrintStream ps = new PrintStream(bos, true);
				e.printStackTrace(ps);
				
				message = bos.toString();
			}
			
			// 失敗報告ダイアログを生成
			MessageDialog.openError(null, Messages.getString("failed"), message);
		}
	}

	@Override
	public void dispose() {
		for (ICloudRepository repository: cloudRepositories.getCloudRepositories()) {
			repository.getHinemosManager().getModelWatch().removeWatcher(repository, watcher);
		}
		
		if (service != null)
			service.dispose();
		
		getSite().setSelectionProvider(null);

		super.dispose();
	}

	@Override
	public void update() {
		cloudRepositories.update(false);
	}

	protected void refresh() {
		treeViewer.refresh();
		// Expand tree to level 3 (account)
		treeViewer.expandToLevel(3);
	}

	@Override
	protected StructuredViewer getViewer() {
		return treeViewer;
	}

	@Override
	public String getId() {
		return Id;
	}

	@Override
	public void setFocus() {
		super.setFocus();
		CloudOptionSourceProvider.setActiveHinemosManagerToProvider(currentHinemosManager);
		CloudOptionSourceProvider.setActiveCloudScopeToProvider(currentScope);
		CloudOptionSourceProvider.setActiveLocationToProvider(currentLocation);
		
		if (service == null)
			service = new FacilityRootUpdateService();
	}
	
	@Override
	public void init(IViewSite site, IMemento memento) throws PartInitException {
		super.init(site, memento);
		
		if (memento == null)
			return;
		
		String pathString = memento.getString("path");
		if (pathString != null) {
			ObjectMapper om = new ObjectMapper();
			ObjectReader or = om.reader(new TypeReference<List<List<String>>>(){});

			try {
				initPaths = or.readValue(pathString);
			} catch (IOException e) {
				Logger.getLogger(this.getClass()).warn(e.getMessage(), e);
			}
		}
	}

	@Override
	public void saveState(IMemento memento) {
		super.saveState(memento);
		TreePath[] path = treeViewer.getExpandedTreePaths();
		
		List<List<String>> pathList = new ArrayList<>();
		for (TreePath p: path) {
			List<String> pl = new ArrayList<>();
			for (int i = 0; i <  p.getSegmentCount(); ++i) {
				Object o = p.getSegment(i);
				if (o instanceof ICloudRepository) {
					ICloudRepository r = (ICloudRepository)o;
					pl.add(r.getHinemosManager().getUrl());
				} else if (o instanceof IScope) {
					IFacility f = (IFacility)o;
					pl.add(f.getFacilityId());
				} else if (o instanceof INode) {
					IFacility f = (IFacility)o;
					pl.add(f.getFacilityId());
				}
			}
			pathList.add(pl);
		}
		
		ObjectMapper om = new ObjectMapper();
		ObjectWriter ow = om.writer();
		try {
			String pathString = ow.writeValueAsString(pathList);
			memento.putString("path", pathString);
		} catch (JsonProcessingException e) {
			Logger.getLogger(this.getClass()).warn(e.getMessage(), e);
		}
	}
}