/*

 Copyright (C) 2011 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.agent.log;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.SendQueue;
import com.clustercontrol.agent.util.AgentProperties;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.ws.agent.OutputBasicInfo;
import com.clustercontrol.ws.monitor.LogfileCheckInfo;
import com.clustercontrol.ws.monitor.MonitorInfo;

/**
 * ログ転送スレッドを管理するクラス<BR>
 * 
 * 転送対象ログファイル情報を受け取り、ログ転送スレッドを制御します。
 * 
 */
public class LogfileMonitorManager {

	//ロガー
	private static Log m_log = LogFactory.getLog(LogfileMonitorManager.class);

	/** ファイルパスとファイルの読み込み状態を保持しているマップ */
	private static ConcurrentHashMap<String, LogfileMonitor> m_logfileMonitorCache =
			new ConcurrentHashMap<String, LogfileMonitor>();

	/** 読み込み中のファイル正規表現一覧。
	 * ここに含まれていないファイル正規表現　→　途中から読み始める。
	 * ここに含まれているファイル正規表現　→　新規ファイルが生成されたと見なし、最初から読む。
	 */
	private static ArrayList<String> monitoringFileList = new ArrayList<String>();
	
	/** 監視項目ごとのディレクトリの存在状態を保持しているマップ。
	 */
	private static ConcurrentHashMap<String, Boolean> directoryExistsMap =
			new ConcurrentHashMap<String, Boolean>();

	/** Queue送信  */
	private static SendQueue m_sendQueue;

	/** ログファイル監視間隔 */
	private static int m_runInterval = 10000; // 10sec

	/** 監視可能ファイル数 */
	private static int m_fileMax = 500;

	/** このエージェントの監視設定 */
	private static ArrayList<MonitorInfo> m_monitorList = new ArrayList<MonitorInfo>();

	/**
	 * コンストラクタ
	 * 
	 * @param ejbConnectionManager EJBコネクション管理
	 * @param sendQueue 監視管理Queue送信
	 * @param props ログ転送エージェントプロパティ
	 */
	public static void setSendQueue(SendQueue sendQueue) {
		m_sendQueue = sendQueue;
	}

	static {
		String key1 = "logfile.run.interval";
		try {
			String runIntervalStr = AgentProperties.getProperty(key1, Integer.toString(m_runInterval));
			m_runInterval = Integer.parseInt(runIntervalStr);
		} catch (Exception e) {
			m_log.warn("LogfileThread : " + e.getMessage());
		}
		String key2 = "logfile.file.max";
		try {
			String fileMaxStr = AgentProperties.getProperty(key2, Integer.toString(m_fileMax));
			m_fileMax = Integer.parseInt(fileMaxStr);
		} catch (Exception e) {
			m_log.warn("LogfileThread : " + e.getMessage());
		}
		m_log.info(key1 + "=" + m_runInterval + ", " + key2 + "=" + m_fileMax);
	}

	/**
	 * 監視設定をスレッドに反映します。<BR>
	 * 
	 * @param list 転送対象ログファイル情報一覧
	 */
	public static void setLogfileMonitor(ArrayList<MonitorInfo> monitorList) {
		String filename = "";
		ConcurrentHashMap<String, Boolean> tmpDirectoryExixstsMap = new ConcurrentHashMap<String, Boolean>();
		for (MonitorInfo info : monitorList) {
			LogfileCheckInfo check = info.getLogfileCheckInfo();
			filename += "[" + check.getDirectory() + "," + check.getFileName() + "]";
			
			// 監視設定の対象ディレクトリ存在チェック
			String directoryStr = check.getDirectory();
			File directory = new File(directoryStr);
			tmpDirectoryExixstsMap.put(check.getMonitorId(), directory.isDirectory());
			m_log.debug("setLogfileMonitor() : directoryExistsMap put monitorId = " + check.getMonitorId() + 
					", directoryStr = " + directoryStr +
					", exists = " + directory.isDirectory());
			
		}
		m_log.info("setLogfileMonitor() : m_monitorList=" + filename);

		directoryExistsMap = tmpDirectoryExixstsMap;
		m_monitorList = monitorList;
	}

	private static void refresh() {
		HashMap <String, ArrayList<MonitorInfo>> newMonitorMap =
				new HashMap<String, ArrayList<MonitorInfo>>();

		/*
		 * logfileMonitorはログファイルごとにオブジェクトが生成される。
		 * logfileMonitor.monitorInfoListに監視設定が登録される。
		 * (logfileMonitorとmonitorInfoは1対多の関係)
		 */
		/*
		 * 1. logfileMonitorを生成する。
		 */
		int fileN = 0;
		m_log.debug("refresh() : m_monitorList.size=" + m_monitorList.size());
		ArrayList<String> tmpKeyList = new ArrayList<String>();
		for (MonitorInfo monitorInfo : m_monitorList) {
			if (monitorInfo.getMonitorFlg() == ValidConstant.TYPE_INVALID) {
				continue;
			}
			String directoryStr = monitorInfo.getLogfileCheckInfo().getDirectory();
			String filenamePattern = monitorInfo.getLogfileCheckInfo().getFileName();
			m_log.debug("refresh() directory=" + directoryStr +
					", filenamePattern=" + filenamePattern);

			// 最初から読むか否か判定する。
			// 既存のファイル正規表現であり、新規ファイルの時は最初から読む。
			boolean readTopFlag = true;
			String key = directoryStr + "///" + filenamePattern;
			tmpKeyList.add(key);
			if (!monitoringFileList.contains(key)) {
				monitoringFileList.add(key);
				readTopFlag = false;
			}

			Pattern pattern = null;
			// 大文字・小文字を区別しない場合
			try {
				pattern = Pattern.compile(filenamePattern, Pattern.DOTALL | Pattern.CASE_INSENSITIVE);
				// 大文字・小文字を区別する場合
				// pattern = Pattern.compile(filenamePattern, Pattern.DOTALL);
			} catch (Exception e) {
				m_log.warn("refresh() : " + e.getMessage());
				continue;
			}

			File directory = new File(directoryStr);
			if (!directory.isDirectory()) {
				m_log.info("refresh() " + directoryStr + " is not directory");
				directoryExistsMap.put(monitorInfo.getMonitorId(), directory.isDirectory());
				continue;
			}
			else {
				// ディレクトリが現在存在しており、かつ前回のチェック時に存在していなかった場合は、
				// そのディレクトリ内のファイルは途中から読む（監視対象のフェイルオーバー対応）
				if(!directoryExistsMap.get(monitorInfo.getMonitorId())) {
					m_log.info("refresh() " + directoryStr + " is new found");
					readTopFlag = false;
				}
				directoryExistsMap.put(monitorInfo.getMonitorId(), directory.isDirectory());
			}
			
			for (File file : directory.listFiles()) {
				m_log.debug("refresh() file=" + file.getName());
				if (!file.isFile()) {
					m_log.debug(file.getName() + " is not file");
					continue;
				}
				Matcher matcher = pattern.matcher(file.getName());
				if(!matcher.matches()) {
					m_log.debug("refresh() don't match. filename=" + file.getName() + ", pattern=" + filenamePattern);
					continue;
				}
				fileN ++;
				if (fileN > m_fileMax) {
					m_log.warn("refresh() too many files for logfile. not-monitoring file=" + file.getName());
					continue;
				}
				String filePath = file.getAbsolutePath();
				LogfileMonitor logfileMonitor = m_logfileMonitorCache.get(filePath);

				if(logfileMonitor == null){
					// ファイル監視オブジェクトを生成。
					logfileMonitor = new LogfileMonitor(filePath, readTopFlag);
					m_logfileMonitorCache.put(filePath, logfileMonitor);
				}
				ArrayList<MonitorInfo> list = newMonitorMap.get(filePath);
				if (list == null){
					list = new ArrayList<MonitorInfo> ();
					newMonitorMap.put(filePath, list);
				}
				list.add(monitorInfo);
			}
		}

		/*
		 * 2. logfileMonitor.monitorInfoListを登録する。
		 */
		ArrayList<String> noMonitorFileList = new ArrayList<String>();
		for (String filePath : m_logfileMonitorCache.keySet()) {
			LogfileMonitor thread = m_logfileMonitorCache.get(filePath);
			ArrayList<MonitorInfo> list = newMonitorMap.get(filePath);
			thread.setMonitor(list);
			/*
			 * 監視設定が登録されていないファイルは閉じる。
			 */
			if (thread.clean()) {
				noMonitorFileList.add(filePath);
			}
		}
		for (String file : noMonitorFileList) {
			m_logfileMonitorCache.remove(file);
		}

		// monitoringFileListの整理。
		// すでに不要になったキーの削除。
		while (true) {
			boolean flag = true;
			for (String pattern : monitoringFileList) {
				if (!tmpKeyList.contains(pattern)) {
					monitoringFileList.remove(pattern);
					flag = false;
					break;
				}
			}
			if (flag) {
				break;
			}
		}
	}

	public static void start() {
		LogfileThread thread = new LogfileThread();
		thread.setName("LogFileMonitor start()");
		thread.start();
	}

	private static class LogfileThread extends Thread {
		@Override
		public void run() {
			m_log.info("run LogfileThread");
			while (true) {
				try {
					refresh();
					for (String filePath : m_logfileMonitorCache.keySet()) {
						LogfileMonitor logfileMonitor = m_logfileMonitorCache.get(filePath);
						logfileMonitor.run();
					}
				} catch (Exception e) {
					m_log.warn("LogfileThread : " + e.getClass().getCanonicalName() + ", " +
							e.getMessage(), e);
				} catch (Throwable e) {
					m_log.error("LogfileThread : " + e.getClass().getCanonicalName() + ", " +
							e.getMessage(), e);
				}
				try {
					Thread.sleep(m_runInterval);
				} catch (InterruptedException e) {
					m_log.info("LogfileThread is Interrupted");
					break;
				}
			}
		}
	}

	/**
	 * 監視管理のJMSに情報を通知します。<BR>
	 * 
	 * @param priority 重要度
	 * @param app アプリケーション
	 * @param msgId メッセージID
	 * @param msg メッセージ
	 * @param msgOrg オリジナルメッセージ
	 */
	public static void sendMessage(String filePath, int priority, String app, String msgId, String msg, String msgOrg, String monitorId) {
		// ログ出力情報
		OutputBasicInfo output = new OutputBasicInfo();
		output.setPluginId(HinemosModuleConstant.MONITOR_LOGFILE);
		output.setPriority(priority);
		output.setApplication(app);
		output.setMessageId(msgId);
		output.setMessage(msg);
		output.setMessageOrg(msgOrg);

		output.setGenerationDate(new Date().getTime());
		output.setMonitorId(monitorId);
		output.setFacilityId(""); // マネージャがセットする。
		output.setScopeText(""); // マネージャがセットする。

		m_sendQueue.put(output);
	}

	protected static int getRunInterval() {
		return m_runInterval;
	}
}
