package org.postgresforest;

import java.sql.*;
import java.util.*;
import java.util.concurrent.*;
import java.util.regex.Pattern;

import org.postgresforest.apibase.*;
import org.postgresforest.constant.ErrorStr;
import org.postgresforest.exception.ForestResourceDisposedException;
import org.postgresforest.util.*;

import net.jcip.annotations.*;

@NotThreadSafe public class ForestStatement implements Statement, ForestCloseable {
    
    protected final static Pattern selectSqlRegex =
        Pattern.compile("\\A\\s*[Ss][Ee][Ll][Ee][Cc][Tt]\\s");
    
    protected final static Pattern forUpdateSqlRegex =
        Pattern.compile("[Ff][Oo][Rr]\\s+[Uu][Pp][Dd][Aa][Tt][Ee]");
    
    protected final static Pattern commitSqlRegex =
        Pattern.compile("\\A\\s*[Cc][Oo][Mm][Mm][Ii][Tt]");
    
    protected final EntrypointCommonResource epCommonResource;
    
    /**
     * このForestStatementに紐づく各系へのStatement。
     * Statement#cancel()関数は単一スレッド以外からアクセスされる可能性が高いため
     * この変数はスレッドセーフ（特にスレッド間のメモリ可視性）を担保するために
     * CopyOnWriteArrayListを使用する
     */
    protected final CopyOnWriteArrayList<Statement> stmts;
    private ForestResultSet currentForestResultSet = null;
    protected void setForestResultSet(ForestResultSet newForestResultSet) {
        if (currentForestResultSet != null) {
            try { currentForestResultSet.close(); } catch (SQLException e) { }
        }
        currentForestResultSet = newForestResultSet;
    }
    private boolean isClosed = false;
    /** 
     * このオブジェクトが閉じられていれば例外を投げる 
     * @throws ForestResourceDisposedException このForestオブジェクトが閉じられている場合
     */
    protected void checkObjectClosed() throws ForestResourceDisposedException {
        if (isClosed == true) {
            throw new ForestResourceDisposedException(ErrorStr.RESOURCE_CLOSED.toString());
        }
    }
    
    /**
     * 系固定で実行するタイプのAPIを実行するサーバID。0/1を取る。
     * （-1の場合は結果取得の準備が整っていない、つまり、execute系を実行されていない）
     */
    private int fixExecuteId = -1;
    protected int getFixExecuteId() {
        return fixExecuteId;
    }
    protected void setFixExecuteId(int fixExecuteId) {
        this.fixExecuteId = fixExecuteId;
    }
    
    public ForestStatement(final EntrypointCommonResource epCommonResource, final List<? extends Statement> stmts) {
        this.epCommonResource = epCommonResource;
        // Statement（あるいはその派生クラス）は、cancel関数により並行アクセスされる
        // 可能性を含んでいる。そのため、CopyOnWriteArrayListに格納することで、
        // スレッド間のメモリ可視性を担保する
        this.stmts = new CopyOnWriteArrayList<Statement>(stmts);
    }
    
    public void addBatch(String sql) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void cancel() throws SQLException {
        // リソース全体がまだあるか否かのチェックは行う。これは、実行中のSQLが
        // あるにも関わらずStatementのキャンセルが呼ばれずにcloseされた場合など
        // closeが最後まで行ききれない場合などに備えるため。
        // closeが最後まで行ききれていない場合は、epCommonResource中のexecuteは
        // まだ実行可能なので、epCommonResourceに実行するか否かの判断を任せる
        final Callable<Void> task0 = new StatementTask.Cancel(stmts.get(0));
        final Callable<Void> task1 = new StatementTask.Cancel(stmts.get(1));
        epCommonResource.executeAllApiWithoutBroken(task0, task1);
    }
    
    public void clearBatch() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void clearWarnings() throws SQLException {
        checkObjectClosed();
        // 全系実行
        final Callable<Void> task0 = new StatementTask.ClearWarnings(stmts.get(0));
        final Callable<Void> task1 = new StatementTask.ClearWarnings(stmts.get(1));
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public boolean execute(String sql) throws SQLException {
        checkObjectClosed();
        // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        final Callable<Boolean> task0 = new StatementTask.Execute(stmts.get(0), sql);
        final Callable<Boolean> task1 = new StatementTask.Execute(stmts.get(1), sql);
        final Callable<Boolean> dummy = new StatementTask.Execute(null, null);
        final List<Boolean> result = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy);
        // 次のgetUpdateCountなどの固定系実行のAPIのために実行成功した系をセットする
        // 優先実行系で結果が取得できていればその系を次の実行系IDとし、そうでなければ
        // 逆のサーバID（0/1の2値なので1の補数を取ればよい）次の実行系とする
        final int prefServerId = epCommonResource.getPreferentialExecServerId();
        setFixExecuteId( (result.get(prefServerId) != null) ? prefServerId : (1 - prefServerId) );
        return result.get(getFixExecuteId()).booleanValue();
    }
    
    public boolean execute(String sql, int autoGeneratedKeys)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public boolean execute(String sql, String[] columnNames)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int[] executeBatch() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public ResultSet executeQuery(String sql) throws SQLException {
        checkObjectClosed();
        // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        // 全系実行か任意一系実行かの振り分けを行う（パース）
        final boolean isAllExec;
        switch (parseQuery(sql)) {
            case SELECT: {
                isAllExec = false;
                break;
            }
            case COMMIT: {
                epCommonResource.setIsTxBorder(true);
                isAllExec = true;
                break;
            }
            default: {
                isAllExec = true;
                break;
            }
        }
        
        final Callable<ResultSet> task0 = new StatementTask.ExecuteQuery(stmts.get(0), sql);
        final Callable<ResultSet> task1 = new StatementTask.ExecuteQuery(stmts.get(1), sql);
        final Callable<ResultSet> dummy = new StatementTask.ExecuteQuery(null, null);
        final List<ResultSet> results;
        if (isAllExec) {
            results = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy);
        } else {
            results = epCommonResource.executeAnyApiWithPreCheck(task0, task1, dummy);
        }
        
        // 優先実行系の結果が存在していればそちらを使い、結果がない場合は逆を使う。
        // ここで取得したResultSetを基にForestResultSetを作成し、固定実行系を定める。
        final int prefExecId = epCommonResource.getPreferentialExecServerId();
        setFixExecuteId( (results.get(prefExecId) != null) ? prefExecId : 1 - prefExecId );
        final ForestResultSet newResult = new ForestResultSet(epCommonResource, results, getFixExecuteId());
        setForestResultSet(newResult);
        return newResult;
    }
    
    public int executeUpdate(String sql) throws SQLException {
        checkObjectClosed();
        // 行儀の悪いアプリケーション（トランザクションを跨ってStatementを使いまわすアプリ）
        // のために、リカバリ最中で、かつトランザクションの境界であるならば、executeの
        // 実行直前でリカバリ完了を待ち、リソースの回復をする。この場合、このStatementは
        // 閉塞されているためエラーが返る
        epCommonResource.checkRecoveryAndRecoverObjects();
        
        // SQLがCOMMITなら、Txの境目であることを宣言する
        if (parseQuery(sql) == SQLType.COMMIT) {
            epCommonResource.setIsTxBorder(true);
        }
        
        final Callable<Integer> task0 = new StatementTask.ExecuteUpdate(stmts.get(0), sql);
        final Callable<Integer> task1 = new StatementTask.ExecuteUpdate(stmts.get(1), sql);
        final Callable<Integer> dummy = new StatementTask.ExecuteUpdate(null, null);
        final List<Integer> result = epCommonResource.executeAllApiWithPreCheck(task0, task1, dummy);
        
        // 次のgetUpdateCountなどの固定系実行のAPIのために実行系をセットする
        // 優先実行系で結果が取得できていればその系を次の実行系とし、そうでなければ
        // 逆のサーバID（0/1の2値なので1の補数を取ればよい）次の実行系とする
        final int prefServerId = epCommonResource.getPreferentialExecServerId();
        setFixExecuteId( (result.get(prefServerId) != null) ? prefServerId : (1 - prefServerId) );
        return result.get(getFixExecuteId());
    }
    
    public int executeUpdate(String sql, int autoGeneratedKeys)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int executeUpdate(String sql, int[] columnIndexes)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int executeUpdate(String sql, String[] columnNames)
            throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public Connection getConnection() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int getFetchDirection() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int getFetchSize() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public ResultSet getGeneratedKeys() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int getMaxFieldSize() throws SQLException {
        checkObjectClosed();
        // 任意系実行（その後の実行系に影響無し）
        final Callable<Integer> task0 = new StatementTask.GetMaxFieldSize(stmts.get(0));
        final Callable<Integer> task1 = new StatementTask.GetMaxFieldSize(stmts.get(1));
        final List<Integer> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get( (results.get(0) != null) ? 0 : 1 ).intValue();
    }
    
    public int getMaxRows() throws SQLException {
        checkObjectClosed();
        // 任意系実行（その後の実行系に影響無し）
        final Callable<Integer> task0 = new StatementTask.GetMaxRows(stmts.get(0));
        final Callable<Integer> task1 = new StatementTask.GetMaxRows(stmts.get(1));
        final List<Integer> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get( (results.get(0) != null) ? 0 : 1 ).intValue();
    }
    
    public boolean getMoreResults() throws SQLException {
        checkObjectClosed();
        // 固定系実行
        // executeを１回も実行されていないならfalseを返す
        if (getFixExecuteId() == -1) {
            return false;
        }
        final Callable<Boolean> task = new StatementTask.GetMoreResults(stmts.get(getFixExecuteId()));
        return epCommonResource.executeOneApi(task, getFixExecuteId()).booleanValue();
    }
    
    public boolean getMoreResults(int current) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int getQueryTimeout() throws SQLException {
        checkObjectClosed();
        // 任意系実行（その後の実行系に影響無し）
        final Callable<Integer> task0 = new StatementTask.GetQueryTimeout(stmts.get(0));
        final Callable<Integer> task1 = new StatementTask.GetQueryTimeout(stmts.get(1));
        final List<Integer> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get( (results.get(0) != null) ? 0 : 1 ).intValue();
    }
    
    public ResultSet getResultSet() throws SQLException {
        checkObjectClosed();
        // 固定系実行
        // executeを１回も実行されていないならnullを返す
        if (getFixExecuteId() == -1) {
            return null;
        }
        final Callable<ResultSet> task = new StatementTask.GetResultSet(stmts.get(getFixExecuteId()));
        final ResultSet newResultSet = epCommonResource.executeOneApi(task, getFixExecuteId());
        
        // 新しく取得したResultSetから、ForestResultSetオブジェクトを作り出し、
        // ForestStatementへのResultSetの登録を行う
        final List<ResultSet> resultSetList = new ArrayList<ResultSet>(2);
        resultSetList.add(null);
        resultSetList.add(null);
        resultSetList.set(getFixExecuteId(), newResultSet);
        final ForestResultSet newForestResultSet = new ForestResultSet(epCommonResource, resultSetList, getFixExecuteId());
        this.setForestResultSet(newForestResultSet);
        return newForestResultSet;
    }
    
    public int getResultSetConcurrency() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int getResultSetHoldability() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int getResultSetType() throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public int getUpdateCount() throws SQLException {
        checkObjectClosed();
        // 固定系実行
        // executeを１回も実行されていないなら-1を返す
        if (getFixExecuteId() == -1) {
            return -1;
        }
        final Callable<Integer> task = new StatementTask.GetUpdateCount(stmts.get(getFixExecuteId()));
        return epCommonResource.executeOneApi(task, getFixExecuteId());
    }
    
    public SQLWarning getWarnings() throws SQLException {
        checkObjectClosed();
        // 任意系実行（その後の実行系に影響無し）
        final Callable<SQLWarning> task0 = new StatementTask.GetWarnings(stmts.get(0));
        final Callable<SQLWarning> task1 = new StatementTask.GetWarnings(stmts.get(1));
        final List<SQLWarning> results = epCommonResource.executeAnyApi(task0, task1);
        return results.get( (results.get(0) != null) ? 0 : 1 );
    }
    
    public void setCursorName(String name) throws SQLException {
        checkObjectClosed();
        // 全系実行（その後の実行系に影響無し）
        final Callable<Void> task0 = new StatementTask.SetCursorName(stmts.get(0), name);
        final Callable<Void> task1 = new StatementTask.SetCursorName(stmts.get(1), name);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setEscapeProcessing(boolean enable) throws SQLException {
        checkObjectClosed();
        // 全系実行（その後の実行系に影響無し）
        final Callable<Void> task0 = new StatementTask.SetEscapeProcessing(stmts.get(0), enable);
        final Callable<Void> task1 = new StatementTask.SetEscapeProcessing(stmts.get(1), enable);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setFetchDirection(int direction) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setFetchSize(int rows) throws SQLException {
        throw new SQLException(org.postgresforest.constant.ErrorStr.NOT_SUPPORTED.toString());
    }
    
    public void setMaxFieldSize(int max) throws SQLException {
        checkObjectClosed();
        // 全系実行（その後の実行系に影響無し）
        final Callable<Void> task0 = new StatementTask.SetMaxFieldSize(stmts.get(0), max);
        final Callable<Void> task1 = new StatementTask.SetMaxFieldSize(stmts.get(1), max);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setMaxRows(int max) throws SQLException {
        checkObjectClosed();
        // 全系実行（その後の実行系に影響無し）
        final Callable<Void> task0 = new StatementTask.SetMaxRows(stmts.get(0), max);
        final Callable<Void> task1 = new StatementTask.SetMaxRows(stmts.get(1), max);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void setQueryTimeout(int seconds) throws SQLException {
        checkObjectClosed();
        // 全系実行（その後の実行系に影響無し）
        final Callable<Void> task0 = new StatementTask.SetQueryTimeout(stmts.get(0), seconds);
        final Callable<Void> task1 = new StatementTask.SetQueryTimeout(stmts.get(1), seconds);
        epCommonResource.executeAllApi(task0, task1);
        return;
    }
    
    public void close() throws SQLException {
        if (isClosed == false) {
            if (currentForestResultSet != null) {
                ((ForestCloseable) currentForestResultSet).close();
            }
            final Callable<Void> task0 = new StatementTask.Close(stmts.get(0));
            final Callable<Void> task1 = new StatementTask.Close(stmts.get(1));
            // closeは縮退対象としない。これは、close関数の中で仮に縮退などが発生すると
            // 内部から再度closeを呼ばれることになり、無限ループが発生するため
            epCommonResource.executeAllApiWithoutBroken(task0, task1);
        }
        isClosed = true;
    }
    
    public void closeOneSide(int serverId) {
        if (isClosed == false && currentForestResultSet != null) {
            ((ForestCloseable) currentForestResultSet).closeOneSide(serverId);
            // PostgreSQLのstatementは、nullをセットするものの、既存のものを閉塞はしない
            // これは既存のStatementのResultSetが、既に答えをDBから取り出し終わっている場合に
            // そのままデータを取り出し続けることができるようにするため
            stmts.set(serverId, null);
        }
    }
    
    /**
     * parseQueryにて出力されるSQLの種類
     */
    protected enum SQLType {
        /** FOR UPDATEの無いSELECT */
        SELECT,
        /** SELECT FOR UPDATE */
        SELECT_FORUP,
        /** COMMIT */
        COMMIT,
        /** SELECT/SELECT FOR UPDATE/COMMIT 以外のSQL */
        OTHER;
    }
    
    /**
     * SQLを解析して、SQLTypeにマッピングする
     * @param sql 解析対象のSQL
     * @return SQLに対応したSQLTypeの値
     */
    protected SQLType parseQuery(final String sql) {
        if (selectSqlRegex.matcher(sql).find()) {
            if (forUpdateSqlRegex.matcher(sql).find()) {
                return SQLType.SELECT_FORUP;
            } else {
                return SQLType.SELECT;
            }
        } else {
            if (commitSqlRegex.matcher(sql).find()) {
                return SQLType.COMMIT;
            } else {
                return SQLType.OTHER;
            }
        }
    }
}
