/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng;

import java.io.StringWriter;
import java.text.MessageFormat;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;

/**
 * syslog-ng監視プラグインクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class SyslogNGPlugin extends AbstractUIPlugin {

    /** プラグインのインスタンス。 */
    private static SyslogNGPlugin plugin = null;

    /**
     * syslog-ng監視プラグインのインスタンスを返します。
     * 
     * @return 自身のインスタンス
     */ 
    public static SyslogNGPlugin getDefault() {
        return plugin;
    }

    /**
     * syslog-ng監視プラグインのリソースバンドルから、
     * キーに該当する文字列を返します。
     * 
     * @param key キー
     * @return 文字列。リソースバンドルが無い場合、もしくは指定した文字列 が存在しない場合は、keyを返します。
     */
    public static String getResourceString(String key) {
        ResourceBundle bundle = getDefault().getResourceBundle();
        try {
            return (bundle != null) ? bundle.getString(key) : key;
        } catch (MissingResourceException e) {
            return key;
        }
    }

    /**
     * syslog-ng監視プラグインのリソースバンドルから、
     * キーに該当する文字列を返します。
     * 
     * @param key キー
     * @param arguments 置き換える文字列
     * @return 文字列。リソースバンドルが無い場合、もしくは指定した文字列 が存在しない場合は、keyを返します。
     * @see java.text.MessageFormat
     */
    public static String getResourceString(String key, Object[] arguments) {
        ResourceBundle bundle = getDefault().getResourceBundle();
        try {
            String message = (bundle != null) ? bundle.getString(key) : key;
            return MessageFormat.format(message, arguments);
        } catch (MissingResourceException e) {
            return key;
        }
    }

    /**
     * ログを出力します。
     * 
     * @param message ログメッセージ文字列
     * @param e 発生した例外
     */
    public static void log(String message, Exception e) {
        IStatus status = new Status(IStatus.ERROR, getPluginId(),
                IStatus.ERROR, message, e);
        getDefault().getLog().log(status);
    }

    /**
     * ログを出力します。
     * 
     * @param message ログメッセージ文字列
     * @see #log(String, Exception)
     */
    public static void log(String message) {
        log(message, null);
    }

    /**
     * ログを出力します。
     * 
     * @param e 発生した例外
     * @see #log(String, Exception)
     */
    public static void log(Exception e) {
        StringWriter stringWriter = new StringWriter();
        String message = stringWriter.getBuffer().toString();
        log(message, e);
    }

    /**
     * プラグインIDとして利用できる、固有のIDを返します。
     * 
     * @return プラグインID
     * @see Bundle#getSymbolicName()
     */
    public static String getPluginId() {
        return getDefault().getBundle().getSymbolicName();
    }

    /** リソースバンドルインスタンス。 */
    private ResourceBundle resourceBundle = null;


    /**
	 * コンストラクタ。<BR>
	 * リソースバンドルを取得します。
	 */
    public SyslogNGPlugin() {
        super();
        plugin = this;
        try {
            resourceBundle = ResourceBundle
                    .getBundle("com.clustercontrol.syslogng.SyslogNGPluginResources");
        } catch (MissingResourceException x) {
            resourceBundle = null;
        }
    }

    /**
     * プラグイン 開始。
     * プラグインの活性化の際に呼ばれます。
     * 
     * @param context BundleContextオブジェクト
     * @throws Exception
     * 
     * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
     */
    public void start(BundleContext context) throws Exception {
        super.start(context);
    }

    /**
     * プラグイン 停止。
     * プラグインの非活性化の際に呼ばれます。
     * 
     * @param context BundleContextオブジェクト
     * @throws Exception
     * 
     * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
     */
    public void stop(BundleContext context) throws Exception {
        super.stop(context);
    }

    /**
	 * syslog-ng監視プラグインのリソースバンドルを返します。
	 * 
	 * @return リソースバンドル
	 */
    public ResourceBundle getResourceBundle() {
        return resourceBundle;
    }
}