/*
                                                                                                
Copyright (C) 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.poller;

import java.lang.reflect.Constructor;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.ejb.FinderException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.impl.StdScheduler;

import com.clustercontrol.sharedtable.DataTable;
import com.clustercontrol.sharedtable.DataTableNotFoundException;
import com.clustercontrol.sharedtable.SharedTable;
import com.clustercontrol.vm.bean.VmMethodTypeConstant;
import com.clustercontrol.vm.ejb.entity.VmMethodMstLocal;
import com.clustercontrol.vm.ejb.entity.VmMethodMstPK;
import com.clustercontrol.vm.ejb.entity.VmMethodMstUtil;
import com.clustercontrol.vm.VmPollerImplInterface;
import com.clustercontrol.bean.PollerProtocolConstant;
import com.clustercontrol.poller.cfg.PollerConfig;
import com.clustercontrol.poller.cfg.SnmpPollerConfig;
import com.clustercontrol.poller.cfg.VmPollerConfig;
import com.clustercontrol.poller.cfg.WbemPollerConfig;
import com.clustercontrol.poller.impl.SnmpPollerImpl;
import com.clustercontrol.poller.impl.WbemPollerImpl;
import com.clustercontrol.poller.job.PollingJob;

public class PollingController {
	private static Log m_log = LogFactory.getLog( PollingController.class );
	
	private Object modifyLock = new Object();

	private final static String SHARED_TABLE_JNDI_NAME = "SharedTable";
	private final static String POLLER_MANAGER_JNDI_NAME = "PollerManager";
	
	private final static int RUNNING = 1;
	private final static int STOPPED = 2;
	
	private String m_quartzJndiName = "QuartzRAM";
	private String m_quartzJobName;  // ポーラ名とする
	private String m_quartzGroupName;  // ポーラグループとする
	
	private String m_pollerGroup;
	
	private String m_pollerName;
	
	private PollingControllerConfig m_pollingConfig;
	
	private String m_tableGroup;
	
	private String m_tableName;
	
	// ポーラの状態を示す
	volatile private int m_status;
	
	public PollingController(
			String pollerGroup, 
			String pollerName, 
			PollerConfig pollerConfig, 
			boolean indexCheckFlg,
			String tableGroup, 
			String tableName)
	throws NotInitializedException, DataTableNotFoundException{
		m_pollerGroup = pollerGroup;
		m_pollerName = pollerName;
		
		PollingControllerConfig pollingConfig = new PollingControllerConfig(pollerConfig, indexCheckFlg);
		
		m_pollingConfig = pollingConfig;
		m_tableGroup = tableGroup;
		m_tableName = tableName;

		m_quartzJobName = m_pollerName;
		m_quartzGroupName = m_pollerGroup;
		
		m_status = PollingController.STOPPED;
		
		try {
			// 書き込み対象テーブルが存在するか否かを確認する
			SharedTable sharedTable = lookupSharedTable();
			
			if(!sharedTable.containsDataTable(tableGroup, tableName)){
				// テーブルが存在しないため例外を投げる
				throw new DataTableNotFoundException(tableGroup, tableName);
			}
		} catch (NamingException e) {
			m_log.error("get SharedTable:" + e.getMessage());
			throw new NotInitializedException(
					"Table is not initialized. (" + tableName + " in " + tableGroup + " )");
		}
	}
	
	protected void scheduleJob(){
		// 収集間隔を取得する
		int interval = m_pollingConfig.getMinPollingInterval();

		// デバッグ出力
		m_log.debug("scheduleJob : " + interval);
		
		//QuartzのSchedulerをルックアップ
		Scheduler scheduler = null;
		try {
			InitialContext ctx = new InitialContext();
			Object obj = ctx.lookup(m_quartzJndiName);
			scheduler = (Scheduler)PortableRemoteObject.narrow(obj, StdScheduler.class);

		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
		
		// スケジューラの取得に失敗した場合は終了
		if(scheduler == null){
			return;
		}
		
		// 収集間隔が-1の場合は新規に登録しない
		if(interval == -1){
			try {
				// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
				scheduler.deleteJob(m_quartzJobName, m_quartzGroupName);

				// ステータス変更
				m_status = PollingController.STOPPED;
			} catch (SchedulerException e) {
				m_log.error(e.getMessage(), e);
			}
			return;
		}
		
		// 新規にジョブを生成
		JobDetail job = new JobDetail(m_quartzJobName, m_quartzGroupName, PollingJob.class);
		//ジョブ完了時に削除されないようにする
		job.setDurability(true);
		
		//JobDetailに変数を設定
		job.getJobDataMap().put("jndiName", POLLER_MANAGER_JNDI_NAME);
		job.getJobDataMap().put("pollerGroup", m_pollerGroup);
		job.getJobDataMap().put("pollerName", m_pollerName);
		
		
		//CronTriggerを作成
		CronTrigger cronTrigger = new CronTrigger(m_quartzJobName, m_quartzGroupName);
		
        //起動失敗した場合は、次の起動予定時刻をセットするように設定
        cronTrigger.setMisfireInstruction(CronTrigger.MISFIRE_INSTRUCTION_DO_NOTHING);
		
		try {
			String cronString = PollingInterval.parseCronExpression(interval);
			cronTrigger.setCronExpression(cronString);
		} catch (ParseException e) {
			// エラー処理
			m_log.error(e.getMessage(), e);
		}
		

		try {
			// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
			scheduler.deleteJob(m_quartzJobName, m_quartzGroupName);

			// ジョブを登録する
			scheduler.scheduleJob(job, cronTrigger);

			// ジョブを実行する
			scheduler.triggerJob(m_quartzJobName, m_quartzGroupName);
			
			// ステータス変更
			m_status = PollingController.RUNNING;
		} catch (SchedulerException e) {
			// エラー処理
			m_log.error(e.getMessage(), e);
		}
	}
	
	protected void polling(){
		try {
			//QuartzのSchedulerをルックアップ
			InitialContext ctx = new InitialContext();
			Object obj = ctx.lookup(m_quartzJndiName);
			Scheduler scheduler = (Scheduler)PortableRemoteObject.narrow(obj, StdScheduler.class);

			// ジョブを実行する
			scheduler.triggerJob(m_quartzJobName, m_quartzGroupName);
			
			// ステータス変更
			m_status = PollingController.RUNNING;
		} catch (SchedulerException e) {
			// エラー処理
			m_log.error(e.getMessage(), e);
		} catch (NamingException e) {
			// エラー処理
			m_log.error(e.getMessage(), e);
		}
	}
	
	/**
	 * ポーリングを開始する
	 * @param collectorName 収集名
	 * @param interval 収集間隔
	 * @param maps 収集プロトコルとポーリング対象のマップ
	 * @throws CollectorAlreadyExistException 
	 * @throws NamingException 
	 */
	public void startPolling(
			String collectorName,
			int interval, 
			HashMap<String, List<String>> maps)
	throws CollectorAlreadyExistException{
		synchronized (modifyLock) {
			// 指定の収集名が既に登録されているかどうかを確認する
			if(m_pollingConfig.containsCollectorName(collectorName)){
				// 既に収集名が設定されているため例外を投げる
				throw new CollectorAlreadyExistException(collectorName);
			}

			try {
				// 書き込み対象テーブルが存在するか否かを確認する
				SharedTable stable = lookupSharedTable();

				if(!stable.containsCollectorName(m_tableGroup, m_tableName, collectorName)){
					// ない場合は新規に作成する
					stable.registerCollector(m_tableGroup, m_tableName, collectorName, interval);
				}
			} catch (DataTableNotFoundException e) {
				m_log.error(e.getMessage(), e);
				return;
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				return;
			}
			
			// 指定の収集名で、収集間隔、収集プロトコルとポーリング対象のマップを登録する
			m_pollingConfig.addPollingTargets(collectorName, interval, maps);

			// ポーリングのスケジュールを設定
			scheduleJob();
		}
	}

	/**
	 * ポーリングを終了する
	 * @param collectorName 収集名
	 * @throws FacilityNotFoundException
	 */
	public void stopPolling(String collectorName){
		synchronized (modifyLock) {
			m_log.debug("stop polling : " + m_pollerGroup + ", " + m_pollerName + ", " + collectorName);

			// 指定の収集名で、収集間隔、ポーリング対象のリストを削除する
			boolean modifyFlg = m_pollingConfig.removePollingTargets(collectorName);

			// 最小収集間隔が変更されている場合はスケジューリング
			if(modifyFlg){
				scheduleJob();
			}

			try {
				// 書き込み対象テーブルの登録情報を削除する
				SharedTable stable = lookupSharedTable();

				stable.unregisterCollector(m_tableGroup, m_tableName, collectorName);
			} catch (DataTableNotFoundException e) {
				m_log.error(e.getMessage(), e);
				return;
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				return;
			}
		}
	}
	
	/**
	 * 全ての収集名のポーリングを終了する
	 * @throws FacilityNotFoundException
	 */
	public void stopPollingAll(){
		synchronized (modifyLock) {
			m_log.debug("stop polling : " + m_pollerGroup + ", " + m_pollerName);

			// 全ての収集間隔、ポーリング対象のリストを削除する
			boolean modifyFlg = m_pollingConfig.removePollingAllTargets();

			// 最小収集間隔が変更されている場合はスケジューリング
			if(modifyFlg){
				scheduleJob();
			}

			try {
				// 書き込み対象テーブルを削除する
				SharedTable sharedtable = lookupSharedTable();
				if(sharedtable != null){
					sharedtable.removeDataTable(m_tableGroup, m_tableName);
				}
			} catch (NamingException e) {
				m_log.error(e.getMessage(), e);
				return;
			}
		}
	}	
	
	/**
	 * ポーリングを実行する
	 */
	public void run() {
		// デバッグログ出力
		m_log.debug("run start : " + m_pollerGroup + "  " + m_pollerName);
	
		SharedTable sharedTable = null;
		// 収集値格納用テーブルをルックアップ
		try {
			// 共有テーブルをルックアップして取得する
			sharedTable = lookupSharedTable();
			if(sharedTable == null){
				// 書き込み対象テーブルを取得できないため終了する
				return;
			}
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		}
		
		try {
			/* ポーリングしても意味のないポーリング対象がないかチェックする */

			/* ポーラが管理している収集名のうち、書き込み対象である共有テーブルでは
			 * 既に参照対象でなくなっている収集名を特定する */
			
			// 共有テーブルで参照される収集名のセットを取得
			Set<String> retainCollectorNames = 
				new HashSet<String>(sharedTable.getCollectorNames(m_tableGroup, m_tableName));

			// ポーラで保持している収集名のセットを取得
			Set<String> pollerCollectorNames = m_pollingConfig.getCollectorNames();
			
			// 共通要素を抽出
			retainCollectorNames.retainAll(pollerCollectorNames);
			
			// 削除対象の収集名
			Set<String> removeCollectoName = 
				new HashSet<String>(m_pollingConfig.getCollectorNames());
			removeCollectoName.removeAll(retainCollectorNames);
			
			synchronized (modifyLock) {
				Iterator<String> itr = removeCollectoName.iterator();
				
				boolean modifyFlg = false;
				while(itr.hasNext()){
					String collectorName = itr.next();
					// 指定の収集名で、収集間隔、ポーリング対象のリストを削除する
					modifyFlg = m_pollingConfig.removePollingTargets(collectorName);
				}

				// 最小収集間隔が変更されている場合はスケジューリング
				if(modifyFlg){
					scheduleJob();
				}
			}
		} catch (DataTableNotFoundException e) {
			m_log.error(e.getMessage(), e);
			// 書き込み対象テーブルを取得できないため終了する
			return;
		}

		// 共有テーブルが保持しているデータホルダのうち有効である収集間隔のセットを取得
		Set<Integer> holderIntervals;
		try {
			holderIntervals = sharedTable.getIntervals(m_tableGroup, m_tableName);
		} catch (DataTableNotFoundException e) {
			m_log.error(e.getMessage(), e);
			// 書き込み対象テーブルを取得できないため終了する
			return;
		}
	
		// 更新すべきテーブルがない場合は処理終了
		if(holderIntervals.size() <= 0){
			return;
		}		
	
		// 現在時刻を取得
		long now = System.currentTimeMillis();

		// 今のタイミングで更新すべきテーブルを特定するため、
		// 更新の必要のある収集間隔のリストを取得する
		List<Integer> intervals = m_pollingConfig.getCurrentRefreshIntervals(now);
	
		ArrayList<Integer> tmpList = new ArrayList<Integer>(intervals);
	
		// 全エンティティ内を調べるループをまわすためテンポラリのリストを作成
		Iterator<Integer> itr = tmpList.iterator();
		while(itr.hasNext()){
			int interval = itr.next();
			// 存在しない場合は収集対象から削除する
			if(!holderIntervals.contains(interval)){
				m_log.debug("Remove : " + interval);
				intervals.remove(new Integer(interval));
			}
		}
	
		// 更新すべきテーブルがない場合は処理終了
		if(intervals.size() <= 0){
			return;
		}
	
		// ポーリング対象の中から今のタイミングで収集すべきものを抽出する
		HashMap<String, List<String>> pollingTargetMap = m_pollingConfig.getCurrentTargetMap(now);
	
		m_log.debug("run() pollingTargetMap : " + pollingTargetMap);
		// 収集すべきポーリング対象がない場合は処理終了
		if(pollingTargetMap.size() <= 0){
			m_log.debug("polling targets are nothing.");
			return;
		}
		
		PollerConfig pollerConfig = m_pollingConfig.getPollerConfig();
		
		
		// ポーリングを行い値を収集
		DataTable dataTable = new DataTable();
		
		// *****************************
		// SNMPのpollerの設定
		// *****************************
		if(pollingTargetMap.get(PollerProtocolConstant.PROTOCOL_SNMP) != null) {
			SnmpPollerConfig snmpConfig = pollerConfig.getSnmpConfig();
			SnmpPollerImpl poller = new SnmpPollerImpl();
			poller.polling(
					pollerConfig.getAddress(), 
					snmpConfig.getPort(),
					snmpConfig.getVersion(),
					snmpConfig.getComunity(),
					snmpConfig.getRetries(),
					snmpConfig.getTimeout(),
					pollingTargetMap.get(PollerProtocolConstant.PROTOCOL_SNMP), 
					m_pollingConfig.isIndexCheckFlg(),
					dataTable);
		}
		
		// *****************************
		// WBEMのpollerの設定
		// *****************************
		if(pollingTargetMap.get(PollerProtocolConstant.PROTOCOL_WBEM) != null) {
			WbemPollerConfig wbemConfig = pollerConfig.getWbemConfig();
			WbemPollerImpl poller = new WbemPollerImpl();
			poller.polling(
					pollerConfig.getAddress(), 
					wbemConfig.getPort(),
					wbemConfig.getProtocol(),
					wbemConfig.getUserName(),
					wbemConfig.getPassword(),
					wbemConfig.getNameSpace(),
					wbemConfig.getRetries(),
					wbemConfig.getTimeout(),
					pollingTargetMap.get(PollerProtocolConstant.PROTOCOL_WBEM),
					dataTable);
		}
		
		// *****************************
		// 仮想化のpollerの設定
		// *****************************
		Iterator<String> pollerProtocolItr = pollingTargetMap.keySet().iterator();
		while (pollerProtocolItr.hasNext()) {
			String pollerProtocol = pollerProtocolItr.next();
			m_log.debug("pollerProtocol : " + pollerProtocol);
			if (pollerProtocol.equals(PollerProtocolConstant.PROTOCOL_SNMP) ||
					pollerProtocol.equals(PollerProtocolConstant.PROTOCOL_WBEM)) {
				continue;
			}
			VmPollerConfig vmConfig = pollerConfig.getVmConfig();
			VmMethodMstLocal local = null;
			try {
				local = VmMethodMstUtil.getLocalHome().findByPrimaryKey(new VmMethodMstPK(
					pollerProtocol, VmMethodTypeConstant.POLLERIMPL));
			} catch (FinderException e) {
				e.printStackTrace();
			} catch (NamingException e) {
				e.printStackTrace();
			} 
			if(local == null){
				m_log.warn(pollerProtocol + "の実行クラスが内部DBに登録されていません。(" +
						VmMethodTypeConstant.POLLERIMPL +
						")");
				continue;
			}
			
			// pollerProtocolの実行クラス名をcc_vm_solution_mstから取得。
			// 存在していない場合、中止。
			String className = local.getClassName();
			m_log.debug("action() : className = " + className);
			if(className == null) {
				m_log.warn(pollerProtocol + "の実行クラスが内部DBに登録されていません。。");
				continue;
			}
			
			// コンストラクタの生成
			// 失敗したら、中止。
			Constructor pollerConstructor = null;
			try {
				pollerConstructor = Class.forName(className).getConstructor();
			} catch (SecurityException e) {
				e.printStackTrace();
			} catch (NoSuchMethodException e) {
				e.printStackTrace();
			} catch (ClassNotFoundException e) {
				m_log.warn(className + " is not poller class");
				continue;
			}
			if (pollerConstructor == null) {
				m_log.warn("action() : pollerConstructor is null");
				continue;
			}
				
			// ポーラーの取得。
			// 取得できたら、ポーリング実行。
			VmPollerImplInterface poller = null;
			try {
				poller = (VmPollerImplInterface) pollerConstructor.newInstance();
			} catch (Exception e) {
				e.printStackTrace();
			}
			if (poller == null) {
				m_log.warn("action() : poller is null");
				continue;
			}
					
			poller.init(pollerProtocol);
			poller.polling(
					pollerConfig.getAddress(),
					vmConfig.getVmNodeType(),
					vmConfig.getVmManagementNodeIp(),
					vmConfig.getVmName(),
					vmConfig.getVmUser(),
					vmConfig.getVmUserPassword(),
					vmConfig.getVmProtocol(),
					pollingTargetMap.get(pollerProtocol), dataTable);
		}
		
		m_log.debug("dataTable" + dataTable);
		
		// 収集値を共有テーブルに格納する
		itr = intervals.iterator();			
		while(itr.hasNext()){
			int interval = itr.next();
			
			// 新規収集した情報をテーブルの先頭に挿入する
			m_log.debug("insert " + m_tableGroup + ", "
					+ m_tableName + ", " + interval);
			try {
				// テーブルホルダに収集値を挿入する
				// IPアドレスを識別キーとすることで、ファシリティで定義されているIPアドレスが変更になった場合に
				// 変更前と変更後の収集値が同じテーブルホルダに格納されることを防ぐ
				sharedTable.insertDataTable(
						m_tableGroup,
						m_tableName, 
						interval, 
						dataTable,
						pollerConfig.getAddress().toString());
			} catch (DataTableNotFoundException e) {
				m_log.error(e.getMessage(), e);
			}
		}
	
		// デバッグログ出力
		m_log.debug("run end   : " + m_pollerGroup + "  " + m_pollerName);
	}

	/**
	 * 指定のOIDに対してポーリングを実行します。
	 * 
	 * @param targetValues ポーリング対象
	 * @return 収集したMIB値が格納されたデータテーブル
	 */
	public DataTable polling(int timeout, int retries, HashMap<String, List<String>> targetMaps){
		
		Set<String> setKeys = targetMaps.keySet();
		Iterator<String> itrKeys = setKeys.iterator();
		
		// 各ポーラーで共通で使用するDataTableを定義する
		DataTable dataTable = new DataTable();
		
		while(itrKeys.hasNext()){
			
			String collectMethod = itrKeys.next();
			PollerConfig pollerConfig = m_pollingConfig.getPollerConfig();
			
			// *****************************
			// SNMPのpollingの設定
			// *****************************
			if(collectMethod.equals(PollerProtocolConstant.PROTOCOL_SNMP)){
				
				// ポーリングを行い値を収集
				SnmpPollerConfig snmpConfig = pollerConfig.getSnmpConfig();
				List<String> pollingTargets = targetMaps.get(PollerProtocolConstant.PROTOCOL_SNMP);
				
				SnmpPollerImpl poller = new SnmpPollerImpl();
				poller.polling(
						pollerConfig.getAddress(), 
						snmpConfig.getPort(),
						snmpConfig.getVersion(),
						snmpConfig.getComunity(),
						retries,
						timeout,
						pollingTargets, 
						m_pollingConfig.isIndexCheckFlg(),
						dataTable);
				
			}
			// *****************************
			// WBEMのpollingの設定
			// *****************************
			else if(collectMethod.equals(PollerProtocolConstant.PROTOCOL_WBEM)){
				
				// ポーリングを行い値を収集
				WbemPollerConfig wbemConfig = pollerConfig.getWbemConfig();
				List<String> pollingTargets = targetMaps.get(PollerProtocolConstant.PROTOCOL_WBEM);
				
				WbemPollerImpl poller = new WbemPollerImpl();
				poller.polling(
						pollerConfig.getAddress(), 
						wbemConfig.getPort(),
						wbemConfig.getProtocol(),
						wbemConfig.getUserName(),
						wbemConfig.getPassword(),
						wbemConfig.getNameSpace(),
						retries,
						timeout,
						pollingTargets,
						dataTable);
			}
			// *****************************
			// 仮想化のpollingの設定
			// *****************************
			/*
			else if(collectMethod.equals(PollerProtocolConstant.PROTOCOL_VM)){
				
				
				// FIXME: 仮想化対応箇所
				
				
			}
			*/
			
		}
		
		return dataTable;
	}

	protected SharedTable lookupSharedTable() throws NamingException{
		try {
			// 指定の値格納用テーブルが存在するか確認する
			InitialContext ctx = new InitialContext();
			
			Object obj = ctx.lookup(SHARED_TABLE_JNDI_NAME);

			SharedTable table = 
				(SharedTable)PortableRemoteObject.narrow(obj, SharedTable.class);
			
			return table;
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			throw e;
		}
	}
	
	/**
	 * ポーラグループ名を返します。
	 * @return ポーラグループ名
	 */
	public String getPollerGroup() {
		return m_pollerGroup;
	}

	/**
	 * ポーラ名を返します。
	 * @return ポーラ名
	 */
	public String getPollerName() {
		return m_pollerName;
	}
	
	/**
	 * ポーリング設定を返します。
	 * @return ポーリング設定
	 */
	public PollingControllerConfig getPollingConfig() {
		return m_pollingConfig;
	}

	public String getQuartzJndiName() {
		return m_quartzJndiName;
	}

	public void setQuartzJndiName(String jndiName) {
		m_quartzJndiName = jndiName;
	}

	/**
	 * ポーラの実行状態を返します。
	 * @return 実行状態
	 */
	public int getStatus() {
		return m_status;
	}
}
