/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.dao;

import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.CommunicationException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.util.LdapConnectionManager;
import com.clustercontrol.accesscontrol.util.TimeKeeper;
import com.clustercontrol.accesscontrol.util.TimeKeeperService;
import com.clustercontrol.repository.bean.FileSystemInfoAttributeConstant;
import com.clustercontrol.repository.ejb.entity.FileSystemInfoBean;
import com.clustercontrol.repository.ejb.entity.FileSystemInfoPK;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * FileSystemInfo(BMP Entity Bean)のDAO実装クラス
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class FileSystemInfoDAOImpl implements FileSystemInfoDAO {
	protected static Log m_log = LogFactory.getLog( FileSystemInfoDAOImpl.class );
	
	protected static final String OBJECTCLASS = "objectClass";
	protected static final String BASE_DN = "ou=facility";
	protected static final String OBJECT_CLASS_FILE_SYSTEM_INFO = "ccFileSystemInfo";
	/** LDAP用コンテキスト取得時のリトライ上限 */	
	private static int MAX_RETRY = 3;
	
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#init()
	 */
	public void init() {

	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#load(com.clustercontrol.repository.ejb.FileSystemInfoPK, com.clustercontrol.repository.ejb.FileSystemInfoBean)
	 */
	public void load(FileSystemInfoPK pk, FileSystemInfoBean ejb) throws EJBException {
		javax.naming.directory.DirContext ctx = null;
		
		int retryCount=0;
		while(MAX_RETRY > retryCount){
			TimeKeeper timer = null;
			try {
				clearBean(ejb);
				
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				// LDAPへのアクセスが無応答になった場合に処理を中断するためのタイマースレッドを起動する。
				timer = TimeKeeperService.start();
				
				//属性を取得
				javax.naming.directory.Attributes attrs = ctx.getAttributes(pk.getDn());
	
				//dn取得
				ejb.setDn(pk.getDn());
				//cn取得
				ejb.setCn(new Integer((String)attrs.get(FileSystemInfoAttributeConstant.CN).get(0)));
				
				//ccSnmpOID取得
				if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMSNMPOID) != null){
					ejb.setSnmpOID((String)attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMSNMPOID).get(0));
				}
				//ccFileSystemMountPoint取得
				if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMMOUNTPOINT) != null){
					ejb.setFileSystemMountPoint((String)attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMMOUNTPOINT).get(0));
				}
				//ccFileSystemType取得
				if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMTYPE) != null){
					ejb.setFileSystemType((String)attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMTYPE).get(0));
				}
				//description取得
				if(attrs.get(FileSystemInfoAttributeConstant.DESCRIPTION) != null){
					ejb.setDescription((String)attrs.get(FileSystemInfoAttributeConstant.DESCRIPTION).get(0));
				}
				//ccDisplayName取得
				if(attrs.get(FileSystemInfoAttributeConstant.DISPLAYNAME) != null){
					ejb.setDisplayName((String)attrs.get(FileSystemInfoAttributeConstant.DISPLAYNAME).get(0));
				}
				//dnからファシリティIDを抽出する
				//dnは下記の内容となるため、初めに出現する","以前と最後に出現する","以降を削除したあと
				//"ccFacilityId="を削除する。
				//dnの内容：
				//　　"cn=eth0,ccFacilityId=HOST,ou=facility"
				String dn = ejb.getDn();
				dn = dn.replaceFirst("[^,]*,", "");
				dn = dn.replaceFirst(",[\\s\\w=]*", "");
				dn = dn.replaceFirst(FileSystemInfoAttributeConstant.FACILITYID + "=", "");
				ejb.setFacilityId(dn);
				
			} catch ( javax.naming.NamingException ex ) {
			    if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
				m_log.debug("load(): catch NamingException" + ex.getMessage());
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
			}catch (NullPointerException e){
				//jreのバグ（？）により、NullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("load() : LDAP query result is null");
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("load() context close failed : " + e.getMessage());
				}
				
				// タイマースレッドを停止
				TimeKeeperService.stop(timer);
			}
			return;
		}
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#store(com.clustercontrol.repository.ejb.FileSystemInfoBean)
	 */
	public void store(FileSystemInfoBean ejb) throws EJBException {
		javax.naming.directory.DirContext ctx = null;
		int retryCount=0;
		while(MAX_RETRY > retryCount){
			TimeKeeper timer = null;
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				// LDAPへのアクセスが無応答になった場合に処理を中断するためのタイマースレッドを起動する。
				timer = TimeKeeperService.start();
				
				//属性を取得
				javax.naming.directory.Attributes attrs = ctx.getAttributes(ejb.getDn());
				
				javax.naming.directory.Attributes addAttrs = 
					new javax.naming.directory.BasicAttributes();
				javax.naming.directory.Attributes replaceAttrs = 
					new javax.naming.directory.BasicAttributes();
				javax.naming.directory.Attributes removeAttrs = 
					new javax.naming.directory.BasicAttributes();
				
				//description設定
				if(attrs.get(FileSystemInfoAttributeConstant.DESCRIPTION) == null && ejb.getDescription() != null){
					addAttrs.put(FileSystemInfoAttributeConstant.DESCRIPTION, ejb.getDescription());
				}
				else if(attrs.get(FileSystemInfoAttributeConstant.DESCRIPTION) != null && ejb.getDescription() == null){
					removeAttrs.put(FileSystemInfoAttributeConstant.DESCRIPTION, null);
				}
				else if(attrs.get(FileSystemInfoAttributeConstant.DESCRIPTION) != null && ejb.getDescription() != null){
					replaceAttrs.put(FileSystemInfoAttributeConstant.DESCRIPTION, ejb.getDescription());
				}
				//ccFileSystemSnmpOID設定
				if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMSNMPOID) == null && ejb.getSnmpOID() != null){
					addAttrs.put(FileSystemInfoAttributeConstant.FILESYSTEMSNMPOID, ejb.getSnmpOID());
				}
				else if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMSNMPOID) != null && ejb.getSnmpOID() == null){
					removeAttrs.put(FileSystemInfoAttributeConstant.FILESYSTEMSNMPOID, null);
				}
				else if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMSNMPOID) != null && ejb.getSnmpOID() != null){
					replaceAttrs.put(FileSystemInfoAttributeConstant.FILESYSTEMSNMPOID, ejb.getSnmpOID());
				}
//				//ccFileSystemIndex設定
//				if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMINDEX) == null && ejb.getFileSystemIndex() != null){
//					addAttrs.put(FileSystemInfoAttributeConstant.FILESYSTEMINDEX, ejb.getFileSystemIndex().toString());
//				}
//				else if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMINDEX) != null && ejb.getFileSystemIndex() == null){
//					removeAttrs.put(FileSystemInfoAttributeConstant.FILESYSTEMINDEX, null);
//				}
//				else if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMINDEX) != null && ejb.getFileSystemIndex() != null){
//					replaceAttrs.put(FileSystemInfoAttributeConstant.FILESYSTEMINDEX, ejb.getFileSystemIndex().toString());
//				}
				//ccFileSystemMountPoint設定
				if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMMOUNTPOINT) == null && ejb.getFileSystemMountPoint() != null){
					addAttrs.put(FileSystemInfoAttributeConstant.FILESYSTEMMOUNTPOINT, ejb.getFileSystemMountPoint());
				}
				else if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMMOUNTPOINT) != null && ejb.getFileSystemMountPoint() == null){
					removeAttrs.put(FileSystemInfoAttributeConstant.FILESYSTEMMOUNTPOINT, null);
				}
				else if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMMOUNTPOINT) != null && ejb.getFileSystemMountPoint() != null){
					replaceAttrs.put(FileSystemInfoAttributeConstant.FILESYSTEMMOUNTPOINT, ejb.getFileSystemMountPoint());
				}
				//ccFileSystemType設定
				if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMTYPE) == null && ejb.getFileSystemType() != null){
					addAttrs.put(FileSystemInfoAttributeConstant.FILESYSTEMTYPE, ejb.getFileSystemType());
				}
				else if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMTYPE) != null && ejb.getFileSystemType() == null){
					removeAttrs.put(FileSystemInfoAttributeConstant.FILESYSTEMTYPE, null);
				}
				else if(attrs.get(FileSystemInfoAttributeConstant.FILESYSTEMTYPE) != null && ejb.getFileSystemType() != null){
					replaceAttrs.put(FileSystemInfoAttributeConstant.FILESYSTEMTYPE, ejb.getFileSystemType());
				}
				//ccDisplayName設定
				if(attrs.get(FileSystemInfoAttributeConstant.DISPLAYNAME) == null && ejb.getDisplayName() != null){
					addAttrs.put(FileSystemInfoAttributeConstant.DISPLAYNAME, ejb.getDisplayName());
				}
				else if(attrs.get(FileSystemInfoAttributeConstant.DISPLAYNAME) != null && ejb.getDisplayName() == null){
					removeAttrs.put(FileSystemInfoAttributeConstant.DISPLAYNAME, null);
				}
				else if(attrs.get(FileSystemInfoAttributeConstant.DISPLAYNAME) != null && ejb.getDisplayName() != null){
					replaceAttrs.put(FileSystemInfoAttributeConstant.DISPLAYNAME, ejb.getDisplayName());
				}
				//属性を変更する（追加）
				ctx.modifyAttributes(
						ejb.getDn(), 
						javax.naming.directory.DirContext.ADD_ATTRIBUTE, 
						addAttrs);
				//属性を変更する（置換）
				ctx.modifyAttributes(
						ejb.getDn(), 
						javax.naming.directory.DirContext.REPLACE_ATTRIBUTE, 
						replaceAttrs);
				//属性を変更する（削除）
				ctx.modifyAttributes(
						ejb.getDn(), 
						javax.naming.directory.DirContext.REMOVE_ATTRIBUTE, 
						removeAttrs);
				
			} catch ( javax.naming.NamingException ex ) {
			    if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} catch (NullPointerException e) {
				//jreのバグ（？）により、NullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("store() : LDAP query result is null");
				
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("store() context close failed : " + e.getMessage());
				}
				
				// タイマースレッドを停止
				TimeKeeperService.stop(timer);
			}
			return;
		}
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#remove(com.clustercontrol.repository.ejb.FileSystemInfoPK)
	 */
	public void remove(FileSystemInfoPK pk) throws RemoveException, EJBException {
		javax.naming.directory.DirContext ctx = null;
		int retryCount=0;
		while(MAX_RETRY > retryCount){
			TimeKeeper timer = null;
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				// LDAPへのアクセスが無応答になった場合に処理を中断するためのタイマースレッドを起動する。
				timer = TimeKeeperService.start();
				
				//エントリを削除
				ctx.destroySubcontext(pk.getDn());
				return;
				
			} catch ( javax.naming.NamingException ex ) {
			    if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
	            AplLogger apllog = new AplLogger("REP", "rep");
	            String[] args = {pk.getDn()};
	            apllog.put("SYS", "004", args);
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} catch (NullPointerException e){
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("remove() : LDAP query result is null");
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("remove() context close failed : " + e.getMessage());
				}
				
				// タイマースレッドを停止
				TimeKeeperService.stop(timer);
			}
		}
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#create(com.clustercontrol.repository.ejb.FileSystemInfoBean)
	 */
	public FileSystemInfoPK create(FileSystemInfoBean ejb) throws CreateException, EJBException {
		FileSystemInfoPK pk = null;
		javax.naming.directory.DirContext ctx = null;
		int retryCount=0;	
		while(MAX_RETRY > retryCount){
			TimeKeeper timer = null;
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				// Attributes for new entry
				javax.naming.directory.Attributes attrs =
					new javax.naming.directory.BasicAttributes();
				
				attrs.put(OBJECTCLASS, OBJECT_CLASS_FILE_SYSTEM_INFO);
				attrs.put(FileSystemInfoAttributeConstant.CN, ejb.getCn().toString());
				
				// dn文字列作成
				StringBuffer dn = new StringBuffer();
				dn.append(FileSystemInfoAttributeConstant.CN);
				dn.append("=");
				dn.append(ejb.getCn());
				dn.append(",");
				dn.append(FileSystemInfoAttributeConstant.FACILITYID);
				dn.append("=");
				dn.append(ejb.getFacilityId());
				dn.append(",");
				dn.append(BASE_DN);
				
				//dn設定
				ejb.setDn(dn.toString());
	
				// LDAPへのアクセスが無応答になった場合に処理を中断するためのタイマースレッドを起動する。
				timer = TimeKeeperService.start();
				
				//エントリ作成
				ctx.createSubcontext(ejb.getDn(), attrs);
	
				pk = new FileSystemInfoPK(ejb.getDn());
				return pk;
				
			} catch ( javax.naming.NamingException ex ) {
				if(ex instanceof CommunicationException){
					LdapConnectionManager.getConnectionManager().setDirContext(null);
				}
				AplLogger apllog = new AplLogger("REP", "rep");
				String[] args = {ejb.getDn()};
				apllog.put("SYS", "003", args);
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
//				ver 2.1.1改修分をマージした際にコメントアウト 7/4 kozakai
//	            AplLogger apllog = new AplLogger("REP", "rep");
//	            String[] args = {ejb.getDn()};
//	            apllog.put("SYS", "003", args);
//			    if(ex instanceof CommunicationException){
//			        LdapConnectionManager.getConnectionManager().setDirContext(null);
//			    }
//			    else if(ex instanceof NameAlreadyBoundException){
//			    	throw new DuplicateKeyException(ex.getMessage());
//	            }
//				throw new CreateException(ex.getMessage());
				
				
			} catch (NullPointerException e){
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("create() : LDAP query result is null");
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}

			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("create() context close failed : " + e.getMessage());
				}
				
				// タイマースレッドを停止
				TimeKeeperService.stop(timer);
			}
		}
		return pk;
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		ArrayList ret = new ArrayList();

		m_log.debug("findAll()");
		
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// BaseDN以下にぶら下がるすべてのレベルを検索の対象
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_FILE_SYSTEM_INFO);
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		
		int retryCount=0;
		while(MAX_RETRY > retryCount){
			TimeKeeper timer = null;
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				// LDAPへのアクセスが無応答になった場合に処理を中断するためのタイマースレッドを起動する。
				timer = TimeKeeperService.start();
				
				results = ctx.search(BASE_DN, search.toString(), constraints);
				
				if (results == null ) {
					m_log.debug("findAll() : No Search Result");
					return ret;
				}
	
				//検索結果を取得
				while ( results.hasMoreElements() ) {
					javax.naming.directory.SearchResult aResult = (javax.naming.directory.SearchResult)results.nextElement();
					FileSystemInfoPK pk = new FileSystemInfoPK(aResult.getName() + "," + BASE_DN);
					ret.add(pk);
				}
				
			} catch (NamingException e) {
			    if(e instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
				m_log.debug("findAll() : " + e.getMessage());
//				中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
					
			} catch (NullPointerException e){
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findAll() : LDAP query result is null");
	//				中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(results != null)
						results.close();
					if(ctx != null)
						ctx.close();
				} catch (NamingException e) {
					m_log.error("findAll() context close failed : " + e.getMessage());
				}
				
				// タイマースレッドを停止
				TimeKeeperService.stop(timer);
			}
		}
		return ret;
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#findByPrimaryKey(com.clustercontrol.repository.ejb.FacilityPK)
	 */
	public FileSystemInfoPK findByPrimaryKey(FileSystemInfoPK pk) throws FinderException {
		javax.naming.directory.DirContext ctx = null;
		
		int retryCount=0;
		
		while(MAX_RETRY > retryCount){
			TimeKeeper timer = null;
			try {
				m_log.debug("findByPrimaryKey() : dn = " + pk.getDn());
				
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				// LDAPへのアクセスが無応答になった場合に処理を中断するためのタイマースレッドを起動する。
				timer = TimeKeeperService.start();
				
				//属性を取得
				javax.naming.directory.Attributes attrs = ctx.getAttributes(pk.getDn());
				
	//			2.1.1改修分マージ時にコメントアウト 7/4 kozakai
	//			return pk;
				
			} catch ( javax.naming.NamingException ex ) {
			    if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
				m_log.debug("findByPrimaryKey() : " + ex.getMessage());
					
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
//				2.1.1改修分マージ時にコメントアウト 7/4 kozakai
//				m_log.debug("findByPrimaryKey() : " + ex.getMessage());
//				throw new FinderException("dn=" + pk.getDn() + " not found.");
				
			}catch (NullPointerException e){
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findByFacilityIdAndName() : LDAP query result is null");
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(ctx != null){
						ctx.close();
					}
				} catch (NamingException e) {
					m_log.error("findByPrimaryKey() context close failed : " + e.getMessage());
				}
				
				// タイマースレッドを停止
				TimeKeeperService.stop(timer);
			}
			return pk;
		}
		return pk;
	}

	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#findByFacilityId(java.lang.String)
	 */
	public Collection findByFacilityId(String facilityId) throws FinderException {
		ArrayList ret = new ArrayList();

		m_log.debug("findByFacilityId() : FacilityId = " + facilityId);
		
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// BaseDN以下にぶら下がるすべてのレベルを検索の対象
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);

		// dn文字列作成
		StringBuffer baseDn = new StringBuffer();
		baseDn.append(FileSystemInfoAttributeConstant.FACILITYID);
		baseDn.append("=");
		baseDn.append(facilityId);
		baseDn.append(",");
		baseDn.append(BASE_DN);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_FILE_SYSTEM_INFO);
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		
		int retryCount=0;
		
		while(MAX_RETRY > retryCount){
			TimeKeeper timer = null;
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				// LDAPへのアクセスが無応答になった場合に処理を中断するためのタイマースレッドを起動する。
				timer = TimeKeeperService.start();
				
				results = ctx.search(baseDn.toString(), search.toString(), constraints);
				
				if (results == null ) {
						m_log.warn("findByFacilityId() : No Search Result : FacilityId = " + facilityId);
					return ret;
				}
	
				//検索結果を取得
				while ( results.hasMoreElements() ) {
					javax.naming.directory.SearchResult aResult = (javax.naming.directory.SearchResult)results.nextElement();
					FileSystemInfoPK pk = new FileSystemInfoPK(aResult.getName() + "," + baseDn.toString());
					ret.add(pk);
				}
				
			} catch (NamingException e) {
			    if(e instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
				m_log.debug("findByFacilityId() : " + e.getMessage());
				//中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} catch (NullPointerException e) {
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findByFacilityId() : LDAP query result is null");
	//				中身を空にします。
				ret.clear();
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(results != null)
						results.close();
					if(ctx != null)
						ctx.close();
				} catch (NamingException e) {
					m_log.error("findByFacilityId() context close failed : " + e.getMessage());
				}
				
				// タイマースレッドを停止
				TimeKeeperService.stop(timer);
			}
			return ret;
		}
		return ret;
	}
	/* (non-Javadoc)
	 * @see com.clustercontrol.repository.dao.FileSystemInfoDAO#findByFacilityIdAndName(java.lang.String, java.lang.String)
	 */
	public FileSystemInfoPK findByFacilityIdAndName(String facilityId, String name) throws FinderException {
		m_log.debug("findByFacilityIdAndName() : FacilityId = " + facilityId + ", FacilityName = " + name);
		
		// エントリの検索
		javax.naming.directory.SearchControls constraints = 
			new javax.naming.directory.SearchControls();

		// BaseDN以下にぶら下がるすべてのレベルを検索の対象
		constraints.setSearchScope(javax.naming.directory.SearchControls.SUBTREE_SCOPE);

		// dn文字列作成
		StringBuffer baseDn = new StringBuffer();
		baseDn.append(FileSystemInfoAttributeConstant.FACILITYID);
		baseDn.append("=");
		baseDn.append(facilityId);
		baseDn.append(",");
		baseDn.append(BASE_DN);

		// 検索文字列作成
		StringBuffer search = new StringBuffer();
		search.append("(&(");
		search.append(OBJECTCLASS);
		search.append("=");
		search.append(OBJECT_CLASS_FILE_SYSTEM_INFO);
		search.append(")(");
		search.append(FileSystemInfoAttributeConstant.CN);
		search.append("=");
		search.append(name);
		search.append("))");
		
		//検索実行
		javax.naming.NamingEnumeration results = null;
		javax.naming.directory.DirContext ctx = null;
		FileSystemInfoPK pk = null;
		
		int retryCount=0;
		while(MAX_RETRY > retryCount){
			TimeKeeper timer = null;
			try {
				// LDAP用のコンテキストの作成
				ctx = LdapConnectionManager.getConnectionManager().getDirContext();
				
				// LDAPへのアクセスが無応答になった場合に処理を中断するためのタイマースレッドを起動する。
				timer = TimeKeeperService.start();
				
				results = ctx.search(baseDn.toString(), search.toString(), constraints);
				
				//検索結果を取得
				if ( results.hasMoreElements() ) {
					javax.naming.directory.SearchResult aResult = (javax.naming.directory.SearchResult)results.nextElement();
					pk = new FileSystemInfoPK(aResult.getName() + "," + baseDn.toString());
				}
				else{
					throw new FinderException("FacilityId = " + facilityId + ", FacilityName = " + name + " not found.");
				}
				
			} catch ( javax.naming.NamingException ex ) {
			    if(ex instanceof CommunicationException){
			        LdapConnectionManager.getConnectionManager().setDirContext(null);
			    }
				m_log.debug("findByFacilityIdAndName() : " + ex.getMessage());
					
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(ex);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
//				2.1.1改修分のマージ時にコメントアウト 7/4 kozakai
//				m_log.debug("findByFacilityIdAndName() : " + ex.getMessage());
//				throw new FinderException("FacilityId = " + facilityId + ", FacilityName = " + name + " not found.");
				
			} catch (NullPointerException e){
				//jreのバグ（？）により、hasMoreElements()でNullPointerExceptionが発生することがある。
				//根本対処の方法がないため、NullPointerExcetionをcatchする。（2007/06/19　宮本）
				m_log.debug("findByFacilityIdAndName() : LDAP query result is null");
				
				// whileループの判定の前に、メッソッドを抜けるかのチェックをするため、
				// ここでカウントアップする。
				retryCount++;
				if(retryCount >= MAX_RETRY){
					throw new EJBException(e);
				} else {
					//リトライ上限以下の場合には再度LDAPに問い合わせをします。
					//whlie loopに戻ります。
					continue;
				}
				
			} finally {
				try {
					if(results != null)
						results.close();
					if(ctx != null)
						ctx.close();
				} catch (NamingException e) {
					m_log.error("findByFacilityIdAndName() context close failed : " + e.getMessage());
				}
				
				// タイマースレッドを停止
				TimeKeeperService.stop(timer);
			}
			return pk;
		}
		return pk;
	}
	/**
	 * Beanクリア処理
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param ejb
	 */
	protected void clearBean(FileSystemInfoBean ejb){
	    ejb.setCn(null);
	    ejb.setDescription(null);
	    ejb.setFileSystemMountPoint(null);
	    ejb.setFileSystemType(null);
	    ejb.setDn(null);
	    ejb.setFacilityId(null);
	    ejb.setSnmpOID(null);
	    ejb.setDisplayName(null);
	}
}
