/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.logagent;

import java.util.Properties;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.notify.bean.QueueConstant;

/**
 * マネージャとのやり取りに利用するQueue送信クラス<BR>
 * 
 * ログ転送エージェントはHinemosイベントと転送対象ファイルのリクエストを
 * マネージャに送信する。<BR>
 * 
 * Queueとしては、<BR>
 * com.clustercontrol.logtransfer.bean.QueueConstant.QUEUE_NAME_FILE_MANAGE<BR>
 * と<BR>
 * com.clustercontrol.monitor.bean.QueueConstant.QUEUE_NAME_LOG<BR>
 * に接続する。
 *
 * @version 3.0.0
 * @since 2.1.0
 */
public class SendQueue implements ExceptionListener {

	private static final String QUEUE_CON_FACTORY = "ConnectionFactory";

	private static final String QUEUE_USER_NAME = "queue.user.name";

	private static final String QUEUE_USER_PASSWORD = "queue.user.password";

	private Agent m_agent;
	
	private Properties m_props;

	private QueueConnection m_con;

	private Queue m_queueLog;

	private Queue m_queueFacility;

	private QueueSession m_session;

	private QueueConnectionFactory m_factory;

	private boolean m_isErr = false;
	
	private long m_interval = 10000;

	//ロガー
	static private Log log = LogFactory.getLog(SendQueue.class);
    
	/**
	 * コンストラクタ
	 * @param props プロパティファイル情報
	 */
	public SendQueue(Agent agent, Properties props) {
		super();
		m_agent = agent;
		m_props = props;
		
		// ReciveTopic再接続処理実行間隔取得
		String interval = m_props.getProperty("receivetopic.reconnection.interval");
		if (interval != null) {
			try {
				// プロパティファイルには秒で記述されているため、1000をかける
				m_interval = Integer.parseInt(interval) * 1000;
			} catch (NumberFormatException e) {
				log.error("receivetopic.reconnection.interval",e);
			}
		}
		
		initial();
	}

	/**
	 *  再接続処理
	 */
	synchronized private boolean reInitial() {

		boolean ret = false;

		log.info("再接続処理!");

		terminate();

		if (initial()) {

			ret = true;

			log.info("再接続処理:成功!");

			//エラーフラグ解除
			m_isErr = false;


		} else {
			log.info("再接続処理:失敗!");
		}

		return ret;
	}
	/**
	 * サーバ接続の終了処理
	 *  
	 */
	public void terminate() {

		try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
		}

		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e1) {
		}
	}

    /**
     * メッセージをマネージャに送信します。<BR>
     * 
	 * 実行応答メッセージを送信する。<BR>
	 * 処理失敗は1回再試行する。
	 * 
	 * @param msg
	 */
	synchronized public boolean put(Object info) {

		if (m_isErr) {
			reInitial();
		}

		QueueSender sender = null;
		Message mess = null;

//		for(int i = 0; i < 2 ;i ++) {
//			try {
//				
//				if (info instanceof LogOutputInfo) {
//					//監視機能へ送信
//					//Sender作成
//					sender = m_session.createSender(m_queueLog);
//					//メッセージ作成
//					mess = m_session.createObjectMessage((LogOutputInfo)info);
//					
//					if (log.isDebugEnabled()) {
//						LogOutputInfo loginfo = (LogOutputInfo)info;
//						log.debug("create message : "
//								+ loginfo.getFacilityId() + ", "
//								+ loginfo.getMonitorId() + ", "
//								+ loginfo.getMessageId());
//					}
//					
//				}else if (info instanceof String) {
//					//ログ転送へファシリティID送信
//					//Sender作成
//					sender = m_session.createSender(m_queueFacility);
//					//メッセージ作成
//					mess = m_session.createTextMessage((String)info);
//
//					log.debug("create message : notify Fasility ID : " + info);
//				}else{
//					return false;
//				}
//				
//				//送信
//				log.info("オブジェクトメッセージ送信!");
//				sender.send(mess);
//				
//				return true;
//	
//			} catch (JMSException e) {
//				log.error(e);
//				log.info("オブジェクトメッセージ送信リトライ");
//				
//				try {
//					wait(m_interval);
//				} catch (InterruptedException e1) {
//				}
//				
//				if(!reInitial()){
//					return false;
//				}
//	
//			} finally {
//				try {
//					if (sender != null)
//						sender.close();
//				} catch (Exception e1) {
//				}
//			}
//		}
		while(true) {
			try {
				
				if (info instanceof OutputNotifyGroupInfo) {
					//監視機能へ送信
					//Sender作成
					sender = m_session.createSender(m_queueLog);
					//メッセージ作成
					mess = m_session.createObjectMessage((OutputNotifyGroupInfo)info);
					
					if (log.isDebugEnabled()) {
						OutputNotifyGroupInfo loginfo = (OutputNotifyGroupInfo)info;
						log.debug("オブジェクトメッセージ送信! : "
								+ "Facility ID : " + loginfo.getFacilityId() + ", "
								+ "Monitor ID : " + loginfo.getMonitorId() + ", "
								+ "Priority : " + loginfo.getPriority() + ", "
								+ "Message ID : " + loginfo.getMessageId() + ", "
								+ "Message : " + loginfo.getMessage());
					}
					
				}else if (info instanceof String) {
					//ログ転送へファシリティID送信
					//Sender作成
					sender = m_session.createSender(m_queueFacility);
					//メッセージ作成
					mess = m_session.createTextMessage((String)info);

					log.debug("オブジェクトメッセージ送信! : notify Fasility ID : " + info);
				}else{
					return false;
				}
				

				// 詳細情報を上記のlog出力含めるため、ここではコメントアウトとする
//				log.info("オブジェクトメッセージ送信!");
				//送信
				sender.send(mess);
				
				return true;
	
			} catch (JMSException e) {
				log.error(e);
				log.info("オブジェクトメッセージ送信リトライ");
				
				try {
					wait(m_interval);
				} catch (InterruptedException e1) {
				}
				
				if(!reInitial()){
					return false;
				}
	
			} finally {
				try {
					if (sender != null)
						sender.close();
				} catch (Exception e1) {
				}
			}
		}
		// for文をwhile文に変更したため、コメントアウトした
//		return false;
	}


	/**
	 * 接続初期化します。<BR>
	 * 
	 * @return
	 */
    private boolean initial() {

		log.info("EJB接続初期化");

		InitialContext con = null;

		try {
			//InitialContextの生成
			con = new InitialContext(m_props);

			//コネクションファクトリ生成
			m_factory = (QueueConnectionFactory) con.lookup(QUEUE_CON_FACTORY);

			//コネクション生成
			if (m_props.getProperty(QUEUE_USER_NAME) != null) {
				//ユーザ認証
				m_con = m_factory.createQueueConnection(m_props
						.getProperty(QUEUE_USER_NAME), m_props
						.getProperty(QUEUE_USER_PASSWORD));
			} else {
				//ユーザ認証なし
				m_con = m_factory.createQueueConnection();
			}

			
			//セッション生成
			m_session = m_con.createQueueSession(false,
					Session.AUTO_ACKNOWLEDGE);
			
			//エラーハンドラセット
	        m_con.setExceptionListener(this);
			
			//コネクション開始
			m_con.start();

			//メッセージQueue取得
			m_queueLog = (Queue) con.lookup(QueueConstant.QUEUE_NAME_EVENT);
			m_queueFacility = (Queue) con.lookup(com.clustercontrol.logtransfer.bean.QueueConstant.QUEUE_NAME_FILE_MANAGE);

			
			

		} catch (Exception e) {
			log.error("Init", e);
			m_isErr = true;
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e1) {
			}
		}
		return true;

	}
	
    /* 通信エラーハンドラ
     * (non-Javadoc)
     * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
     */
    public void onException(JMSException arg0) {

    	log.error(arg0.getMessage(), arg0);

    	m_agent.reConnection();

    }
    
	/**
	 * 再接続処理を行う
	 */
	synchronized public void reconnect() {
		
    	m_isErr = true;
        
        terminate();
        
	}
}
