/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.accesscontrol.composite;

import java.util.ArrayList;

import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.accesscontrol.action.GetUserList;
import com.clustercontrol.accesscontrol.action.GetUserListTableDefine;
import com.clustercontrol.accesscontrol.bean.UserListTableDefine;
import com.clustercontrol.accesscontrol.composite.action.UserDoubleClickListener;
import com.clustercontrol.bean.Property;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.util.Messages;

/**
 * アクセス[ユーザ]ビュー用のコンポジットクラスです。
 * 
 * クライアントのビューを構成します。
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class UserListComposite extends Composite {
	/** テーブルビューア */
    protected CommonTableViewer m_viewer = null;
    /** ヘッダ用ラベル */
    protected Label m_labelType = null;
    /** 件数用ラベル */
    protected Label m_labelCount = null;
    /** ユーザID */
    protected String m_uid = null;

    /**
     * コンストラクタ
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public UserListComposite(Composite parent, int style) {
        super(parent, style);
        initialize();
    }

    /**
     * コンポジットを配置します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        m_labelType = new Label(this, SWT.LEFT);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        m_labelType.setLayoutData(gridData);

        
        Table table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.SINGLE);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        table.setLayoutData(gridData);

        m_labelCount = new Label(this, SWT.RIGHT);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        m_labelCount.setLayoutData(gridData);

        m_viewer = new CommonTableViewer(table);
        m_viewer.createTableColumn(GetUserListTableDefine.get(),
        		UserListTableDefine.SORT_COLUMN_INDEX,
        		UserListTableDefine.SORT_ORDER);

        m_viewer.addDoubleClickListener(new UserDoubleClickListener(this));
    }

    /**
     * テーブルビューアーを更新します。<BR>
     * ユーザ一覧情報を取得し、共通テーブルビューアーにセットします。
     *  
     * @see com.clustercontrol.accesscontrol.action.GetUserList#getAll()
     */
    public void update() {
        ArrayList userInfo = null;

        //ユーザ一覧情報取得
        GetUserList getUserList = new GetUserList();
        userInfo = getUserList.getAll();
        if(userInfo == null){
        	userInfo = new ArrayList();
        }
        
        m_viewer.setInput(userInfo);

        m_labelType.setText("");
        Object[] args = { new Integer(userInfo.size()) };
        m_labelCount.setText(Messages.getString("records", args));
    }

    /**
     * テーブルビューアーを更新します。<BR>
     * フィルタ条件に一致するユーザ一覧情報を取得し、<BR>
     * 共通テーブルビューアーにセットします。
     *  
     * @param condition フィルタ条件
     * 
     * @see com.clustercontrol.accesscontrol.action.GetUserList#get(Property)
     */
    public void update(Property condition) {
        ArrayList userInfo = null;

        //ジョブ履歴情報取得
        GetUserList getUserList = new GetUserList();
        userInfo = getUserList.get(condition);

        m_viewer.setInput(userInfo);

        m_labelType.setText(Messages.getString("filtered.list"));
        Object[] args = { new Integer(userInfo.size()) };
        m_labelCount.setText(Messages.getString("filtered.records", args));
    }

    /**
     * このコンポジットが利用するテーブルビューアを返します。
     * 
     * @return テーブルビューア
     */
    public TableViewer getTableViewer() {
        return m_viewer;
    }

    /**
     * このコンポジットが利用するテーブルを返します。
     * 
     * @return テーブル
     */
    public Table getTable() {
        return m_viewer.getTable();
    }

    /**
     * ユーザIDを返します。
     * 
     * @return ユーザID
     */
    public String getUid() {
        return m_uid;
    }

    /**
     * ユーザIDを設定します。
     * 
     * @param uid ユーザID
     */
    public void setUid(String uid) {
    	m_uid = uid;
    }
}