/*
 *  @(#)DefaultMonitor.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.autodoc.v1.testserver;


import java.util.Hashtable;

import org.apache.log4j.Logger;


/**
 * This implemenation of <tt>Monitor</tt> uses a Hashtable to store the
 * test data.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since      March 17, 2002
 * @version    $Date: 2003/05/27 13:30:16 $
 */
public class DefaultMonitor implements Monitor
{
    private static final Logger LOG = Logger.getLogger( DefaultMonitor.class );
    
    private Hashtable testData = new Hashtable();
    private TestDataFactory dataFactory;
    private Server server;
    
    
    /**
     * Constructor for a Monitor requiring a reference to the Server which will
     * receive the completed <tt>TestData</tt> structures, and a reference to
     * a factory for creating new <tt>TestData</tt> instances for the particular
     * framework this Monitor belongs to.
     *
     * @param s the server to receive completed <tt>TestData</tt> entities.
     *      This cannot be <tt>null</tt>.
     * @param f the factory in charge of creating new <tt>TestData</tt>
     *      entities.  This cannot be <tt>null</tt>.
     * @exception IllegalArgumentException if <tt>s</tt> or <tt>f</tt> is
     *      <tt>null</tt>.
     */
    public DefaultMonitor( Server s, TestDataFactory f )
    {
        if (s == null || f == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        this.server = s;
        this.dataFactory = f;
    }
    
    
    /**
     * Adds a new <tt>TestData</tt> instance related to the given
     * <tt>TestInfo</tt>.  If there already is a <tt>TestData</tt> for the
     * given <tt>TestInfo</tt>, then an exception is thrown.
     *
     * @param info the unique test identifier to create a new <tt>TestData</tt>
     *      instance for.
     * @exception IllegalStateException if <tt>info</tt> is already
     *      been added without having been sent.
     * @exception IllegalArgumentException if <tt>info</tt> is <tt>null</tt>.
     */
    public void addTestData( TestInfo info )
    {
        assertNotNull( info );
        synchronized (this.testData)
        {
            if (this.testData.containsKey( info ))
            {
                throw new IllegalStateException( "Test "+info+
                    " has already been registered." );
            }
            LOG.debug("**** Monitor "+this+" adding test "+info);
            TestData td = createTestData( info );
            this.testData.put( info, td );
        }
    }
    
    
    /**
     * Retrieves the data associated with the given <tt>TestInfo</tt>, as was
     * created through {@link #addTestData( TestInfo )}.  If the <tt>info</tt>
     * was never passed to the add method, then an exception is thrown.
     *
     * @param info the unique test identifier
     * @exception IllegalStateException if <tt>info</tt> has not been added,
     *      or has been removed through the send call.
     * @exception IllegalArgumentException if <tt>info</tt> is <tt>null</tt>.
     */
    public TestData getTestData( TestInfo info )
    {
        assertNotNull( info );
        synchronized (this.testData)
        {
            TestData td = (TestData)this.testData.get( info );
            assertNotNull( td );
            return td;
        }
    }
    
    
    /**
     * Sends the <tt>TestData</tt> associated with <tt>info</tt> to the
     * inner server, and removes the data from the inner cache.  If the
     * <tt>info</tt> has not been added, then an exception is thrown.
     *
     * @param info the unique test identifier
     * @exception IllegalStateException if <tt>info</tt> has not been added,
     *      or has been removed through the send call.
     * @exception IllegalArgumentException if <tt>info</tt> is <tt>null</tt>.
     */
    public void sendTestData( TestInfo info )
    {
        assertNotNull( info );
        TestData td;
        synchronized (this.testData)
        {
            td = (TestData)this.testData.remove( info );
        }
        LOG.debug("**** Monitor "+this+" removing test "+info);
        sendTestDataToServer( td );
    }
    
    
    /**
     * Creates a new TestData instance for the info object through the
     * factory.  This is guaranteed to never return <tt>null</tt>.
     *
     * @param info the unique test identifier
     * @exception IllegalStateException if the factory returns <tt>null</tt>.
     */
    protected TestData createTestData( TestInfo info )
    {
        TestData td = this.dataFactory.createTestData( info );
        if (td == null)
        {
            throw new IllegalStateException("factory returned null");
        }
        return td;
    }
    
    
    /**
     * Retrieves the registered TestData instance for the info.  This
     * may return <tt>null</tt>.
     *
     * @param info the unique test identifier
     */
    protected TestData retrieveTestData( TestInfo info )
    {
        TestData td = (TestData)this.testData.get( info );
        return td;
    }
    
    
    /**
     * Sends off the test data to the server.
     */
    protected void sendTestDataToServer( TestData td )
    {
        assertNotNull( td );
        this.server.addTestData( td );
    }
    
    
    /**
     * Ensures that <tt>info</tt> is not <tt>null</tt>.
     */
    protected void assertNotNull( TestInfo info )
    {
        if (info == null)
        {
            throw new IllegalArgumentException( "TestInfo was null" );
        }
    }
    
    
    /**
     * Ensures that <tt>td</tt> is not <tt>null</tt>.
     */
    protected void assertNotNull( TestData td )
    {
        if (td == null)
        {
            throw new IllegalStateException( "Test is not registered." );
        }
    }
}

