/* -*- Mode: Vala; indent-tabs-mode: nil; tab-width: 2 -*-
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 * SPDX-FileCopyrightText: Michael Terry
 */

using GLib;

[GtkTemplate (ui = "/org/gnome/DejaDup/OverviewPage.ui")]
public class OverviewPage : BasicNavigationPage
{
  public signal void browse(bool all);
  public signal void show_schedule();
  public signal void show_folders();
  public signal void show_storage();

  [GtkChild]
  unowned Adw.ActionRow schedule_row;
  [GtkChild]
  unowned Adw.ActionRow folders_row;
  [GtkChild]
  unowned Adw.ActionRow storage_row;
  [GtkChild]
  unowned HelpButton storage_warning_button;
  [GtkChild]
  unowned Gtk.Label storage_warning_label;

  DejaDup.FilteredSettings settings;
  uint timeout_id;
  construct {
    var backend_watcher = DejaDup.BackendWatcher.get_instance();
    backend_watcher.changed.connect(() => {update_storage_row.begin();});
    update_storage_row.begin();

    settings = DejaDup.get_settings();
    settings.changed[DejaDup.INCLUDE_LIST_KEY].connect(update_folders_row);
    update_folders_row();

    settings.changed[DejaDup.PERIODIC_KEY].connect(update_schedule_row);
    settings.changed[DejaDup.PERIODIC_PERIOD_KEY].connect(update_schedule_row);
    settings.changed[DejaDup.LAST_BACKUP_KEY].connect(update_schedule_row);
    update_schedule_row();

    // Update the label every hour to lazily catch cross-day boundaries
    timeout_id = Timeout.add_seconds(60 * 60, update_schedule_row_timer);
  }

  ~OverviewPage()
  {
    Source.remove(timeout_id);
  }

  async void update_storage_row()
  {
    var backend = DejaDup.Backend.get_default();
    storage_row.subtitle = yield backend.get_location_pretty();

    string reason;
    if (backend.is_acceptable(out reason)) {
      storage_warning_button.visible = false;
      storage_row.remove_css_class("error");
    }
    else {
      storage_warning_button.visible = true;
      storage_warning_label.label = reason;
      storage_row.add_css_class("error");
    }
  }

  [GtkCallback]
  void on_browse_all_activated()
  {
    browse(true);
  }

  [GtkCallback]
  void on_schedule_activated()
  {
    show_schedule();
  }

  [GtkCallback]
  void on_folders_activated()
  {
    show_folders();
  }

  [GtkCallback]
  void on_storage_activated()
  {
    show_storage();
  }

  bool update_schedule_row_timer()
  {
    update_schedule_row();
    return Source.CONTINUE;
  }

  void update_schedule_row()
  {
    if (settings.get_boolean(DejaDup.PERIODIC_KEY)) {
      schedule_row.subtitle = ConfigAutoBackupRow.get_next_backup_desc();
      schedule_row.remove_css_class("warning");
    } else {
      schedule_row.subtitle = _("Automatic backups disabled");
      schedule_row.add_css_class("warning");
    }
  }

  void update_folders_row()
  {
    update_folders_row_async.begin();
  }

  async void update_folders_row_async()
  {
    var include_list = settings.get_file_list(DejaDup.INCLUDE_LIST_KEY);
    string[] nicknames = {};
    foreach (var include in include_list) {
      nicknames += yield DejaDup.get_nickname(include);
    }

    if (nicknames.length > 0) {
      folders_row.subtitle = string.joinv(", ", nicknames);
      folders_row.remove_css_class("warning");
    } else {
      folders_row.subtitle = _("None");
      folders_row.add_css_class("warning");
    }
  }
}
