//===-- Range reduction for double precision sin/cos/tan -*- C++ --------*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIBC_SRC_MATH_GENERIC_RANGE_REDUCTION_DOUBLE_COMMON_H
#define LLVM_LIBC_SRC_MATH_GENERIC_RANGE_REDUCTION_DOUBLE_COMMON_H

#include "src/__support/FPUtil/FPBits.h"
#include "src/__support/FPUtil/double_double.h"
#include "src/__support/FPUtil/dyadic_float.h"
#include "src/__support/FPUtil/multiply_add.h"
#include "src/__support/FPUtil/nearest_integer.h"
#include "src/__support/common.h"
#include "src/__support/integer_literals.h"
#include "src/__support/macros/config.h"

namespace LIBC_NAMESPACE_DECL {

namespace generic {

using LIBC_NAMESPACE::fputil::DoubleDouble;
using Float128 = LIBC_NAMESPACE::fputil::DyadicFloat<128>;

LIBC_INLINE constexpr Float128 PI_OVER_128_F128 = {
    Sign::POS, -133, 0xc90f'daa2'2168'c234'c4c6'628b'80dc'1cd1_u128};

// Note: The look-up tables ONE_TWENTY_EIGHT_OVER_PI is selected to be either
// from fma:: or nofma:: namespace.

// For large range |x| >= 2^32, we use the exponent of x to find 3 double-chunks
// of 128/pi c_hi, c_mid, c_lo such that:
//   1) ulp(round(x * c_hi, D, RN)) >= 256,
//   2) If x * c_hi = ph_hi + ph_lo and x * c_mid = pm_hi + pm_lo, then
//        min(ulp(ph_lo), ulp(pm_hi)) >= 2^-53.
//   3) ulp(round(x * c_lo, D, RN)) <= 2^-7x.
// This will allow us to do quick computations as:
//   (x * 256/pi) ~ x * (c_hi + c_mid + c_lo)    (mod 256)
//                ~ ph_lo + pm_hi + pm_lo + (x * c_lo)
// Then,
//   round(x * 128/pi) = round(ph_lo + pm_hi)    (mod 256)
// And the high part of fractional part of (x * 128/pi) can simply be:
//   {x * 128/pi}_hi = {ph_lo + pm_hi}.
// To prevent overflow when x is very large, we simply scale up
// (c_hi, c_mid, c_lo) by a fixed power of 2 (based on the index) and scale down
// x by the same amount.

template <bool NO_FMA> struct LargeRangeReduction {
  // Calculate the high part of the range reduction exactly.
  LIBC_INLINE unsigned compute_high_part(double x) {
    using FPBits = typename fputil::FPBits<double>;
    FPBits xbits(x);

    // TODO: The extra exponent gap of 62 below can be reduced a bit for non-FMA
    // with a more careful analysis, which in turn will reduce the error bound
    // for non-FMA
    int x_e_m62 = xbits.get_biased_exponent() - (FPBits::EXP_BIAS + 62);
    idx = static_cast<unsigned>((x_e_m62 >> 4) + 3);
    // Scale x down by 2^(-(16 * (idx - 3))
    xbits.set_biased_exponent((x_e_m62 & 15) + FPBits::EXP_BIAS + 62);
    // 2^62 <= |x_reduced| < 2^(62 + 16) = 2^78
    x_reduced = xbits.get_val();
    // x * c_hi = ph.hi + ph.lo exactly.
    DoubleDouble ph =
        fputil::exact_mult<NO_FMA>(x_reduced, ONE_TWENTY_EIGHT_OVER_PI[idx][0]);
    // x * c_mid = pm.hi + pm.lo exactly.
    DoubleDouble pm =
        fputil::exact_mult<NO_FMA>(x_reduced, ONE_TWENTY_EIGHT_OVER_PI[idx][1]);
    // Extract integral parts and fractional parts of (ph.lo + pm.hi).
    double kh = fputil::nearest_integer(ph.lo);
    double ph_lo_frac = ph.lo - kh; // Exact
    double km = fputil::nearest_integer(pm.hi + ph_lo_frac);
    double pm_hi_frac = pm.hi - km; // Exact
    // x * 128/pi mod 1 ~ y_hi + y_lo
    y_hi = ph_lo_frac + pm_hi_frac; // Exact
    pm_lo = pm.lo;
    return static_cast<unsigned>(static_cast<int64_t>(kh) +
                                 static_cast<int64_t>(km));
  }

  LIBC_INLINE DoubleDouble fast() const {
    // y_lo = x * c_lo + pm.lo
    double y_lo = fputil::multiply_add(x_reduced,
                                       ONE_TWENTY_EIGHT_OVER_PI[idx][2], pm_lo);
    DoubleDouble y = fputil::exact_add(y_hi, y_lo);

    // Digits of pi/128, generated by Sollya with:
    // > a = round(pi/128, D, RN);
    // > b = round(pi/128 - a, D, RN);
    constexpr DoubleDouble PI_OVER_128_DD = {0x1.1a62633145c07p-60,
                                             0x1.921fb54442d18p-6};

    // Error bound: with {a} denote the fractional part of a, i.e.:
    //   {a} = a - round(a)
    // Then,
    //   | {x * 128/pi} - (y_hi + y_lo) | <  2 * ulp(x_reduced *
    //                                         * ONE_TWENTY_EIGHT_OVER_PI[idx][2])
    // For FMA:
    //   | {x * 128/pi} - (y_hi + y_lo) | <= 2 * 2^77 * 2^-103 * 2^-52
    //                                    =  2^-77.
    //   | {x mod pi/128} - (u.hi + u.lo) | < 2 * 2^-6 * 2^-77.
    //                                      = 2^-82.
    // For non-FMA:
    //   | {x * 128/pi} - (y_hi + y_lo) | <= 2 * 2^77 * 2^-99 * 2^-52
    //                                    =  2^-73.
    //   | {x mod pi/128} - (u.hi + u.lo) | < 2 * 2^-6 * 2^-73.
    //                                      = 2^-78.
    return fputil::quick_mult<NO_FMA>(y, PI_OVER_128_DD);
  }

  LIBC_INLINE Float128 accurate() const {
    // y_lo = x * c_lo + pm.lo
    Float128 y_lo_0(x_reduced * ONE_TWENTY_EIGHT_OVER_PI[idx][3]);
    Float128 y_lo_1 = fputil::quick_mul(
        Float128(x_reduced), Float128(ONE_TWENTY_EIGHT_OVER_PI[idx][2]));
    Float128 y_lo_2(pm_lo);
    Float128 y_hi_f128(y_hi);

    Float128 y = fputil::quick_add(
        y_hi_f128,
        fputil::quick_add(y_lo_2, fputil::quick_add(y_lo_1, y_lo_0)));

    return fputil::quick_mul(y, PI_OVER_128_F128);
  }

private:
  // Index of x in the look-up table ONE_TWENTY_EIGHT_OVER_PI.
  unsigned idx;
  // x scaled down by 2^(-16 *(idx - 3))).
  double x_reduced;
  // High part of (x * 128/pi) mod 1.
  double y_hi;
  // Low part of x * ONE_TWENTY_EIGHT_OVER_PI[idx][1].
  double pm_lo;
};

LIBC_INLINE Float128 range_reduction_small_f128(double x) {
  double prod_hi = x * ONE_TWENTY_EIGHT_OVER_PI[3][0];
  double kd = fputil::nearest_integer(prod_hi);

  Float128 mk_f128(-kd);
  Float128 x_f128(x);
  Float128 p_hi =
      fputil::quick_mul(x_f128, Float128(ONE_TWENTY_EIGHT_OVER_PI[3][0]));
  Float128 p_mid =
      fputil::quick_mul(x_f128, Float128(ONE_TWENTY_EIGHT_OVER_PI[3][1]));
  Float128 p_lo =
      fputil::quick_mul(x_f128, Float128(ONE_TWENTY_EIGHT_OVER_PI[3][2]));
  Float128 s_hi = fputil::quick_add(p_hi, mk_f128);
  Float128 s_lo = fputil::quick_add(p_mid, p_lo);
  Float128 y = fputil::quick_add(s_hi, s_lo);

  return fputil::quick_mul(y, PI_OVER_128_F128);
}

LIBC_INLINE constexpr Float128 SIN_K_PI_OVER_128_F128[65] = {
    {Sign::POS, 0, 0},
    {Sign::POS, -133, 0xc90a'afbd'1b33'efc9'c539'edcb'fda0'cf2c_u128},
    {Sign::POS, -132, 0xc8fb'2f88'6ec0'9f37'6a17'954b'2b7c'5171_u128},
    {Sign::POS, -131, 0x96a9'0496'70cf'ae65'f775'7409'4d3c'35c4_u128},
    {Sign::POS, -131, 0xc8bd'35e1'4da1'5f0e'c739'6c89'4bbf'7389_u128},
    {Sign::POS, -131, 0xfab2'72b5'4b98'71a2'7047'29ae'56d7'8a37_u128},
    {Sign::POS, -130, 0x9640'8374'7309'd113'000a'89a1'1e07'c1fe_u128},
    {Sign::POS, -130, 0xaf10'a224'59fe'32a6'3fee'f3bb'58b1'f10d_u128},
    {Sign::POS, -130, 0xc7c5'c1e3'4d30'55b2'5cc8'c00e'4fcc'd850_u128},
    {Sign::POS, -130, 0xe05c'1353'f27b'17e5'0ebc'61ad'e6ca'83cd_u128},
    {Sign::POS, -130, 0xf8cf'cbd9'0af8'd57a'4221'dc4b'a772'598d_u128},
    {Sign::POS, -129, 0x888e'9315'8fb3'bb04'9841'56f5'5334'4306_u128},
    {Sign::POS, -129, 0x94a0'3176'acf8'2d45'ae4b'a773'da6b'f754_u128},
    {Sign::POS, -129, 0xa09a'e4a0'bb30'0a19'2f89'5f44'a303'cc0b_u128},
    {Sign::POS, -129, 0xac7c'd3ad'58fe'e7f0'811f'9539'84ef'f83e_u128},
    {Sign::POS, -129, 0xb844'2987'd22c'f576'9cc3'ef36'746d'e3b8_u128},
    {Sign::POS, -129, 0xc3ef'1535'754b'168d'3122'c2a5'9efd'dc37_u128},
    {Sign::POS, -129, 0xcf7b'ca1d'476c'516d'a812'90bd'baad'62e4_u128},
    {Sign::POS, -129, 0xdae8'804f'0ae6'015b'362c'b974'182e'3030_u128},
    {Sign::POS, -129, 0xe633'74c9'8e22'f0b4'2872'ce1b'fc7a'd1cd_u128},
    {Sign::POS, -129, 0xf15a'e9c0'37b1'd8f0'6c48'e9e3'420b'0f1e_u128},
    {Sign::POS, -129, 0xfc5d'26df'c4d5'cfda'27c0'7c91'1290'b8d1_u128},
    {Sign::POS, -128, 0x839c'3cc9'17ff'6cb4'bfd7'9717'f288'0abf_u128},
    {Sign::POS, -128, 0x88f5'9aa0'da59'1421'b892'ca83'61d8'c84c_u128},
    {Sign::POS, -128, 0x8e39'd9cd'7346'4364'bba4'cfec'bff5'4867_u128},
    {Sign::POS, -128, 0x9368'2a66'e896'f544'b178'2191'1e71'c16e_u128},
    {Sign::POS, -128, 0x987f'bfe7'0b81'a708'19ce'c845'ac87'a5c6_u128},
    {Sign::POS, -128, 0x9d7f'd149'0285'c9e3'e25e'3954'9638'ae68_u128},
    {Sign::POS, -128, 0xa267'9928'48ee'b0c0'3b51'67ee'359a'234e_u128},
    {Sign::POS, -128, 0xa736'55df'1f2f'489e'149f'6e75'9934'68a3_u128},
    {Sign::POS, -128, 0xabeb'49a4'6764'fd15'1bec'da80'89c1'a94c_u128},
    {Sign::POS, -128, 0xb085'baa8'e966'f6da'e4ca'd00d'5c94'bcd2_u128},
    {Sign::POS, -128, 0xb504'f333'f9de'6484'597d'89b3'754a'be9f_u128},
    {Sign::POS, -128, 0xb968'41bf'7ffc'b21a'9de1'e3b2'2b8b'f4db_u128},
    {Sign::POS, -128, 0xbdae'f913'557d'76f0'ac85'320f'528d'6d5d_u128},
    {Sign::POS, -128, 0xc1d8'705f'fcbb'6e90'bdf0'715c'b8b2'0bd7_u128},
    {Sign::POS, -128, 0xc5e4'0358'a8ba'05a7'43da'25d9'9267'326b_u128},
    {Sign::POS, -128, 0xc9d1'124c'931f'da7a'8335'241b'e169'3225_u128},
    {Sign::POS, -128, 0xcd9f'023f'9c3a'059e'23af'31db'7179'a4aa_u128},
    {Sign::POS, -128, 0xd14d'3d02'313c'0eed'744f'ea20'e8ab'ef92_u128},
    {Sign::POS, -128, 0xd4db'3148'750d'1819'f630'e8b6'dac8'3e69_u128},
    {Sign::POS, -128, 0xd848'52c0'a80f'fcdb'24b9'fe00'6635'74a4_u128},
    {Sign::POS, -128, 0xdb94'1a28'cb71'ec87'2c19'b632'53da'43fc_u128},
    {Sign::POS, -128, 0xdebe'0563'7ca9'4cfb'4b19'aa71'fec3'ae6d_u128},
    {Sign::POS, -128, 0xe1c5'978c'05ed'8691'f4e8'a837'2f8c'5810_u128},
    {Sign::POS, -128, 0xe4aa'5909'a08f'a7b4'1227'85ae'67f5'515d_u128},
    {Sign::POS, -128, 0xe76b'd7a1'e63b'9786'1251'2952'9d48'a92f_u128},
    {Sign::POS, -128, 0xea09'a68a'6e49'cd62'15ad'45b4'a1b5'e823_u128},
    {Sign::POS, -128, 0xec83'5e79'946a'3145'7e61'0231'ac1d'6181_u128},
    {Sign::POS, -128, 0xeed8'9db6'6611'e307'86f8'c20f'b664'b01b_u128},
    {Sign::POS, -128, 0xf109'0827'b437'25fd'6712'7db3'5b28'7316_u128},
    {Sign::POS, -128, 0xf314'4762'4708'8f74'a548'6bdc'455d'56a2_u128},
    {Sign::POS, -128, 0xf4fa'0ab6'316e'd2ec'163c'5c7f'03b7'18c5_u128},
    {Sign::POS, -128, 0xf6ba'073b'424b'19e8'2c79'1f59'cc1f'fc23_u128},
    {Sign::POS, -128, 0xf853'f7dc'9186'b952'c7ad'c6b4'9888'91bb_u128},
    {Sign::POS, -128, 0xf9c7'9d63'272c'4628'4504'ae08'd19b'2980_u128},
    {Sign::POS, -128, 0xfb14'be7f'bae5'8156'2172'a361'fd2a'722f_u128},
    {Sign::POS, -128, 0xfc3b'27d3'8a5d'49ab'2567'78ff'cb5c'1769_u128},
    {Sign::POS, -128, 0xfd3a'abf8'4528'b50b'eae6'bd95'1c1d'abbe_u128},
    {Sign::POS, -128, 0xfe13'2387'0cfe'9a3d'90cd'1d95'9db6'74ef_u128},
    {Sign::POS, -128, 0xfec4'6d1e'8929'2cf0'4139'0efd'c726'e9ef_u128},
    {Sign::POS, -128, 0xff4e'6d68'0c41'd0a9'0f66'8633'f1ab'858a_u128},
    {Sign::POS, -128, 0xffb1'0f1b'cb6b'ef1d'421e'8eda'af59'453e_u128},
    {Sign::POS, -128, 0xffec'4304'2668'65d9'5657'5523'6696'1732_u128},
    {Sign::POS, 0, 1},
};

} // namespace generic

} // namespace LIBC_NAMESPACE_DECL

#endif // LLVM_LIBC_SRC_MATH_GENERIC_RANGE_REDUCTION_DOUBLE_COMMON_H
