! This file is part of xtb.
!
! Copyright (C) 2023 Marcel Mueller
!
! xtb is free software: you can redistribute it and/or modify it under
! the terms of the GNU Lesser General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.
!
! xtb is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU Lesser General Public License for more details.
!
! You should have received a copy of the GNU Lesser General Public License
! along with xtb.  If not, see <https://www.gnu.org/licenses/>.

#ifndef WITH_TBLITE
#define WITH_TBLITE 0
#endif

!> PTB basis set data

module xtb_ptb_vdzp

   !> mctc-lib
   use mctc_env, only: error_type, wp
   use mctc_io, only: structure_type
   use mctc_io_constants, only: pi
#if WITH_TBLITE

   use tblite_basis_type, only: cgto_type, basis_type
   use tblite_basis_ortho, only: orthogonalize

   implicit none
   private

   public :: add_vDZP_basis, nshell, max_shell, max_elem, ang_shell, new_basis

   !> Maximum number of shells supported by PTB/vDZP
   integer, parameter :: max_shell = 7

   !> Highest element supported by PTB/vDZP
   integer, parameter :: max_elem = 86

   !> Maximal number of primitives per CGTO
   integer, parameter :: max_prim = 5

   integer, parameter :: nshell(max_elem) = [ &
   & 3, 3, 5, 5, 5, 5, 5, 5, 5, 5, 6, 6, 5, 5, 5, 5, 5, 5, 6, 6, &   ! 1-20
   & 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 5, 5, 5, 5, 5, 5, 6, 6, 7, 7, &   ! 21-40
   & 7, 7, 7, 7, 7, 7, 7, 7, 5, 5, 5, 5, 5, 5, 6, 6, 7, 0, 0, 0, &   ! 41-60
   & 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 7, 7, 7, 7, 7, 7, 7, 7, 7, &   ! 61-80
   & 5, 5, 5, 5, 5, 5]                                               ! 81-86

   !> Angular momenta of basis functions (-> CGTOs)
   integer, parameter :: ang_shell(max_shell, max_elem) = reshape([&
   & 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, & ! up to element: 3
   & 0, 0, 0, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, & ! up to element: 6
   & 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, & ! up to element: 9
   & 0, 0, 1, 1, 2, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, & ! up to element: 12
   & 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, & ! up to element: 15
   & 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, & ! up to element: 18
   & 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 2, & ! up to element: 21
   & 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, & ! up to element: 24
   & 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, & ! up to element: 27
   & 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, & ! up to element: 30
   & 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, & ! up to element: 33
   & 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, & ! up to element: 36
   & 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 2, & ! up to element: 39
   & 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, & ! up to element: 42
   & 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, & ! up to element: 45
   & 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, & ! up to element: 48
   & 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, & ! up to element: 51
   & 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, & ! up to element: 54
   & 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 2, & ! up to element: 57
   & 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, & ! up to element: 60
   & 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, & ! up to element: 63
   & 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, & ! up to element: 66
   & 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, & ! up to element: 69
   & 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 2, & ! up to element: 72
   & 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, & ! up to element: 75
   & 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, & ! up to element: 78
   & 0, 0, 0, 1, 1, 2, 2, 0, 0, 0, 1, 1, 2, 2, 0, 0, 1, 1, 2, 0, 0, & ! up to element: 81
   & 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0, & ! up to element: 84
   & 0, 0, 1, 1, 2, 0, 0, 0, 0, 1, 1, 2, 0, 0], shape(ang_shell))

   !> Contraction length (n_prim) of basis functions (-> CGTOs)
   integer, parameter :: n_prim(max_shell, max_elem) = reshape([&
   & 5, 3, 2, 0, 0, 0, 0, 5, 3, 2, 0, 0, 0, 0, 5, 3, 2, 3, 2, 0, 0, & ! up to element: 3
   & 5, 3, 2, 3, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, & ! up to element: 6
   & 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, & ! up to element: 9
   & 4, 2, 4, 2, 2, 0, 0, 4, 3, 2, 5, 3, 2, 0, 4, 3, 2, 5, 3, 2, 0, & ! up to element: 12
   & 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, & ! up to element: 15
   & 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, & ! up to element: 18
   & 4, 3, 2, 4, 3, 2, 0, 4, 3, 2, 4, 3, 2, 0, 5, 2, 2, 4, 2, 4, 2, & ! up to element: 21
   & 5, 2, 2, 4, 2, 4, 2, 5, 2, 2, 4, 2, 4, 2, 5, 2, 2, 4, 2, 4, 2, & ! up to element: 24
   & 5, 2, 2, 4, 2, 4, 2, 5, 2, 2, 4, 2, 4, 2, 5, 2, 2, 4, 2, 4, 2, & ! up to element: 27
   & 5, 2, 2, 4, 2, 4, 2, 5, 2, 2, 4, 2, 4, 2, 5, 2, 2, 4, 2, 4, 2, & ! up to element: 30
   & 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, & ! up to element: 33
   & 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, & ! up to element: 36
   & 4, 3, 2, 4, 3, 2, 0, 4, 3, 2, 4, 3, 2, 0, 5, 2, 2, 4, 2, 4, 2, & ! up to element: 39
   & 5, 2, 2, 4, 2, 4, 2, 5, 2, 2, 4, 2, 4, 2, 5, 2, 2, 4, 2, 4, 2, & ! up to element: 42
   & 5, 2, 2, 4, 2, 4, 2, 5, 2, 2, 4, 2, 4, 2, 5, 2, 2, 4, 2, 4, 2, & ! up to element: 45
   & 5, 2, 2, 4, 2, 4, 2, 5, 2, 2, 4, 2, 4, 2, 5, 2, 2, 4, 2, 4, 2, & ! up to element: 48
   & 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, & ! up to element: 51
   & 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, & ! up to element: 54
   & 4, 3, 2, 4, 3, 2, 0, 4, 3, 2, 4, 3, 2, 0, 4, 2, 2, 4, 2, 4, 2, & ! up to element: 57
   & 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, & ! up to element: 60
   & 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, & ! up to element: 63
   & 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, & ! up to element: 66
   & 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, & ! up to element: 69
   & 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 2, 2, 4, 2, 4, 2, & ! up to element: 72
   & 4, 2, 2, 4, 2, 4, 2, 4, 2, 2, 4, 2, 4, 2, 4, 2, 2, 4, 2, 4, 2, & ! up to element: 75
   & 4, 2, 2, 4, 2, 4, 2, 4, 2, 2, 4, 2, 4, 2, 4, 2, 2, 4, 2, 4, 2, & ! up to element: 78
   & 4, 2, 2, 4, 2, 4, 2, 4, 2, 2, 4, 2, 4, 2, 4, 2, 4, 2, 2, 0, 0, & ! up to element: 81
   & 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0, & ! up to element: 84
   & 4, 2, 4, 2, 2, 0, 0, 4, 2, 4, 2, 2, 0, 0], shape(n_prim))

   real(wp), parameter :: val_el(max_elem) = [ &
     & 1.0_wp, 2.0_wp, & !He
     & 3.0_wp, 4.0_wp, 3.0_wp, 4.0_wp, 5.0_wp, 6.0_wp, 7.0_wp, 8.0_wp, & !Ne
     & 9.0_wp, 10.0_wp, 3.0_wp, 4.0_wp, 5.0_wp, 6.0_wp, 7.0_wp, 8.0_wp, & !Ar
     & 9.0_wp, 10.0_wp, 11.0_wp, 12.0_wp, 13.0_wp, 14.0_wp, 15.0_wp, 16.0_wp, 17.0_wp, 18.0_wp, 19.0_wp, 20.0_wp, &
     & 3.0_wp, 4.0_wp, 5.0_wp, 6.0_wp, 7.0_wp, 8.0_wp, &  !Kr
     & 9.0_wp, 10.0_wp, 11.0_wp, 12.0_wp, 13.0_wp, 14.0_wp, 15.0_wp, 16.0_wp, 17.0_wp, 18.0_wp, 19.0_wp, 20.0_wp, &
     & 3.0_wp, 4.0_wp, 5.0_wp, 6.0_wp, 7.0_wp, 8.0_wp, &  !Xe
     & 9.0_wp, 10.0_wp, 11.0_wp, &
     & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, &
     & 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, 0.0_wp, & !LNs
     & 12.0_wp, 13.0_wp, 14.0_wp, 15.0_wp, 16.0_wp, 17.0_wp, 18.0_wp, 19.0_wp, 20.0_wp, &
     & 3.0_wp, 4.0_wp, 5.0_wp, 6.0_wp, 7.0_wp, 8.0_wp]  !Rn

   !> CGTO exponents (Exponent of the primitive Gaussian functions)
   real(wp), protected :: exponents(max_prim, max_shell, max_elem) = 0.0_wp

   !> CGTO coefficients (Contraction coefficients of the primitive Gaussian functions,
   !> might contain normalization)
   real(wp), protected :: coefficients(max_prim, max_shell, max_elem) = 0.0_wp

   interface add_vDZP_basis
      module procedure :: add_vDZP_basis_scaling
      module procedure :: add_vDZP_basis_noscaling
   end interface add_vDZP_basis

contains

   subroutine add_vDZP_basis_noscaling(mol, bas)
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Basis set type
      type(basis_type), intent(out) :: bas
      real(wp), allocatable :: expscal(:, :)

      allocate (expscal(max_shell, mol%nat), source=1.0_wp)
      call add_vDZP_basis_scaling(mol, expscal, bas)

   end subroutine add_vDZP_basis_noscaling

   subroutine add_vDZP_basis_scaling(mol, expscal, bas)
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Exponent scaling factor
      real(wp), intent(in) :: expscal(:, :)
      !> Basis set type
      type(basis_type), intent(out) :: bas
      !> Array of CGTOs
      type(cgto_type), allocatable :: cgto(:, :)
      !> Array of nshells per atom ID
      integer, allocatable :: nsh_id(:)
      !> help integers
      integer :: iat, izp, ish, nprim, il, ng, iid

      !> Two over pi
      real(wp), parameter :: top = 2.0_wp/pi
      !> Double factorial, see OEIS A001147
      real(wp), parameter :: dfactorial(8) = &
         & [1.0_wp, 1.0_wp, 3.0_wp, 15.0_wp, 105.0_wp, 945.0_wp, 10395.0_wp, 135135.0_wp]
      real(wp) :: norm
      
      !> debug mode
      logical, parameter :: debug(2) = [ .false., .false. ]

      !> Initialize full parameter set
      !> set up the array of CGTOs for the molecule of interest
      call setCGTOexponents()
      call setCGTOcoefficients()

      nsh_id = nshell(mol%num)
      allocate (cgto(maxval(nsh_id), mol%nat))

      do iat = 1, mol%nat
         iid = mol%id(iat)
         izp = mol%num(iid)
         do ish = 1, nsh_id(iid)

            il = ang_shell(ish, izp)
            nprim = n_prim(ish, izp)

            cgto(ish, iat)%ang = il
            cgto(ish, iat)%nprim = nprim
            cgto(ish, iat)%alpha(1:nprim) = exponents(1:nprim, ish, izp) * expscal(ish, iat)
            cgto(ish, iat)%coeff(1:nprim) = coefficients(1:nprim, ish, izp)

            do ng = 1, nprim
               norm = (top*exponents(ng, ish, izp))**0.75_wp* &
                           & sqrt(4*exponents(ng, ish, izp))**il/sqrt(dfactorial(il + 1))
               
               if(debug(1)) & !##### DEV WRITE #####
                   write(*,*) "norm: ", norm
               
               cgto(ish, iat)%coeff(ng) = cgto(ish, iat)%coeff(ng)*norm
            end do

            if (debug(2)) then !##### DEV WRITE #####
               write (*, *) 'CGTOs for atom ', iat, ' shell ', ish, ' with ', nprim, ' primitives'
               write (*, *) cgto(ish, iat)%alpha
            end if

         end do
      end do
      call new_basis(bas, mol, nsh_id, cgto, 1.0_wp)

   end subroutine add_vDZP_basis_scaling

   !> Create a new basis set
   !> TAKEN OVER FROM TBLITE BUT MODIFIED FOR ATOM-IN-MOLECULE SPECIFIC BASIS SETS
   !> Central change is basically that CGTO is not species specific but atom specific
   subroutine new_basis(self, mol, nshell_id, cgto, acc)
      !> Instance of the basis set data
      type(basis_type), intent(out) :: self
      !> Molecular structure data
      type(structure_type), intent(in) :: mol
      !> Number of shells per species
      integer, intent(in) :: nshell_id(:)
      !> Contracted Gaussian basis functions for each shell and atom (not species)
      type(cgto_type), intent(in) :: cgto(:, :)
      !> Calculation accuracy
      real(wp), intent(in) :: acc

      integer :: iat, isp, ish, iao, ii
      real(wp) :: min_alpha

      self%nsh_id = nshell_id
      self%cgto = cgto
      self%intcut = integral_cutoff(acc)

      ! Make count of shells for each atom
      self%nsh_at = nshell_id(mol%id)

      ! Create mapping between atoms and shells
      self%nsh = sum(self%nsh_at)
      allocate(self%ish_at(mol%nat), self%sh2at(self%nsh))
      ii = 0
      do iat = 1, mol%nat
         self%ish_at(iat) = ii
         do ish = 1, self%nsh_at(iat)
            self%sh2at(ii+ish) = iat
         end do
         ii = ii + self%nsh_at(iat)
      end do

      ! Make count of spherical orbitals for each shell
      allocate(self%nao_sh(self%nsh))
      do iat = 1, mol%nat
         isp = mol%id(iat)
         ii = self%ish_at(iat)
         do ish = 1, self%nsh_at(iat)
            self%nao_sh(ii+ish) = 2*cgto(ish, iat)%ang + 1
         end do
      end do

      ! Create mapping between shells and spherical orbitals, also map directly back to atoms
      self%nao = sum(self%nao_sh)
      allocate(self%iao_sh(self%nsh), self%ao2sh(self%nao), self%ao2at(self%nao))
      ii = 0
      do ish = 1, self%nsh
         self%iao_sh(ish) = ii
         do iao = 1, self%nao_sh(ish)
            self%ao2sh(ii+iao) = ish
            self%ao2at(ii+iao) = self%sh2at(ish)
         end do
         ii = ii + self%nao_sh(ish)
      end do

      ii = 0
      do iat = 1, mol%nat
         isp = mol%id(iat)
         do ish = 1, nshell_id(isp)
            self%iao_sh(ish+self%ish_at(iat)) = ii
            ii = ii + 2*cgto(ish, iat)%ang + 1
         end do
      end do

      min_alpha = huge(acc)
      do iat = 1, mol%nat
         isp = mol%id(iat)
         do ish = 1, nshell_id(isp)
            self%maxl = max(self%maxl, cgto(ish, iat)%ang)
            min_alpha = min(min_alpha, minval(cgto(ish, iat)%alpha(:cgto(ish, iat)%nprim)))
         end do
      end do

      self%min_alpha = min_alpha

   end subroutine new_basis

   !> Create integral cutoff from accuracy value
   pure function integral_cutoff(acc) result(intcut)
      !> Accuracy for the integral cutoff
      real(wp), intent(in) :: acc
      !> Integral cutoff
      real(wp) :: intcut

      real(wp), parameter :: min_intcut = 5.0_wp, max_intcut = 25.0_wp, &
         & max_acc = 1.0e-4_wp, min_acc = 1.0e+3_wp

      intcut = clip(max_intcut - 10*log10(clip(acc, min_acc, max_acc)), min_intcut, max_intcut)
   end function integral_cutoff

   pure function clip(val, min_val, max_val) result(res)
      real(wp), intent(in) :: val, min_val, max_val
      real(wp) :: res
      res = min(max(val, min_val), max_val)
   end function clip

   subroutine setCGTOexponents()
      ! set up the array of CGTOs during initialization of the basis set data
      exponents(:, :, 1) = reshape([&
      &   81.8867808750_wp, 12.2310638614_wp, 2.7868151442_wp, 0.7757866774_wp, 0.2234336928_wp, &
      &    0.3310974836_wp, 0.1074553508_wp, 0.0506805084_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.4170436842_wp, 0.2907814067_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 2) = reshape([&
      &  248.3043592663_wp, 39.2573598600_wp, 9.2902428730_wp, 2.6506789483_wp, 0.8115962676_wp, &
      &    0.2686071619_wp, 0.3450258059_wp, 0.1020071229_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.3100417126_wp, 0.2650087254_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 3) = reshape([&
      &  261.5043973958_wp, 39.4350606126_wp, 8.9030026289_wp, 2.3121285367_wp, 0.6735607405_wp, &
      &    0.6379379194_wp, 0.0630472157_wp, 0.0201349309_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0291876837_wp, 0.0147628243_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.6076858090_wp, 0.2613136142_wp, 0.0787044417_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.2485737044_wp, 0.0884158491_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 4) = reshape([&
      &  510.0323980656_wp, 76.4055567974_wp, 17.2552505089_wp, 4.5860100183_wp, 1.3564305120_wp, &
      &    1.6289233715_wp, 0.1584849748_wp, 0.0639028925_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0569848988_wp, 0.0291546539_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.6588941345_wp, 0.4493798008_wp, 0.1130668801_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.4580820800_wp, 0.1182910232_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 5) = reshape([&
      &    1.5486107510_wp, 1.2308725272_wp, 0.2889361609_wp, 0.1059455346_wp, 0.0000000000_wp, &
      &    0.1494302661_wp, 0.0460783786_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    6.8246256422_wp, 1.7868414679_wp, 0.5294456855_wp, 0.1712887032_wp, 0.0000000000_wp, &
      &    0.1378551304_wp, 0.0456389269_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.6859529283_wp, 0.2171774103_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 6) = reshape([&
      &    2.1749877903_wp, 1.8522269465_wp, 0.4590984876_wp, 0.1691289865_wp, 0.0000000000_wp, &
      &    0.2274550226_wp, 0.0869185910_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   12.8700853337_wp, 3.2954435830_wp, 0.9766457178_wp, 0.3484251971_wp, 0.0000000000_wp, &
      &    0.1464894811_wp, 0.0637339143_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.0253179254_wp, 0.2502490645_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 7) = reshape([&
      &    2.7353785829_wp, 2.2198005987_wp, 0.5927086715_wp, 0.2274069999_wp, 0.0000000000_wp, &
      &    0.2804354568_wp, 0.1053733938_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   19.1045284762_wp, 4.8310138463_wp, 1.4549695059_wp, 0.4947790465_wp, 0.0000000000_wp, &
      &    0.1785303261_wp, 0.0686186522_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.2308129548_wp, 0.3708519692_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 8) = reshape([&
      &    3.5436418206_wp, 2.7127174755_wp, 0.6831845251_wp, 0.2559767392_wp, 0.0000000000_wp, &
      &    0.2865000254_wp, 0.1230120195_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   27.0038418204_wp, 6.7626549662_wp, 2.0117045986_wp, 0.6344283489_wp, 0.0000000000_wp, &
      &    0.1990676604_wp, 0.0656436220_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.3661093404_wp, 0.4619316413_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 9) = reshape([&
      &    7.0348611514_wp, 4.8729322959_wp, 1.3308783230_wp, 0.5254697227_wp, 0.0000000000_wp, &
      &    0.8212660476_wp, 0.1983411850_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   35.5261000398_wp, 9.0935705426_wp, 2.6854132264_wp, 0.8336783951_wp, 0.0000000000_wp, &
      &    0.2622797707_wp, 0.0823962412_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.4549124511_wp, 0.4464982920_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 10) = reshape([&
      &    8.5110251424_wp, 6.9447096523_wp, 1.5610416509_wp, 0.5994966328_wp, 0.0000000000_wp, &
      &    0.6095110352_wp, 0.2456830341_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   56.1111907854_wp, 14.5420587171_wp, 4.4689928374_wp, 1.5047863724_wp, 0.0000000000_wp, &
      &    0.4988083466_wp, 0.1562832516_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.6788770498_wp, 0.4160602900_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 11) = reshape([&
      &   17.8594699798_wp, 1.7415894920_wp, 0.6712166624_wp, 0.3567486799_wp, 0.0000000000_wp, &
      &    1.0405322386_wp, 0.3635864812_wp, 0.0307886700_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0633885947_wp, 0.0278865900_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   94.4089009077_wp, 22.8330353439_wp, 7.3923085900_wp, 2.7004815238_wp, 0.9800414697_wp, &
      &    2.6672915282_wp, 0.3369608432_wp, 0.0552979862_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3177362815_wp, 0.0503994489_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 12) = reshape([&
      &   21.7460780009_wp, 2.0546939637_wp, 0.7753782019_wp, 0.6467550534_wp, 0.0000000000_wp, &
      &    1.5180358741_wp, 0.6266484724_wp, 0.0613424691_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1145257817_wp, 0.0491437987_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   98.1499215778_wp, 22.9869017746_wp, 7.0737044133_wp, 2.3574633603_wp, 0.7767370816_wp, &
      &    3.7713018110_wp, 0.1891668746_wp, 0.0684198793_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.4018066250_wp, 0.1024411340_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 13) = reshape([&
      &    3.3070579509_wp, 1.0649906666_wp, 0.2170949676_wp, 0.1033990508_wp, 0.0000000000_wp, &
      &    0.0913265492_wp, 0.0403846847_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.9669792197_wp, 0.4186026984_wp, 0.1790164620_wp, 0.0927745320_wp, 0.0000000000_wp, &
      &    0.1012752781_wp, 0.0329036919_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.6535205058_wp, 0.1779641308_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 14) = reshape([&
      &    4.2060617638_wp, 1.3790297537_wp, 0.2657401936_wp, 0.1125026975_wp, 0.0000000000_wp, &
      &    0.1336736168_wp, 0.0482978594_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.3675054594_wp, 0.3913149432_wp, 0.1582289307_wp, 0.0689376672_wp, 0.0000000000_wp, &
      &    0.1178174482_wp, 0.0410676559_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.2298032859_wp, 0.3040628459_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 15) = reshape([&
      &    7.4926909130_wp, 1.5209796925_wp, 0.3899864695_wp, 0.1859442469_wp, 0.0000000000_wp, &
      &    0.4262774313_wp, 0.0756411554_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1824498024_wp, 0.7168151222_wp, 0.2843560028_wp, 0.1198446629_wp, 0.0000000000_wp, &
      &    0.1041747369_wp, 0.0429965406_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.6951629604_wp, 0.3518225286_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 16) = reshape([&
      &    7.6013552695_wp, 1.9271212187_wp, 0.5214563075_wp, 0.2305322405_wp, 0.0000000000_wp, &
      &    0.4700053479_wp, 0.0834692104_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.7116392578_wp, 1.1572761602_wp, 0.4497270776_wp, 0.1952907287_wp, 0.0000000000_wp, &
      &    0.1210424109_wp, 0.0610439153_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.7777511183_wp, 0.4044670696_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 17) = reshape([&
      &   14.1511314132_wp, 2.0576977438_wp, 0.8257208749_wp, 0.3024429018_wp, 0.0000000000_wp, &
      &    0.5148884525_wp, 0.0987063230_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.8441742713_wp, 1.3775156286_wp, 0.6191239331_wp, 0.2619220957_wp, 0.0000000000_wp, &
      &    0.1264217887_wp, 0.0739880460_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.9222108147_wp, 0.4283187367_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 18) = reshape([&
      &   14.9401011223_wp, 2.2431656363_wp, 1.4389709266_wp, 0.4068764863_wp, 0.0000000000_wp, &
      &    0.4402596528_wp, 0.1618758735_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    3.1654845112_wp, 1.1111117198_wp, 0.4146494376_wp, 0.1626164943_wp, 0.0000000000_wp, &
      &    0.1543477957_wp, 0.0712708915_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.4678130378_wp, 0.3993964034_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 19) = reshape([&
      &    2.7126821319_wp, 0.9595271592_wp, 0.4158613963_wp, 0.2224283178_wp, 0.0000000000_wp, &
      &    0.7458301087_wp, 0.4477365220_wp, 0.0272429672_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0326606807_wp, 0.0243518102_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   10.0654991445_wp, 0.9740194810_wp, 0.3745963723_wp, 0.1569215209_wp, 0.0000000000_wp, &
      &    4.0420120880_wp, 0.0872590926_wp, 0.0547142578_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.8666385929_wp, 0.1570859979_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 20) = reshape([&
      &    2.7193013515_wp, 1.8832756514_wp, 0.6421639036_wp, 0.3364878673_wp, 0.0000000000_wp, &
      &    3.0263243203_wp, 1.2422832265_wp, 0.0534883483_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0583733675_wp, 0.0299216423_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   10.4115130396_wp, 1.1812018529_wp, 0.4728065452_wp, 0.2063636713_wp, 0.0000000000_wp, &
      &    1.1039635255_wp, 0.0922106260_wp, 0.0076550591_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.4665723495_wp, 0.2615472894_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 21) = reshape([&
      &   12.8945211402_wp, 5.0716862742_wp, 1.1671818996_wp, 0.6639420132_wp, 0.3096545797_wp, &
      &    1.0408857678_wp, 0.5680970016_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0706176928_wp, 0.0272484062_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    6.3685784963_wp, 1.9490040974_wp, 0.8153016946_wp, 0.3138159598_wp, 0.0000000000_wp, &
      &    0.0377104305_wp, 0.0268417048_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   11.6872259113_wp, 3.3147583797_wp, 1.0463367384_wp, 0.3139680731_wp, 0.0000000000_wp, &
      &    0.0963695352_wp, 0.0310540568_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 22) = reshape([&
      &   16.4613478868_wp, 5.3713868155_wp, 1.4489857569_wp, 0.6389195593_wp, 0.2416191513_wp, &
      &    1.0517364425_wp, 0.5899282883_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0752507458_wp, 0.0266020252_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    7.1224296555_wp, 2.3831538366_wp, 1.0288785478_wp, 0.4160699615_wp, 0.0000000000_wp, &
      &    0.1794407196_wp, 0.0767042843_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   16.2855230972_wp, 4.9031864300_wp, 1.6442890149_wp, 0.5252815477_wp, 0.0000000000_wp, &
      &    0.1700966392_wp, 0.0641035253_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 23) = reshape([&
      &   16.2553192652_wp, 6.4050631277_wp, 1.4571636250_wp, 0.4999264247_wp, 0.3011854170_wp, &
      &    1.0126054160_wp, 0.5478869529_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0909336793_wp, 0.0367442926_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    8.1030985723_wp, 2.6298365591_wp, 1.1122023904_wp, 0.4298125109_wp, 0.0000000000_wp, &
      &    0.1341703442_wp, 0.0455417317_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   20.9918521476_wp, 6.4829737952_wp, 2.2596079777_wp, 0.7873884613_wp, 0.0000000000_wp, &
      &    0.2662996354_wp, 0.0858487752_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 24) = reshape([&
      &   19.3268078729_wp, 6.8953520801_wp, 1.8696402671_wp, 0.8438064455_wp, 0.3115868775_wp, &
      &    1.2261441671_wp, 0.5367843617_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1036874552_wp, 0.0404539436_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    9.2749733046_wp, 2.8348032171_wp, 1.1886719048_wp, 0.4526724525_wp, 0.0000000000_wp, &
      &    0.0999434419_wp, 0.0206906499_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   25.3435184433_wp, 7.8371946741_wp, 2.7037309907_wp, 0.9233293323_wp, 0.0000000000_wp, &
      &    0.3017290372_wp, 0.0956591182_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 25) = reshape([&
      &   20.8432280800_wp, 7.7604194191_wp, 1.9045164135_wp, 1.0013204339_wp, 0.4819374725_wp, &
      &    1.2682684204_wp, 0.8572412139_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1091400288_wp, 0.0453189924_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   10.5124368681_wp, 3.0990731358_wp, 1.3229488320_wp, 0.5270553101_wp, 0.0000000000_wp, &
      &    0.3445070603_wp, 0.1183613613_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   31.8182274570_wp, 10.0374216667_wp, 3.5345164179_wp, 1.2497617734_wp, 0.0000000000_wp, &
      &    0.4196638122_wp, 0.1314896584_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 26) = reshape([&
      &   22.3586551512_wp, 8.7267222560_wp, 1.8835113460_wp, 0.8980988298_wp, 0.4991414954_wp, &
      &    1.4146727331_wp, 0.8389699617_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1269671899_wp, 0.0482976870_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   11.7373357965_wp, 3.4917405584_wp, 1.5196703141_wp, 0.6122302450_wp, 0.0000000000_wp, &
      &    0.2707405127_wp, 0.1109565205_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   37.6024365136_wp, 11.9804690214_wp, 4.3158628480_wp, 1.5838176619_wp, 0.0000000000_wp, &
      &    0.5483572563_wp, 0.1714773606_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 27) = reshape([&
      &   25.3353156945_wp, 9.4883550548_wp, 2.1057708661_wp, 1.1527415631_wp, 0.6001346519_wp, &
      &    1.7285930484_wp, 0.9651354088_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1382175379_wp, 0.0487996195_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   13.0198468256_wp, 3.8985691282_wp, 1.6919120417_wp, 0.6631943270_wp, 0.0000000000_wp, &
      &    0.2662337363_wp, 0.0876826023_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   46.4671865021_wp, 14.8447279630_wp, 5.3247819103_wp, 1.9461930200_wp, 0.0000000000_wp, &
      &    0.6636977666_wp, 0.1975387031_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 28) = reshape([&
      &   28.1173073289_wp, 10.3231141462_wp, 2.2665013347_wp, 1.0316983380_wp, 0.4297631165_wp, &
      &    1.5850961164_wp, 0.9276884260_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1427239817_wp, 0.0522885623_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   14.6492634274_wp, 4.4872485553_wp, 1.9265222019_wp, 0.7555711824_wp, 0.0000000000_wp, &
      &    0.3209233380_wp, 0.1245974388_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   55.7955968733_wp, 17.6717134428_wp, 6.4588843706_wp, 2.3754495803_wp, 0.0000000000_wp, &
      &    0.8085979068_wp, 0.2392562094_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 29) = reshape([&
      &   30.0956544648_wp, 11.1894654151_wp, 2.5414521184_wp, 1.2905107640_wp, 0.6089618157_wp, &
      &    2.0184304785_wp, 0.8491389749_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1556092510_wp, 0.0551002639_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   16.1979168969_wp, 4.4879083134_wp, 1.9283660776_wp, 0.7639432366_wp, 0.0000000000_wp, &
      &    0.2998402276_wp, 0.1167734034_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   60.4872979389_wp, 19.2330464057_wp, 6.9650906898_wp, 2.5689033725_wp, 0.0000000000_wp, &
      &    0.8758096281_wp, 0.2576905092_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 30) = reshape([&
      &   33.8744737144_wp, 11.9726229564_wp, 2.9617143960_wp, 1.8299819125_wp, 0.8705547357_wp, &
      &    1.7392038607_wp, 0.7849581068_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1737774828_wp, 0.0589361576_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   18.4288527365_wp, 4.7578285162_wp, 2.0515049111_wp, 0.8112900902_wp, 0.0000000000_wp, &
      &    0.2803295274_wp, 0.0749676294_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   69.3457839202_wp, 22.3446434630_wp, 8.2300737468_wp, 3.1296076376_wp, 0.0000000000_wp, &
      &    1.1154453442_wp, 0.3504384339_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 31) = reshape([&
      &    3.4618510997_wp, 1.6733742718_wp, 0.2113183680_wp, 0.0978281533_wp, 0.0000000000_wp, &
      &    0.1516270107_wp, 0.0556043763_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.2240721969_wp, 0.5703406870_wp, 0.1763421819_wp, 0.0693381271_wp, 0.0000000000_wp, &
      &    0.1032021487_wp, 0.0325213621_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.4341367587_wp, 0.1364369508_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 32) = reshape([&
      &    3.2174961017_wp, 1.9181665726_wp, 0.2149038643_wp, 0.0675899303_wp, 0.0000000000_wp, &
      &    0.4656962588_wp, 0.0601740251_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    3.9141795570_wp, 2.3464686306_wp, 0.2738611119_wp, 0.1059900713_wp, 0.0000000000_wp, &
      &    0.2021350652_wp, 0.0612146871_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3700096331_wp, 0.1329762014_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 33) = reshape([&
      &    3.5076936430_wp, 1.8883830260_wp, 0.2881112405_wp, 0.1431800881_wp, 0.0000000000_wp, &
      &    0.3524385142_wp, 0.0731264985_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.3582554091_wp, 0.9346477028_wp, 0.2926544377_wp, 0.1229273186_wp, 0.0000000000_wp, &
      &    0.1187675371_wp, 0.0476051037_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3144879389_wp, 0.2184939652_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 34) = reshape([&
      &    3.6508657306_wp, 2.2109036092_wp, 0.3351709281_wp, 0.1516388367_wp, 0.0000000000_wp, &
      &    0.5996388222_wp, 0.0755821343_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.5071482574_wp, 0.9133674146_wp, 0.3412219787_wp, 0.1504325873_wp, 0.0000000000_wp, &
      &    0.1570828545_wp, 0.0612724269_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3675975026_wp, 0.2230131874_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 35) = reshape([&
      &    4.2140146350_wp, 2.4121886986_wp, 0.3910670270_wp, 0.1646742527_wp, 0.0000000000_wp, &
      &    0.4855996647_wp, 0.0745989946_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.7899472531_wp, 1.2920361874_wp, 0.4493832522_wp, 0.1945901673_wp, 0.0000000000_wp, &
      &    0.1227597600_wp, 0.0667386257_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.4424381466_wp, 0.2458490188_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 36) = reshape([&
      &    3.9865114406_wp, 2.9439017448_wp, 0.4299018667_wp, 0.1518218389_wp, 0.0000000000_wp, &
      &    0.3347118130_wp, 0.1117451689_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.1539576862_wp, 1.2837486855_wp, 0.5016855617_wp, 0.2375202680_wp, 0.0000000000_wp, &
      &    0.1401131612_wp, 0.0892049086_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.5263012853_wp, 0.2031002668_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 37) = reshape([&
      &    3.8699995998_wp, 2.1309046361_wp, 0.7380020268_wp, 0.4095287428_wp, 0.0000000000_wp, &
      &    1.3505807268_wp, 0.6721234614_wp, 0.2150260402_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0417275737_wp, 0.0181730854_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.7674104175_wp, 0.6902792725_wp, 0.2721346865_wp, 0.0998613142_wp, 0.0000000000_wp, &
      &    0.4838684259_wp, 0.0649903391_wp, 0.0263441670_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.4829529086_wp, 0.1185996801_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 38) = reshape([&
      &    1.6927066375_wp, 1.1497957858_wp, 0.6691314779_wp, 0.2482200767_wp, 0.0000000000_wp, &
      &    1.8862665114_wp, 0.4001643902_wp, 0.0879286172_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0594431968_wp, 0.0285846861_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.8066476993_wp, 0.8193765247_wp, 0.3481582678_wp, 0.1460036470_wp, 0.0000000000_wp, &
      &    0.9351504983_wp, 0.0877863570_wp, 0.0454209931_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.6884063529_wp, 0.1927046470_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 39) = reshape([&
      &    7.2965498919_wp, 2.6884221856_wp, 2.1776425724_wp, 0.6590906836_wp, 0.3180003131_wp, &
      &    0.6748129575_wp, 0.2903533626_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0616678237_wp, 0.0256735122_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.4130739059_wp, 1.9911183354_wp, 0.6448850290_wp, 0.2621581495_wp, 0.0000000000_wp, &
      &    0.0902518344_wp, 0.0387345409_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.6406670823_wp, 1.3772316933_wp, 0.5211703035_wp, 0.1913179457_wp, 0.0000000000_wp, &
      &    0.0665611720_wp, 0.0210587140_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 40) = reshape([&
      &    7.6367554594_wp, 3.0594606047_wp, 2.7342894697_wp, 0.9740305298_wp, 0.3212377682_wp, &
      &    0.6661183133_wp, 0.3820049300_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0786389850_wp, 0.0309777078_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.6353045367_wp, 2.1397016730_wp, 0.7012702619_wp, 0.2941175758_wp, 0.0000000000_wp, &
      &    0.1539634610_wp, 0.0882621012_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    3.2010475777_wp, 1.4850487280_wp, 0.5595742535_wp, 0.1966188689_wp, 0.0000000000_wp, &
      &    0.0624518154_wp, 0.0203259288_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 41) = reshape([&
      &    8.2269076724_wp, 5.2588539382_wp, 4.0249090030_wp, 1.1951299632_wp, 0.3596657345_wp, &
      &    0.7809721116_wp, 0.4681497899_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0843303000_wp, 0.0349186666_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.8713515900_wp, 2.2809355967_wp, 0.7435514907_wp, 0.2980692503_wp, 0.0000000000_wp, &
      &    0.1525956055_wp, 0.0683274547_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    3.3678803063_wp, 1.7565647813_wp, 0.6587350177_wp, 0.2310593964_wp, 0.0000000000_wp, &
      &    0.0722162264_wp, 0.0248010999_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 42) = reshape([&
      &    9.6737525129_wp, 3.5722821467_wp, 1.5308497549_wp, 0.6319968398_wp, 0.2888393295_wp, &
      &    0.7784179969_wp, 0.5341726528_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0876265252_wp, 0.0335626819_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    3.0937076734_wp, 2.4036418734_wp, 0.8182403149_wp, 0.3362058800_wp, 0.0000000000_wp, &
      &    0.1419361620_wp, 0.0511420579_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    4.2487066377_wp, 2.0430767011_wp, 0.8192195992_wp, 0.3006232872_wp, 0.0000000000_wp, &
      &    0.0999249399_wp, 0.0338375854_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 43) = reshape([&
      &    8.7344882315_wp, 3.8358990951_wp, 3.4697517447_wp, 0.9671245162_wp, 0.4241826040_wp, &
      &    1.0039245272_wp, 0.5430131163_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0880942370_wp, 0.0336827749_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    3.4050239849_wp, 2.5642650740_wp, 0.8635816252_wp, 0.3374110300_wp, 0.0000000000_wp, &
      &    0.0917955177_wp, 0.0298792828_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    5.0977321108_wp, 2.1871383332_wp, 0.9766244759_wp, 0.3960519350_wp, 0.0000000000_wp, &
      &    0.1481100405_wp, 0.0596886916_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 44) = reshape([&
      &    9.6965004814_wp, 3.9864902077_wp, 3.5072705542_wp, 1.0443986380_wp, 0.4487519200_wp, &
      &    1.0644030456_wp, 0.6277799828_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1017931840_wp, 0.0347764120_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    3.7634042951_wp, 2.7708724580_wp, 0.9885724310_wp, 0.4085978545_wp, 0.0000000000_wp, &
      &    0.1876870043_wp, 0.0515047396_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    5.5500320903_wp, 2.4463558194_wp, 1.0733420255_wp, 0.4293894049_wp, 0.0000000000_wp, &
      &    0.1650526452_wp, 0.0688448973_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 45) = reshape([&
      &   10.6122771600_wp, 4.1757113576_wp, 3.5401982647_wp, 1.0547124576_wp, 0.4634769418_wp, &
      &    1.1209878856_wp, 0.5053544402_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1077990558_wp, 0.0355937083_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    3.9157627601_wp, 3.2149462727_wp, 1.1383900143_wp, 0.4839002065_wp, 0.0000000000_wp, &
      &    0.1738017907_wp, 0.0589550838_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    6.3968274588_wp, 2.9083247683_wp, 1.3676661206_wp, 0.5730745730_wp, 0.0000000000_wp, &
      &    0.2154179849_wp, 0.0746944756_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 46) = reshape([&
      &   11.2600422052_wp, 4.4475544079_wp, 3.7989504299_wp, 1.1251912398_wp, 0.4525560293_wp, &
      &    1.1515059615_wp, 0.7580228873_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1143411069_wp, 0.0416220716_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    4.2802023309_wp, 3.2583939379_wp, 1.1654549903_wp, 0.4882210994_wp, 0.0000000000_wp, &
      &    0.2012753948_wp, 0.0570535843_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    6.9393350390_wp, 3.0041306102_wp, 1.3655900496_wp, 0.5705188226_wp, 0.0000000000_wp, &
      &    0.2352113063_wp, 0.0985486147_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 47) = reshape([&
      &   11.1373801858_wp, 4.9896862939_wp, 4.4759201941_wp, 1.2014863432_wp, 0.5103926679_wp, &
      &    1.1688652945_wp, 0.7038925085_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1093741729_wp, 0.0337765840_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    4.5563726846_wp, 3.5131662613_wp, 1.2447948640_wp, 0.5120477386_wp, 0.0000000000_wp, &
      &    0.1764168001_wp, 0.0523720328_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    7.4464533918_wp, 3.2559279762_wp, 1.4581773858_wp, 0.5963057185_wp, 0.0000000000_wp, &
      &    0.2298255135_wp, 0.0901519699_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 48) = reshape([&
      &   12.0172445048_wp, 5.1142178256_wp, 4.4469401154_wp, 1.2166283492_wp, 0.5320521267_wp, &
      &    1.0551059874_wp, 0.6041483293_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1463833943_wp, 0.0470464794_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    4.7572332466_wp, 3.8673724081_wp, 1.3588230771_wp, 0.5770987793_wp, 0.0000000000_wp, &
      &    0.1823608158_wp, 0.0624192760_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    8.1100200590_wp, 3.4517267686_wp, 1.5817082818_wp, 0.6734079757_wp, 0.0000000000_wp, &
      &    0.2726344689_wp, 0.1133398673_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 49) = reshape([&
      &    1.4247532048_wp, 0.9677765183_wp, 0.1893165802_wp, 0.0696829867_wp, 0.0000000000_wp, &
      &    0.2620640235_wp, 0.0644587710_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.8104490271_wp, 1.0466434967_wp, 0.1849875781_wp, 0.0718616198_wp, 0.0000000000_wp, &
      &    0.1380981709_wp, 0.0412475809_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1382420051_wp, 0.0797773004_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 50) = reshape([&
      &    2.4010620788_wp, 1.1533161155_wp, 0.2269147644_wp, 0.1100379485_wp, 0.0000000000_wp, &
      &    0.3172254872_wp, 0.0610539972_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.5668984255_wp, 1.4992604360_wp, 0.2270347476_wp, 0.0921794989_wp, 0.0000000000_wp, &
      &    0.1734073471_wp, 0.0486745497_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.2411023860_wp, 0.1165513360_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 51) = reshape([&
      &    1.9012914964_wp, 1.4494751203_wp, 0.2551796634_wp, 0.1595573014_wp, 0.0000000000_wp, &
      &    0.3551141266_wp, 0.0673931419_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.3756898517_wp, 1.3480627744_wp, 0.2437446368_wp, 0.1071669900_wp, 0.0000000000_wp, &
      &    0.2054194487_wp, 0.0629162913_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.2052842489_wp, 0.1611946851_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 52) = reshape([&
      &    2.0835141307_wp, 1.7145079016_wp, 0.2437218570_wp, 0.1306813705_wp, 0.0000000000_wp, &
      &    0.3157944051_wp, 0.0882213312_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.4810128106_wp, 1.3991692055_wp, 0.2299463082_wp, 0.1274346383_wp, 0.0000000000_wp, &
      &    0.5038164194_wp, 0.0568096234_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1920181830_wp, 0.0644378649_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 53) = reshape([&
      &    2.4676377559_wp, 1.6969281624_wp, 0.2988362153_wp, 0.1391992270_wp, 0.0000000000_wp, &
      &    0.2414376128_wp, 0.0782099064_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.9544178660_wp, 0.6635898595_wp, 0.2367501342_wp, 0.1305847395_wp, 0.0000000000_wp, &
      &    0.2291682425_wp, 0.0672208846_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3279416236_wp, 0.2285145459_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 54) = reshape([&
      &    2.7550959248_wp, 1.7383123670_wp, 0.3580466505_wp, 0.1421246036_wp, 0.0000000000_wp, &
      &    0.2754788502_wp, 0.1009070576_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1178397193_wp, 0.6912679101_wp, 0.2792677131_wp, 0.1559554989_wp, 0.0000000000_wp, &
      &    0.2653520270_wp, 0.0758927443_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.4027802085_wp, 0.2436328269_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 55) = reshape([&
      &    2.1724333233_wp, 1.2135594299_wp, 0.9130741810_wp, 0.3138804982_wp, 0.0000000000_wp, &
      &    1.2944343903_wp, 1.0887562635_wp, 0.1419902019_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1104868186_wp, 0.0210362262_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.2966120628_wp, 0.5115876495_wp, 0.2015500923_wp, 0.0653384781_wp, 0.0000000000_wp, &
      &    0.2996757223_wp, 0.0634708433_wp, 0.0293637535_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.2880131530_wp, 0.0947761533_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 56) = reshape([&
      &    2.1652244396_wp, 1.5636066630_wp, 0.4683886601_wp, 0.1824906303_wp, 0.0000000000_wp, &
      &    0.9698118812_wp, 0.4383358076_wp, 0.0514139275_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0384206855_wp, 0.0348271566_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.2468676773_wp, 0.6197191042_wp, 0.3147221920_wp, 0.2182148726_wp, 0.0000000000_wp, &
      &    0.8888232380_wp, 0.1123394250_wp, 0.0489172515_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3763627984_wp, 0.1226783911_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 57) = reshape([&
      &    2.9615188153_wp, 2.1168988873_wp, 0.5211524650_wp, 0.2300693986_wp, 0.0000000000_wp, &
      &    0.5795461193_wp, 0.2905097529_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0489167626_wp, 0.0186047375_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    3.2452456432_wp, 2.4708904354_wp, 0.5453038570_wp, 0.2165379667_wp, 0.0000000000_wp, &
      &    0.0332993691_wp, 0.0067457242_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.5098488156_wp, 0.7430098754_wp, 0.3398791350_wp, 0.1375122865_wp, 0.0000000000_wp, &
      &    0.0604322522_wp, 0.0214467711_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 72) = reshape([&
      &    5.7452639133_wp, 3.8941425537_wp, 3.1210273359_wp, 0.6354967021_wp, 0.0000000000_wp, &
      &    0.7648443510_wp, 0.2903396851_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0820913252_wp, 0.0255054811_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    8.0473497562_wp, 5.3531549053_wp, 0.9897313028_wp, 0.3730849556_wp, 0.0000000000_wp, &
      &    0.0429307807_wp, 0.0182045448_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    3.4306862559_wp, 1.2846387566_wp, 0.4337185176_wp, 0.1502958512_wp, 0.0000000000_wp, &
      &    0.0725037975_wp, 0.0270618213_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 73) = reshape([&
      &    5.9638037077_wp, 3.9611166778_wp, 3.0857666326_wp, 0.6658087014_wp, 0.0000000000_wp, &
      &    0.8059723198_wp, 0.2689305320_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0865534297_wp, 0.0277911573_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    8.1829042084_wp, 5.4190107277_wp, 1.0808973054_wp, 0.4271018590_wp, 0.0000000000_wp, &
      &    0.1236651665_wp, 0.0746313915_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    3.9387808580_wp, 1.3674991382_wp, 0.6299359454_wp, 0.2730924079_wp, 0.0000000000_wp, &
      &    0.0999256830_wp, 0.0379438526_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 74) = reshape([&
      &    6.1061259471_wp, 4.1277025702_wp, 3.2749708173_wp, 0.7330044729_wp, 0.0000000000_wp, &
      &    0.8414440672_wp, 0.2717802127_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1003318127_wp, 0.0296327328_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    8.2855535298_wp, 5.6156781876_wp, 1.1585730468_wp, 0.4676173803_wp, 0.0000000000_wp, &
      &    0.1372706277_wp, 0.0822936787_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    4.1921268470_wp, 1.4484135624_wp, 0.6266278251_wp, 0.2491246702_wp, 0.0000000000_wp, &
      &    0.0974213903_wp, 0.0364470548_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 75) = reshape([&
      &    6.4887820034_wp, 4.2175684596_wp, 3.2628794282_wp, 0.7547842228_wp, 0.0000000000_wp, &
      &    0.9410817346_wp, 0.2727123557_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1111268601_wp, 0.0400093551_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    7.5204976817_wp, 6.4982585062_wp, 1.2164273942_wp, 0.4951584383_wp, 0.0000000000_wp, &
      &    0.1361864415_wp, 0.0824413577_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    4.4242760906_wp, 1.5375286439_wp, 0.7208507104_wp, 0.3123777034_wp, 0.0000000000_wp, &
      &    0.1227230957_wp, 0.0503446455_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 76) = reshape([&
      &    6.5893015210_wp, 4.4834108264_wp, 3.5987847476_wp, 0.8304828739_wp, 0.0000000000_wp, &
      &    0.9744019859_wp, 0.3370998802_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1193714779_wp, 0.0403346528_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    8.0188685081_wp, 6.5163330882_wp, 1.2986616050_wp, 0.5370470081_wp, 0.0000000000_wp, &
      &    0.1634255953_wp, 0.0467524570_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    4.5620228869_wp, 1.7004981704_wp, 0.7839635055_wp, 0.3330344193_wp, 0.0000000000_wp, &
      &    0.1283203114_wp, 0.0514946900_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 77) = reshape([&
      &    6.9906964809_wp, 4.6576288266_wp, 3.8221332403_wp, 0.8888256223_wp, 0.0000000000_wp, &
      &    1.0237937702_wp, 0.3091993156_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1279674794_wp, 0.0400885609_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    8.6890881851_wp, 6.5773665586_wp, 1.3564133544_wp, 0.5449280936_wp, 0.0000000000_wp, &
      &    0.1407943817_wp, 0.0488516269_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    4.6651786647_wp, 1.8060504241_wp, 0.7873946253_wp, 0.3213739876_wp, 0.0000000000_wp, &
      &    0.1216252992_wp, 0.0491056990_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 78) = reshape([&
      &    7.6794126159_wp, 4.7090877596_wp, 3.6453135177_wp, 0.9534209508_wp, 0.0000000000_wp, &
      &    1.0277697705_wp, 0.8958271695_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1395552974_wp, 0.0493405229_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    8.2891445388_wp, 7.3163111403_wp, 1.4491099803_wp, 0.6092433881_wp, 0.0000000000_wp, &
      &    0.1936831197_wp, 0.0441661444_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    4.9103329499_wp, 1.9386914265_wp, 0.8647475616_wp, 0.3573322836_wp, 0.0000000000_wp, &
      &    0.1379463366_wp, 0.0502428297_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 79) = reshape([&
      &    8.4822473298_wp, 5.0088873714_wp, 3.9534630804_wp, 1.0166742808_wp, 0.0000000000_wp, &
      &    1.2445323177_wp, 0.3098858253_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1475963975_wp, 0.0477196468_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    8.8315546905_wp, 7.5662827943_wp, 1.5067004098_wp, 0.6128226931_wp, 0.0000000000_wp, &
      &    0.1762219651_wp, 0.0619940122_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    5.1498883740_wp, 2.0382878178_wp, 0.8980172069_wp, 0.3626358585_wp, 0.0000000000_wp, &
      &    0.1384587406_wp, 0.0467155913_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 80) = reshape([&
      &    9.5623002294_wp, 5.0793955320_wp, 3.9828504715_wp, 0.9611274009_wp, 0.0000000000_wp, &
      &    1.3778030600_wp, 0.3814406712_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1723514344_wp, 0.0639438407_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    9.8931019325_wp, 7.4238124397_wp, 1.6007386060_wp, 0.6526770962_wp, 0.0000000000_wp, &
      &    0.1834980871_wp, 0.0645928586_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    5.3156924429_wp, 2.2455545985_wp, 1.0019472664_wp, 0.4139028865_wp, 0.0000000000_wp, &
      &    0.1672136426_wp, 0.0670981617_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 81) = reshape([&
      &    1.5053130494_wp, 0.9264066389_wp, 0.1954724419_wp, 0.0791975715_wp, 0.0000000000_wp, &
      &    0.1794233670_wp, 0.0720618462_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.3404453909_wp, 0.8666047851_wp, 0.2171234410_wp, 0.0914086728_wp, 0.0000000000_wp, &
      &    0.5918172747_wp, 0.0436824964_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1173148370_wp, 0.0544590363_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 82) = reshape([&
      &    1.3417530042_wp, 1.1054976895_wp, 0.2116990015_wp, 0.1295478557_wp, 0.0000000000_wp, &
      &    0.5497060340_wp, 0.0632848495_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.4148039892_wp, 1.0240692151_wp, 0.1860290058_wp, 0.0827963010_wp, 0.0000000000_wp, &
      &    0.3310882696_wp, 0.0464829162_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.2004936720_wp, 0.0882621230_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 83) = reshape([&
      &    1.5984523550_wp, 1.0240319342_wp, 0.2116943619_wp, 0.1883153252_wp, 0.0000000000_wp, &
      &    0.3024257522_wp, 0.0849940521_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.4947083461_wp, 1.2543674099_wp, 0.2071753160_wp, 0.0750395300_wp, 0.0000000000_wp, &
      &    0.1999851312_wp, 0.0623732155_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1256429674_wp, 0.0928094533_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 84) = reshape([&
      &    1.8993154122_wp, 1.3302056843_wp, 0.2954490299_wp, 0.1994519224_wp, 0.0000000000_wp, &
      &    0.3150252995_wp, 0.1049884112_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.6119096388_wp, 1.0998677631_wp, 0.2066178999_wp, 0.1209901326_wp, 0.0000000000_wp, &
      &    0.4711977042_wp, 0.0626035323_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1654339419_wp, 0.0626740491_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 85) = reshape([&
      &    1.9241485807_wp, 1.3466887051_wp, 0.3594391132_wp, 0.1887821167_wp, 0.0000000000_wp, &
      &    0.4367330461_wp, 0.0866602027_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.0825443537_wp, 1.1602957126_wp, 0.3505634635_wp, 0.1602206951_wp, 0.0000000000_wp, &
      &    0.1913146944_wp, 0.0692409349_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.2861954909_wp, 0.1687580258_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      exponents(:, :, 86) = reshape([&
      &    1.9740197826_wp, 1.4783565006_wp, 0.3382927196_wp, 0.1380506620_wp, 0.0000000000_wp, &
      &    0.2392385224_wp, 0.1151567344_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.8660461127_wp, 1.5723348408_wp, 0.3177162471_wp, 0.1412398740_wp, 0.0000000000_wp, &
      &    0.2003539513_wp, 0.0771900579_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3330655580_wp, 0.2141086265_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

   end subroutine setCGTOexponents

   subroutine setCGTOcoefficients()
      ! set up the array of CGTOs during initialization of the basis set data
      coefficients(:, :, 1) = reshape([&
      &    0.0084239542_wp, 0.0648612854_wp, 0.3114008836_wp, 0.9853080817_wp, 1.2568199629_wp, &
      &    0.0522923008_wp, 0.1041393028_wp, 0.2451157144_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.7597658486_wp, 1.5228446261_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 2) = reshape([&
      &    0.0050137918_wp, 0.0349837015_wp, 0.1629731956_wp, 0.4896910164_wp, 0.9328993507_wp, &
      &    0.2228774686_wp, 0.0621111128_wp, 0.0913424292_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.6969879844_wp, 0.1784886288_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 3) = reshape([&
      &    0.0079958373_wp, 0.0587533458_wp, 0.2510129190_wp, 0.6011369304_wp, 0.4782661246_wp, &
      &   -0.1464729241_wp, 0.8216610557_wp, 0.0347735148_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.0626458835_wp, 1.0955097156_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0888083392_wp, 0.4240666998_wp, 0.7379982447_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.6947435151_wp, 0.7887153535_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 4) = reshape([&
      &    0.0058051846_wp, 0.0433420461_wp, 0.1889081431_wp, 0.4696335722_wp, 0.3958965761_wp, &
      &   -0.1344930671_wp, 0.9447373319_wp, 0.1710320121_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.8936588255_wp, 1.4630004992_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0253021648_wp, 0.1449091249_wp, 0.2005520187_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.8813959265_wp, 0.9934250366_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 5) = reshape([&
      &   -0.3754535307_wp, 0.2963831947_wp, 0.4002811324_wp, 0.2523396505_wp, 0.0000000000_wp, &
      &    0.6319072629_wp, 0.7178197535_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0320550548_wp, 0.1686295904_wp, 0.5082224532_wp, 0.4492004551_wp, 0.0000000000_wp, &
      &    1.1227628920_wp, 1.6880879207_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.2023177121_wp, 1.3781368298_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 6) = reshape([&
      &   -0.2760589108_wp, 0.2401547511_wp, 0.1911248622_wp, 0.0872944675_wp, 0.0000000000_wp, &
      &    0.9879381267_wp, 1.0166507890_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0247414382_wp, 0.1521920799_wp, 0.4486742981_wp, 0.4947569383_wp, 0.0000000000_wp, &
      &    1.5944158113_wp, 0.8078591428_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.8331878654_wp, 0.8582635673_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 7) = reshape([&
      &   -0.2395937966_wp, 0.2249896934_wp, 0.1945177612_wp, 0.0981255757_wp, 0.0000000000_wp, &
      &    0.8535154775_wp, 1.1307272405_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0117926375_wp, 0.0729002872_wp, 0.2055427628_wp, 0.2583582066_wp, 0.0000000000_wp, &
      &    1.2421466979_wp, 0.4290510283_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.2618941500_wp, 1.1816595669_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 8) = reshape([&
      &   -0.2635542930_wp, 0.2656788314_wp, 0.2790872926_wp, 0.1128685562_wp, 0.0000000000_wp, &
      &    1.0275945773_wp, 1.1591665161_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0124701545_wp, 0.0745207762_wp, 0.2100798910_wp, 0.2913197287_wp, 0.0000000000_wp, &
      &    1.5124136595_wp, 0.3126349791_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.8582942309_wp, 0.8460346661_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 9) = reshape([&
      &   -0.4460464175_wp, 0.2411203826_wp, 0.6665933072_wp, 0.7819171613_wp, 0.0000000000_wp, &
      &    0.1355442440_wp, 0.9935540101_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0268182499_wp, 0.1335966811_wp, 0.3763263551_wp, 0.5166814524_wp, 0.0000000000_wp, &
      &    0.9312219312_wp, 0.2006698726_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.3694287271_wp, 1.4665640355_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 10) = reshape([&
      &   -0.3464077314_wp, 0.2424283468_wp, 0.4568106077_wp, 0.4139917398_wp, 0.0000000000_wp, &
      &    0.1756686950_wp, 0.8548929299_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0186171975_wp, 0.1199645115_wp, 0.3515700926_wp, 0.5527585569_wp, 0.0000000000_wp, &
      &    0.9596176552_wp, 0.2937398160_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.5625013690_wp, 1.4506867461_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 11) = reshape([&
      &   -0.0946393954_wp, 0.4543680948_wp, 0.3261102196_wp, 0.0997333483_wp, 0.0000000000_wp, &
      &    1.1289284600_wp, 0.9712702544_wp, 0.7250212779_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.4610747063_wp, 0.8652179678_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0222855204_wp, 0.1410948143_wp, 0.4564529358_wp, 0.8119317011_wp, 0.8656371592_wp, &
      &   -0.0058244669_wp, 0.5247693421_wp, 0.0117672347_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.2163057446_wp, 1.0845205284_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 12) = reshape([&
      &   -0.1077124154_wp, 0.6050399451_wp, 0.3582532187_wp, 0.1244565874_wp, 0.0000000000_wp, &
      &    0.9533253035_wp, 0.8850528467_wp, 0.6632005974_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1028211537_wp, 1.2628271970_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0200305323_wp, 0.1306459616_wp, 0.4163059034_wp, 0.6646215749_wp, 0.4609532743_wp, &
      &   -0.0080329346_wp, 0.3403696886_wp, 0.2601278563_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.8466228930_wp, 1.5338934586_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 13) = reshape([&
      &    0.0358876794_wp, -0.3014343231_wp, 0.4080036741_wp, 0.5049445652_wp, 0.0000000000_wp, &
      &    0.2672501233_wp, 0.4240937855_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0935935101_wp, 0.1860419116_wp, 0.3645718593_wp, 0.2700830937_wp, 0.0000000000_wp, &
      &    0.5625226590_wp, 0.8295159399_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.5450977055_wp, 1.1492282501_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 14) = reshape([&
      &    0.0360461004_wp, -0.2786216598_wp, 0.5140323046_wp, 0.3657775520_wp, 0.0000000000_wp, &
      &    0.2518606786_wp, 1.0019122172_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0453117942_wp, 0.2885813853_wp, 0.3722645445_wp, 0.1007765756_wp, 0.0000000000_wp, &
      &    0.5117843044_wp, 0.7906991822_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3174905262_wp, 1.1608021024_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 15) = reshape([&
      &    0.0362112569_wp, -0.4103465246_wp, 0.5904311605_wp, 0.7000264691_wp, 0.0000000000_wp, &
      &    0.0669462096_wp, 0.6628689776_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0707083346_wp, 0.1797981374_wp, 0.3471855375_wp, 0.1861561034_wp, 0.0000000000_wp, &
      &    0.6421736782_wp, 0.5929794893_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.2574706574_wp, 1.0890941431_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 16) = reshape([&
      &    0.0464446643_wp, -0.4644218072_wp, 0.5758906709_wp, 0.8265845025_wp, 0.0000000000_wp, &
      &    0.1360721060_wp, 1.0191214160_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.1721506776_wp, 0.2565103812_wp, 0.5141222973_wp, 0.4498981316_wp, 0.0000000000_wp, &
      &    0.7000574937_wp, 0.8244654819_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3019833288_wp, 1.0987929459_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 17) = reshape([&
      &    0.0176036558_wp, -0.3790806863_wp, 0.3740234991_wp, 0.7210033000_wp, 0.0000000000_wp, &
      &    0.1700892014_wp, 0.9451135771_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.1357624952_wp, 0.2234357537_wp, 0.7281511265_wp, 0.8190170579_wp, 0.0000000000_wp, &
      &    0.9907179187_wp, 0.6966079996_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.5886296117_wp, 1.9378519271_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 18) = reshape([&
      &    0.0115218612_wp, -0.5589680809_wp, 0.4733423415_wp, 0.6312932650_wp, 0.0000000000_wp, &
      &    0.0692493083_wp, 1.6030235429_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0772616665_wp, 0.4711817330_wp, 0.7510591023_wp, 0.2232011629_wp, 0.0000000000_wp, &
      &    1.4725647742_wp, 0.5811269410_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.6444613542_wp, 1.4014205832_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 19) = reshape([&
      &   -0.1381205112_wp, 0.1836477871_wp, 0.0545041514_wp, 0.0799930722_wp, 0.0000000000_wp, &
      &   -0.0136848845_wp, 0.5075696248_wp, 0.9125362054_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.7396727005_wp, 1.1042889800_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0384915374_wp, 0.6743323939_wp, 0.6125382359_wp, 0.1725209432_wp, 0.0000000000_wp, &
      &   -0.0070598231_wp, 0.0778063744_wp, 0.6885375338_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.5020401090_wp, 1.0552718784_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 20) = reshape([&
      &   -0.0789786448_wp, 0.1871158634_wp, 0.5895786417_wp, 0.3708042346_wp, 0.0000000000_wp, &
      &   -0.2336218176_wp, 0.1660110184_wp, 0.0358510557_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.0944269804_wp, 0.4311566050_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0217611052_wp, 0.3078191546_wp, 0.2692569853_wp, 0.0684067019_wp, 0.0000000000_wp, &
      &   -0.0093667786_wp, 0.5886351750_wp, 0.0286550189_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3330108773_wp, 0.7403398467_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 21) = reshape([&
      &    0.0792369965_wp, -0.4580786257_wp, 0.3364128767_wp, 0.3327363778_wp, 0.2114569153_wp, &
      &    0.4639782910_wp, 0.4351485772_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.4825159147_wp, 0.6393481117_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0935458737_wp, 0.3412380502_wp, 0.6011800094_wp, 0.2696034958_wp, 0.0000000000_wp, &
      &    0.9360371639_wp, 1.2883161603_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0486577834_wp, 0.2022250288_wp, 0.3967872318_wp, 0.4585126820_wp, 0.0000000000_wp, &
      &    0.7050414365_wp, 0.5783608933_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 22) = reshape([&
      &    0.0820167749_wp, -0.6504804409_wp, 0.5387065379_wp, 0.4504409532_wp, 0.1549958663_wp, &
      &    0.6277567947_wp, 0.4871645749_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.6637805272_wp, 0.3996890639_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0378736917_wp, 0.1149752808_wp, 0.2212333081_wp, 0.1175422001_wp, 0.0000000000_wp, &
      &    1.2698937471_wp, 0.5852465788_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0577173597_wp, 0.2533884287_wp, 0.5458626996_wp, 0.6746017999_wp, 0.0000000000_wp, &
      &    1.1662328419_wp, 0.6226421314_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 23) = reshape([&
      &    0.0940237090_wp, -0.5371870569_wp, 0.5202589266_wp, 0.1225207414_wp, 0.0727366742_wp, &
      &    1.0985503247_wp, 0.6073033618_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.5951895522_wp, 0.9485086968_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0812343184_wp, 0.2670042758_wp, 0.4859824582_wp, 0.2380173965_wp, 0.0000000000_wp, &
      &    0.8653192116_wp, 1.0015121242_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0479407090_wp, 0.2180683685_wp, 0.4830086248_wp, 0.5960055686_wp, 0.0000000000_wp, &
      &    0.8733220093_wp, 0.4124553680_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 24) = reshape([&
      &    0.1159287465_wp, -0.7899815893_wp, 0.5637073390_wp, 0.4451972680_wp, 0.0706486201_wp, &
      &    1.4304126937_wp, 0.8172005610_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.2783622365_wp, 0.9988381372_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0676268821_wp, 0.2484912484_wp, 0.4183092510_wp, 0.1906432380_wp, 0.0000000000_wp, &
      &    0.5728058175_wp, 0.8298461093_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0434092502_wp, 0.2085717515_wp, 0.4812803159_wp, 0.6001677294_wp, 0.0000000000_wp, &
      &    1.1487193065_wp, 0.6104281344_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 25) = reshape([&
      &    0.1034023772_wp, -0.6561416378_wp, 0.6657987881_wp, 0.3965224285_wp, 0.2920241547_wp, &
      &    1.5654970384_wp, 1.2013231888_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.4577418739_wp, 1.3576805482_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0975569992_wp, 0.3751691727_wp, 0.5972278516_wp, 0.2797263925_wp, 0.0000000000_wp, &
      &    0.6237905123_wp, 1.2508523096_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0470477265_wp, 0.2383587935_wp, 0.5795769580_wp, 0.7554726662_wp, 0.0000000000_wp, &
      &    1.1074566492_wp, 0.6259072837_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 26) = reshape([&
      &    0.0928715404_wp, -0.5431756879_wp, 0.7880081914_wp, 0.5061183399_wp, 0.1514679379_wp, &
      &    1.0480908785_wp, 0.7019447086_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1105846640_wp, 0.9601831144_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0874570715_wp, 0.3211949303_wp, 0.5259142305_wp, 0.2636909548_wp, 0.0000000000_wp, &
      &    0.8028906463_wp, 0.6149718893_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0357880628_wp, 0.1852568365_wp, 0.4555180344_wp, 0.6103556002_wp, 0.0000000000_wp, &
      &    1.5449660768_wp, 0.8104768500_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 27) = reshape([&
      &    0.0877668560_wp, -0.5506927166_wp, 0.7312085692_wp, 0.4355912312_wp, 0.2609723269_wp, &
      &    0.7437387523_wp, 0.9441499730_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1522561787_wp, 1.3227885140_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.1066877688_wp, 0.3792872112_wp, 0.6460704677_wp, 0.3210800228_wp, 0.0000000000_wp, &
      &    0.5167135519_wp, 0.6105448228_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0367389367_wp, 0.2064009869_wp, 0.5459647040_wp, 0.7723139516_wp, 0.0000000000_wp, &
      &    1.7648412581_wp, 0.9819594462_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 28) = reshape([&
      &    0.0950542040_wp, -0.6193523790_wp, 0.9030460676_wp, 0.5726699719_wp, 0.1079361121_wp, &
      &    0.5364718212_wp, 0.4664705278_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.0529351327_wp, 0.8522290712_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.1235722348_wp, 0.3730493985_wp, 0.6666186945_wp, 0.3343576923_wp, 0.0000000000_wp, &
      &    1.1097969005_wp, 1.0639861253_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0320162550_wp, 0.2065585243_wp, 0.5768807366_wp, 0.8536329780_wp, 0.0000000000_wp, &
      &    1.5770049833_wp, 0.9040572666_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 29) = reshape([&
      &    0.0881110025_wp, -0.5668646183_wp, 0.6906016130_wp, 0.5266429870_wp, 0.1322278424_wp, &
      &    0.9461947443_wp, 0.9618528159_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.2055055912_wp, 1.0402755812_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.1059459732_wp, 0.4254179136_wp, 0.6426327796_wp, 0.2923179217_wp, 0.0000000000_wp, &
      &    1.2928568214_wp, 1.0612055715_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0320263274_wp, 0.1869749174_wp, 0.5193877530_wp, 0.7624258197_wp, 0.0000000000_wp, &
      &    0.8778776635_wp, 0.5129917287_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 30) = reshape([&
      &    0.0712776169_wp, -0.5067587298_wp, 0.4933789987_wp, 0.3073121404_wp, 0.2491083979_wp, &
      &    1.1891265840_wp, 0.3404971497_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.0561668667_wp, 0.8755841026_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0727832544_wp, 0.3204870062_wp, 0.4649551234_wp, 0.2020724277_wp, 0.0000000000_wp, &
      &    0.2906807134_wp, 0.3408326927_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0388944758_wp, 0.2259367600_wp, 0.6251991173_wp, 0.9424303159_wp, 0.0000000000_wp, &
      &    1.3000390639_wp, 0.7254285781_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 31) = reshape([&
      &    0.0865966347_wp, -0.2705873627_wp, 0.3518193940_wp, 0.2854166868_wp, 0.0000000000_wp, &
      &    0.2979026501_wp, 0.8114592123_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.1423342474_wp, 0.1563009573_wp, 0.4142334733_wp, 0.1872025643_wp, 0.0000000000_wp, &
      &    0.7816802007_wp, 1.1399041023_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.6307242930_wp, 1.7353764401_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 32) = reshape([&
      &    0.2209570134_wp, -0.5116837537_wp, 0.8316223550_wp, 0.2050189587_wp, 0.0000000000_wp, &
      &    0.2427153162_wp, 1.5703504561_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0714488775_wp, -0.1960142538_wp, 0.7224779766_wp, 0.3198090816_wp, 0.0000000000_wp, &
      &    0.2193175962_wp, 1.9737428249_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.0712585034_wp, 1.0603605067_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 33) = reshape([&
      &    0.1580318540_wp, -0.4908057960_wp, 0.7295408932_wp, 0.3266735798_wp, 0.0000000000_wp, &
      &    0.1687923741_wp, 1.0587797108_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.1650610386_wp, 0.1541195684_wp, 0.2697527709_wp, 0.1745092282_wp, 0.0000000000_wp, &
      &    0.8786809977_wp, 1.1623428922_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.8315475391_wp, 1.0334238677_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 34) = reshape([&
      &    0.2314574450_wp, -0.5635338552_wp, 0.7298620738_wp, 0.3156571652_wp, 0.0000000000_wp, &
      &    0.1260251573_wp, 1.0692525790_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.1626054224_wp, 0.1834999183_wp, 0.3722353215_wp, 0.2719590036_wp, 0.0000000000_wp, &
      &    0.5308163057_wp, 1.5983911839_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.0609097352_wp, 0.8930145843_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 35) = reshape([&
      &    0.1656920285_wp, -0.4768771932_wp, 0.7102996020_wp, 0.3620930364_wp, 0.0000000000_wp, &
      &    0.1458258776_wp, 0.7680873587_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.2073568282_wp, 0.1861079692_wp, 0.3325243332_wp, 0.3037787211_wp, 0.0000000000_wp, &
      &    0.8041347342_wp, 1.3199195267_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1586727683_wp, 1.0213662278_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 36) = reshape([&
      &    0.2905599570_wp, -0.5161110299_wp, 0.5649715867_wp, 0.2723516949_wp, 0.0000000000_wp, &
      &    0.1930542103_wp, 0.8975463618_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.1785654239_wp, 0.1727154436_wp, 0.4007007159_wp, 0.3165281986_wp, 0.0000000000_wp, &
      &    0.9403571381_wp, 0.8469222248_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1744183286_wp, 0.9753028848_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 37) = reshape([&
      &    0.1223198886_wp, -0.6668468311_wp, 0.4180469061_wp, 0.5161218982_wp, 0.0000000000_wp, &
      &    0.4412289932_wp, -0.1801831911_wp, 0.9409888972_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1948232324_wp, 0.2407849081_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.1009914611_wp, 0.5231021123_wp, 0.5095861563_wp, 0.1052813569_wp, 0.0000000000_wp, &
      &   -0.0219673194_wp, 0.1921985632_wp, 0.2547788031_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.8149189457_wp, 1.1602724817_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 38) = reshape([&
      &   -0.3933417481_wp, 0.2084078101_wp, 0.4105604738_wp, 0.2235689400_wp, 0.0000000000_wp, &
      &    0.4032351936_wp, -0.0772447749_wp, 0.2640205158_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1734095339_wp, 1.2141147112_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0592109274_wp, 0.2635314830_wp, 0.2538391981_wp, 0.0547260188_wp, 0.0000000000_wp, &
      &   -0.0099179439_wp, 0.7314549213_wp, 0.2279162835_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.4909035377_wp, 0.7636626845_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 39) = reshape([&
      &    0.1953417898_wp, -1.8383591718_wp, 1.0239508989_wp, 0.6776125162_wp, 0.2868032538_wp, &
      &    0.5856652855_wp, 0.2920720521_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.4711397844_wp, 0.4340892859_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.8961099014_wp, 0.9209148832_wp, 0.7472498010_wp, 0.3589675782_wp, 0.0000000000_wp, &
      &    1.5876945998_wp, 1.1249891698_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0514103295_wp, 0.2596916097_wp, 0.6476701350_wp, 0.7227720322_wp, 0.0000000000_wp, &
      &    0.6355755341_wp, 0.2131988528_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 40) = reshape([&
      &    0.2255943878_wp, -2.0217709346_wp, 1.0439877615_wp, 0.6494815136_wp, 0.2719785428_wp, &
      &    0.6100983727_wp, 0.2793668017_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.5730000693_wp, 0.5911694787_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.8512036283_wp, 0.8921204690_wp, 0.7104777533_wp, 0.2925282135_wp, 0.0000000000_wp, &
      &    1.4385600094_wp, 0.9957253875_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0388924984_wp, 0.2756924682_wp, 0.6464365095_wp, 0.6665361134_wp, 0.0000000000_wp, &
      &    0.6499480044_wp, 0.1948098456_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 41) = reshape([&
      &    0.1738692146_wp, 0.4007164216_wp, -1.2924428831_wp, 0.4742868724_wp, 0.2729189857_wp, &
      &    0.6094566248_wp, 0.3099105290_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.2708480513_wp, 0.5911896311_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.8732174739_wp, 0.9405434823_wp, 0.7602372584_wp, 0.2695564855_wp, 0.0000000000_wp, &
      &    1.4016876619_wp, 1.0105826966_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0516726059_wp, 0.2845162518_wp, 0.6642329336_wp, 0.6588253218_wp, 0.0000000000_wp, &
      &    0.6969667497_wp, 0.1966241476_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 42) = reshape([&
      &    0.2936395398_wp, -1.8576148711_wp, 1.2248641285_wp, 0.6604561062_wp, 0.2961533522_wp, &
      &    0.6036324144_wp, 0.3018713060_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.3170877837_wp, 0.5807431794_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.8266315775_wp, 0.9006748608_wp, 0.7118285177_wp, 0.2694899293_wp, 0.0000000000_wp, &
      &    1.3031295892_wp, 1.2628816558_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0560474220_wp, 0.2604820789_wp, 0.6458917785_wp, 0.6807581707_wp, 0.0000000000_wp, &
      &    0.9341675525_wp, 0.2259628556_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 43) = reshape([&
      &    0.1918291416_wp, -2.1057751165_wp, 1.5184666476_wp, 0.7322926856_wp, 0.2741396334_wp, &
      &    0.6016133320_wp, 0.2951450099_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.3012565358_wp, 0.5378236539_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.8549829967_wp, 0.9593190096_wp, 0.8288466152_wp, 0.2551445261_wp, 0.0000000000_wp, &
      &    1.1014669878_wp, 1.9371154846_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0546676799_wp, 0.2641824101_wp, 0.6238310014_wp, 0.6633616508_wp, 0.0000000000_wp, &
      &    1.0298023557_wp, 0.2613935603_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 44) = reshape([&
      &    0.1763689959_wp, -1.9813118301_wp, 1.4137707813_wp, 0.6572540909_wp, 0.2523933741_wp, &
      &    0.5667616487_wp, 0.3028115876_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.3348938177_wp, 0.6059671674_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.8594444710_wp, 0.9555262318_wp, 0.8939402453_wp, 0.3345640182_wp, 0.0000000000_wp, &
      &    1.5667491852_wp, 0.9135252413_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0578467610_wp, 0.2536384787_wp, 0.5834908784_wp, 0.5895921304_wp, 0.0000000000_wp, &
      &    1.1162962972_wp, 0.2652571062_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 45) = reshape([&
      &    0.1685031025_wp, -1.8473388366_wp, 1.3110673502_wp, 0.6979938683_wp, 0.1858655312_wp, &
      &    0.6309537031_wp, 0.2631659342_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.8950583090_wp, 0.4577141319_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -1.1554459315_wp, 1.2030306721_wp, 0.7920749164_wp, 0.3619559700_wp, 0.0000000000_wp, &
      &    1.4677765457_wp, 1.8886998340_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0484093002_wp, 0.2670355302_wp, 0.6940205834_wp, 0.7929618904_wp, 0.0000000000_wp, &
      &    1.1661518494_wp, 0.2649375312_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 46) = reshape([&
      &    0.1640975771_wp, -1.8667225025_wp, 1.3597368627_wp, 0.5655300462_wp, 0.1887728876_wp, &
      &    0.9446526249_wp, 0.2043510493_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.9544372199_wp, 0.4712085851_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -1.1773773405_wp, 1.2887357387_wp, 1.0606857029_wp, 0.4099538707_wp, 0.0000000000_wp, &
      &    1.7757171210_wp, 1.1739355611_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0456261495_wp, 0.3118394110_wp, 0.7180555574_wp, 0.7097393456_wp, 0.0000000000_wp, &
      &    0.7615446376_wp, 0.2526508019_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 47) = reshape([&
      &    0.1701133941_wp, -1.8877354596_wp, 1.4343568127_wp, 0.8685286127_wp, 0.1921226014_wp, &
      &    0.9812396645_wp, 0.3012214403_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.0539939791_wp, 0.4115219835_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -1.2029333559_wp, 1.3132965888_wp, 1.0363129102_wp, 0.3832942704_wp, 0.0000000000_wp, &
      &    0.9737035601_wp, 0.9129104875_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0456858488_wp, 0.3343212176_wp, 0.7590371455_wp, 0.7296687434_wp, 0.0000000000_wp, &
      &    1.4580044533_wp, 0.2691010107_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 48) = reshape([&
      &    0.1693399663_wp, -1.8985434431_wp, 1.4508537949_wp, 0.9335239845_wp, 0.1939684640_wp, &
      &    1.2728082541_wp, 0.2291426379_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.6967542112_wp, 0.3589609059_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -1.0769106875_wp, 1.1517797297_wp, 0.7214321378_wp, 0.2787900973_wp, 0.0000000000_wp, &
      &    0.8049884200_wp, 0.5981933460_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0252224128_wp, 0.2118852973_wp, 0.4475423505_wp, 0.4186681660_wp, 0.0000000000_wp, &
      &    1.1164488479_wp, 0.1753271368_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 49) = reshape([&
      &    0.1952163696_wp, -0.3903359184_wp, 0.3320665124_wp, 0.4819122045_wp, 0.0000000000_wp, &
      &    0.0519287745_wp, 0.8019063172_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0613449224_wp, -0.1931571298_wp, 0.4583900966_wp, 0.1941581309_wp, 0.0000000000_wp, &
      &    0.1046317339_wp, 0.8684932449_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.5733956371_wp, 1.0162198062_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 50) = reshape([&
      &    0.1495271971_wp, -0.5027813324_wp, 0.5152421592_wp, 0.3836208295_wp, 0.0000000000_wp, &
      &    0.1573531454_wp, 0.9824057623_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0550512009_wp, -0.1697387699_wp, 0.4931832107_wp, 0.3270337863_wp, 0.0000000000_wp, &
      &    0.2256068770_wp, 2.0653004641_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1010903100_wp, 1.4744657354_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 51) = reshape([&
      &    0.3723909867_wp, -0.5898216645_wp, 0.2380205250_wp, 0.2826020745_wp, 0.0000000000_wp, &
      &    0.1267017940_wp, 1.0466208005_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0511225842_wp, -0.1610757793_wp, 0.4057651527_wp, 0.1864865330_wp, 0.0000000000_wp, &
      &    0.1402915697_wp, 1.9269343285_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.9234488314_wp, 1.0840752768_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 52) = reshape([&
      &    0.8022855784_wp, -1.1474320023_wp, 0.7093727995_wp, 0.1269650847_wp, 0.0000000000_wp, &
      &    0.1906003747_wp, 0.8367042242_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0739016921_wp, -0.2387836277_wp, 0.3404555009_wp, 0.2479512998_wp, 0.0000000000_wp, &
      &    0.9196767161_wp, 0.7975712645_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.5713976557_wp, 0.9697775092_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 53) = reshape([&
      &    0.2701910394_wp, -0.5148535819_wp, 0.4460319745_wp, 0.1939932914_wp, 0.0000000000_wp, &
      &    0.2140603254_wp, 0.8533261137_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.2678904497_wp, 0.2943071949_wp, 0.2460219797_wp, 0.1387733778_wp, 0.0000000000_wp, &
      &    0.0631058497_wp, 2.0300565207_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1596268804_wp, 1.0649107315_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 54) = reshape([&
      &    0.2498969270_wp, -0.5376316607_wp, 0.4891017389_wp, 0.2680060377_wp, 0.0000000000_wp, &
      &    0.1933227003_wp, 0.8878592718_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.2763670794_wp, 0.3231787560_wp, 0.2983586288_wp, 0.2268852294_wp, 0.0000000000_wp, &
      &    0.0673367659_wp, 1.8302437814_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1613785962_wp, 1.0437448357_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 55) = reshape([&
      &    0.0939448971_wp, -0.5748768519_wp, 0.3802933507_wp, 0.2755391259_wp, 0.0000000000_wp, &
      &   -0.2468820083_wp, 0.3524935370_wp, 1.0138844204_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1429481916_wp, 0.2899551775_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.1582052704_wp, 0.4114808131_wp, 0.4142625583_wp, 0.0723021758_wp, 0.0000000000_wp, &
      &   -0.0973555233_wp, 0.0700642976_wp, 0.6602215762_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1304177083_wp, 0.9566975471_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 56) = reshape([&
      &    0.1721269727_wp, -0.4124985207_wp, 0.4034337925_wp, 0.1539359150_wp, 0.0000000000_wp, &
      &    0.3321739327_wp, -0.0935987128_wp, 0.2758294758_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3482492066_wp, 1.0492229933_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0907133522_wp, 0.1860653469_wp, 0.0533459173_wp, 0.1240388865_wp, 0.0000000000_wp, &
      &   -0.0092669017_wp, 0.5200291537_wp, 0.2947536541_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.7407473001_wp, 0.5440855478_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 57) = reshape([&
      &    0.3444365958_wp, -0.6246222761_wp, 0.5151272377_wp, 0.2462184580_wp, 0.0000000000_wp, &
      &    0.9377095154_wp, 1.1643673314_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1059957769_wp, 0.4649671119_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1767690945_wp, -0.3327136467_wp, 0.5267575193_wp, 0.3366107677_wp, 0.0000000000_wp, &
      &    1.0440226728_wp, 0.0673539017_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0916224635_wp, 0.1791963444_wp, 0.4854274676_wp, 0.4631670320_wp, 0.0000000000_wp, &
      &    0.8666161087_wp, 0.4368730724_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 72) = reshape([&
      &    0.2763985054_wp, -0.8398984895_wp, 0.4742404310_wp, 0.1587779188_wp, 0.0000000000_wp, &
      &    1.7074689779_wp, 0.5191376022_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.9589105718_wp, 0.4085381413_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1343747279_wp, -0.3046128778_wp, 0.6078625667_wp, 0.4339349354_wp, 0.0000000000_wp, &
      &    1.3713066772_wp, 0.8095630077_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0677895496_wp, 0.2830644941_wp, 0.6195080240_wp, 0.3907105912_wp, 0.0000000000_wp, &
      &    0.7947192693_wp, 0.3422856891_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 73) = reshape([&
      &    0.2793640731_wp, -0.8502608092_wp, 0.4795688403_wp, 0.1933941874_wp, 0.0000000000_wp, &
      &    1.6882468484_wp, 0.3791336605_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.5952635292_wp, 0.5333830994_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1366252069_wp, -0.3170910501_wp, 0.5874632106_wp, 0.4359889813_wp, 0.0000000000_wp, &
      &    1.3481178576_wp, 0.7798292868_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0579007685_wp, 0.2258945458_wp, 0.3792190308_wp, 0.5083819390_wp, 0.0000000000_wp, &
      &    0.8129113122_wp, 0.3323995803_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 74) = reshape([&
      &    0.2770253408_wp, -0.8447303220_wp, 0.4831422449_wp, 0.1580065990_wp, 0.0000000000_wp, &
      &    1.6449098834_wp, 0.4095688280_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.8695898151_wp, 0.5289299321_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1573542119_wp, -0.3528549526_wp, 0.6124209349_wp, 0.4467229784_wp, 0.0000000000_wp, &
      &    0.9648750399_wp, 0.2465864559_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0548992156_wp, 0.2169978922_wp, 0.4329824144_wp, 0.4231238259_wp, 0.0000000000_wp, &
      &    1.0902662907_wp, 0.4787890583_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 75) = reshape([&
      &    0.2577362154_wp, -0.8228756201_wp, 0.4790670328_wp, 0.2031472409_wp, 0.0000000000_wp, &
      &    1.8817641031_wp, 0.4293411418_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    2.2553222264_wp, 1.2595317684_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.4969273598_wp, -0.6768091953_wp, 0.5723646899_wp, 0.3963404989_wp, 0.0000000000_wp, &
      &    4.1653415457_wp, 0.9086392715_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0754169646_wp, 0.2897406928_wp, 0.5116744995_wp, 0.5430267468_wp, 0.0000000000_wp, &
      &    0.4191302347_wp, 0.1248313571_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 76) = reshape([&
      &    0.2623240149_wp, -0.8241515428_wp, 0.4882368833_wp, 0.1489592899_wp, 0.0000000000_wp, &
      &    1.7042764041_wp, 0.4199393658_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1904161707_wp, 0.4150984079_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3669919807_wp, -0.5725124600_wp, 0.6343997445_wp, 0.4372961237_wp, 0.0000000000_wp, &
      &    2.8468635307_wp, 0.3494642070_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0758453991_wp, 0.2620124183_wp, 0.4892530510_wp, 0.4806070510_wp, 0.0000000000_wp, &
      &    1.9465277518_wp, 0.2913417329_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 77) = reshape([&
      &    0.2385659413_wp, -0.8244257419_wp, 0.5076133569_wp, 0.1191573877_wp, 0.0000000000_wp, &
      &    1.4574272861_wp, 0.2815435129_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1713913625_wp, 0.5219015664_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.2431193228_wp, -0.4445443606_wp, 0.6456552486_wp, 0.4027046304_wp, 0.0000000000_wp, &
      &    0.9832022698_wp, 0.5357084022_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0810602852_wp, 0.2745318060_wp, 0.5093129620_wp, 0.4219073402_wp, 0.0000000000_wp, &
      &    1.5420321459_wp, 0.2153559523_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 78) = reshape([&
      &    0.2308779087_wp, -0.8616547828_wp, 0.5419045574_wp, 0.4725531253_wp, 0.0000000000_wp, &
      &    1.5438163332_wp, 0.2761753206_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.2119339711_wp, 0.5479717380_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.7128480396_wp, -0.9404143677_wp, 0.7103097767_wp, 0.4569199799_wp, 0.0000000000_wp, &
      &    2.2925001443_wp, 0.2242100894_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0796251494_wp, 0.2678131364_wp, 0.4925357570_wp, 0.4150365016_wp, 0.0000000000_wp, &
      &    2.0060600385_wp, 0.2620756083_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 79) = reshape([&
      &    0.2290370072_wp, -0.9074052091_wp, 0.5327294630_wp, 0.2887194121_wp, 0.0000000000_wp, &
      &    1.1027692844_wp, 0.3439120221_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.0993671534_wp, 0.5226749192_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.5094751380_wp, -0.7210747764_wp, 0.6983880674_wp, 0.4090740976_wp, 0.0000000000_wp, &
      &    2.7302732725_wp, 1.1228951186_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0769406844_wp, 0.2661484888_wp, 0.4799125212_wp, 0.3774713685_wp, 0.0000000000_wp, &
      &    2.4621111810_wp, 0.2507382490_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 80) = reshape([&
      &    0.1812507711_wp, -0.8693504440_wp, 0.4863761813_wp, 0.2589833052_wp, 0.0000000000_wp, &
      &    1.7221063944_wp, 0.2504557267_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.1003349417_wp, 0.6778239803_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.2180673210_wp, -0.4202807631_wp, 0.6693056584_wp, 0.3867819255_wp, 0.0000000000_wp, &
      &    2.4664336639_wp, 1.5617596305_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &   -0.0785741529_wp, 0.2435599452_wp, 0.4511518005_wp, 0.3459238274_wp, 0.0000000000_wp, &
      &    2.3007794128_wp, 0.2106858638_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 81) = reshape([&
      &    0.3136626350_wp, -0.7227461451_wp, 0.6342492651_wp, 0.6164473385_wp, 0.0000000000_wp, &
      &    0.1447818682_wp, 0.8900593377_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0909783953_wp, -0.2595438069_wp, 0.3621595563_wp, 0.3653157094_wp, 0.0000000000_wp, &
      &    0.0188480498_wp, 1.4161039445_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.6041782457_wp, 0.8860138258_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 82) = reshape([&
      &    0.7755786109_wp, -1.2027653089_wp, 0.7237056706_wp, 0.2567612681_wp, 0.0000000000_wp, &
      &    0.1318863871_wp, 1.5104859876_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1527501041_wp, -0.2888771254_wp, 0.4372199728_wp, 0.1185222738_wp, 0.0000000000_wp, &
      &    0.2017682399_wp, 3.7504888202_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.8350233723_wp, 1.0954056050_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 83) = reshape([&
      &    0.0905053831_wp, -0.5055560649_wp, 0.7337168566_wp, 0.1120260819_wp, 0.0000000000_wp, &
      &    0.2095105736_wp, 0.5972485894_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.3959538106_wp, -0.5659318484_wp, 0.6385590224_wp, 0.1417978736_wp, 0.0000000000_wp, &
      &    0.1168066246_wp, 3.0860071180_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    1.0094389994_wp, 0.7399342711_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 84) = reshape([&
      &    0.3988020993_wp, -0.8198104390_wp, 0.5302666706_wp, 0.1834329643_wp, 0.0000000000_wp, &
      &    0.2245164118_wp, 0.8187649588_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1554993959_wp, -0.3991222115_wp, 0.5477830199_wp, 0.2071803261_wp, 0.0000000000_wp, &
      &    0.7788540607_wp, 1.0663979232_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.9603002617_wp, 0.7863182294_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 85) = reshape([&
      &    0.5258566612_wp, -1.0677206211_wp, 0.6387237748_wp, 0.5029538178_wp, 0.0000000000_wp, &
      &    0.0869263896_wp, 0.9435032626_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0625746766_wp, -0.2584552762_wp, 0.4582578093_wp, 0.5156832729_wp, 0.0000000000_wp, &
      &    0.1047040373_wp, 0.9292841182_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.1741097938_wp, 1.0489783689_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

      coefficients(:, :, 86) = reshape([&
      &    0.6587585031_wp, -1.2040129364_wp, 0.9767723434_wp, 0.2152472334_wp, 0.0000000000_wp, &
      &    0.0785569744_wp, 1.1522387982_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.2553787677_wp, -0.3776812501_wp, 0.5314758067_wp, 0.2850723186_wp, 0.0000000000_wp, &
      &    0.1017994703_wp, 0.8819072086_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.2457188879_wp, 1.2198972012_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, &
      &    0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp, 0.0000000000_wp], (/max_prim, max_shell/))

   end subroutine setCGTOcoefficients

#endif
end module xtb_ptb_vdzp
