/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/test_run_result.hpp
 *
 * @brief [LEVEL: beta] @ref diagnostics::unittest::Test_Run_Result
 * class
 *
 * $Id: test_run_result.hpp,v 1.20 2005/06/23 09:54:27 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @test diagnostics/unittest/test_system/test_run_result.t.cpp
 */

#ifndef DIAGNOSTICS__UNITTEST__TEST_SYSTEM__TEST_RUN_RESULT_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_SYSTEM__TEST_RUN_RESULT_HPP__INCLUDE_GUARD

#include <diagnostics/unittest/namespace.hpp>

// used in the interface by reference
// used in the implementation by value
#include <diagnostics/frame/record.hpp>

// used in the interface by reference
#include <string>

// used in the interface by reference
// used in the implementation by value
#include <vector>

// used in the interface by reference
#include <iostream>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

/**
 * @class Test_Run_Result diagnostics/unittest/test_system/test_run_result.hpp
 *
 * @brief a Test_Run_Result represents the data on a single run of a
 * single Test_Case generated by a @ref Run_Test_Suite_Traversal.
 * This class embodies the testing policy, i.e., which execution of a
 * @ref Test_Case is considered successful, failed or invalid.
 *
 * @nosubgrouping
 */
class Test_Run_Result
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Types
     * @{
     */
private:
    typedef Test_Run_Result Self;

public:
    /**
     * @brief the state of the result...
     */
    typedef enum {
	/**
	 * @brief no records are there, i.e., records().size()==0
	 */
	STATE_EMPTY,
	/**
	 * @brief records are there -- but the TYPE_TESTCASE_EXIT is missing
	 */
	STATE_INCOMPLETE,
	/**
	 * @brief a complete execution execution trace is stored in this
	 */
	STATE_COMPLETE
    } Abstract_state_t;


    typedef ::std::vector<Record> Records_t;

    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Disabled Creation
     * @{
     */
private:
    Test_Run_Result();
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Creation
     * @{
     */
public:
    /**
     * @brief Cstr starting with @ref STATE_EMPTY
     *
     * @throw Test_System_Exception if the precondition is not met
     *
     * @pre @a build_level != @ref LEVEL_TEST 
     * @pre @a target_level != @ref LEVEL_TEST 
     * @pre @a build_level != @ref LEVEL_SYSTEM 
     * @pre @a target_level != @ref LEVEL_SYSTEM
     * @pre @a target_level <= build_level
     *
     * @post build_level() == @a build_level
     * @post target_level() == @a target_level
     * @post abstract_state() == @ref STATE_EMPTY
     */
    Test_Run_Result(Level_t const build_level,
		    Level_t const target_level);


    /**
     * @brief to build a Test_Run_Result from scratch
     * 
     * @throw Test_System_Exception if the preconditions are not met
     *
     * @pre @a build_level != @ref LEVEL_TEST 
     * @pre @a target_level != @ref LEVEL_TEST
     * @pre @a build_level != @ref LEVEL_SYSTEM 
     * @pre @a target_level != @ref LEVEL_SYSTEM
     * @pre @a target_level <= build_level
     * @pre @a records is valid the sense that copying record by
     * record from the beginning to end with add_record succeeds.
     *
     * @post build_level() == @a build_level
     * @post target_level() == @a target_level
     * @post records() == @a records
     * @post all other fields are set according to records(). 
     */
    Test_Run_Result(Level_t const build_level,
		    Level_t const target_level,
		    Records_t const & records);
    
    /**
     * @throw never
     */
    Test_Run_Result(Self const & other);
    /**
     * @throw never
     */
    Self & operator=(Self const & other);
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Comparision
     * @{
     */
public:
    /**
     * @throw never
     */
    bool operator==(Self const & r) const;

    /**
     * @throw never
     */
    inline bool operator!=(Self const & r) const { return !operator==(r); }
    // @}


    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Accessors
     * @{
     */
public:
    /**
     * @brief the build level of the executed @ref Test_Case
     *
     * @throw never
     */
    inline Level_t build_level() const { return m_build_level; }

    /**
     * @brief the target level of the executed @ref Test_Case
     *
     * @throw never
     */
    inline Level_t target_level() const { return m_target_level; }

    /**
     * @brief whether there occured invalidations, i.e., whether it
     * could be executed in a conclusive way yielding a failure or
     * success (e.g., an execution of a @ref Test_Case is invalid, if
     * it could access files it would have required).
     *
     * @throw never
     */
    inline int invalidation_count() const { return m_invalidation_count; }

    /**
     * @brief the number of failures which occured during the
     * execution of the @ref Test_Case. 
     * 
     * @throw never
     */
    inline int failure_count() const { return m_failure_count; }    

    /**
     * @brief returns the current abstract_state
     *
     * @throw never
     */
    inline Abstract_state_t abstract_state() const { return m_abstract_state; }

    /**
     * @brief the records which were collected during the execution of
     * the @ref Test_Case.
     *
     * @throw never
     */
    inline Records_t const & records() const { return m_records; }


#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    /**
     * @brief the runtime of the execution (seconds) -- only defined
     * if abstract_state() == @ref STATE_COMPLETE
     *
     * @throw never
     */
    inline Sec_t sec() const { return m_sec; }

    /**
     * @brief the runtime of the execution (microseconds) -- only
     * defined if abstract_state() == @ref STATE_COMPLETE
     *
     * @throw never
     */
    inline Usec_t usec() const { return m_usec; }
#endif
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Modifiers
     * @{
     */
public:
    /**
     * @brief adds a @ref Record to the sequence of @ref records(). 
     *
     * @throw Test_System_Exception if the new record is not a valid
     * continuation of the @ref records(), i.e., if one of the
     * preconditions is not met.
     *
     * @attention records of @ref TYPE_LOG_OPEN, @ref TYPE_LOG_CLOSE
     * and of @ref LEVEL_SYSTEM are completely ignored!
     *
     * For all OTHER types, the following rules apply:
     *
     * @pre abstract_state () == @ref STATE_EMPTY <--> @a record.type() == @ref TYPE_TESTCASE_ENTER
     * @pre @a record.type() != @ref TYPE_TESTCASE_ENTER --> abstract_state() == @ref STATE_INCOMPLETE
     *
     * @post PRE.records().push_back(@a record) == POST.records()
     * @post @a record.type() == @ref TYPE_TESTCASE_ENTER --> POST.abstract_state() == @ref STATE_INCOMPLETE
     * @post @a record.type() == @ref TYPE_TESTCASE_EXIT  --> POST.abstract_state() == @ref STATE_COMPLETE
     * @post @a record.type() == @ref TYPE_FAILED_ASSERTION --> PRE.failure_count()+1 == POST.failure_count()
     * @post @a record.type() == @ref TYPE_UNEXPECTED_EXCEPTION --> PRE.failure_count()+1 == POST.failure_count()
     * @post @a record.type() == @ref TYPE_FAILED_CHECK && @a record.level() == @ref LEVEL_TEST --> 
     *           PRE.invalidation_count()+1 == POST.invalidation_count()
     * @post @a record.type() == @ref TYPE_WRONG_EXCEPTION && @a record.level() == @ref LEVEL_TEST --> 
     *           PRE.failure_count()+1 == POST.failure_count()
     * @post @a record.type() == @ref TYPE_WRONG_EXCEPTION && @a record.level() == @ref LEVEL_TEST --> 
     *           PRE.failure_count()+1 == POST.failure_count()
     * @post @a record.type() == @ref TYPE_FAILED_CHECK && @a record.level() > target_level() 
     *                                                  && @a record.level() != @ref LEVEL_TEST --> 
     *           PRE.failure_count()+1 == POST.failure_count()
     */
    void add_record(Record const & record);
    // @}


    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Encapsulated State
     * @{
     */
private:
    Abstract_state_t m_abstract_state;
    int m_failure_count;
    int m_invalidation_count;
    
    Level_t m_build_level;
    Level_t m_target_level;
    Records_t m_records;
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    Sec_t m_sec;
    Usec_t m_usec;
#endif
    // @}
};

UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

/**
 * @brief prints a @ref diagnostics::unittest::Test_Run_Result into an stream (multiline
 * output, tailing newline added)
 */
::std::ostream & operator<<(::std::ostream & stream, ::diagnostics::unittest::Test_Run_Result const & r);

#endif
// vim:ts=4:sw=4
